/* msm-dri2.c
 *
 * Copyright (c) 2009, Code Aurora Forum. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *     * Neither the name of Code Aurora nor
 *       the names of its contributors may be used to endorse or promote
 *       products derived from this software without specific prior written
 *       permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NON-INFRINGEMENT ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <sys/ioctl.h>
#include <sys/mman.h>

#include "msm.h"
#include "xf86drm.h"
#include "dri2.h"

#include "msm-drm.h"

#if DRI2INFOREC_VERSION >= 2
#define USE_DRI2_2
#endif

struct msm_dri2_priv {
    PixmapPtr pixmap;
};

#ifndef USE_DRI2_2

static DRI2BufferPtr MSMDRI2CreateBuffers(DrawablePtr drawable,
					  unsigned int *attachments,
					  int count)
{
    ScreenPtr pScreen = drawable->pScreen;
    DRI2BufferPtr buffers;
    struct msm_dri2_priv *privates;
    struct msm_pixmap_priv *pixpriv;
    PixmapPtr pixmap;
    int i;

    buffers = xcalloc(count, sizeof(*buffers));
    if (buffers == NULL)
	return NULL;

    privates = xcalloc(count, sizeof(struct msm_dri2_priv));

    if (privates == NULL) {
	xfree(buffers);
	return NULL;
    }


    for(i = 0; i < count; i++) {


	if (attachments[i] == DRI2BufferFrontLeft ||
	    attachments[i] == DRI2BufferFakeFrontLeft) {

		if (drawable->type == DRAWABLE_PIXMAP)
			pixmap = (PixmapPtr) drawable;
		else
			pixmap = pScreen->GetWindowPixmap((WindowPtr) drawable);

	    pixmap->refcnt++;
	}
	else {

	    pixmap = (*pScreen->CreatePixmap)(pScreen,
					      drawable->width,
					      drawable->height,
					      drawable->depth,
					      0);
	}

	if (pixmap == NULL)
	    return NULL;

	pixpriv = exaGetPixmapDriverPrivate(pixmap);

	if (pixpriv)
	    msm_drm_bo_flink(pixpriv->bo, &buffers[i].name);

	buffers[i].attachment = attachments[i];
	buffers[i].pitch = pixmap->devKind;
	buffers[i].cpp = pixmap->drawable.bitsPerPixel / 8;
	buffers[i].attachment = attachments[i];
	buffers[i].driverPrivate = &privates[i];
	buffers[i].flags = 0;

	privates[i].pixmap = pixmap;
    }

    return buffers;
}
#else
static DRI2Buffer2Ptr
MSMDRI2CreateBuffer(DrawablePtr drawable, unsigned int attachment,
		    unsigned int format)
{
    DRI2Buffer2Ptr buffer;
    struct msm_dri2_priv *private;
    ScreenPtr pScreen = drawable->pScreen;
    struct msm_pixmap_priv *pixpriv;
    PixmapPtr pixmap;

    buffer = xcalloc(1, sizeof(*buffer));
    if (buffer == NULL)
	return NULL;

    private = xcalloc(1, sizeof(struct msm_dri2_priv));

    if (private == NULL) {
	xfree(buffer);
	return NULL;
    }

    if (attachment == DRI2BufferFrontLeft ||
	attachment == DRI2BufferFakeFrontLeft) {
	if (drawable->type == DRAWABLE_PIXMAP)
	    pixmap = (PixmapPtr) drawable;
	else
	    pixmap = pScreen->GetWindowPixmap((WindowPtr) drawable);
	pixmap->refcnt++;
    }
    else {
	/* BackLeft and BackRight */
	pixmap = (*pScreen->CreatePixmap)(pScreen,
					  drawable->width,
					  drawable->height,
					  (format != 0) ? format :
					  drawable->depth,
					  0);

	if (pixmap == NULL)
	    return NULL;

	/* To begin with, put the back buffers into EBI memory -
	 * eventually this needs to shift to KMEM */

        pixpriv = exaGetPixmapDriverPrivate(pixmap);

	if (pixpriv)
	    msm_drm_bo_set_memtype(pixpriv->bo, MSM_DRM_MEMTYPE_EBI);
    }

    pixpriv = exaGetPixmapDriverPrivate(pixmap);

    if (pixpriv)
	msm_drm_bo_flink(pixpriv->bo, &buffer->name);

    buffer->pitch = pixmap->devKind;
    buffer->cpp = pixmap->drawable.bitsPerPixel / 8;
    buffer->attachment = attachment;
    buffer->driverPrivate = private;
    buffer->format = format;
    buffer->flags = 0;

    private->pixmap = pixmap;

    return buffer;
}
#endif

#ifndef USE_DRI2_2
static void
MSMDRI2DestroyBuffers(DrawablePtr drawable,
		      DRI2BufferPtr buffers,
		      int count)
{
    ScreenPtr pScreen = drawable->pScreen;
    struct msm_dri2_priv *priv;
    int i;

    if (!buffers || !count)
	return;

    for(i = 0; i < count; i++) {
	priv = buffers[i].driverPrivate;
	(*pScreen->DestroyPixmap)(priv->pixmap);
    }

    xfree(buffers[0].driverPrivate);
    xfree(buffers);
}
#else
static void
MSMDRI2DestroyBuffer(DrawablePtr drawable, DRI2Buffer2Ptr buffers)
{
    ScreenPtr pScreen = drawable->pScreen;
    struct msm_dri2_priv *priv;

    if (buffers == NULL)
	return;

    priv = buffers->driverPrivate;
    (*pScreen->DestroyPixmap)(priv->pixmap);

    xfree(buffers->driverPrivate);
    xfree(buffers);
}
#endif

static void
MSMDRI2CopyRegion(DrawablePtr pDraw, RegionPtr pRegion,
		  DRI2BufferPtr pDstBuffer, DRI2BufferPtr pSrcBuffer)
{
    ScreenPtr pScreen = pDraw->pScreen;
    struct msm_dri2_priv *srcpriv, *dstpriv;
    PixmapPtr srcPix, dstPix;
    RegionPtr copyRegion;

    GCPtr gc;

    srcpriv = pSrcBuffer->driverPrivate;
    dstpriv = pDstBuffer->driverPrivate;

    if (srcpriv == NULL || dstpriv == NULL)
	return;

    srcPix = srcpriv->pixmap;
    dstPix = dstpriv->pixmap;

    if (pSrcBuffer->attachment == DRI2BufferFakeFrontLeft ||
        pDstBuffer->attachment == DRI2BufferFakeFrontLeft)
	return;

    if (pSrcBuffer->attachment == DRI2BufferFrontLeft)
	srcPix = (PixmapPtr) pDraw;

    if (pDstBuffer->attachment == DRI2BufferFrontLeft)
	dstPix = (PixmapPtr) pDraw;

    gc = GetScratchGC(pDraw->depth, pScreen);
    copyRegion = REGION_CREATE(pScreen, NULL, 0);
    REGION_COPY(pScreen, copyRegion, pRegion);

    (*gc->funcs->ChangeClip)(gc, CT_REGION, copyRegion, 0);
    ValidateGC(&dstPix->drawable, gc);

    (*gc->ops->CopyArea)(&srcPix->drawable, &dstPix->drawable, gc,
			 0, 0, pDraw->width, pDraw->height, 0, 0);

    FreeScratchGC(gc);
}

Bool
MSMDRI2ScreenInit(ScreenPtr pScreen)
{
    DRI2InfoRec info;
    ScrnInfoPtr pScrn = xf86Screens[pScreen->myNum];
    MSMPtr pMsm = MSMPTR(pScrn);

    if (pMsm->drmFD <= 0) {
	ErrorF("DRI2: DRM is not initialized\n");
	return FALSE;
    }

    info.driverName = "yamato";
    info.deviceName = pMsm->drmDevName;
    info.fd = pMsm->drmFD;

#ifndef USE_DRI2_2
    info.version = 1;
    info.CreateBuffers = MSMDRI2CreateBuffers;
    info.DestroyBuffers = MSMDRI2DestroyBuffers;
#else
    info.version = DRI2INFOREC_VERSION;
    info.CreateBuffer = MSMDRI2CreateBuffer;
    info.DestroyBuffer = MSMDRI2DestroyBuffer;
#endif

    info.CopyRegion = MSMDRI2CopyRegion;

    return DRI2ScreenInit(pScreen, &info);
}

void
MSMDRI2ScreenClose(ScreenPtr pScreen)
{
    DRI2CloseScreen(pScreen);
}
