/**
 * geis_xcb.c
 *
 * Copyright 2010 Canonical Ltd.
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option) any
 * later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.	 See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */
#include "geis_xcb.h"

#include "geis_instance.h"
#include "geis_instance_table.h"
#include "geis_logging.h"
#include <grail.h>
#include <grail-types.h>
#include <stdlib.h>
#include <string.h>
#include <X11/Xlib-xcb.h>
#include <X11/X.h>
#include "xcb_gesture.h"
#include <xcb/xcb.h>

struct _GeisXcb
{
  GeisXcbWinInfo   *win_info;
  Display          *display;
  xcb_connection_t *connection;
  int               fd;
  GeisInstanceTable instance_table;
  GeisInputFuncs   *input_funcs;
  void             *input_cookie;
};

typedef struct _GrailAttrMap
{
  const char *name;
  int         attr;
} GrailAttrMap;

static const GrailAttrMap s_grail_drag_attrs[] =
{
  { GEIS_GESTURE_ATTRIBUTE_DELTA_X,         GRAIL_PROP_DRAG_DX }, 
  { GEIS_GESTURE_ATTRIBUTE_DELTA_Y,         GRAIL_PROP_DRAG_DY },
  { GEIS_GESTURE_ATTRIBUTE_VELOCITY_X,      GRAIL_PROP_DRAG_VX },
  { GEIS_GESTURE_ATTRIBUTE_VELOCITY_Y,      GRAIL_PROP_DRAG_VY },
  { GEIS_GESTURE_ATTRIBUTE_POSITION_X,      GRAIL_PROP_DRAG_X },
  { GEIS_GESTURE_ATTRIBUTE_POSITION_Y,      GRAIL_PROP_DRAG_Y },
  { GEIS_GESTURE_ATTRIBUTE_BOUNDINGBOX_X1,  GRAIL_PROP_DRAG_X1 },
  { GEIS_GESTURE_ATTRIBUTE_BOUNDINGBOX_Y1,  GRAIL_PROP_DRAG_Y1 },
  { GEIS_GESTURE_ATTRIBUTE_BOUNDINGBOX_X2,  GRAIL_PROP_DRAG_X2 },
  { GEIS_GESTURE_ATTRIBUTE_BOUNDINGBOX_Y2,  GRAIL_PROP_DRAG_Y2 },
  { GEIS_GESTURE_ATTRIBUTE_TOUCH_0_ID,      GRAIL_PROP_DRAG_ID_T0 },
  { GEIS_GESTURE_ATTRIBUTE_TOUCH_0_X,       GRAIL_PROP_DRAG_X_T0 },
  { GEIS_GESTURE_ATTRIBUTE_TOUCH_0_Y,       GRAIL_PROP_DRAG_Y_T0 },
  { GEIS_GESTURE_ATTRIBUTE_TOUCH_1_ID,      GRAIL_PROP_DRAG_ID_T1 },
  { GEIS_GESTURE_ATTRIBUTE_TOUCH_1_X,       GRAIL_PROP_DRAG_X_T1 },
  { GEIS_GESTURE_ATTRIBUTE_TOUCH_1_Y,       GRAIL_PROP_DRAG_Y_T1 },
  { GEIS_GESTURE_ATTRIBUTE_TOUCH_2_ID,      GRAIL_PROP_DRAG_ID_T2 },
  { GEIS_GESTURE_ATTRIBUTE_TOUCH_2_X,       GRAIL_PROP_DRAG_X_T2 },
  { GEIS_GESTURE_ATTRIBUTE_TOUCH_2_Y,       GRAIL_PROP_DRAG_Y_T2 },
  { GEIS_GESTURE_ATTRIBUTE_TOUCH_3_ID,      GRAIL_PROP_DRAG_ID_T3 },
  { GEIS_GESTURE_ATTRIBUTE_TOUCH_3_X,       GRAIL_PROP_DRAG_X_T3 },
  { GEIS_GESTURE_ATTRIBUTE_TOUCH_3_Y,       GRAIL_PROP_DRAG_Y_T3 },
  { GEIS_GESTURE_ATTRIBUTE_TOUCH_4_ID,      GRAIL_PROP_DRAG_ID_T4 },
  { GEIS_GESTURE_ATTRIBUTE_TOUCH_4_X,       GRAIL_PROP_DRAG_X_T4 },
  { GEIS_GESTURE_ATTRIBUTE_TOUCH_4_Y,       GRAIL_PROP_DRAG_Y_T4 },
};

static const GrailAttrMap s_grail_pinch_attrs[] =
{
  { GEIS_GESTURE_ATTRIBUTE_RADIUS_DELTA,    GRAIL_PROP_PINCH_DR },
  { GEIS_GESTURE_ATTRIBUTE_RADIAL_VELOCITY, GRAIL_PROP_PINCH_VR },
  { GEIS_GESTURE_ATTRIBUTE_RADIUS,          GRAIL_PROP_PINCH_R },
  { GEIS_GESTURE_ATTRIBUTE_BOUNDINGBOX_X1,  GRAIL_PROP_PINCH_X1 },
  { GEIS_GESTURE_ATTRIBUTE_BOUNDINGBOX_Y1,  GRAIL_PROP_PINCH_Y1 },
  { GEIS_GESTURE_ATTRIBUTE_BOUNDINGBOX_X2,  GRAIL_PROP_PINCH_X2 },
  { GEIS_GESTURE_ATTRIBUTE_BOUNDINGBOX_Y2,  GRAIL_PROP_PINCH_Y2 },
  { GEIS_GESTURE_ATTRIBUTE_TOUCH_0_ID,      GRAIL_PROP_PINCH_ID_T0 },
  { GEIS_GESTURE_ATTRIBUTE_TOUCH_0_X,       GRAIL_PROP_PINCH_X_T0 },
  { GEIS_GESTURE_ATTRIBUTE_TOUCH_0_Y,       GRAIL_PROP_PINCH_Y_T0 },
  { GEIS_GESTURE_ATTRIBUTE_TOUCH_1_ID,      GRAIL_PROP_PINCH_ID_T1 },
  { GEIS_GESTURE_ATTRIBUTE_TOUCH_1_X,       GRAIL_PROP_PINCH_X_T1 },
  { GEIS_GESTURE_ATTRIBUTE_TOUCH_1_Y,       GRAIL_PROP_PINCH_Y_T1 },
  { GEIS_GESTURE_ATTRIBUTE_TOUCH_2_ID,      GRAIL_PROP_PINCH_ID_T2 },
  { GEIS_GESTURE_ATTRIBUTE_TOUCH_2_X,       GRAIL_PROP_PINCH_X_T2 },
  { GEIS_GESTURE_ATTRIBUTE_TOUCH_2_Y,       GRAIL_PROP_PINCH_Y_T2 },
  { GEIS_GESTURE_ATTRIBUTE_TOUCH_3_ID,      GRAIL_PROP_PINCH_ID_T3 },
  { GEIS_GESTURE_ATTRIBUTE_TOUCH_3_X,       GRAIL_PROP_PINCH_X_T3 },
  { GEIS_GESTURE_ATTRIBUTE_TOUCH_3_Y,       GRAIL_PROP_PINCH_Y_T3 },
  { GEIS_GESTURE_ATTRIBUTE_TOUCH_4_ID,      GRAIL_PROP_PINCH_ID_T4 },
  { GEIS_GESTURE_ATTRIBUTE_TOUCH_4_X,       GRAIL_PROP_PINCH_X_T4 },
  { GEIS_GESTURE_ATTRIBUTE_TOUCH_4_Y,       GRAIL_PROP_PINCH_Y_T4 },
};

static const GrailAttrMap s_grail_rotate_attrs[] =
{
  { GEIS_GESTURE_ATTRIBUTE_ANGLE_DELTA,      GRAIL_PROP_ROTATE_DA },
  { GEIS_GESTURE_ATTRIBUTE_ANGULAR_VELOCITY, GRAIL_PROP_ROTATE_VA },
  { GEIS_GESTURE_ATTRIBUTE_ANGLE,            GRAIL_PROP_ROTATE_A },
  { GEIS_GESTURE_ATTRIBUTE_BOUNDINGBOX_X1,   GRAIL_PROP_ROTATE_X1 },
  { GEIS_GESTURE_ATTRIBUTE_BOUNDINGBOX_Y1,   GRAIL_PROP_ROTATE_Y1 },
  { GEIS_GESTURE_ATTRIBUTE_BOUNDINGBOX_X2,   GRAIL_PROP_ROTATE_X2 },
  { GEIS_GESTURE_ATTRIBUTE_BOUNDINGBOX_Y2,   GRAIL_PROP_ROTATE_Y2 },
  { GEIS_GESTURE_ATTRIBUTE_TOUCH_0_ID,       GRAIL_PROP_ROTATE_ID_T0 },
  { GEIS_GESTURE_ATTRIBUTE_TOUCH_0_X,        GRAIL_PROP_ROTATE_X_T0 },
  { GEIS_GESTURE_ATTRIBUTE_TOUCH_0_Y,        GRAIL_PROP_ROTATE_Y_T0 },
  { GEIS_GESTURE_ATTRIBUTE_TOUCH_1_ID,       GRAIL_PROP_ROTATE_ID_T1 },
  { GEIS_GESTURE_ATTRIBUTE_TOUCH_1_X,        GRAIL_PROP_ROTATE_X_T1 },
  { GEIS_GESTURE_ATTRIBUTE_TOUCH_1_Y,        GRAIL_PROP_ROTATE_Y_T1 },
  { GEIS_GESTURE_ATTRIBUTE_TOUCH_2_ID,       GRAIL_PROP_ROTATE_ID_T2 },
  { GEIS_GESTURE_ATTRIBUTE_TOUCH_2_X,        GRAIL_PROP_ROTATE_X_T2 },
  { GEIS_GESTURE_ATTRIBUTE_TOUCH_2_Y,        GRAIL_PROP_ROTATE_Y_T2 },
  { GEIS_GESTURE_ATTRIBUTE_TOUCH_3_ID,       GRAIL_PROP_ROTATE_ID_T3 },
  { GEIS_GESTURE_ATTRIBUTE_TOUCH_3_X,        GRAIL_PROP_ROTATE_X_T3 },
  { GEIS_GESTURE_ATTRIBUTE_TOUCH_3_Y,        GRAIL_PROP_ROTATE_Y_T3 },
  { GEIS_GESTURE_ATTRIBUTE_TOUCH_4_ID,       GRAIL_PROP_ROTATE_ID_T4 },
  { GEIS_GESTURE_ATTRIBUTE_TOUCH_4_X,        GRAIL_PROP_ROTATE_X_T4 },
  { GEIS_GESTURE_ATTRIBUTE_TOUCH_4_Y,        GRAIL_PROP_ROTATE_Y_T4 },
};

static const GrailAttrMap s_grail_tap_attrs[] =
{
  { GEIS_GESTURE_ATTRIBUTE_TAP_TIME,        GRAIL_PROP_TAP_DT },
  { GEIS_GESTURE_ATTRIBUTE_POSITION_X,      GRAIL_PROP_TAP_X },
  { GEIS_GESTURE_ATTRIBUTE_POSITION_Y,      GRAIL_PROP_TAP_Y },
  { GEIS_GESTURE_ATTRIBUTE_BOUNDINGBOX_X1,  GRAIL_PROP_TAP_X1 },
  { GEIS_GESTURE_ATTRIBUTE_BOUNDINGBOX_Y1,  GRAIL_PROP_TAP_Y1 },
  { GEIS_GESTURE_ATTRIBUTE_BOUNDINGBOX_X2,  GRAIL_PROP_TAP_X2 },
  { GEIS_GESTURE_ATTRIBUTE_BOUNDINGBOX_Y2,  GRAIL_PROP_TAP_Y2 },
  { GEIS_GESTURE_ATTRIBUTE_TOUCH_0_ID,      GRAIL_PROP_TAP_ID_T0 },
  { GEIS_GESTURE_ATTRIBUTE_TOUCH_0_X,       GRAIL_PROP_TAP_X_T0 },
  { GEIS_GESTURE_ATTRIBUTE_TOUCH_0_Y,       GRAIL_PROP_TAP_Y_T0 },
  { GEIS_GESTURE_ATTRIBUTE_TOUCH_1_ID,      GRAIL_PROP_TAP_ID_T1 },
  { GEIS_GESTURE_ATTRIBUTE_TOUCH_1_X,       GRAIL_PROP_TAP_X_T1 },
  { GEIS_GESTURE_ATTRIBUTE_TOUCH_1_Y,       GRAIL_PROP_TAP_Y_T1 },
  { GEIS_GESTURE_ATTRIBUTE_TOUCH_2_ID,      GRAIL_PROP_TAP_ID_T2 },
  { GEIS_GESTURE_ATTRIBUTE_TOUCH_2_X,       GRAIL_PROP_TAP_X_T2 },
  { GEIS_GESTURE_ATTRIBUTE_TOUCH_2_Y,       GRAIL_PROP_TAP_Y_T2 },
  { GEIS_GESTURE_ATTRIBUTE_TOUCH_3_ID,      GRAIL_PROP_TAP_ID_T3 },
  { GEIS_GESTURE_ATTRIBUTE_TOUCH_3_X,       GRAIL_PROP_TAP_X_T3 },
  { GEIS_GESTURE_ATTRIBUTE_TOUCH_3_Y,       GRAIL_PROP_TAP_Y_T3 },
  { GEIS_GESTURE_ATTRIBUTE_TOUCH_4_ID,      GRAIL_PROP_TAP_ID_T4 },
  { GEIS_GESTURE_ATTRIBUTE_TOUCH_4_X,       GRAIL_PROP_TAP_X_T4 },
  { GEIS_GESTURE_ATTRIBUTE_TOUCH_4_Y,       GRAIL_PROP_TAP_Y_T4 },
};

typedef struct _GrailTypeMap
{
  int                 type;
  const char         *name;
  int                 attr_count;
  const GrailAttrMap *attrs;
} GrailTypeMap;

#define GEIS_GESTURE_TYPE_DRAG   GRAIL_TYPE_DRAG1
#define GEIS_GESTURE_TYPE_PINCH  GRAIL_TYPE_PINCH1
#define GEIS_GESTURE_TYPE_ROTATE GRAIL_TYPE_ROTATE1
#define GEIS_GESTURE_TYPE_TAP    GRAIL_TYPE_TAP1

static const GrailTypeMap s_grail_type_map[] =
{
  {
    GEIS_GESTURE_TYPE_DRAG,
    GEIS_GESTURE_DRAG,
    sizeof(s_grail_drag_attrs) / sizeof(GrailAttrMap),
    s_grail_drag_attrs
  },
  {
    GEIS_GESTURE_TYPE_PINCH,
    GEIS_GESTURE_PINCH,
    sizeof(s_grail_pinch_attrs) / sizeof(GrailAttrMap),
    s_grail_pinch_attrs
  },
  {
    GEIS_GESTURE_TYPE_ROTATE,
    GEIS_GESTURE_ROTATE,
    sizeof(s_grail_rotate_attrs) / sizeof(GrailAttrMap),
    s_grail_rotate_attrs
  },
  {
    GEIS_GESTURE_TYPE_TAP,
    GEIS_GESTURE_TAP,
    sizeof(s_grail_tap_attrs) / sizeof(GrailAttrMap),
    s_grail_tap_attrs
  },
  {
    GRAIL_TYPE_DRAG2,
    GEIS_GESTURE_TYPE_DRAG2,
    sizeof(s_grail_drag_attrs) / sizeof(GrailAttrMap),
    s_grail_drag_attrs
  },
  {
    GRAIL_TYPE_PINCH2,
    GEIS_GESTURE_TYPE_PINCH2,
    sizeof(s_grail_pinch_attrs) / sizeof(GrailAttrMap),
    s_grail_pinch_attrs
  },
  {
    GRAIL_TYPE_ROTATE2,
    GEIS_GESTURE_TYPE_ROTATE2,
    sizeof(s_grail_rotate_attrs) / sizeof(GrailAttrMap),
    s_grail_rotate_attrs
  },
  {
    GRAIL_TYPE_DRAG3,
    GEIS_GESTURE_TYPE_DRAG3,
    sizeof(s_grail_drag_attrs) / sizeof(GrailAttrMap),
    s_grail_drag_attrs
  },
  {
    GRAIL_TYPE_PINCH3,
    GEIS_GESTURE_TYPE_PINCH3,
    sizeof(s_grail_pinch_attrs) / sizeof(GrailAttrMap),
    s_grail_pinch_attrs
  },
  {
    GRAIL_TYPE_ROTATE3,
    GEIS_GESTURE_TYPE_ROTATE3,
    sizeof(s_grail_rotate_attrs) / sizeof(GrailAttrMap),
    s_grail_rotate_attrs
  },
  {
    GRAIL_TYPE_DRAG4,
    GEIS_GESTURE_TYPE_DRAG4,
    sizeof(s_grail_drag_attrs) / sizeof(GrailAttrMap),
    s_grail_drag_attrs
  },
  {
    GRAIL_TYPE_PINCH4,
    GEIS_GESTURE_TYPE_PINCH4,
    sizeof(s_grail_pinch_attrs) / sizeof(GrailAttrMap),
    s_grail_pinch_attrs
  },
  {
    GRAIL_TYPE_ROTATE4,
    GEIS_GESTURE_TYPE_ROTATE4,
    sizeof(s_grail_rotate_attrs) / sizeof(GrailAttrMap),
    s_grail_rotate_attrs
  },
  {
    GRAIL_TYPE_DRAG5,
    GEIS_GESTURE_TYPE_DRAG5,
    sizeof(s_grail_drag_attrs) / sizeof(GrailAttrMap),
    s_grail_drag_attrs
  },
  {
    GRAIL_TYPE_PINCH5,
    GEIS_GESTURE_TYPE_PINCH5,
    sizeof(s_grail_pinch_attrs) / sizeof(GrailAttrMap),
    s_grail_pinch_attrs
  },
  {
    GRAIL_TYPE_ROTATE5,
    GEIS_GESTURE_TYPE_ROTATE5,
    sizeof(s_grail_rotate_attrs) / sizeof(GrailAttrMap),
    s_grail_rotate_attrs
  },
  {
    GRAIL_TYPE_TAP1,
    GEIS_GESTURE_TYPE_TAP1,
    sizeof(s_grail_tap_attrs) / sizeof(GrailAttrMap),
    s_grail_tap_attrs
  },
  {
    GRAIL_TYPE_TAP2,
    GEIS_GESTURE_TYPE_TAP2,
    sizeof(s_grail_tap_attrs) / sizeof(GrailAttrMap),
    s_grail_tap_attrs
  },
  {
    GRAIL_TYPE_TAP3,
    GEIS_GESTURE_TYPE_TAP3,
    sizeof(s_grail_tap_attrs) / sizeof(GrailAttrMap),
    s_grail_tap_attrs
  },
  {
    GRAIL_TYPE_TAP4,
    GEIS_GESTURE_TYPE_TAP4,
    sizeof(s_grail_tap_attrs) / sizeof(GrailAttrMap),
    s_grail_tap_attrs
  },
  {
    GRAIL_TYPE_TAP5,
    GEIS_GESTURE_TYPE_TAP5,
    sizeof(s_grail_tap_attrs) / sizeof(GrailAttrMap),
    s_grail_tap_attrs
  },
  // special (undocumented) gestures available in Grail
  {
    GRAIL_TYPE_EDRAG,
    "EDrag",
    sizeof(s_grail_drag_attrs) / sizeof(GrailAttrMap),
    s_grail_drag_attrs
  },
  {
    GRAIL_TYPE_EPINCH,
    "EPinch",
    sizeof(s_grail_drag_attrs) / sizeof(GrailAttrMap),
    s_grail_drag_attrs
  },
  {
    GRAIL_TYPE_EROTATE,
    "ERotate",
    sizeof(s_grail_drag_attrs) / sizeof(GrailAttrMap),
    s_grail_drag_attrs
  },
  {
    GRAIL_TYPE_MDRAG,
    "MDrag",
    sizeof(s_grail_drag_attrs) / sizeof(GrailAttrMap),
    s_grail_drag_attrs
  },
  {
    GRAIL_TYPE_MPINCH,
    "MPinch",
    sizeof(s_grail_drag_attrs) / sizeof(GrailAttrMap),
    s_grail_drag_attrs
  },
  {
    GRAIL_TYPE_MROTATE,
    "MRotate",
    sizeof(s_grail_drag_attrs) / sizeof(GrailAttrMap),
    s_grail_drag_attrs
  },
  { 0, NULL, 0, NULL }
};



GeisGestureAttr s_name_attr[2] =
{
  { GEIS_GESTURE_ATTRIBUTE_GESTURE_NAME, GEIS_ATTR_TYPE_STRING, 0 },
  { GEIS_GESTURE_ATTRIBUTE_TOUCHES, GEIS_ATTR_TYPE_INTEGER, 0 }
};

static GeisInteger
geis_xcb_determine_primitive_type(int xcb_gesture_type)
{
  switch (xcb_gesture_type)
  {
    case GRAIL_TYPE_DRAG1:
    case GRAIL_TYPE_DRAG2:
    case GRAIL_TYPE_DRAG3:
    case GRAIL_TYPE_DRAG4:
    case GRAIL_TYPE_DRAG5:
    case GRAIL_TYPE_EDRAG:
    case GRAIL_TYPE_MDRAG:
      return GEIS_GESTURE_TYPE_DRAG;

    case GRAIL_TYPE_PINCH1:
    case GRAIL_TYPE_PINCH2:
    case GRAIL_TYPE_PINCH3:
    case GRAIL_TYPE_PINCH4:
    case GRAIL_TYPE_PINCH5:
    case GRAIL_TYPE_EPINCH:
    case GRAIL_TYPE_MPINCH:
      return GEIS_GESTURE_TYPE_PINCH;

    case GRAIL_TYPE_ROTATE1:
    case GRAIL_TYPE_ROTATE2:
    case GRAIL_TYPE_ROTATE3:
    case GRAIL_TYPE_ROTATE4:
    case GRAIL_TYPE_ROTATE5:
    case GRAIL_TYPE_EROTATE:
    case GRAIL_TYPE_MROTATE:
      return GEIS_GESTURE_TYPE_ROTATE;

    case GRAIL_TYPE_TAP1:
    case GRAIL_TYPE_TAP2:
    case GRAIL_TYPE_TAP3:
    case GRAIL_TYPE_TAP4:
    case GRAIL_TYPE_TAP5:
      return GEIS_GESTURE_TYPE_TAP;
  }

  return xcb_gesture_type;
}

static GeisInteger
geis_xcb_determine_fingers(int gesture_type)
{
  switch (gesture_type)
  {
    case GRAIL_TYPE_DRAG1:
    case GRAIL_TYPE_PINCH1:
    case GRAIL_TYPE_ROTATE1:
    case GRAIL_TYPE_TAP1:
      return 1;

    case GRAIL_TYPE_DRAG2:
    case GRAIL_TYPE_PINCH2:
    case GRAIL_TYPE_ROTATE2:
    case GRAIL_TYPE_TAP2:
      return 2;

    case GRAIL_TYPE_DRAG3:
    case GRAIL_TYPE_PINCH3:
    case GRAIL_TYPE_ROTATE3:
    case GRAIL_TYPE_TAP3:
    case GRAIL_TYPE_EDRAG:
    case GRAIL_TYPE_EPINCH:
    case GRAIL_TYPE_EROTATE:
      return 3;

    case GRAIL_TYPE_DRAG4:
    case GRAIL_TYPE_PINCH4:
    case GRAIL_TYPE_ROTATE4:
    case GRAIL_TYPE_TAP4:
    case GRAIL_TYPE_MDRAG:
    case GRAIL_TYPE_MPINCH:
    case GRAIL_TYPE_MROTATE:
      return 4;

    case GRAIL_TYPE_DRAG5:
    case GRAIL_TYPE_PINCH5:
    case GRAIL_TYPE_ROTATE5:
    case GRAIL_TYPE_TAP5:
      return 5;
  }

  return 0;
}

static const char *
geis_xcb_get_gesture_type_primitive_name(GeisInteger gesture_type)
{
  GeisInteger primitive_type = geis_xcb_determine_primitive_type(gesture_type);
  const GrailTypeMap *type = s_grail_type_map;
  for (; type->name; ++type)
  {
    if (type->type == primitive_type)
    {
      return type->name;
    }
  }
  return "unknown";
}

/**
 * Dispatches gesture events to the instance's callback.
 */
static void
geis_xcb_dispatch_gesture_type(GeisInstance instance,
                               const char **gesture_list)
{
  if (gesture_list == GEIS_ALL_GESTURES)
  {
      const GrailTypeMap *type = s_grail_type_map;
      for (; type->name; ++type)
      {
	s_name_attr[0].string_val = 
	      geis_xcb_get_gesture_type_primitive_name(type->type);
	s_name_attr[1].integer_val = geis_xcb_determine_fingers(type->type);
	if (instance->gesture_funcs && instance->gesture_funcs->added)
	{
	  instance->gesture_funcs->added(instance->gesture_cookie,
	                                 type->type,
	                                 0,
	                                 sizeof(s_name_attr) / sizeof(GeisGestureAttr),
	                                 s_name_attr);
	}
      }
  }
  else
  {
    const char **gesture = NULL;
    for (gesture = gesture_list; *gesture; ++gesture)
    {
      const GrailTypeMap *type = s_grail_type_map;
      for (; type->name; ++type)
      {
	if (strcmp(type->name, *gesture) == 0)
	{
	  s_name_attr[0].string_val = 
	      geis_xcb_get_gesture_type_primitive_name(type->type);
	  s_name_attr[1].integer_val = geis_xcb_determine_fingers(type->type);
	  if (instance->gesture_funcs && instance->gesture_funcs->added)
	  {
	    instance->gesture_funcs->added(instance->gesture_cookie,
	                                   type->type,
	                                   0,
	                                   sizeof(s_name_attr) / sizeof(GeisGestureAttr),
	                                   s_name_attr);
	  }

	  break;
	}
      }
    }
  }
}

/**
 * Dispatches gesture events to the instance's callback.
 */
static void
geis_xcb_dispatch_gesture(GeisInstance                instance,
                          xcb_gesture_notify_event_t *event)
{
  static const int num_standard_attrs = 9;
  int num_attrs = num_standard_attrs + event->num_props;
  float *properties = NULL;
  GeisInteger gesture_type = geis_xcb_determine_primitive_type(event->gesture_type);

  GeisGestureAttr *attrs = calloc(num_attrs, sizeof(GeisGestureAttr));
  if (!attrs)
  {
    return;
  }

  attrs[0].name = GEIS_GESTURE_ATTRIBUTE_DEVICE_ID;
  attrs[0].type = GEIS_ATTR_TYPE_INTEGER;
  attrs[0].integer_val = event->device_id;
  attrs[1].name = GEIS_GESTURE_ATTRIBUTE_TIMESTAMP;
  attrs[1].type = GEIS_ATTR_TYPE_INTEGER;
  attrs[1].integer_val = event->time;
  attrs[2].name = GEIS_GESTURE_ATTRIBUTE_ROOT_WINDOW_ID;
  attrs[2].type = GEIS_ATTR_TYPE_INTEGER;
  attrs[2].integer_val = event->root;
  attrs[3].name = GEIS_GESTURE_ATTRIBUTE_EVENT_WINDOW_ID;
  attrs[3].type = GEIS_ATTR_TYPE_INTEGER;
  attrs[3].integer_val = event->event;
  attrs[4].name = GEIS_GESTURE_ATTRIBUTE_CHILD_WINDOW_ID;
  attrs[4].type = GEIS_ATTR_TYPE_INTEGER;
  attrs[4].integer_val = event->child;
  attrs[5].name = GEIS_GESTURE_ATTRIBUTE_FOCUS_X;
  attrs[5].type = GEIS_ATTR_TYPE_FLOAT;
  attrs[5].float_val = event->focus_x;
  attrs[6].name = GEIS_GESTURE_ATTRIBUTE_FOCUS_Y;
  attrs[6].type = GEIS_ATTR_TYPE_FLOAT;
  attrs[6].float_val = event->focus_y;

  attrs[8].name = GEIS_GESTURE_ATTRIBUTE_TOUCHES;
  attrs[8].type = GEIS_ATTR_TYPE_INTEGER;
  attrs[8].integer_val = geis_xcb_determine_fingers(event->gesture_type);

  {
    const GrailTypeMap *type = s_grail_type_map;
    properties = (float *)(event + 1);

    attrs[7].name = GEIS_GESTURE_ATTRIBUTE_GESTURE_NAME;
    attrs[7].type = GEIS_ATTR_TYPE_STRING;
    attrs[7].string_val = "unknown";
    for (; type->name; ++type)
    {
      if (type->type == gesture_type)
      {
	int i;
	attrs[7].string_val = type->name;

	for (i=0; i < event->num_props; ++i)
	{
	  if (i < type->attr_count)
	  {
	    attrs[num_standard_attrs + i].name = type->attrs[i].name;
	    attrs[num_standard_attrs + i].type = GEIS_ATTR_TYPE_FLOAT;
	    attrs[num_standard_attrs + i].float_val = properties[i];
	  }
	  else
	  {
	    attrs[num_standard_attrs + i].name = "unknown";
	    attrs[num_standard_attrs + i].type = GEIS_ATTR_TYPE_FLOAT;
	    attrs[num_standard_attrs + i].float_val = properties[i];
	  }
	}

	break;
      }
    }
  }

  switch (event->status)
  {
    case GRAIL_STATUS_BEGIN:
      if (instance->gesture_funcs &&
	  instance->gesture_funcs->start)
      {
	instance->gesture_funcs->start(instance->gesture_cookie,
	                               gesture_type,
	                               event->gesture_id,
	                               num_attrs, attrs);
      }

      break;
    case GRAIL_STATUS_UPDATE:
      if (instance->gesture_funcs &&
	  instance->gesture_funcs->update)
      {
	instance->gesture_funcs->update(instance->gesture_cookie,
	                                gesture_type,
	                                event->gesture_id,
	                                num_attrs, attrs);
      }

      break;
    case GRAIL_STATUS_END:
      if (instance->gesture_funcs &&
	  instance->gesture_funcs->finish)
      {
	instance->gesture_funcs->finish(instance->gesture_cookie,
	                                gesture_type,
	                                event->gesture_id,
	                                num_attrs, attrs);
      }

      break;
  }

  free(attrs);
}


GeisXcb
geis_xcb_new(GeisXcbWinInfo *win_info)
{
  GeisXcb xcb = calloc(1, sizeof(struct _GeisXcb));
  if (!xcb)
  {
    geis_error("error allocating GeisXcb");
    return NULL;
  }

  xcb->win_info = win_info;
  xcb->display = XOpenDisplay(win_info->display_name);
  if (!xcb->display)
  {
    geis_error("error opening X server.");
    goto error_exit;
  }

  xcb->connection = XGetXCBConnection(xcb->display);
  if (!xcb->connection)
  {
    geis_error("error connecting to X server.");
    goto error_exit;
  }

  {
    xcb_gesture_query_version_cookie_t  version_cookie;
    xcb_gesture_query_version_reply_t  *version_reply = NULL;
    xcb_generic_error_t                *error = NULL;

    version_cookie = xcb_gesture_query_version(xcb->connection,
                                               XCB_GESTURE_MAJOR_VERSION,
                                               XCB_GESTURE_MINOR_VERSION);
    version_reply = xcb_gesture_query_version_reply(xcb->connection,
                                                    version_cookie,
                                                    &error);
    if (!version_reply)
    {
      geis_error("failed to receive gesture version reply\n");
      goto error_exit;
    }

    if (version_reply->major_version != XCB_GESTURE_MAJOR_VERSION
     && version_reply->minor_version != XCB_GESTURE_MINOR_VERSION)
    {
      geis_error("server supports unrecognized version: %d.%d\n",
                 version_reply->major_version, version_reply->minor_version);
    }

   free(version_reply);
  }


  xcb->fd = xcb_get_file_descriptor(xcb->connection);
  xcb->instance_table = geis_instance_table_new(1);
  if (!xcb->instance_table)
  {
    geis_error("error constructing instance table");
    goto error_exit;
  }

  goto final_exit;

error_exit:
  free(xcb);

final_exit:
  return xcb;
}


/**
 * Create a new GeisInstance for a windowId.
 *
 * @param[in] An X11 windowId.
 */
GeisInstance
geis_xcb_create_instance(GeisXcb xcb, uint32_t window_id)
{
  GeisInstance instance = geis_instance_new(window_id);
  if (!instance)
  {
    geis_error("failure to create GeisInstance");
  }
  else
  {
    geis_instance_table_add(xcb->instance_table, instance);
  }
  return instance;
}


/**
 * Gets the file description on which events will appear.
 */
int
geis_xcb_fd(GeisXcb xcb)
{
  return xcb->fd;
}


/**
 * Dispatches events.
 */
void
geis_xcb_dispatch(GeisXcb xcb)
{
  if (xcb->connection)
  {
    const xcb_query_extension_reply_t *extension_info;
    extension_info = xcb_get_extension_data(xcb->connection, &xcb_gesture_id);

    xcb_generic_event_t *event = xcb_poll_for_event(xcb->connection);
    while (event)
    {
      xcb_gesture_notify_event_t *gesture_event = NULL;
      if (event->response_type != GenericEvent) {
	geis_warning("received non-generic event type: %d\n",
	             event->response_type);
	continue;
      }

      gesture_event = (xcb_gesture_notify_event_t*)event;
      if (gesture_event->extension != extension_info->major_opcode)
      {
	geis_warning("received non-gesture extension event: %d\n",
	             gesture_event->extension);
	continue;
      }

      if (gesture_event->event_type != XCB_GESTURE_NOTIFY)
      {
	geis_warning("received unrecognized gesture event type: %d\n",
	             gesture_event->event_type);
	continue;
      }

      GeisInstance instance = geis_instance_table_get(xcb->instance_table,
                                                      gesture_event->event);
      if (!instance)
      {
	geis_error("no instance found for window_id 0x%08x\n",
	           gesture_event->event);
      }
      else
      {
	geis_xcb_dispatch_gesture(instance, gesture_event);
      }

      event = xcb_poll_for_event(xcb->connection);
    }
  }
}


void
geis_xcb_free(GeisXcb xcb)
{
  xcb_disconnect(xcb->connection);
  XCloseDisplay(xcb->display);
}


static void
geis_xcb_map_gestures_to_mask(const char **gesture_list,
                              uint32_t **gesture_mask,
                              uint32_t *gesture_mask_len)
{
  *gesture_mask_len = 1;
  *gesture_mask = calloc(sizeof(uint32_t), *gesture_mask_len);
  if (!gesture_mask)
    return;

  const char **g = gesture_list;
  if (g == GEIS_ALL_GESTURES)
  {
    uint32_t all = 0xffffffff;
    memcpy(*gesture_mask, &all, sizeof(uint32_t));
  }
  else
  {
    while (*g)
    {
      const GrailTypeMap *type = s_grail_type_map;
      for (; type->name; ++type)
      {
	if (strcmp(type->name, *g) == 0)
	{
	  if (type->type < 32)
	  {
	    **gesture_mask |= (1 << type->type);
	  }
	  break;
	}
      }
      ++g;
    }
  }
}


static GeisStatus
geis_xcb_verify_event_selection(GeisXcb   xcb,
                                uint32_t  window_id,
                                uint16_t  device_id,
                                uint32_t  mask_len,
                                uint32_t *mask)
{
  GeisStatus                                result = GEIS_UNKNOWN_ERROR;
  xcb_generic_error_t                      *error;
  xcb_gesture_get_selected_events_cookie_t  events_cookie;
  xcb_gesture_get_selected_events_reply_t  *events_reply = NULL;
  xcb_gesture_event_mask_iterator_t         event_mask_it;
  uint32_t                                  mask_len_reply = 0;
  uint32_t                                 *mask_reply = NULL;
  int                                       device_is_found = 0;

  events_cookie = xcb_gesture_get_selected_events(xcb->connection,
                                                  window_id);
  events_reply = xcb_gesture_get_selected_events_reply(xcb->connection,
                                                       events_cookie,
                                                       &error);
  if (!events_reply)
  {
    geis_error("failed to receive selected events reply\n");
    goto really_done;
  }

  for (event_mask_it = xcb_gesture_get_selected_events_masks_iterator(events_reply);
       event_mask_it.rem;
       xcb_gesture_event_mask_next(&event_mask_it))
  {
    xcb_gesture_event_mask_t *event_mask = event_mask_it.data;
    if (event_mask->device_id == device_id)
    {
      mask_len_reply = xcb_gesture_event_mask_mask_data_length(event_mask);
      if (mask_len_reply != mask_len)
      {
	geis_error("incorrect mask length returned by server - expected %d, got %d\n",
	           mask_len, mask_len_reply);
	goto done;
      }

      mask_reply = xcb_gesture_event_mask_mask_data(event_mask);
      if (memcmp(mask, mask_reply, mask_len * 4) != 0)
      {
	geis_error("incorrect mask returned by server\n");
	goto done;
      }

      device_is_found = 1;
    }
  }

  if (!device_is_found)
  {
    geis_error("gesture mask for device %d not returned by server\n", device_id);
    goto done;
  }

  result = GEIS_STATUS_SUCCESS;
done:
  free(events_reply);
really_done:
  return result;
}


GeisStatus
geis_xcb_subscribe(GeisXcb        xcb,
                   GeisInstance   instance,
                   uint16_t       device_id,
                   const char   **gesture_list)
{
  GeisStatus            result = GEIS_UNKNOWN_ERROR;
  uint32_t              mask_len = 1;
  uint32_t             *mask = 0;
  const char          **g;
  xcb_generic_error_t  *error;
  xcb_void_cookie_t     select_cookie;
  xcb_window_t          window_id = geis_instance_get_window_id(instance);

  if (gesture_list == GEIS_ALL_GESTURES)
  {
    geis_debug("subscribing device %d for the all gestures\n", device_id);
  }
  else
  {
    geis_debug("subscribing device %d for the following gestures:\n", device_id);
    for (g = gesture_list; *g; ++g)
    {
      geis_debug("\t\"%s\"\n", *g);
    }
  }

  geis_xcb_map_gestures_to_mask(gesture_list, &mask, &mask_len);
  if (!mask)
  {
    geis_error("failed to allocate mask\n");
    goto really_done;
  }

  select_cookie = xcb_gesture_select_events_checked(xcb->connection,
                                                    window_id,
                                                    device_id,
                                                    mask_len,
                                                    mask);
  error = xcb_request_check(xcb->connection, select_cookie);
  if (error)
  {
    geis_error("failed to select events for window 0x%08x\n", window_id);
    goto done;
  }

  result = geis_xcb_verify_event_selection(xcb, window_id, device_id, mask_len, mask);
  if (result == GEIS_STATUS_SUCCESS)
  {
    geis_xcb_dispatch_gesture_type(instance, gesture_list);
  }

done:
  free(mask);
really_done:
  return result;
}


GeisStatus
geis_xcb_input_devices(GeisXcb xcb, GeisInputFuncs *funcs, void *cookie)
{
  xcb->input_funcs  = funcs;
  xcb->input_cookie = cookie;
  return GEIS_STATUS_SUCCESS;
}


