# -*- coding: utf-8 -*-
#
# Author: Guillermo Gonzalez <guillermo.gonzalez@canonical.com>
#
# Copyright 2009 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

""" Tests for the Volume Manager """
from __future__ import with_statement
import logging
import os
import uuid
import sys

from ubuntuone.storageprotocol.client import ListShares
from ubuntuone.storageprotocol.sharersp import (
    NotifyShareHolder,
    ShareResponse,
)
from contrib.testing.testcase import (
    FakeMain,
    BaseTwistedTestCase,
)
from ubuntuone.syncdaemon.volume_manager import (
    Share,
    VolumeManager,
    ShareFileShelf,
)
from twisted.internet import defer, reactor

# grab the metadata version before tests fiddle with it
CURRENT_METADATA_VERSION = VolumeManager.METADATA_VERSION


class BaseVolumeManagerTests(BaseTwistedTestCase):
    """ Bas TestCase for Volume Manager tests """

    def setUp(self):
        """ setup the test """
        BaseTwistedTestCase.setUp(self)
        self.log = logging.getLogger("ubuntuone.SyncDaemon.TEST")
        self.log.info("starting test %s.%s", self.__class__.__name__,
                      self._testMethodName)
        self.root_dir = self.mktemp('root_dir')
        self.data_dir = self.mktemp('data_dir')
        self.shares_dir = self.mktemp('shares_dir')
        self.partials_dir = self.mktemp('partials_dir')
        self.main = FakeMain(self.root_dir, self.shares_dir,
                             self.data_dir, self.partials_dir)
        self.vm = self.main.vm

    def tearDown(self):
        """ cleanup main and remove the temp dir """
        self.main.shutdown()
        self.rmtree(self.root_dir)
        self.rmtree(self.data_dir)
        self.rmtree(self.shares_dir)
        self.log.info("finished test %s.%s", self.__class__.__name__,
                      self._testMethodName)
        VolumeManager.METADATA_VERSION = CURRENT_METADATA_VERSION
        return BaseTwistedTestCase.tearDown(self)


class VolumeManagerTests(BaseVolumeManagerTests):
    """ Tests for Volume Manager internal API. """

    def test_on_server_root(self):
        """ check that list_shares is called in on_server_root """
        d = defer.Deferred()
        # helper method, pylint: disable-msg=C0111
        def list_shares():
            mdobj = self.main.fs.get_by_path(self.root_dir)
            d.callback(mdobj)
        self.main.action_q.list_shares = list_shares
        self.vm.on_server_root('root_uuid')
        return d

    def test_add_share(self):
        """ test the add_share method. """
        # initialize the the root
        self.vm.on_server_root('root_uuid')
        share_path = os.path.join(self.shares_dir, 'fake_share')
        share = Share(share_path, share_id='share_id')
        self.vm.add_share(share)
        self.assertIn(share.id, self.vm.shares)

    def test_share_deleted(self):
        """ Check that a share is deleted from the share mapping. """
        # initialize the the root
        self.vm.on_server_root('root_uuid')
        share_path = os.path.join(self.shares_dir, 'fake_share')
        share = Share(share_path, share_id='share_id')
        self.vm.add_share(share)
        self.assertIn(share.id, self.vm.shares)
        self.vm.share_deleted(share.id)
        self.assertNotIn(share.id, self.vm.shares)

    def test_share_changed(self):
        """ check that VM.share_changed updates the access_level. """
        share_holder = NotifyShareHolder.from_params('share_id', None,
                                                     'fake_share',
                                                     'test_username',
                                                     'visible_name', 'Modify')
        # initialize the the root
        self.vm.on_server_root('root_uuid')
        share_path = os.path.join(self.shares_dir, share_holder.share_name)
        share = Share(share_path, share_id=share_holder.share_id,
                      access_level='View')
        self.vm.add_share(share)
        self.vm.share_changed(share_holder)
        self.assertEquals('Modify', self.vm.shares[share.id].access_level)

    def test_handle_AQ_SHARES_LIST(self):
        """ test the handling of the AQ_SHARE_LIST event. """
        share_id = uuid.uuid4()
        share_response = ShareResponse.from_params(share_id, 'to_me',
                                                   'fake_share_uuid',
                                                   'fake_share', 'username',
                                                   'visible_username', 'yes',
                                                   'View')
        # initialize the the root
        self.vm.on_server_root('root_uuid')
        response = ListShares(None)
        response.shares = [share_response]
        self.vm.handle_AQ_SHARES_LIST(response)
        self.assertEquals(2, len(self.vm.shares)) # the new shares and root
        # check that the share is in the shares dict
        self.assertTrue(str(share_id) in self.vm.shares)
        share = self.vm.shares[str(share_id)]
        self.assertEquals('fake_share', share.name)
        self.assertEquals('fake_share_uuid', share.subtree)



    def test_handle_SV_SHARE_CHANGED(self):
        """ test the handling of the AQ_SHARE_LIST event. """
        share_id = uuid.uuid4()
        share_holder = NotifyShareHolder.from_params(share_id, None,
                                                     'fake_share',
                                                     'test_username',
                                                     'visible_name', 'Modify')
        # initialize the the root
        self.vm.on_server_root('root_uuid')
        # create a share
        share_path = os.path.join(self.shares_dir, share_holder.share_name)
        share = Share(share_path, share_id=str(share_holder.share_id),
                      access_level='View')
        self.vm.add_share(share)
        self.vm.handle_SV_SHARE_CHANGED(message='changed', info=share_holder)
        self.assertEquals('Modify', self.vm.shares[str(share_id)].access_level)
        self.vm.handle_SV_SHARE_CHANGED('deleted', share_holder)
        self.assertNotIn('share_id', self.vm.shares)

    def test_persistence(self):
        """ Test that the persistence of shares works as expected. """
        # create the folders layout
        share_path = os.path.join(self.shares_dir, 'my_share')
        share = Share(share_path, share_id='a_share_id', access_level='View')
        self.vm.add_share(share)
        other_vm = VolumeManager(self.main)
        for key in self.vm.shares:
            self.assertEquals(self.vm.shares[key].__dict__,
                              other_vm.shares[key].__dict__)

    def test_handle_AQ_SHARES_LIST_shared(self):
        """test the handling of the AQ_SHARE_LIST event, with a shared dir."""
        share_id = uuid.uuid4()
        share_response = ShareResponse.from_params(share_id, 'to_me',
                                                   'fake_share_uuid',
                                                   'fake_share', 'username',
                                                   'visible_username', 'yes',
                                                   'View')
        shared_id = uuid.uuid4()
        shared_response = ShareResponse.from_params(shared_id, 'from_me',
                                                   'shared_uuid',
                                                   'fake_shared', 'myname',
                                                   'my_visible_name', 'yes',
                                                   'Modify')
        # initialize the the root
        self.vm.on_server_root('root_uuid')
        shared_dir = os.path.join(self.root_dir, 'shared_dir')
        self.main.fs.create(path=shared_dir, share_id="", is_dir=True)
        self.main.fs.set_node_id(shared_dir, shared_response.subtree)
        response = ListShares(None)
        response.shares = [share_response, shared_response]
        self.vm.handle_AQ_SHARES_LIST(response)
        self.assertEquals(2, len(self.vm.shares)) # the new shares and root
        self.assertEquals(1, len(self.vm.shared)) # the new shares and root
        shared = self.vm.shared[str(shared_id)]
        self.assertEquals('fake_shared', shared.name)
        # check that the uuid is stored in fs
        mdobj = self.main.fs.get_by_path(shared.path)
        self.assertEquals(shared.subtree, mdobj.node_id)

    def test_add_shared(self):
        """ Test VolumeManager.add_shared """
        path = os.path.join(self.vm.root.path, 'shared_path')
        self.main.fs.create(path, "")
        self.main.fs.set_node_id(path, 'node_id')
        self.main.fs.get_by_node_id("", 'node_id')

        # helper method, pylint: disable-msg=C0111
        def fake_create_share(node_id, user, name, access_level, marker):
            self.assertTrue(marker in self.vm.marker_share_map)
            self.vm.handle_AQ_CREATE_SHARE_OK(share_id='share_id',
                                              marker=marker)
        self.main.action_q.create_share = fake_create_share
        self.vm.create_share(path, 'fake_user', 'shared_name', 'View')

        self.assertTrue(self.vm.shared.get('share_id') is not None)
        share = self.vm.shared.get('share_id')
        self.assertEquals('fake_user', share.other_username)
        self.assertEquals('shared_name', share.name)
        self.assertEquals('View', share.access_level)
        self.assertEquals('node_id', share.subtree)
        self.assertEquals('share_id', share.id)

    def test_create_share(self):
        """ Test VolumeManager.create_share """
        path = os.path.join(self.vm.root.path, 'shared_path')
        self.main.fs.create(path, "")
        self.main.fs.set_node_id(path, 'node_id')
        self.main.fs.get_by_node_id("", 'node_id')

        # helper method, pylint: disable-msg=C0111
        def fake_create_share(node_id, user, name, access_level, marker):
            self.assertEquals('node_id', node_id)
            self.assertEquals('fake_user', user)
            self.assertEquals('shared_name', name)
            self.assertEquals('View', access_level)
            self.assertTrue(marker is not None)
            share = self.vm.marker_share_map[marker]
            self.assertEquals(path, share.path)
            self.assertEquals('View', share.access_level)
            self.assertEquals(marker, share.id)
            self.assertEquals('fake_user', share.other_username)
            self.assertEquals('node_id', share.subtree)

        self.main.action_q.create_share = fake_create_share
        self.vm.create_share(path, 'fake_user', 'shared_name', 'View')

    def test_create_share_error(self):
        """ Test VolumeManager.create_share """
        path = os.path.join(self.vm.root.path, 'shared_path')
        self.main.fs.create(path, "")
        self.main.fs.set_node_id(path, 'node_id')
        self.main.fs.get_by_node_id("", 'node_id')

        # helper method, pylint: disable-msg=C0111
        def fake_create_share(node_id, user, name, access_level, marker):
            self.vm.handle_AQ_CREATE_SHARE_ERROR(marker, 'a fake error')

        self.main.action_q.create_share = fake_create_share
        self.vm.create_share(path, 'fake_user', 'shared_name', 'View')

    def test_accept_share(self):
        """ Test the accept_share method. """
        d = defer.Deferred()
        self.vm.on_server_root('root_uuid')
        share_path = os.path.join(self.shares_dir, 'fake_share')
        share = Share(share_path, share_id='share_id', subtree="node_id")
        self.vm.add_share(share)
        self.assertIn(share.id, self.vm.shares)
        self.assertEquals(False, share.accepted)
        # helper method, pylint: disable-msg=C0111
        def answer_share(share_id, answer):
            reactor.callLater(0.2, d.callback, (share_id, answer))
            return d
        self.main.action_q.answer_share = answer_share
        def callback(result):
            share_id, answer = result
            self.assertEquals(share.id, share_id)
            self.assertEquals('Yes', answer)
        d.addCallback(callback)
        self.vm.accept_share(share.id, True)
        return d

    def test_handle_AQ_SHARES_LIST_shared_missing_md(self):
        """test the handling of the AQ_SHARE_LIST event, when the md
        isn't there yet.
        """
        shared_response = ShareResponse.from_params('shared_id', 'from_me',
                                                   'shared_uuid',
                                                   'fake_shared', 'myname',
                                                   'my_visible_name', 'yes',
                                                   'Modify')
        # initialize the the root
        self.vm.on_server_root('root_uuid')
        response = ListShares(None)
        response.shares = [shared_response]
        self.vm.handle_AQ_SHARES_LIST(response)
        self.assertEquals(1, len(self.vm.shared)) # the new shares and root
        shared = self.vm.shared['shared_id']
        self.assertEquals('fake_shared', shared.name)
        # check that the uuid is stored in fs
        self.assertEquals(shared_response.subtree, shared.subtree)
        self.assertEquals(None, shared.path)

    def test_handle_SV_SHARE_ANSWERED(self):
        """ test the handling of the AQ_SHARE_ANSWERED. """
        path = os.path.join(self.vm.root.path, 'shared_path')
        self.main.fs.create(path, "")
        self.main.fs.set_node_id(path, 'node_id')
        self.main.fs.get_by_node_id("", 'node_id')
        # initialize the the root
        self.vm.on_server_root('root_uuid')
        # add the shared folder
        share = Share(path, share_id='share_id', access_level='View')
        self.vm.add_shared(share)
        self.assertEquals(False, self.vm.shared['share_id'].accepted)
        # check that a answer notify of a missing share don't blowup
        self.vm.handle_SV_SHARE_ANSWERED('share_id', 'Yes')
        self.assertEquals(True, self.vm.shared['share_id'].accepted)

    def test_handle_SV_SHARE_ANSWERED_missing(self):
        """ test the handling of the AQ_SHARE_ANSWERED. """
        path = os.path.join(self.vm.root.path, 'shared_path')
        self.main.fs.create(path, "")
        self.main.fs.set_node_id(path, 'node_id')
        self.main.fs.get_by_node_id("", 'node_id')
        # initialize the the root
        self.vm.on_server_root('root_uuid')
        self.assertNotIn('share_id', self.vm.shared)
        # check that a answer notify of a missing share don't blowup
        self.vm.handle_SV_SHARE_ANSWERED('share_id', 'Yes')
        self.assertNotIn('share_id', self.vm.shared)

    def test_delete_share(self):
        """tests the deletion of a share and if it's removed from the metadata
        """
        share_response = ShareResponse.from_params('share_id', 'to_me',
                                                   'fake_share_uuid',
                                                   'fake_share', 'username',
                                                   'visible_username', 'yes',
                                                   'View')
        share_response_1 = ShareResponse.from_params('share_id_1', 'to_me',
                                                   'fake_share_uuid_1',
                                                   'fake_share_1', 'username',
                                                   'visible_username', 'yes',
                                                   'View')
        # initialize the the root
        self.vm.on_server_root('root_uuid')
        response = ListShares(None)
        response.shares = [share_response, share_response_1]
        self.vm.handle_AQ_SHARES_LIST(response)
        self.assertEquals(3, len(self.vm.shares)) # the new shares and root
        # check that the share is in the shares dict
        self.assertTrue('share_id' in self.vm.shares)
        self.assertTrue('share_id_1' in self.vm.shares)
        share = self.vm.shares['share_id']
        self.assertEquals('fake_share', share.name)
        self.assertEquals('fake_share_uuid', share.subtree)
        share = self.vm.shares['share_id_1']
        self.assertEquals('fake_share_1', share.name)
        self.assertEquals('fake_share_uuid_1', share.subtree)
        # inject a new ListShares response
        new_response = ListShares(None)
        new_response.shares = [share_response]
        self.vm.handle_AQ_SHARES_LIST(new_response)
        # check that the missing share was removed
        self.assertTrue('share_id' in self.vm.shares)
        self.assertFalse('share_id_1' in self.vm.shares)
        share = self.vm.shares['share_id']
        self.assertEquals('fake_share', share.name)
        self.assertEquals('fake_share_uuid', share.subtree)
        self.assertEquals(None, self.vm.shares.get('share_id_1'))


class VolumeManagerUnicodeTests(BaseVolumeManagerTests):
    """Tests for Volume Manager unicode capabilities."""

    def test_handle_SHARES_sharename(self):
        """test the handling of AQ_SHARE_LIST with non-ascii share name."""
        share_response = ShareResponse.from_params('share_id', 'to_me',
                                                   'fake_share_uuid',
                                                   u'montón', 'username',
                                                   'visible', 'yes',
                                                   'View')
        # initialize the the root
        self.vm.on_server_root('root_uuid')
        response = ListShares(None)
        response.shares = [share_response]
        self.vm.handle_AQ_SHARES_LIST(response)

        # check
        share = self.vm.shares['share_id']
        shouldbe_dir = os.path.join(self.shares_dir,
                                    u"montón".encode("utf8") + " from visible")
        self.assertEquals(shouldbe_dir, share.path)

    def test_handle_SHARES_visible_username(self):
        """test the handling of AQ_SHARE_LIST with non-ascii visible uname."""
        share_response = ShareResponse.from_params('share_id', 'to_me',
                                                   'fake_share_uuid',
                                                   'sharename', 'username',
                                                   u'Darío Toño', 'yes',
                                                   'View')
        # initialize the the root
        self.vm.on_server_root('root_uuid')
        response = ListShares(None)
        response.shares = [share_response]
        self.vm.handle_AQ_SHARES_LIST(response)

        # check
        share = self.vm.shares['share_id']
        shouldbe_dir = os.path.join(self.shares_dir,
                            "sharename from " + u"Darío Toño".encode("utf8"))
        self.assertEquals(shouldbe_dir, share.path)

    def test_handle_SV_SHARE_CHANGED_sharename(self):
        """test the handling of SV_SHARE_CHANGED for non-ascii share name."""
        share_holder = NotifyShareHolder.from_params('share_id', None,
                                                     u'año',
                                                     'test_username',
                                                     'visible', 'Modify')
        self.vm.on_server_root('root_uuid')
        self.vm.handle_SV_SHARE_CHANGED('changed', share_holder)
        shouldbe_dir = os.path.join(self.shares_dir,
                                    u"año".encode("utf8") + " from visible")
        self.assertEquals(shouldbe_dir, self.vm.shares['share_id'].path)

    def test_handle_SV_SHARE_CHANGED_visible(self):
        """test the handling of SV_SHARE_CHANGED for non-ascii visible name."""
        share_holder = NotifyShareHolder.from_params('share_id', None,
                                                     'share',
                                                     'test_username',
                                                     u'Ramón', 'Modify')
        self.vm.on_server_root('root_uuid')
        self.vm.handle_SV_SHARE_CHANGED('changed', share_holder)
        shouldbe_dir = os.path.join(self.shares_dir,
                                    "share from " + u"Ramón".encode("utf8"))
        self.assertEquals(shouldbe_dir, self.vm.shares['share_id'].path)



class ShareShelfUpgradeTests(BaseTwistedTestCase):
    """ Tests for shares shelf upgrades"""

    def setUp(self):
        """ setup the test """
        BaseTwistedTestCase.setUp(self)
        self.root_dir = self.mktemp('Ubuntu One')
        self.data_dir = self.mktemp('data_dir')
        self.partials_dir = self.mktemp('partials_dir')
        self.shares_dir = self.mktemp(os.path.join('Ubuntu One',
                                                   'Shared with Me'))

    def tearDown(self):
        """ cleanup main and remove the temp dir """
        main = getattr(self, 'main', None)
        if main:
            main.shutdown()
        if os.path.exists(self.root_dir):
            self.rmtree(self.root_dir)
        if os.path.exists(self.data_dir):
            self.rmtree(self.data_dir)
        if os.path.exists(self.shares_dir):
                self.rmtree(self.shares_dir)
        VolumeManager.METADATA_VERSION = CURRENT_METADATA_VERSION
        return BaseTwistedTestCase.tearDown(self)

    def check_version(self):
        """ check if the current version in the .version file is the lastone.
        """
        with open(os.path.join(self.data_dir, 'vm', '.version'), 'r') as fd:
            self.assertEquals(CURRENT_METADATA_VERSION, fd.read().strip())

    def test_0_to_1(self):
        """ Test the upgrade from the first shelf layout version to v. 1"""
        # ensure a clean data_dir
        self.rmtree(self.data_dir)
        vm_data_dir = os.path.join(self.data_dir, 'vm')
        old_shelf = ShareFileShelf(vm_data_dir)
        # add the root_uuid key
        root_share = Share(self.root_dir)
        root_share.access_level = 'Modify'
        old_shelf[''] = root_share
        for idx in range(1, 10):
            old_shelf[str(uuid.uuid4())] = \
                    Share(os.path.join(self.shares_dir, str(idx)))
        # ShareFileShelf.keys returns a generator
        old_keys = [key for key in old_shelf.keys()]
        self.assertEquals(10, len(old_keys))
        # we want to keep a refernece to main in order to shutdown
        # pylint: disable-msg=W0201
        self.main = FakeMain(self.root_dir, self.shares_dir,
                             self.data_dir, self.partials_dir)
        new_keys = [new_key for new_key in self.main.vm.shares.keys()]
        self.assertEquals(10, len(new_keys))
        for new_key in new_keys:
            self.assertIn(new_key, old_keys)
        # check the old data is still there (in the backup)
        backup_shelf = ShareFileShelf(os.path.join(vm_data_dir, '0.bkp'))
        backup_keys = [key for key in backup_shelf.keys()]
        for old_key in old_keys:
            self.assertIn(old_key, backup_keys)
        for new_key in new_keys:
            self.assertIn(new_key, backup_keys)
        self.check_version()

    def test_1_to_2(self):
        """ Test the upgrade from v.1 of the metadata to v.2"""
        # ensure a clean data_dir
        self.rmtree(self.data_dir)
        vm_data_dir = os.path.join(self.data_dir, 'vm')
        vm_shares_dir = os.path.join(vm_data_dir, 'shares')
        os.makedirs(vm_data_dir)
        # write the .version file with v.1
        with open(os.path.join(vm_data_dir, '.version'), 'w') as fd:
            fd.write('1')

        share_file = os.path.join(vm_shares_dir,
                                  '0/6/6/0664f050-9254-45c5-9f31-3482858709e4')
        os.makedirs(os.path.dirname(share_file))
        # this is the str of a version 2 pickle
        share_value = "\x80\x02ccanonical.ubuntuone.storage.syncdaemon." + \
                "volume_manager\nShare\nq\x01)\x81q\x02}q\x03(U\x04nameq" + \
                "\x04U\tfakeshareq\x05U\x0eother_usernameq\x06U\x08fakeu" + \
                "serq\x07U\x07subtreeq\x08U$beb0c48c-6755-4fbd-938f-3d20" + \
                "fa7b102bq\tU\x12other_visible_nameq\nU\tfake userq\x0bU" + \
                "\x0caccess_levelq\x0cU\x04Viewq\rU\x04pathq\x0eU=/home/" + \
                "auser/Ubuntu One/Shared With Me/fakeshare from fakeuser" + \
                "q\x0fU\x08acceptedq\x10\x88U\x02idq\x11U$0664f050-9254-" + \
                "45c5-9f31-3482858709e4q\x12ub."
        with open(share_file, 'w') as fd:
            fd.write(share_value)

        # fake the old namespace
        sys.modules['canonical.ubuntuone.storage.syncdaemon.volume_manager'] = \
                sys.modules['ubuntuone.syncdaemon.volume_manager']
        # try to load the shelf
        old_shelf = ShareFileShelf(vm_shares_dir)
        share = old_shelf['0664f050-9254-45c5-9f31-3482858709e4']
        self.assertTrue(share is not None)
        del sys.modules['canonical.ubuntuone.storage.syncdaemon.volume_manager']
        # now use the real VolumeManager
        # we want to keep a refernece to main in order to shutdown
        # pylint: disable-msg=W0201
        self.main = FakeMain(self.root_dir, self.shares_dir,
                             self.data_dir, self.partials_dir)
        new_keys = [new_key for new_key in self.main.vm.shares.keys()]
        self.assertEquals(2, len(new_keys)) # the fake share plus root
        for key in ['', share.id]:
            self.assertIn(key, new_keys)
        self.check_version()

    def test_2_to_3(self):
        """ Test the upgrade from v.2 of the metadata to v.3"""
        vm_data_dir = os.path.join(self.data_dir, 'vm')
        os.makedirs(vm_data_dir)
        with open(os.path.join(vm_data_dir, '.version'), 'w') as fd:
            fd.write('2')
        open(self.root_dir + '/foo.conflict', 'w').close()
        open(self.root_dir + '/foo.conflict.23', 'w').close()
        open(self.shares_dir + '/bar.partial', 'w').close()
        os.mkdir(self.shares_dir + '/baz/')
        open(self.shares_dir + '/baz/baz.conflict', 'w').close()
        os.chmod(self.shares_dir + '/baz/', 0500)
        self.main = FakeMain(self.root_dir, self.shares_dir,
                             self.data_dir, self.partials_dir)
        self.assertTrue(os.path.exists(self.root_dir + '/foo.u1conflict'))
        self.assertTrue(os.path.exists(self.root_dir + '/foo.u1conflict.23'))
        self.assertTrue(os.path.exists(self.shares_dir + '/.u1partial.bar'))
        self.assertTrue(os.path.exists(self.shares_dir + '/baz/baz.u1conflict'))

    def test_2_to_3_more(self):
        """ Test the upgrade from v.2 of the metadata to v.3 some more"""
        vm_data_dir = os.path.join(self.data_dir, 'vm')
        os.makedirs(vm_data_dir)
        with open(os.path.join(vm_data_dir, '.version'), 'w') as fd:
            fd.write('2')

        expected = []

        for dirname in self.root_dir, self.shares_dir:
            # a plain .conflict...
            # ...on a file
            open(dirname + '/1a.conflict', 'w').close()
            expected.append(dirname + '/1a.u1conflict')
            # ...on an empty directory
            os.mkdir(dirname + '/1b.conflict')
            expected.append(dirname + '/1b.u1conflict')
            # ...on a directory with content
            os.mkdir(dirname + '/1c.conflict')
            os.mkdir(dirname + '/1c.conflict/1c')
            expected.append(dirname + '/1c.u1conflict/1c')
            # ...in a readonly directory
            os.mkdir(dirname + '/1d')
            os.mkdir(dirname + '/1d/1d.conflict')
            os.chmod(dirname + '/1d', 0500)
            expected.append(dirname + '/1d/1d.u1conflict')
            # ...in a directory that is also a .conflict
            os.mkdir(dirname + '/1e.conflict')
            os.mkdir(dirname + '/1e.conflict/1e.conflict')
            expected.append(dirname + '/1e.u1conflict/1e.u1conflict')

            # a numbered .conflict...
            # ...on a file
            open(dirname + '/2a.conflict.2', 'w').close()
            expected.append(dirname + '/2a.u1conflict.2')
            # ...on an empty directory
            os.mkdir(dirname + '/2b.conflict.3')
            expected.append(dirname + '/2b.u1conflict.3')
            # ...on a directory with content
            os.mkdir(dirname + '/2c.conflict.4')
            os.mkdir(dirname + '/2c.conflict.4/2c')
            expected.append(dirname + '/2c.u1conflict.4/2c')
            # ...in a readonly directory
            os.mkdir(dirname + '/2d')
            os.mkdir(dirname + '/2d/2d.conflict.5')
            os.chmod(dirname + '/2d', 0500)
            expected.append(dirname + '/2d/2d.u1conflict.5')
            # ...in a directory that is also a .conflict
            os.mkdir(dirname + '/2e.conflict')
            os.mkdir(dirname + '/2e.conflict/2e.conflict.6')
            expected.append(dirname + '/2e.u1conflict/2e.u1conflict.6')

            # a plain .conflict of which there already exists a .u1conflict...
            # ...on a file
            open(dirname + '/3a.conflict', 'w').close()
            open(dirname + '/3a.u1conflict', 'w').close()
            expected.append(dirname + '/3a.u1conflict')
            expected.append(dirname + '/3a.u1conflict.1')
            # ...on an empty directory
            os.mkdir(dirname + '/3b.conflict')
            os.mkdir(dirname + '/3b.u1conflict')
            expected.append(dirname + '/3b.u1conflict')
            expected.append(dirname + '/3b.u1conflict.1')
            # ...on a directory with content
            os.mkdir(dirname + '/3c.conflict')
            os.mkdir(dirname + '/3c.conflict/3c')
            os.mkdir(dirname + '/3c.u1conflict')
            os.mkdir(dirname + '/3c.u1conflict/3c2')
            expected.append(dirname + '/3c.u1conflict.1/3c')
            expected.append(dirname + '/3c.u1conflict/3c2')
            # ...in a readonly directory
            os.mkdir(dirname + '/3d')
            os.mkdir(dirname + '/3d/3d.conflict')
            os.mkdir(dirname + '/3d/3d.u1conflict')
            os.mkdir(dirname + '/3d/3d.u1conflict/3d')
            os.chmod(dirname + '/3d', 0500)
            expected.append(dirname + '/3d/3d.u1conflict/3d')
            expected.append(dirname + '/3d/3d.u1conflict.1')
            # ...in a directory that is also a .conflict
            os.mkdir(dirname + '/3e.conflict')
            os.mkdir(dirname + '/3e.conflict/3e.conflict')
            os.mkdir(dirname + '/3e.conflict/3e.u1conflict')
            os.mkdir(dirname + '/3e.conflict/3e.u1conflict/3e')
            expected.append(dirname + '/3e.u1conflict/3e.u1conflict/3e')
            expected.append(dirname + '/3e.u1conflict/3e.u1conflict.1')

            # a numbered .conflict of which there already exists a .u1conflict...
            # ...on a file
            open(dirname + '/4a.conflict.1', 'w').close()
            open(dirname + '/4a.u1conflict.1', 'w').close()
            expected.append(dirname + '/4a.u1conflict.1')
            expected.append(dirname + '/4a.u1conflict.2')
            # ...on an empty directory
            os.mkdir(dirname + '/4b.conflict.2')
            os.mkdir(dirname + '/4b.u1conflict.2')
            expected.append(dirname + '/4b.u1conflict.2')
            expected.append(dirname + '/4b.u1conflict.3')
            # ...on a directory with content
            os.mkdir(dirname + '/4c.conflict.3')
            os.mkdir(dirname + '/4c.conflict.3/4c')
            os.mkdir(dirname + '/4c.u1conflict.3')
            expected.append(dirname + '/4c.u1conflict.4/4c')
            expected.append(dirname + '/4c.u1conflict.3')
            # ...in a readonly directory
            os.mkdir(dirname + '/4d')
            os.mkdir(dirname + '/4d/4d.conflict.4')
            os.mkdir(dirname + '/4d/4d.u1conflict.4')
            os.chmod(dirname + '/4d', 0500)
            expected.append(dirname + '/4d/4d.u1conflict.4')
            expected.append(dirname + '/4d/4d.u1conflict.5')
            # ...in a directory that is also a .conflict
            os.mkdir(dirname + '/4e.conflict')
            os.mkdir(dirname + '/4e.conflict/4e.conflict.5')
            os.mkdir(dirname + '/4e.conflict/4e.u1conflict.5')
            expected.append(dirname + '/4e.u1conflict/4e.u1conflict.5')
            expected.append(dirname + '/4e.u1conflict/4e.u1conflict.6')

            # a plain .partial...
            # ...of a file
            open(dirname + '/5a.partial', 'w').close()
            expected.append(dirname + '/.u1partial.5a')
            # ...of a directory
            os.mkdir(dirname + '/5b')
            open(dirname + '/5b/.partial', 'w').close()
            expected.append(dirname + '/5b/.u1partial')
            # ...of a readonly directory
            os.mkdir(dirname + '/5c')
            open(dirname + '/5c/.partial', 'w').close()
            os.chmod(dirname + '/5c', 0500)
            expected.append(dirname + '/5c/.u1partial')

            # a plain .partial of which there already exists a .u1partial...
            # ...of a file
            open(dirname + '/6a.partial', 'w').close()
            open(dirname + '/.u1partial.6a', 'w').close()
            expected.append(dirname + '/.u1partial.6a')
            expected.append(dirname + '/.u1partial.6a.1')
            # ...of a directory
            os.mkdir(dirname + '/6b')
            open(dirname + '/6b/.partial', 'w').close()
            open(dirname + '/6b/.u1partial', 'w').close()
            expected.append(dirname + '/6b/.u1partial')
            expected.append(dirname + '/6b/.u1partial.1')
            # ...of a readonly directory
            os.mkdir(dirname + '/6c')
            open(dirname + '/6c/.partial', 'w').close()
            open(dirname + '/6c/.u1partial', 'w').close()
            os.chmod(dirname + '/6c', 0500)
            expected.append(dirname + '/6c/.u1partial')
            expected.append(dirname + '/6c/.u1partial.1')

        self.main = FakeMain(self.root_dir, self.shares_dir,
                             self.data_dir, self.partials_dir)

        for path in expected:
            self.assertTrue(os.path.exists(path), 'missing ' + path)

    def test_missing_version_file_with_version_non_0(self):
        """ Test the upgrade from the first shelf layout version to v3
        while the metadata sould be in v3 format
        """
        # ensure a clean data_dir
        self.rmtree(self.data_dir)
        vm_data_dir = os.path.join(self.data_dir, 'vm', 'shares')
        maybe_old_shelf = ShareFileShelf(vm_data_dir)
        # add the root_uuid key
        root_share = Share(self.root_dir)
        root_share.access_level = 'Modify'
        maybe_old_shelf[''] = root_share
        for idx in range(1, 10):
            maybe_old_shelf[str(uuid.uuid4())] = \
                    Share(os.path.join(self.shares_dir, str(idx)))
        # ShareFileShelf.keys returns a generator
        maybe_old_keys = [key for key in maybe_old_shelf.keys()]
        self.assertEquals(10, len(maybe_old_keys))
        # we want to keep a refernece to main in order to shutdown
        # pylint: disable-msg=W0201
        self.main = FakeMain(self.root_dir, self.shares_dir,
                             self.data_dir, self.partials_dir)
        new_keys = [new_key for new_key in self.main.vm.shares.keys()]
        self.assertEquals(10, len(new_keys))
        for new_key in new_keys:
            self.assertIn(new_key, maybe_old_keys)
        # as we didn't actually upgrade the shelf, just the .version file
        # check the empty 0.bkp
        # check the old data is still there (in the backup)
        backup_shelf = ShareFileShelf(os.path.join(vm_data_dir, '0.bkp'))
        backup_keys = [key for key in backup_shelf.keys()]
        self.assertEquals(0, len(backup_keys))
        self.check_version()

    def test_3_to_4(self):
        """upgrade from version 3 to 4"""
        vm_data_dir = os.path.join(self.data_dir, 'vm')
        os.makedirs(vm_data_dir)
        with open(os.path.join(vm_data_dir, '.version'), 'w') as fd:
            fd.write('3')
        os.rmdir(self.shares_dir)
        # build the old layout
        old_root = os.path.join(self.root_dir, 'My Files')
        old_shares = os.path.join(self.root_dir, 'Shared With Me')
        os.makedirs(os.path.join(old_root, 'test_dir'))
        open(os.path.join(old_root, 'test_file'), 'w').close()
        # create a file in the root
        open(os.path.join(self.root_dir, 'test_file'), 'w').close()
        share_path = os.path.join(old_shares, 'Bla from Foo')
        os.makedirs(share_path)
        os.makedirs(os.path.join(share_path, 'test_dir'))
        open(os.path.join(share_path, 'test_file'), 'w').close()
        # fix permissions
        os.chmod(self.root_dir, 0555)
        os.chmod(old_shares, 0555)
        # migrate the data
        self.main = FakeMain(self.root_dir, self.shares_dir,
                             self.data_dir, self.partials_dir)
        self.assertFalse(os.path.exists(old_root))
        self.assertTrue(os.path.exists(old_shares))
        self.assertTrue(os.path.islink(old_shares))
        self.assertEquals(old_shares, self.main.shares_dir_link)
        self.assertTrue(os.path.exists(os.path.join(self.root_dir,
                                                    'test_dir')))
        self.assertTrue(os.path.exists(os.path.join(self.root_dir,
                                                    'test_file')))
        self.assertTrue(os.path.exists(os.path.join(self.root_dir,
                                                    'test_file.u1conflict')))
        self.assertTrue(os.path.exists(share_path))
        self.assertTrue(os.path.exists(os.path.join(share_path, 'test_dir')))
        self.assertTrue(os.path.exists(os.path.join(share_path, 'test_file')))

    def test_3_to_4_with_symlink_in_myfiles(self):
        """upgrade from version 3 to 4"""
        vm_data_dir = os.path.join(self.data_dir, 'vm')
        os.makedirs(vm_data_dir)
        with open(os.path.join(vm_data_dir, '.version'), 'w') as fd:
            fd.write('3')
        os.rmdir(self.shares_dir)
        # build the old layout
        old_root = os.path.join(self.root_dir, 'My Files')
        old_shares = os.path.join(self.root_dir, 'Shared With Me')
        os.makedirs(os.path.join(old_root, 'test_dir'))
        open(os.path.join(old_root, 'test_file'), 'w').close()
        # create a file in the root
        open(os.path.join(self.root_dir, 'test_file'), 'w').close()
        share_path = os.path.join(old_shares, 'Bla from Foo')
        os.makedirs(share_path)
        os.makedirs(os.path.join(share_path, 'test_dir'))
        open(os.path.join(share_path, 'test_file'), 'w').close()
        # create the Shared with Me symlink in My Files
        os.symlink(old_shares, os.path.join(old_root, 'Shared With Me'))
        # fix permissions
        os.chmod(self.root_dir, 0555)
        os.chmod(old_shares, 0555)
        # migrate the data
        self.shares_dir = os.path.join(self.tmpdir, 'shares')
        self.main = FakeMain(self.root_dir, self.shares_dir,
                             self.data_dir, self.partials_dir)
        self.assertFalse(os.path.exists(old_root))
        self.assertTrue(os.path.exists(old_shares))
        self.assertTrue(os.path.islink(old_shares))
        self.assertEquals(old_shares, self.main.shares_dir_link)
        self.assertTrue(os.path.exists(os.path.join(self.root_dir,
                                                    'test_dir')))
        self.assertTrue(os.path.exists(os.path.join(self.root_dir,
                                                    'test_file')))
        self.assertTrue(os.path.exists(os.path.join(self.root_dir,
                                                    'test_file.u1conflict')))
        self.assertTrue(os.path.exists(share_path))
        self.assertTrue(os.path.exists(os.path.join(share_path, 'test_dir')))
        self.assertTrue(os.path.exists(os.path.join(share_path, 'test_file')))
        self.assertEquals(self.main.shares_dir,
                          os.readlink(self.main.shares_dir_link))

    def test_None_to_4(self):
        """upgrade from version None to 4 (possibly a clean start)"""
        VolumeManager.METADATA_VERSION = '4'
        vm_data_dir = os.path.join(self.data_dir, 'vm')
        version_file = os.path.join(vm_data_dir, '.version')
        if os.path.exists(version_file):
            os.remove(version_file)
        os.rmdir(self.shares_dir)
        os.rmdir(self.root_dir)
        old_root = os.path.join(self.root_dir, 'My Files')
        old_shares = os.path.join(self.root_dir, 'Shared With Me')
        # start and check that everything is ok
        self.main = FakeMain(self.root_dir, self.shares_dir,
                             self.data_dir, self.partials_dir)
        self.assertFalse(os.path.exists(old_root))
        self.assertTrue(os.path.exists(old_shares))
        self.assertTrue(os.path.islink(old_shares))
        self.assertEquals(old_shares, self.main.shares_dir_link)
        if os.path.exists(version_file):
            with open(os.path.join(vm_data_dir, '.version'), 'r') as fd:
                self.assertEquals('4', fd.read())
        else:
            self.fail('missing .version file')

    def test_None_to_4_phantom_share_path(self):
        """upgrade from version None to 4 (possibly a clean start)"""
        VolumeManager.METADATA_VERSION = '4'
        vm_data_dir = os.path.join(self.data_dir, 'vm')
        version_file = os.path.join(vm_data_dir, '.version')
        if os.path.exists(version_file):
            os.remove(version_file)
        os.rmdir(self.shares_dir)
        os.rmdir(self.root_dir)
        old_root = os.path.join(self.root_dir, 'My Files')
        old_shares = os.path.join(self.root_dir, 'Shared With Me')
        # start and check that everything is ok
        self.main = FakeMain(self.root_dir, self.shares_dir,
                             self.data_dir, self.partials_dir)
        root = self.main.vm.shares['']
        # set None to the share path
        root.path = None
        self.main.vm.shares['test'] = root
        if os.path.exists(version_file):
            os.remove(version_file)
        self.main.shutdown()
        # check that it's all OK
        self.main = FakeMain(self.root_dir, self.shares_dir,
                             self.data_dir, self.partials_dir)
        self.assertFalse(os.path.exists(old_root))
        self.assertTrue(os.path.exists(old_shares))
        self.assertTrue(os.path.islink(old_shares))
        self.assertEquals(old_shares, self.main.shares_dir_link)
        if os.path.exists(version_file):
            with open(os.path.join(vm_data_dir, '.version'), 'r') as fd:
                self.assertEquals('4', fd.read())
        else:
            self.fail('missing .version file')

    def test_4_to_5(self):
        """test migration from 4 to 5 (broken symlink in the root)"""
        vm_data_dir = os.path.join(self.data_dir, 'vm')
        os.makedirs(vm_data_dir)
        with open(os.path.join(vm_data_dir, '.version'), 'w') as fd:
            fd.write('4')
        # build the new layout with a broken symlink
        shares_link = os.path.join(self.root_dir, 'Shared With Me')
        os.symlink(shares_link, shares_link)
        self.main = FakeMain(self.root_dir, self.shares_dir,
                             self.data_dir, self.partials_dir)
        self.assertEquals(self.main.shares_dir,
                          os.readlink(self.main.shares_dir_link))
