#!/bin/bash

export LANG=C

readonly BASEDIR=$(dirname "$0")/../lib
. $BASEDIR/device.db
. $BASEDIR/config.sh

init_db()
{
	cat <<EOF
-- known devices
CREATE TABLE devices (
	pciid           TEXT  PRIMARY KEY  NOT NULL,
	familyid        TEXT  NOT NULL,

	CHECK ( length(pciid)=8 and length(familyid)=8 )
);

-- known configs
CREATE TABLE configs (
	id              INTEGER  PRIMARY KEY,
	bios_version    TEXT  NOT NULL, -- bios version: N25(N25ET11W)
	igpu            TEXT  NOT NULL, -- igpu family id: 1002130a
	dgpu            TEXT  NOT NULL, -- dgpu family id, can be empty: 10026664
	gmdate          TEXT  NOT NULL, -- gm data: 2014-12-25
	gmimage         TEXT  NOT NULL, -- gm image: sutton-base-trusty-amd64
	product_version TEXT  NOT NULL, -- product_version: ThinkPad E555
	product_name    TEXT, -- product_name: 01234567
	sku             TEXT, -- : Gordon SIT SKU5
	machine         TEXT, -- : Gordon-AMD-2
	devids          TEXT, -- device ids: 030010021308\n038010026664
	modaliases      BLOB,
	raw_pci_data    BLOB,
	raw_usb_data    BLOB,
	raw_dmi_data    BLOB,

	CHECK( length(bios_version)>0
	   AND ( length(igpu)>0)
	   AND date(gmdate) IS NOT NULL
	)
);
EOF
}

insert_devices_2017()
{
	for pid in ${!DEVICE_TABLE_2017[@]}; do
		fid=${DEVICE_TABLE_2017[$pid]}
		cat <<EOF
INSERT INTO devices ( pciid, familyid) VALUES ( "$pid", "$fid" );
EOF
	done
}

insert_devices_2018()
{
	for pid in ${!DEVICE_TABLE_2018[@]}; do
		fid=${DEVICE_TABLE_2018[$pid]}
		cat <<EOF
INSERT INTO devices ( pciid, familyid) VALUES ( "$pid", "$fid" );
EOF
	done
}

insert_devices_2019()
{
	for pid in ${!DEVICE_TABLE_2019[@]}; do
		fid=${DEVICE_TABLE_2019[$pid]}
		cat <<EOF
INSERT INTO devices ( pciid, familyid) VALUES ( "$pid", "$fid" );
EOF
	done
}

insert_configs_2017()
{
	####################
	### ThinkPad     ###
	####################
	__ '2018-01-08' 'Raven3-2' 'ThinkPad X1 Yoga' 'sutton-wilkes-xenial-amd64' '2017'
	__ '2018-01-08' 'Storm3-2' 'ThinkPad X380 Yoga' 'sutton-wilkes-xenial-amd64' '2017'
	__ '2018-01-08' 'Yoda2-2' 'ThinkPad X1 Carbon 6th' 'sutton-wilkes-xenial-amd64' '2017'
	__ '2018-01-10' 'Finn-2' 'ThinkPad x280' 'sutton-wilkes-xenial-amd64' '2017'
	__ '2018-01-18' 'Kylo-3' 'ThinkPad L380 Yoga' 'sutton-wilkes-xenial-amd64' '2017'
	__ '2017-11-26' 'Solo-2' 'ThinkPad L480' 'sutton-wilkes-xenial-amd64' '2017'
	__ '2017-11-26' 'Lando-2' 'ThinkPad L580' 'sutton-wilkes-xenial-amd64' '2017'
	__ '2018-02-28' 'Leia-2' 'ThinkPad 11e 5th Gen' 'sutton-wilkes-xenial-amd64' '2017'

	__ '2018-01-10' 'Windu2-2' 'ThinkPad T480' 'sutton-hamming-xenial-amd64' '2017'
	__ '2018-01-08' 'Tachi2-T-2' 'ThinkPad T580' 'sutton-hamming-xenial-amd64' '2017'
	__ '2018-01-08' 'Tachi2-P-2' 'ThinkPad P52s' 'sutton-hamming-xenial-amd64' '2017'
	__ '2018-01-10' 'Kolar-2' 'ThinkPad T480s' 'sutton-hamming-xenial-amd64' '2017'
	__ '2018-03-19' 'Walter3-3' 'ThinkPad P52' 'sutton-hamming-xenial-amd64' '2017'

	__ '2018-08-10' 'Windu2-AMD-1' 'ThinkPad A485' 'sutton-wilkinson-bionic-amd64' '2017'
	__ '2018-08-10' 'Finn-AMD-1' 'ThinkPad A285' 'sutton-wilkinson-bionic-amd64' '2017'

	__ '2018-08-10' 'Chiron-4' 'ThinkPad P72' 'sutton-mccarthy-bionic-amd64' '2017'
	__ '2018-08-30' 'Padme-4' 'ThinkPad P1' 'sutton-mccarthy-bionic-amd64' '2017'

	####################
	### LENOVO       ###
	####################
	__ '2018-01-26' 'Tata2-AMD-2' 'LENOVO E41-25' 'sutton-wilkes-xenial-amd64' '2017'
	#__ '2018-01-26' 'Tata2-intel-2' 'LENOVO E41-20' 'sutton-wilkes-xenial-amd64' '2017'
	__ '2018-05-16' 'Tata2-intel-3' 'LENOVO E41-20' 'sutton-wilkes-xenial-amd64' '2017'

	####################
	### LENOVO AIO   ###
	####################
	__ '2018-04-16' 'V330-20ICB-1' 'V330-20ICB AIO' 'sutton-wilkes-xenial-amd64' '2017'
	__ '2018-04-26' 'V530-1' 'V530-15ICB' 'sutton-wilkes-xenial-amd64' '2017'
	__ '2018-05-16' 'V530-dgpu' 'V530-15ICB' 'sutton-hamming-xenial-amd64' '2017'

	####################
	### ThinkStation ###
	####################
	__ '2018-04-18' 'P330-2' 'ThinkStation P330' 'sutton-hamming-xenial-amd64' '2017'
	__ '2018-05-28' 'P330tiny-1' 'ThinkStation P330 Tiny' 'sutton-hamming-xenial-amd64' '2017'

	####################
	### ThinkCentre  ###
	####################
	__ '2017-12-30' 'M710e-1' 'ThinkCentre M710e' 'sutton-minsky-xenial-amd64' '2017'

	# M920t-1 is same BIOS with M720t-1, so just add one.
	__ '2018-04-23' 'M720t-1' 'ThinkCentre M720t' 'sutton-wilkes-xenial-amd64' '2017'
	__ '2018-04-23' 'M920t-1' 'ThinkCentre M920t' 'sutton-wilkes-xenial-amd64' '2017'
	__ '2018-06-13' 'M720q-1' 'ThinkCentre M720q' 'sutton-wilkes-xenial-amd64' '2017'
	__ '2018-06-13' 'M920q-1' 'ThinkCentre M920q' 'sutton-wilkes-xenial-amd64' '2017'

	__ '2018-05-23' 'ThinkCentre-M920z-1' 'ThinkCentre M920z' 'sutton-wilkes-xenial-amd64' '2017'
	__ '2018-05-23' 'ThinkCentre-M820z-1' 'ThinkCentre M820z' 'sutton-wilkes-xenial-amd64' '2017'

	__ '2018-10-20' 'ThinkCentre-M715q-RR-2' 'ThinkCentre M715q' 'sutton-wilkinson-bionic-amd64' '2017'
	__ '2018-08-10' 'ThinkCentre-M725s-1' 'ThinkCentre M725s' 'sutton-mccarthy-bionic-amd64' '2017'

}

insert_configs_2018()
{
	####################
	### ThinkPad     ###
	####################
	# R10
	__ '2018-10-28' 'Kylo2-2' 'ThinkPad L390' 'sutton-wilkinson-bionic-amd64' '2018'

	# N2J, Jazz(T490s) and Sidewipe-1(X390) used the same BIOS.
	#__ '2018-12-08' 'Jazz-1' 'ThinkPad T490s' 'sutton-wilkinson-bionic-amd64' '2018'
	__ '2019-04-15' 'Sideswipe-2' 'ThinkPad X390' 'sutton-wilkinson-bionic-amd64' '2018'

	# N2L
	__ '2019-05-20' 'Bumblebee-2' 'ThinkPad X390 Yoga' 'sutton-wilkinson-bionic-amd64' '2018'

	# R0Z, Lando2(L590) and Solo2(L490) used the same BIOS.
	__ '2019-06-12' 'Solo2-3' 'ThinkPad L490' 'sutton-wilkinson-bionic-amd64' '2018'
	#__ '2019-06-12' 'Lando2-1' 'ThinkPad L590' 'sutton-wilkinson-bionic-amd64' '2018'
	#__ '2019-06-21' 'Lando2.0-3' 'ThinkPad L590' 'sutton-wilkinson-bionic-amd64' '2018'

	# R12
	__ '2019-05-14' 'Drift-AMD-3' 'ThinkPad T495' 'sutton-wilkinson-bionic-amd64' '2018'

	# R13, Sideswipe-AMD-1(X395) and Jazz-AMD-1/2(T495s) used the same BIOS.
	__ '2019-06-21' 'Jazz-AMD-3' 'ThinkPad T495s' 'sutton-wilkinson-bionic-amd64' '2018'
	#__ '2019-03-28' 'Jazz-AMD-1' 'ThinkPad T495s' 'sutton-wilkinson-bionic-amd64' '2018'
	#__ '2019-03-28' 'Sideswipe-AMD-1' 'ThinkPad X395' 'sutton-wilkinson-bionic-amd64' '2018'
	#__ '2019-06-21' 'Sideswipe-AMD-2' 'ThinkPad X395' 'sutton-wilkinson-bionic-amd64' '2018'

	# N2H, Ratchet-1(X1 Yoga 4th) and Knockout-1(X1 Carbon 7th) used the same BIOS.
	__ '2019-06-28' 'Ratchet-2' 'ThinkPad X1 Yoga 4th' 'sutton-dijkstra-bionic-amd64' '2018'
	#__ '2019-06-12' 'Knockout-2' 'ThinkPad X1 Carbon 7th' 'sutton-dijkstra-bionic-amd64' '2018'

	# N2I, Dirft(T490), Ironhide-P-2(P53s) and Ironhide-T-2(T590) used the same BIOS.
	__ '2019-04-15' 'Drift-3' 'ThinkPad T490' 'sutton-mccarthy-bionic-amd64' '2018'
	#__ '2019-04-15' 'Ironhide-P-3' 'ThinkPad P53s' 'sutton-mccarthy-bionic-amd64' '2018'
	#__ '2019-04-15' 'Ironhide-T-2' 'ThinkPad T590' 'sutton-mccarthy-bionic-amd64' '2018'

	# N2N, oem-sutton.mccarthy-bailey-meta
	# Nighthawk and Chiron2.0 used the same BIOS.
	__ '2019-09-25' 'Nighthawk-7' 'ThinkPad P53' 'sutton-mccarthy-bionic-amd64' '2018'
	__ '2019-08-16' 'Chiron2.0-3' 'ThinkPad P73' 'sutton-mccarthy-bionic-amd64' '2018'

	# N2O, oem-sutton.mccarthy-bainbridge-meta
	__ '2019-08-07' 'Padme2.0-3' 'ThinkPad P1 Gen2' 'sutton-mccarthy-bionic-amd64' '2018'
	#__ '2019-08-07' 'Padme2.0-X1E-2' 'ThinkPad X1 Extreme Gen2' 'sutton-mccarthy-bionic-amd64' '2018'

	####################
	###   LENOVO     ###
	####################
	# M22
	__ '2018-12-08' 'V530-07ICB-1' 'V530-07ICB' 'sutton-wilkinson-bionic-amd64' '2018'
	# 03T
	__ '2018-12-25' 'V530-15ARR-1' 'V530-15ARR' 'sutton-wilkinson-bionic-amd64' '2018'

	####################
	### ThinkCentre  ###
	####################
	# M28
	__ '2018-12-24' 'M630e-2' 'ThinkCentre M630e' 'sutton-wilkinson-bionic-amd64' '2018'
	# M2A
	__ '2019-06-26' 'M90n-1' 'ThinkCentre M90n' 'sutton-wilkinson-bionic-amd64' '2018'
	# M1U
	# M920x/q
	__ '2019-06-14' 'M920x-1' 'ThinkCentre M920x' 'sutton-wilkinson-bionic-amd64' '2018'
	# M920t/s
	__ '2019-06-14' 'M920s-1' 'ThinkCentre M920s' 'sutton-wilkinson-bionic-amd64' '2018'
	# M720q/t/s
	__ '2019-06-14' 'M720s-1' 'ThinkCentre M720s' 'sutton-wilkinson-bionic-amd64' '2018'
	__ '2019-06-14' 'M720q-1' 'ThinkCentre M720q' 'sutton-wilkinson-bionic-amd64' '2018'
	# M1N
	__ '2019-06-17' 'M820z-1' 'ThinkCentre M820z' 'sutton-wilkinson-bionic-amd64' '2018'
	# M1M
	__ '2019-06-17' 'M920z-1' 'ThinkCentre M920z' 'sutton-wilkinson-bionic-amd64' '2018'
	# M2P
	__ '2019-07-24' 'EPC300-2' 'ThinkCentre EPC300' 'sutton-wilkinson-bionic-amd64' '2018'
	# M2F
	__ '2019-08-08' 'M75q-1-2' 'ThinkCentre M75q-1' 'sutton-wilkinson-bionic-amd64' '2018'
	# M2C
	__ '2019-08-08' 'M75s-1-1' 'ThinkCentre M75s-1' 'sutton-wilkinson-bionic-amd64' '2018'

	####################
	### ThinkStation ###
	####################
	# S05
	__ '2019-07-24' 'P920-REF-4.0' 'ThinkStation P920' 'sutton-mccarthy-bionic-amd64' '2018'
	#__ '2019-04-15' 'P920-6000s-1' 'ThinkStation P920' 'sutton-mccarthy-bionic-amd64' '2018'
	# S03
	__ '2019-07-24' 'P520-REF-4.0' 'ThinkStation P520' 'sutton-mccarthy-bionic-amd64' '2018'
	__ '2019-07-24' 'P520c-REF-4.0' 'ThinkStation P520c' 'sutton-mccarthy-bionic-amd64' '2018'
	#__ '2019-04-15' 'P520-6000-1' 'ThinkStation P520' 'sutton-mccarthy-bionic-amd64' '2018'

	# M1V
	__ '2019-07-24' 'P330-REF-3.0' 'ThinkStation P330' 'sutton-mccarthy-bionic-amd64' '2018'
	#__ '2019-05-15' 'P330-2.0' 'ThinkStation P330' 'sutton-mccarthy-bionic-amd64' '2018'
	# M1U
	__ '2019-05-15' 'P330-Tiny-2.0' 'ThinkStation P330 Tiny' 'sutton-mccarthy-bionic-amd64' '2018'
}

insert_configs_2019()
{
	####################
	### ThinkPad     ###
	####################
	# R15, oem-sutton.wilkinson-ada-meta
	__ '2019-10-21' 'Ares-1' 'ThinkPad L13' 'sutton-wilkinson-bionic-amd64' '2019'
	# R17, oem-sutton.wilkinson-acton-meta
	# Thor-1 and Odin-1 used the same BIOS.
	__ '2020-06-11' 'Thor-2' 'ThinkPad L15' 'sutton-wilkinson-bionic-amd64' '2019'
	#__ '2020-01-12' 'Odin-1' 'ThinkPad L14' 'sutton-wilkinson-bionic-amd64' '2019'

	# R1B, +oem-sutton.newell-adam-meta
	__ '2020-04-02' 'Drift2-AMD-1' 'ThinkPad T14 Gen 1' 'sutton-newell-focal-amd64' '2019'
	# R1C, +oem-sutton.newell-adalbrechta-meta
	__ '2020-04-02' 'Sideswipe2-AMD-1' 'ThinkPad X13 Gen 1' 'sutton-newell-focal-amd64' '2019'
	#__ '2020-04-02' 'Jazz2-AMD-1' 'ThinkPad T14s Gen 1' 'sutton-newell-focal-amd64' '2019'

	# R1D, oem-sutton.wilkinson-addis-meta
	__ '2019-11-13' 'Leia-GLK-Refresh-1' 'ThinkPad 11e 5th Gen' 'sutton-wilkinson-bionic-amd64' '2019'

	# N2U, oem-sutton.dijkstra-abbott-meta
	__ '2020-02-03' 'Bubblebee2.0-1' 'ThinkPad X13 Yoga' 'sutton-dijkstra-bionic-amd64' '2019'

	# N2V, +oem-sutton.simon-adken-meta
	__ '2020-05-14' 'Padme3-P1-2' 'ThinkPad P1 Gen 3' 'sutton-newell-focal-amd64' '2019'
	#__ '2020-05-14' 'Padme3-P1-1' 'ThinkPad P1 Gen 3' 'sutton-newell-focal-amd64' '2019'

	# N2W, oem-sutton.dijkstra-ackley-meta
	__ '2020-04-24' 'Ratchet2.0-2' 'ThinkPad X1 Yoga Gen 5' 'sutton-dijkstra-bionic-amd64' '2019'
	# N2W, +oem-sutton.newell-ace-meta
	__ '2020-05-25' 'Knockout2.0-2' 'ThinkPad X1 Carbon Gen 8' 'sutton-newell-focal-amd64' '2019'
	#__ '2020-01-08' 'Knockout2.0-1' 'ThinkPad X1 Carbon Gen 8' 'sutton-newell-focal-amd64' '2019'

	# N2X, +oem-sutton.simon-bailee-meta
	# Drift2.0-P-1 and Ironhide2.0-P/T-1 used the same BIOS.
	__ '2020-05-11' 'Drift2.0-P-2' 'ThinkPad P14s' 'sutton-simon-focal-amd64' '2019'
	#__ '2020-05-11' 'Ironhide2.0-P-3' 'ThinkPad P15s' 'sutton-simon-focal-amd64' '2019'
	# N2X, oem-sutton.knuth-abbot-meta
	__ '2020-05-11' 'Ironhide2.0-T-2' 'ThinkPad T15' 'sutton-knuth-bionic-amd64' '2019'
	#__ '2020-04-02' 'Drift2.0-T-1' 'ThinkPad T14' 'sutton-knuth-bionic-amd64' '2019'

	# N2Y, oem-sutton.dijkstra-abbey-meta
	# Sideswipe2.0-1 and Jazz2-1 used the same BIOS.
	__ '2020-04-24' 'Sideswipe2.0-2' 'ThinkPad X13 Gen 1' 'sutton-dijkstra-bionic-amd64' '2019'
	#__ '2020-04-24' 'Jazz2.0-2' 'ThinkPad T14s Gen 1' 'sutton-dijkstra-bionic-amd64' '2019'

	# N30, +oem-sutton.simon-addison-meta
	__ '2020-05-26' 'Blackbird-2' 'ThinkPad P17 Gen 1' 'sutton-simon-focal-amd64' '2019'
	#__ '2020-06-11' 'Raptor-6' 'ThinkPad P15 Gen 1' 'sutton-simon-focal-amd64' '2019'
	#__ '2020-05-26' 'Thor-P-3' 'ThinkPad T15p Gen 1' 'sutton-simon-focal-amd64' '2019'

	####################
	###   LENOVO     ###
	####################
	# M2Y, oem-sutton.wilkinson-cairo-meta
	__ '2019-10-09' 'V530-15ICR-1' 'Lenovo V530-15ICR' 'sutton-wilkinson-bionic-amd64' '2019'
	# CJC, oem-sutton.wilkinson-callie-meta
	__ '2019-11-09' 'ThinkBook-14-1' 'Lenovo ZhaoYang E4-IML' 'sutton-wilkinson-bionic-amd64' '2019'
	# EBC, oem-sutton.wilkinson-calli-meta
	__ '2019-12-10' 'Tata3-1' 'Lenovo E41-45' 'sutton-wilkinson-bionic-amd64' '2019'
	# O51, oem-sutton.wilkinson-calder-meta
	__ '2019-12-10' 'AIO340-IGM-Gen2' 'IdeaCentre AIO340-IGM (Gen2)' 'sutton-wilkinson-bionic-amd64' '2019'
	# O4Y, oem-sutton.wilkinson-cal-meta
	__ '2019-12-18' 'AIO-330-1' 'IdeaCentre AIO330' 'sutton-wilkinson-bionic-amd64' '2019'

	####################
	### ThinkCentre  ###
	####################
	# M2S, +oem-sutton.newell-cadee-meta
	__ '2020-04-02' 'M70a-1' 'ThinkCentre M70a' 'sutton-newell-focal-amd64' '2019'
	# M2T, +oem-sutton.newell-cadby-meta
	__ '2020-05-29' 'M90t-1' 'ThinkCentre M90t' 'sutton-newell-focal-amd64' '2019'
	#__ '2020-05-25' 'M80t-1' 'ThinkCentre M80t' 'sutton-newell-focal-amd64' '2019'
	# M2R, +oem-sutton.newell-cadence-meta
	__ '2020-05-08' 'M90a-1' 'ThinkCentre M90a' 'sutton-newell-focal-amd64' '2019'
	# M30, oem-sutton.wilkinson-cable-meta
	__ '2019-10-09' 'M70e-1-1' 'ThinkCentre M720e' 'sutton-wilkinson-bionic-amd64' '2019'
	# M33, +oem-sutton.newell-cadyna-meta
	__ '2020-05-08' 'M75n-1' 'ThinkCentre M75n' 'sutton-newell-focal-amd64' '2019'
	#__ '2020-05-08' 'M75n-Dali-2' 'ThinkCentre M75n Dali' 'sutton-newell-focal-amd64' '2019'
	#__ '2020-05-08' 'M75n-Dali-1' 'ThinkCentre M75n Dali' 'sutton-newell-focal-amd64' '2019'
	#__ '2020-05-08' 'M75n-Picasso-1' 'ThinkCentre M75n Dali' 'sutton-newell-focal-amd64' '2019'

	# M2W, +oem-sutton.newell-cade-meta
	__ '2020-05-25' 'M70q-1' 'ThinkCentre M70q' 'sutton-newell-focal-amd64' '2019'

	# O4H, +oem-sutton.simon-camellia-meta
	__ '2020-05-28' 'V50t-1' 'ThinkCentre V50t 15IMB' 'sutton-simon-focal-amd64' '2019'

	####################
	### ThinkStation ###
	####################
	# M2W, +oem-sutton.simon-baird-meta
	__ '2018-05-28' 'P340-Tiny-2' 'ThinkStation P340 Tiny' 'sutton-simon-focal-amd64' '2019'
}

__()
{
	gmdate="$1"
	name="$2"
	platform="$3"
	gmimage="$4"
	subdir="$5"

	if [ -f data/${subdir}/${name}/bios_version ]; then
		bios_v3="$(cat data/${subdir}/${name}/bios_version | cut -b1-3)"
	else
		echo "no such machine: $name" >&2
		return 1
	fi

	if [ -f data/${subdir}/${name}/lspci.list ]; then
		devices="$(cat data/${subdir}/${name}/lspci.list)"
	else
		cat <<EOF
-- TODO: collect data for $name
EOF
		echo "no such machine: $name" >&2
		return 1
	fi

	id=$(echo "$devices" | compose_cid)
	#config=$(echo "$devices" | compose_config "$gmdate" "$platform")

	igpu=$(echo "$devices" | filter_igpu_devices | compose_cid)
	dgpu=$(echo "$devices" | filter_dgpu_devices | compose_cid)

	cat <<EOF
-- $name
INSERT INTO configs (
	bios_version,
	igpu,
	dgpu,
	gmdate,
	gmimage,
	product_version,
	machine,
	devids
) VALUES (
	"$bios_v3",
	"$igpu",
	"$dgpu",
	"$gmdate",
	"$gmimage",
	"$platform",
	"$name",
	"$devices"
);
EOF
}

rm -f $BASEDIR/config.db

if :; then

	init_db

	echo "BEGIN;"
	insert_configs_2018
	insert_configs_2019
	echo "END;"

	echo "BEGIN;"
	insert_devices_2017
	insert_devices_2018
	insert_devices_2019
	echo "END;"

fi | sqlite3 $BASEDIR/config.db

chmod a-w $BASEDIR/config.db
