/* specter_SYSLOG.c
 *
 * specter output target for real syslog() logging
 *
 * This target produces a syslog entries identical to the LOG target.
 *
 * (C) 2003 by Harald Welte <laforge@gnumonks.org>
 *
 *  20 Jun 2004, Michal Kwiatkowski <ruby@joker.linuxstuff.pl>:
 *      Fixed for specter.
 */

/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 
 *  as published by the Free Software Foundation
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include <stdio.h>
#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <unistd.h>
#include <string.h>
#include <syslog.h>
#include <specter/specter.h>
#include <conffile/conffile.h>
#include "printpkt.h"


#ifndef SYSLOG_FACILITY_DEFAULT
#define SYSLOG_FACILITY_DEFAULT	"kernel"
#endif

#ifndef SYSLOG_LEVEL_DEFAULT 
#define SYSLOG_LEVEL_DEFAULT	"notice"
#endif


static config_entry_t my_config[] = { 
	{ .key = "facility", .type = CONFIG_TYPE_STRING, .options = CONFIG_OPT_NONE,
		.u = { .string = SYSLOG_FACILITY_DEFAULT } },
	{ .key = "level", .type = CONFIG_TYPE_STRING, .options = CONFIG_OPT_NONE,
		.u = { .string = SYSLOG_LEVEL_DEFAULT } },
	{ .key = "ip_options", .type = CONFIG_TYPE_BOOLEAN, .options = CONFIG_OPT_NONE,
		.u = { .value = 0 } },
	{ .key = "tcp_options", .type = CONFIG_TYPE_BOOLEAN, .options = CONFIG_OPT_NONE,
		.u = { .value = 0 } },
	{ .key = "tcp_seq", .type = CONFIG_TYPE_BOOLEAN, .options = CONFIG_OPT_NONE,
		.u = { .value = 0 } },
	{ .key = "mac_header", .type = CONFIG_TYPE_BOOLEAN, .options = CONFIG_OPT_NONE,
		.u = { .value = 0 } },
};

struct my_data {
	int syslog_level;
	int syslog_facility;
	int flags;
};


static int output_syslog(config_entry_t *ce, void *data)
{
	struct my_data *md = data;
	static char buf[4096];

	if (printpkt_print(buf, md->flags))
		syslog(md->syslog_level|md->syslog_facility, buf);

	return 0;
}
		
static void *syslog_init(config_entry_t *ce)
{
	struct my_data *data;

	if (printpkt_init() == -1) {
		specter_log(SPECTER_FATAL, "printpkt module init failure.\n");
		exit(EXIT_FAILURE);
	}

	if ((data = malloc(sizeof(struct my_data))) == NULL) {
		specter_log(SPECTER_FATAL, "Couldn't allocate data: %s.\n",
				strerror(errno));
		return NULL;
	}

	if (!strcmp(GET_CE(ce,0)->u.string, "daemon"))
		data->syslog_facility = LOG_DAEMON;
	else if (!strcmp(GET_CE(ce,0)->u.string, "kernel"))
		data->syslog_facility = LOG_KERN;
	else if (!strcmp(GET_CE(ce,0)->u.string, "local0"))
		data->syslog_facility = LOG_LOCAL0;
	else if (!strcmp(GET_CE(ce,0)->u.string, "local1"))
		data->syslog_facility = LOG_LOCAL1;
	else if (!strcmp(GET_CE(ce,0)->u.string, "local2"))
		data->syslog_facility = LOG_LOCAL2;
	else if (!strcmp(GET_CE(ce,0)->u.string, "local3"))
		data->syslog_facility = LOG_LOCAL3;
	else if (!strcmp(GET_CE(ce,0)->u.string, "local4"))
		data->syslog_facility = LOG_LOCAL4;
	else if (!strcmp(GET_CE(ce,0)->u.string, "local5"))
		data->syslog_facility = LOG_LOCAL5;
	else if (!strcmp(GET_CE(ce,0)->u.string, "local6"))
		data->syslog_facility = LOG_LOCAL6;
	else if (!strcmp(GET_CE(ce,0)->u.string, "local7"))
		data->syslog_facility = LOG_LOCAL7;
	else if (!strcmp(GET_CE(ce,0)->u.string, "user"))
		data->syslog_facility = LOG_USER;
	else {
		specter_log(SPECTER_FATAL, "Unknown facility '%s'.\n",
			  GET_CE(ce,0)->u.string);
		return NULL;
	}

	if (!strcmp(GET_CE(ce,1)->u.string, "emerg"))
		data->syslog_level = LOG_EMERG;
	else if (!strcmp(GET_CE(ce,1)->u.string, "alert"))
		data->syslog_level = LOG_ALERT;
	else if (!strcmp(GET_CE(ce,1)->u.string, "crit"))
		data->syslog_level = LOG_CRIT;
	else if (!strcmp(GET_CE(ce,1)->u.string, "err"))
		data->syslog_level = LOG_ERR;
	else if (!strcmp(GET_CE(ce,1)->u.string, "warning"))
		data->syslog_level = LOG_WARNING;
	else if (!strcmp(GET_CE(ce,1)->u.string, "notice"))
		data->syslog_level = LOG_NOTICE;
	else if (!strcmp(GET_CE(ce,1)->u.string, "info"))
		data->syslog_level = LOG_INFO;
	else if (!strcmp(GET_CE(ce,1)->u.string, "debug"))
		data->syslog_level = LOG_DEBUG;
	else {
		specter_log(SPECTER_FATAL, "Unknown level '%s'.\n",
			GET_CE(ce,1)->u.string);
		return NULL;
	}

	/* set flags for quicker reference later */
	data->flags = 0x0;
	if (GET_CE(ce, 2)->u.value)
		data->flags |= PRINTPKT_IP_OPT;
	if (GET_CE(ce, 3)->u.value)
		data->flags |= PRINTPKT_TCP_OPT;
	if (GET_CE(ce, 4)->u.value)
		data->flags |= PRINTPKT_TCP_SEQ;
	if (GET_CE(ce, 5)->u.value)
		data->flags |= PRINTPKT_MAC_HDR;

	openlog("specter", LOG_NDELAY|LOG_PID, data->syslog_facility);

	return data;
}

static void syslog_fini(config_entry_t *ce, void *data)
{
	closelog();
	free(data);
}

static specter_output_t syslog_op = { 
	.name = "syslog",
	.ce_base = my_config,
	.ce_num = 6,
	.init = &syslog_init,
	.fini = &syslog_fini,
	.output = &output_syslog
};


void _init(void)
{
	if (register_output(&syslog_op, 0) == -1) {
		specter_log(SPECTER_FATAL, "Couldn't register.\n");
		exit(EXIT_FAILURE);
	}
}

