/*
 * This file is part of signon-plugin-google
 *
 * Copyright (C) 2011 Canonical Ltd.
 *
 * Contact: Alberto Mardegan <alberto.mardegan@canonical.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 */

#include "google-plugin-test.h"

#include "plugin.h"
#include "client-login-data.h"

#include <QtTest/QtTest>

#define SOURCE  QLatin1String("CoolCompany-CoolApp-1.0")

using namespace SignOn;

void GooglePluginTest::initTestCase()
{
    qRegisterMetaType<SignOn::SessionData>();
    qRegisterMetaType<SignOn::UiSessionData>();
    qRegisterMetaType<SignOn::Error>();
}

void GooglePluginTest::cleanupTestCase()
{
}

//prepare each test by creating new plugin
void GooglePluginTest::init()
{
    m_testPlugin = new GooglePlugin::Plugin();
    QObject::connect(m_testPlugin, SIGNAL(result(const SignOn::SessionData&)),
                     this, SLOT(result(const SignOn::SessionData&)),
                     Qt::QueuedConnection);
    QObject::connect(m_testPlugin, SIGNAL(error(const SignOn::Error &)),
                     this, SLOT(pluginError(const SignOn::Error &)),
                     Qt::QueuedConnection);
    QObject::connect(m_testPlugin,
                     SIGNAL(userActionRequired(const SignOn::UiSessionData&)),
                     this, SLOT(uiRequest(const SignOn::UiSessionData&)),
                     Qt::QueuedConnection);
    m_errorEmitted = false;
    m_responseEmitted = false;
    m_userActionRequiredEmitted = false;
}

//finnish each test by deleting plugin
void GooglePluginTest::cleanup()
{
    delete m_testPlugin;
    m_testPlugin=NULL;
}

//slot for receiving result
void GooglePluginTest::result(const SignOn::SessionData& data)
{
    qDebug() << "got result";
    m_responseEmitted = true;
    m_response = data;
    m_loop.exit();
}

//slot for receiving error
void GooglePluginTest::pluginError(const SignOn::Error &err)
{
    qDebug() << "got error" << err.type() << ": " << err.message();
    m_errorEmitted = true;
    m_error = err;
    m_loop.exit();
}

//slot for receiving result
void GooglePluginTest::uiRequest(const SignOn::UiSessionData& data)
{
    Q_UNUSED(data);
    qDebug() << "got ui request";
    m_userActionRequiredEmitted = true;
    m_uiRequest = data;
    m_loop.exit();
}

//slot for store
void GooglePluginTest::store(const SignOn::SessionData &data)
{
    qDebug() << "got store";
    m_stored = data;
}

void GooglePluginTest::aborted(QNetworkReply* reply)
{
    qDebug() << "aborted";
    //we should get error code if request was aborted
    qDebug() << reply->error();
    QVERIFY(reply->error());
    m_loop.exit();
}

// TEST CASES
void GooglePluginTest::testPlugin()
{
    qDebug() << "Checking plugin integrity.";
    QVERIFY(m_testPlugin);
    QVERIFY(m_testPlugin->d_ptr);
}

void GooglePluginTest::testPluginType()
{
    qDebug() << "Checking plugin type.";
    QCOMPARE(m_testPlugin->type(), QLatin1String("google"));
}

void GooglePluginTest::testMechanisms()
{
    qDebug() << "Checking plugin mechanisms.";
    QStringList mechs = m_testPlugin->mechanisms();
    QVERIFY(!mechs.isEmpty());
    QVERIFY(mechs.contains("ClientLogin"));
    qDebug() << mechs;
}

void GooglePluginTest::testCancel()
{
    //does nothing as no active connections
    m_testPlugin->cancel();

    //then real cancel

#if 0 // TODO
    QTimer::singleShot(10*1000, &m_loop, SLOT(quit()));
    m_testPlugin->d->m_manager=new QNetworkAccessManager();
    connect(m_testPlugin->d->m_manager, SIGNAL(finished(QNetworkReply*)),
            this,  SLOT(aborted(QNetworkReply*)),Qt::QueuedConnection);

    QNetworkRequest request(QUrl("http://localhost/"));
    m_testPlugin->d->m_reply = m_testPlugin->d->m_manager->post(request, QByteArray("Query String"));
    m_testPlugin->cancel();
    m_loop.exec();
#endif
}

void GooglePluginTest::testProcessValidate()
{
    GooglePlugin::ClientLoginData data;

    QTimer::singleShot(10*1000, &m_loop, SLOT(quit()));

    // Invalid mechanism
    m_testPlugin->process(data, "DummyMechanism");
    m_loop.exec();
    QVERIFY(m_errorEmitted);
    QCOMPARE(m_error.type(), (int)Error::MechanismNotAvailable);

    //try without params
    m_testPlugin->process(data, "ClientLogin");
    m_loop.exec();
    QVERIFY(m_errorEmitted);
    QCOMPARE(m_error.type(), (int)Error::MissingData);
}

void GooglePluginTest::testProcess()
{
    GooglePlugin::ClientLoginData data;

    QTimer::singleShot(10*1000, &m_loop, SLOT(quit()));

    data.setUserName("signongoogleplugintest@gmail.com");
    data.setSource(SOURCE);

    //try without password
    m_testPlugin->process(data, "ClientLogin");
    m_loop.exec();
    QVERIFY(!m_errorEmitted);
    QVERIFY(!m_responseEmitted);
    QVERIFY(m_userActionRequiredEmitted);
    QCOMPARE(m_uiRequest.QueryUserName(), false);
    QCOMPARE(m_uiRequest.QueryPassword(), true);

    m_uiRequest.setSecret("wrong password");
    m_testPlugin->userActionFinished(m_uiRequest);
    m_loop.exec();
    QVERIFY(m_errorEmitted);
}

//end test cases

QTEST_MAIN(GooglePluginTest)

