/*
 * Copyright (c) 2015 Intel Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "config.h"
#include "ISHService.h"
#include "SensorEnumerator.h"
#include "ISHAccelerometer.h"
#include "ISHAmbientTemperature.h"
#include "ISHGameRotationVector.h"
#include "ISHGeomagneticRotationVector.h"
#include "ISHGlanceGesture.h"
#include "ISHGravity.h"
#include "ISHGyroscope.h"
#include "ISHGyroscopeUncalibrated.h"
#include "ISHHeartRate.h"
#include "ISHLight.h"
#include "ISHLinearAcceleration.h"
#include "ISHMagneticField.h"
#include "ISHMagneticFieldUncalibrated.h"
#include "ISHOrientation.h"
#include "ISHPickUpGesture.h"
#include "ISHPressure.h"
#include "ISHProximity.h"
#include "ISHRelativeHumidity.h"
#include "ISHRotationVector.h"
#include "ISHSignificantMotion.h"
#include "ISHStepCounter.h"
#include "ISHStepDetector.h"
#include "ISHTemperature.h"
#include "ISHTiltDetector.h"
#include "ISHWakeGesture.h"
#include "ISHTerminal.h"
#include "ISHMoveDetector.h"
#include "ISHTapping.h"
#include "ISHLift.h"
#include "ISHShake.h"
#include "ISHGestureFlick.h"
#include "ISHPanZoom.h"
#include "ISHSAR.h"
#include "ISHInstantActivity.h"
#include "ISHPhysicalAccelerometer.h"
#include "ISHPhysicalActivity.h"
#include "ISHPDR.h"
#include "ISHUserDefinedGesture.h"
#include "ISHEnergyExpenditure.h"
#include "ISHDevicePosition.h"
#include "ISHHeatIndex.h"
#include "ISHAltitude.h"
#include "ISHRGB.h"
#include "ISHInfrared.h"
#include "ISHInfraredGesture.h"
#include "ISHPulseOximetry.h"
#include "ISHUltraviolet.h"
#include "ISHSurfaceTemperature.h"
#include "ISHBodyTemperature.h"
#ifdef ENABLE_AUDIO_CLASSIFIER
#include "ISHAudioClassifier.h"
#endif
#include "ISHCustomSensor.h"
#include <fcntl.h>
#include <dirent.h>
#include <unistd.h>
#include <signal.h>
#include <errno.h>
#include <string.h>
#include <ctype.h>
#include "Log.h"

#define SENSOR_EVENT_SELECT_NODE  "/dev/sensor-collection"
#define SENSOR_EVENT_NODE         "/sys/bus/platform/devices/sensor_collection/data/sensors_data"
#define DATA_BUFFER_SIZE          (128 * 1024)

#define NOTIFY_FIRMWARE_RESET (1)
static int notification_fd = -1;
void ISH_firmware_reset_signal_handler(int sig)
{
        ALOGD_IF(DEBUG, "%s line: %d ISH firmware reset signal received.", __FUNCTION__, __LINE__);
        int notification = NOTIFY_FIRMWARE_RESET;
        if (notification_fd > 0)
                write(notification_fd, &notification, sizeof(int));
}

ISHService::ISHService()
{
        ishDriverReady = waitingForISHDriverReady();
        if (!ishDriverReady)
                return;

        initializeISHSensorsTable();

        dataBuf = new char[DATA_BUFFER_SIZE];

        dataFd = open(SENSOR_EVENT_NODE, O_RDONLY);
        if (dataFd < 0) {
                ALOGE("%s line: %d open %s error: %s", __FUNCTION__, __LINE__, SENSOR_EVENT_NODE, strerror(errno));
                return;
        }

        dataSelectFd = open(SENSOR_EVENT_SELECT_NODE, O_RDONLY);
        if (dataSelectFd < 0) {
                ALOGE("%s line: %d open %s error: %s", __FUNCTION__, __LINE__, SENSOR_EVENT_NODE, strerror(errno));
                return;
        }

        // read from event sysfs entry before select.
        int ret = 1;
        while (ret != 0) {
                ret = pread(dataFd, dataBuf, DATA_BUFFER_SIZE, 0);
                if (ret < 0) {
                        ALOGE("%s line: %d read %s error: %s", __FUNCTION__, __LINE__, SENSOR_EVENT_NODE, strerror(errno));
                        close(dataFd);
                        dataFd = -1;
                        return;
                }
        }

        ret = pipe(notifyPipeFd);
        if (ret < 0) {
                ALOGE("%s line: %d pipe error: %s", __FUNCTION__, __LINE__, strerror(errno));
                notifyPipeFd[0] = notifyPipeFd[1] = -1;
                return;
        }
        notification_fd = notifyPipeFd[1];
        struct sigaction action;
        memset(&action, 0, sizeof(action));
        action.sa_handler = ISH_firmware_reset_signal_handler;
        sigemptyset(&action.sa_mask);
        action.sa_flags = SA_NODEFER;
        sigaction(SIGUSR1, &action, NULL);
}

#define WAITING_SENSOR_UP_TIME		50
#define MIN_SENSOR_DIR_NUM		7
bool ISHService::waitingForISHDriverReady()
{
	int try_count = 0;
	int file_num = 0;

	while (try_count < WAITING_SENSOR_UP_TIME) {
		DIR *dp;

		if ((dp = opendir(SENSOR_COLLECTION_DIR)) != NULL){
			struct dirent *dirp;
			int num_tmp = 0;

			while ((dirp = readdir(dp)) != NULL)
				num_tmp++;

			closedir(dp);

			if ((num_tmp != file_num) || (num_tmp <= MIN_SENSOR_DIR_NUM))
				file_num = num_tmp;
			else
				break;
		}

		sleep(1);
		try_count++;
	}

        if (try_count != WAITING_SENSOR_UP_TIME && file_num != 0)
                return true;

        if (try_count == WAITING_SENSOR_UP_TIME)
                ALOGE("%s line: %d waiting for ISH sensor collection timeout!", __FUNCTION__, __LINE__);

        if (file_num == 0)
                ALOGE("%s line: %d No ISH sensor found!", __FUNCTION__, __LINE__);

        return false;
}

#define SENSOR_ENTRY_NAME "sensor_%X_def"
void ISHService::initializeISHSensorsTable()
{
        DIR * dirp;
        struct dirent * entry;
        char path[MAX_STRING_LENGTH];
        int fd;
        char buf[LUID_STRING_LENGTH + LUID_STRING_PREFIX_LENGTH + 1] = { 0 };
        char luid_string[LUID_STRING_LENGTH + 1] = { 0 };
        char simplified_luid_string[LUID_STRING_LENGTH + 1] = { 0 };
        int ret;

        dirp = opendir(SENSOR_COLLECTION_DIR);
        if (dirp == NULL) {
                ALOGE("%s line: %d opendir %s error: %s", __FUNCTION__, __LINE__,
                       SENSOR_COLLECTION_DIR, strerror(errno));
                return;
        }

        while ((entry = readdir(dirp)) != NULL) {
                ALOGD_IF(VERBOSE, "(%s %d) entry_name: %s", __FUNCTION__, __LINE__, entry->d_name);
                if (isSensorEntry(entry->d_name)) {
                        unsigned int serial_number;
                        sscanf(entry->d_name, SENSOR_ENTRY_NAME, &serial_number);
                        snprintf(path, MAX_STRING_LENGTH, SENSOR_PROP_VALUE_FORMAT, serial_number, PROPERTY_SENSOR_SERIAL_NUMBER);
                        fd = open(path, O_RDONLY);
                        if (fd < 0) {
                                ALOGE("%s line: %d open %s error: %s", __FUNCTION__, __LINE__, path, strerror(errno));
                                continue;
                        }
                        ret = read(fd, buf, LUID_STRING_LENGTH + LUID_STRING_PREFIX_LENGTH);
                        if (ret < 0) {
                                ALOGE("%s line: %d read %s error: %s", __FUNCTION__, __LINE__, path, strerror(errno));
                                continue;
                        }
                        memcpy(luid_string, buf + LUID_STRING_PREFIX_LENGTH, LUID_STRING_LENGTH);
                        int i = 0;
                        while (luid_string[i]) {
                                luid_string[i] = tolower(luid_string[i]);
                                i++;
                        }
                        memcpy(simplified_luid_string, luid_string, LUID_STRING_LENGTH);
                        for (int i = 4; i < LUID_STRING_LENGTH - 1; i++) {
                                simplified_luid_string[i] = '0';
                        }
                        unsigned int ish_flag;
                        sscanf(simplified_luid_string + (LUID_STRING_LENGTH - 2), "%x", &ish_flag);
                        ish_flag &= 0x03;
                        snprintf(simplified_luid_string +(LUID_STRING_LENGTH - 2), 3, "%02x", ish_flag);
                        ALOGD_IF(DEBUG, "%s line: %d luid_string: %s simplified_luid_string: %s", __FUNCTION__, __LINE__, luid_string, simplified_luid_string);
                        ishSensorsTable.insert(std::make_pair(luid_string, serial_number));
                        ishSensorsLUID.insert(std::make_pair(simplified_luid_string, luid_string));
                        close(fd);
                }
        }

        closedir(dirp);
}

#define SENSOR_ENTRY_PREFIX         "sensor_"
#define SENSOR_ENTRY_POSTFIX        "_def"
#define SENSOR_ENTRY_PREFIX_LENGTH  7
#define SENSOR_ENTRY_POSTFIX_LENGTH 4
bool ISHService::isSensorEntry(const char * entry_name)
{
        size_t len = strlen(entry_name);
        unsigned int i;

        if (len <= SENSOR_ENTRY_PREFIX_LENGTH + SENSOR_ENTRY_POSTFIX_LENGTH)
                return false;
        if (strncmp(entry_name, SENSOR_ENTRY_PREFIX, SENSOR_ENTRY_PREFIX_LENGTH))
                return false;
        if (strncmp(entry_name + (len - SENSOR_ENTRY_POSTFIX_LENGTH), SENSOR_ENTRY_POSTFIX, SENSOR_ENTRY_POSTFIX_LENGTH))
                return false;
        for (i = SENSOR_ENTRY_PREFIX_LENGTH; i < len - SENSOR_ENTRY_POSTFIX_LENGTH; i++)
                if (entry_name[i] < '0' || (entry_name[i] > '9' && entry_name[i] < 'A') ||
                    (entry_name[i] > 'F' && entry_name[i] < 'a') || entry_name[i] > 'f')
                        return false;

        return true;
}

void ISHService::getDataFds(std::queue<int> &dataFdQueue)
{
        dataFdQueue.push(dataSelectFd);
}

bool ISHService::handleEvents(int fd)
{
        ssize_t ret;
        struct sensor_colloection_data_t *package;

        if (fd != dataSelectFd) {
                ALOGE("%s line: %d error data file descriptor: %d except %d", __FUNCTION__, __LINE__, fd, dataFd);
                return false;
        }

        if (dataFd < 0) {
                ALOGE("%s line: %d error data file descriptor: %d", __FUNCTION__, __LINE__, dataFd);
                return false;
        }

        ret = pread(dataFd, dataBuf, DATA_BUFFER_SIZE, 0);
        for(ssize_t position = 0; position < ret; position += (reinterpret_cast<struct sensor_colloection_data_t *>(dataBuf + position))->size) {
                package = reinterpret_cast<struct sensor_colloection_data_t *>(dataBuf + position);
                unsigned int serial_number = package->id & ~PSEUSDO_EVENT_BIT;
                if (mISHSensorSet.count(serial_number) != 1) {
                        ALOGE("%s line: %d sensor(%X) not exists!", __FUNCTION__, __LINE__, serial_number);
                        continue;
                }
                mISHSensorSet[serial_number]->handleEvents(dataBuf + position, package->size);
        }

        return true;
}

#define DEVICE_INITIALIZED   (1 << 0)
#define ADDINFO_INITIALIZED  (1 << 1)
#define PRIVDATA_INITIALIZED (1 << 2)
int ISHService::enumerateSensors(SensorEnumerator * enumerator, const xmlNodePtr ishRoot, std::vector<Sensor *> &sensors)
{
        struct sensor_device_t device;
        sensor_additional_information_t information;
        ish_private_data_t data;
        int index = 0;
        Sensor *sensor = NULL;

        memset(&device, 0, sizeof(device));

        for (xmlNodePtr node = ishRoot->xmlChildrenNode; node != NULL; node = node->next) {
                xmlNodePtr p = node->xmlChildrenNode;

                xmlChar * typeStr = xmlGetProp(node, reinterpret_cast<const xmlChar*>("type"));
                ALOGD_IF(DEBUG, "%s line: %d sensor type: ******************** %s ********************", __FUNCTION__, __LINE__, typeStr);

                unsigned int result = 0;
                while (p != NULL) {
                        if (!strcmp(reinterpret_cast<const char *>(p->name), "device")) {
                                bool ret = enumerator->initializeSensorDevice(p, device, reinterpret_cast<const char *>(typeStr));
                                p = p->next;
                                if (ret)
                                        result |= DEVICE_INITIALIZED;
                        } else if (!strcmp(reinterpret_cast<const char *>(p->name), "additional_information")) {
                                enumerator->initializeSensorAdditionalInformation(p, information);
                                p = p->next;
                                result |= ADDINFO_INITIALIZED;
                        } else if (!strcmp(reinterpret_cast<const char *>(p->name), "private_data")) {
                                bool ret = initializePrivateData(p, data);
                                p = p->next;
                                if (ret)
                                        result |= PRIVDATA_INITIALIZED;
                        }
                }
                xmlFree(typeStr);
                typeStr = NULL;

                if (result != (DEVICE_INITIALIZED | ADDINFO_INITIALIZED | PRIVDATA_INITIALIZED)) {
                        ALOGD_IF(VERBOSE, "%s line: %d name: %d", __FUNCTION__, __LINE__, result);
                        continue;
                }

                sensor = constructSensorByType(device, information, data);

                if (sensor == NULL) {
                        ALOGE("%s line: %d Cannot construct sensor: %s", __FUNCTION__, __LINE__, device.name);
                        continue;
                }

                sensors.push_back(sensor);
                mISHSensorSet[data.serial_number] = sensor;
                mConfiguredSensors[data.luid_string] = sensor;
                index++;
        }
        ALOGD_IF(DEBUG, "%s line: %d ish configured sensors count: %d", __FUNCTION__, __LINE__, index);

        std::multimap<std::string, unsigned int>::iterator it = ishSensorsTable.begin();
        while (it != ishSensorsTable.end()) {
                unsigned int serial_number = it->second;
                if (mConfiguredSensors.count(it->first)) {
                        Sensor* refSensor = mConfiguredSensors[it->first];
                        sensor = constructSensorByReference(refSensor, serial_number);
                } else {
                        sensor = constructCustomerSensor(enumerator, serial_number, it->first.c_str());
                        if (sensor != NULL)
                                mConfiguredSensors[it->first] = sensor;
                }
                if (sensor == NULL) {
                        ALOGE("%s line: %d Cannot construct disconfigured sensor: %s (serial number: %u)", __FUNCTION__, __LINE__, it->first.c_str(), serial_number);
                } else {
                        sensors.push_back(sensor);
                        mISHSensorSet[serial_number] = sensor;
                        index++;
                }
                ishSensorsTable.erase(it++);
        }

        ALOGD_IF(DEBUG, "%s line: %d ish sensors count: %d", __FUNCTION__, __LINE__, index);

        return index;
}

#define PRIVATE_PROPERTY_SIZE 11 // custom-F02A
bool ISHService::initializePrivateData(xmlNodePtr node, ish_private_data_t &data)
{
        xmlNodePtr p = node->xmlChildrenNode;
        xmlChar *str = NULL;

        memset(data.luid_string, 0, LUID_STRING_LENGTH + 1);
        data.sensitivity = SENSITIVITY_DEFAULT;
        while (p != NULL) {
                if (!strcmp(reinterpret_cast<const char *>(p->name), "luid_string")) {
                        str = xmlNodeGetContent(p);
                        if (str == NULL) {
                                ALOGE("%s line: %d luid is empty!", __FUNCTION__, __LINE__);
                                return false;
                        }
                        strncpy(data.luid_string, reinterpret_cast<char *>(str), LUID_STRING_LENGTH);
                        int i = 0;
                        while (data.luid_string[i]) {
                                data.luid_string[i] = tolower(data.luid_string[i]);
                                i++;
                        }
                        xmlFree(str);
                        if (!initializePrivateData(data, true))
                                return false;
                } else if (!strcmp(reinterpret_cast<const char *>(p->name), "sensitivity")) {
                        str = xmlNodeGetContent(p);
                        if (str != NULL) {
                                data.sensitivity = atoi(reinterpret_cast<char *>(str));
                                xmlFree(str);
                        }
                }
                p = p->next;
        }

        return true;
}

bool ISHService::initializePrivateData(ish_private_data_t &data, bool clean)
{
        DIR * dirp;
        struct dirent * entry;
        unsigned int usageId;
        char simplified_luid_string[LUID_STRING_LENGTH + 1] = { 0 };

        memcpy(simplified_luid_string, data.luid_string, LUID_STRING_LENGTH);
        for (int i = 4; i < LUID_STRING_LENGTH - 1; i++) {
                simplified_luid_string[i] = '0';
        }
        unsigned int ish_flag, instance_id;
        sscanf(simplified_luid_string + (LUID_STRING_LENGTH - 4), "%x", &ish_flag);
        instance_id = ish_flag >> 7;
        ish_flag &= 0x03;
        snprintf(simplified_luid_string + (LUID_STRING_LENGTH - 2), 3, "%02x", ish_flag);

        if (ishSensorsLUID.count(simplified_luid_string) == 0) {
                ALOGE("%s line: %d can't find sensor with luid: %s",
                      __FUNCTION__, __LINE__, simplified_luid_string);
                return false;
        }

        std::multimap<std::string, unsigned int>::iterator it = ishSensorsTable.find(data.luid_string);
        if (it == ishSensorsTable.end()) {
                bool found = false;
                std::multimap<std::string, std::string>::iterator it0;
                for (it0 = ishSensorsLUID.equal_range(simplified_luid_string).first; it0 != ishSensorsLUID.equal_range(simplified_luid_string).second; it0++) {
                        unsigned int curr_instance_id;
                        sscanf(it0->second.c_str() + (LUID_STRING_LENGTH - 4), "%x", &curr_instance_id);
                        curr_instance_id >>= 7;
                        if (curr_instance_id == instance_id) {
                                found = true;
                                break;
                        }
                }
                if (!found) {
                        ALOGE("%s line: %d Can't find instance id: %x (%s)", __FUNCTION__, __LINE__, instance_id, data.luid_string);
                        it0 = ishSensorsLUID.equal_range(simplified_luid_string).first;
                }
                memcpy(data.luid_string, it0->second.c_str(), LUID_STRING_LENGTH);
                it = ishSensorsTable.find(it0->second);
        }
        data.luid = luid_from_string(data.luid_string);
        data.serial_number = it->second;
        if (clean)
                ishSensorsTable.erase(it);

        snprintf(path, MAX_STRING_LENGTH, SENSOR_PROP_DIR_FORMAT, data.serial_number);
        dirp = opendir(path);
        if (dirp == NULL) {
                ALOGW("%s line: %d opendir %s error: %s", __FUNCTION__, __LINE__,
                      path, strerror(errno));
                return false;
        }
        while ((entry = readdir(dirp)) != NULL) {
                if (entry->d_type == DT_DIR) {
                        if (strlen(entry->d_name) != PRIVATE_PROPERTY_SIZE)
                                continue;

                        if (strncmp(entry->d_name, "custom-F", 8) != 0)
                                continue;

                        for (int i = 8; i < 11; i++) {
                                if ((entry->d_name[i] < '0') || (entry->d_name[i] > '9' && entry->d_name[i] < 'A') ||
                                    (entry->d_name[i] > 'F' && entry->d_name[i] < 'a') || (entry->d_name[i] > 'f'))
                                        continue;
                        }

                        sscanf(entry->d_name, "custom-%X", &usageId);
                        data.private_property_usage_ids.insert(usageId);
                }
        }
        closedir(dirp);

        return true;
}

Sensor * ISHService::constructSensorByType(const struct sensor_device_t &device, const sensor_additional_information_t &information, const ish_private_data_t &data)
{
        Sensor * sensor = NULL;
        switch (device.type) {
        case SENSOR_TYPE_ACCELEROMETER:
                sensor = new ISHAccelerometer(device, information, data);
                break;
        case SENSOR_TYPE_MAGNETIC_FIELD:
                sensor = new ISHMagneticField(device, information, data);
                break;
        case SENSOR_TYPE_ORIENTATION:
                sensor = new ISHOrientation(device, information, data);
                break;
        case SENSOR_TYPE_GYROSCOPE:
                sensor = new ISHGyroscope(device, information, data);
                break;
        case SENSOR_TYPE_LIGHT:
                sensor = new ISHLight(device, information, data);
                break;
        case SENSOR_TYPE_PRESSURE:
                sensor = new ISHPressure(device, information, data);
                break;
        case SENSOR_TYPE_TEMPERATURE:
                sensor = new ISHTemperature(device, information, data);
                break;
        case SENSOR_TYPE_PROXIMITY:
                sensor = new ISHProximity(device, information, data);
                break;
        case SENSOR_TYPE_GRAVITY:
                sensor = new ISHGravity(device, information, data);
                break;
        case SENSOR_TYPE_LINEAR_ACCELERATION:
                sensor = new ISHLinearAcceleration(device, information, data);
                break;
        case SENSOR_TYPE_ROTATION_VECTOR:
                sensor = new ISHRotationVector(device, information, data);
                break;
        case SENSOR_TYPE_RELATIVE_HUMIDITY:
                sensor = new ISHRelativeHumidity(device, information, data);
                break;
        case SENSOR_TYPE_AMBIENT_TEMPERATURE:
                sensor = new ISHAmbientTemperature(device, information, data);
                break;
        case SENSOR_TYPE_MAGNETIC_FIELD_UNCALIBRATED:
                sensor = new ISHMagneticFieldUncalibrated(device, information, data);
                break;
        case SENSOR_TYPE_GAME_ROTATION_VECTOR:
                sensor = new ISHGameRotationVector(device, information, data);
                break;
        case SENSOR_TYPE_GYROSCOPE_UNCALIBRATED:
                sensor = new ISHGyroscopeUncalibrated(device, information, data);
                break;
        case SENSOR_TYPE_SIGNIFICANT_MOTION:
                sensor = new ISHSignificantMotion(device, information, data);
                break;
        case SENSOR_TYPE_STEP_DETECTOR:
                sensor = new ISHStepDetector(device, information, data);
                break;
        case SENSOR_TYPE_STEP_COUNTER:
                sensor = new ISHStepCounter(device, information, data);
                break;
        case SENSOR_TYPE_GEOMAGNETIC_ROTATION_VECTOR:
                sensor = new ISHGeomagneticRotationVector(device, information, data);
                break;
        case SENSOR_TYPE_HEART_RATE:
                sensor = new ISHHeartRate(device, information, data);
                break;
        case SENSOR_TYPE_TILT_DETECTOR:
                sensor = new ISHTiltDetector(device, information, data);
                break;
        case SENSOR_TYPE_WAKE_GESTURE:
                sensor = new ISHWakeGesture(device, information, data);
                break;
        case SENSOR_TYPE_GLANCE_GESTURE:
                sensor = new ISHGlanceGesture(device, information, data);
                break;
        case SENSOR_TYPE_PICK_UP_GESTURE:
                sensor = new ISHPickUpGesture(device, information, data);
                break;
        case SENSOR_TYPE_TERMINAL:
                sensor = new ISHTerminal(device, information, data);
                break;
        case SENSOR_TYPE_MOVE_DETECTOR:
                sensor = new ISHMoveDetector(device, information, data);
                break;
        case SENSOR_TYPE_TAPPING:
                sensor = new ISHTapping(device, information, data);
                break;
        case SENSOR_TYPE_LIFT:
                sensor = new ISHLift(device, information, data);
                break;
        case SENSOR_TYPE_SHAKE:
                sensor = new ISHShake(device, information, data);
                break;
        case SENSOR_TYPE_GESTURE_FLICK:
                sensor = new ISHGestureFlick(device, information, data);
                break;
        case SENSOR_TYPE_PAN_ZOOM:
                sensor = new ISHPanZoom(device, information, data);
                break;
        case SENSOR_TYPE_SAR:
                sensor = new ISHSAR(device, information, data);
                break;
        case SENSOR_TYPE_INSTANT_ACTIVITY:
                sensor = new ISHInstantActivity(device, information, data);
                break;
        case SENSOR_TYPE_PHYSICAL_ACCELEROMETER:
                sensor = new ISHPhysicalAccelerometer(device, information, data);
                break;
        case SENSOR_TYPE_PHYSICAL_ACTIVITY:
                sensor = new ISHPhysicalActivity(device, information, data);
                break;
        case SENSOR_TYPE_PDR:
                sensor = new ISHPDR(device, information, data);
                break;
        case SENSOR_TYPE_RGB:
                sensor = new ISHRGB(device, information, data);
                break;
        case SENSOR_TYPE_INFRARED:
                sensor = new ISHInfrared(device, information, data);
                break;
        case SENSOR_TYPE_INFRARED_GESTURE:
                sensor = new ISHInfraredGesture(device, information, data);
                break;
        case SENSOR_TYPE_PULSE_OXIMETRY:
                sensor = new ISHPulseOximetry(device, information, data);
                break;
        case SENSOR_TYPE_ULTRAVIOLET:
                sensor = new ISHUltraviolet(device, information, data);
                break;
        case SENSOR_TYPE_SURFACE_TEMPERATURE:
                sensor = new ISHSurfaceTemperature(device, information, data);
                break;
        case SENSOR_TYPE_BODY_TEMPERATURE:
                sensor = new ISHBodyTemperature(device, information, data);
                break;
        case SENSOR_TYPE_AUDIO_CLASSIFIER:
#ifdef ENABLE_AUDIO_CLASSIFIER
                sensor = new ISHAudioClassifier(device, information, data);
#endif
                break;
        case SENSOR_TYPE_USER_DEFINED_GESTURE:
                sensor = new ISHUserDefinedGesture(device, information, data);
                break;
        case SENSOR_TYPE_ENERGY_EXPENDITURE:
                sensor = new ISHEnergyExpenditure(device, information, data);
                break;
        case SENSOR_TYPE_DEIVCE_POSITION:
                sensor = new ISHDevicePosition(device, information, data);
                break;
        case SENSOR_TYPE_HEAT_INDEX:
                sensor = new ISHHeatIndex(device, information, data);
                break;
        case SENSOR_TYPE_ALTITUDE:
                sensor = new ISHAltitude(device, information, data);
                break;
        default:
                sensor = new ISHCustomSensor(device, information, data);
                break;
        }

        return sensor;
}

Sensor * ISHService::constructSensorByReference(const Sensor * refSensor, const uint32_t serial_number)
{
        Sensor * sensor = NULL;

        switch (refSensor->getType()) {
        case SENSOR_TYPE_ACCELEROMETER:
                sensor = new ISHAccelerometer(refSensor, serial_number);
                break;
        case SENSOR_TYPE_MAGNETIC_FIELD:
                sensor = new ISHMagneticField(refSensor, serial_number);
                break;
        case SENSOR_TYPE_ORIENTATION:
                sensor = new ISHOrientation(refSensor, serial_number);
                break;
        case SENSOR_TYPE_GYROSCOPE:
                sensor = new ISHGyroscope(refSensor, serial_number);
                break;
        case SENSOR_TYPE_LIGHT:
                sensor = new ISHLight(refSensor, serial_number);
                break;
        case SENSOR_TYPE_PRESSURE:
                sensor = new ISHPressure(refSensor, serial_number);
                break;
        case SENSOR_TYPE_TEMPERATURE:
                sensor = new ISHTemperature(refSensor, serial_number);
                break;
        case SENSOR_TYPE_PROXIMITY:
                sensor = new ISHProximity(refSensor, serial_number);
                break;
        case SENSOR_TYPE_GRAVITY:
                sensor = new ISHGravity(refSensor, serial_number);
                break;
        case SENSOR_TYPE_LINEAR_ACCELERATION:
                sensor = new ISHLinearAcceleration(refSensor, serial_number);
                break;
        case SENSOR_TYPE_ROTATION_VECTOR:
                sensor = new ISHRotationVector(refSensor, serial_number);
                break;
        case SENSOR_TYPE_RELATIVE_HUMIDITY:
                sensor = new ISHRelativeHumidity(refSensor, serial_number);
                break;
        case SENSOR_TYPE_AMBIENT_TEMPERATURE:
                sensor = new ISHAmbientTemperature(refSensor, serial_number);
                break;
        case SENSOR_TYPE_MAGNETIC_FIELD_UNCALIBRATED:
                sensor = new ISHMagneticFieldUncalibrated(refSensor, serial_number);
                break;
        case SENSOR_TYPE_GAME_ROTATION_VECTOR:
                sensor = new ISHGameRotationVector(refSensor, serial_number);
                break;
        case SENSOR_TYPE_GYROSCOPE_UNCALIBRATED:
                sensor = new ISHGyroscopeUncalibrated(refSensor, serial_number);
                break;
        case SENSOR_TYPE_STEP_DETECTOR:
                sensor = new ISHStepDetector(refSensor, serial_number);
                break;
        case SENSOR_TYPE_STEP_COUNTER:
                sensor = new ISHStepCounter(refSensor, serial_number);
                break;
        case SENSOR_TYPE_GEOMAGNETIC_ROTATION_VECTOR:
                sensor = new ISHGeomagneticRotationVector(refSensor, serial_number);
                break;
        case SENSOR_TYPE_HEART_RATE:
                sensor = new ISHHeartRate(refSensor, serial_number);
                break;
        case SENSOR_TYPE_TERMINAL:
                sensor = new ISHTerminal(refSensor, serial_number);
                break;
        case SENSOR_TYPE_MOVE_DETECTOR:
                sensor = new ISHMoveDetector(refSensor, serial_number);
                break;
        case SENSOR_TYPE_TAPPING:
                sensor = new ISHTapping(refSensor, serial_number);
                break;
        case SENSOR_TYPE_LIFT:
                sensor = new ISHLift(refSensor, serial_number);
                break;
        case SENSOR_TYPE_SHAKE:
                sensor = new ISHShake(refSensor, serial_number);
                break;
        case SENSOR_TYPE_GESTURE_FLICK:
                sensor = new ISHGestureFlick(refSensor, serial_number);
                break;
        case SENSOR_TYPE_PAN_ZOOM:
                sensor = new ISHPanZoom(refSensor, serial_number);
                break;
        case SENSOR_TYPE_SAR:
                sensor = new ISHSAR(refSensor, serial_number);
                break;
        case SENSOR_TYPE_INSTANT_ACTIVITY:
                sensor = new ISHInstantActivity(refSensor, serial_number);
                break;
        case SENSOR_TYPE_PHYSICAL_ACCELEROMETER:
                sensor = new ISHPhysicalAccelerometer(refSensor, serial_number);
                break;
        case SENSOR_TYPE_PHYSICAL_ACTIVITY:
                sensor = new ISHPhysicalActivity(refSensor, serial_number);
                break;
        case SENSOR_TYPE_PDR:
                sensor = new ISHPDR(refSensor, serial_number);
                break;
        case SENSOR_TYPE_RGB:
                sensor = new ISHRGB(refSensor, serial_number);
                break;
        case SENSOR_TYPE_INFRARED:
                sensor = new ISHInfrared(refSensor, serial_number);
                break;
        case SENSOR_TYPE_INFRARED_GESTURE:
                sensor = new ISHInfraredGesture(refSensor, serial_number);
                break;
        case SENSOR_TYPE_PULSE_OXIMETRY:
                sensor = new ISHPulseOximetry(refSensor, serial_number);
                break;
        case SENSOR_TYPE_ULTRAVIOLET:
                sensor = new ISHUltraviolet(refSensor, serial_number);
                break;
        case SENSOR_TYPE_SURFACE_TEMPERATURE:
                sensor = new ISHSurfaceTemperature(refSensor, serial_number);
                break;
        case SENSOR_TYPE_BODY_TEMPERATURE:
                sensor = new ISHBodyTemperature(refSensor, serial_number);
                break;
        case SENSOR_TYPE_ENERGY_EXPENDITURE:
                sensor = new ISHEnergyExpenditure(refSensor, serial_number);
                break;
        case SENSOR_TYPE_DEIVCE_POSITION:
                sensor = new ISHDevicePosition(refSensor, serial_number);
                break;
        case SENSOR_TYPE_HEAT_INDEX:
                sensor = new ISHHeatIndex(refSensor, serial_number);
                break;
        case SENSOR_TYPE_ALTITUDE:
                sensor = new ISHAltitude(refSensor, serial_number);
                break;
        case SENSOR_TYPE_AUDIO_CLASSIFIER:
        case SENSOR_TYPE_USER_DEFINED_GESTURE:
        case SENSOR_TYPE_SIGNIFICANT_MOTION:
        case SENSOR_TYPE_TILT_DETECTOR:
        case SENSOR_TYPE_WAKE_GESTURE:
        case SENSOR_TYPE_GLANCE_GESTURE:
        case SENSOR_TYPE_PICK_UP_GESTURE:
                break;
        default:
                sensor = new ISHCustomSensor(refSensor, serial_number);
                break;
        }

        return sensor;
}

Sensor * ISHService::constructCustomerSensor(SensorEnumerator * enumerator, const uint32_t serial_number, const char * luidString)
{
        Sensor * sensor = NULL;
        std::string name, vendor;
        int fd, ret;

        snprintf(path, MAX_STRING_LENGTH, SENSOR_PROP_VALUE_FORMAT, serial_number, PROPERTY_FRIENDLY_NAME);
        path[MAX_STRING_LENGTH - 1] = 0;
        fd = open(path, O_RDONLY);
        if (fd < 0) {
                ALOGE("%s line: %d Open %s error: %s", __FUNCTION__, __LINE__, path, strerror(errno));
                return NULL;
        }

        ret = read(fd, valueBuffer, MAX_STRING_LENGTH - 1);
        if (ret < 0) {
                ALOGE("%s line: %d read from %s error: %s", __FUNCTION__, __LINE__, path, strerror(errno));
                close(fd);
                return NULL;
        }
        valueBuffer[ret] = 0;
        close(fd);
        name = valueBuffer;

        snprintf(path, MAX_STRING_LENGTH, SENSOR_PROP_VALUE_FORMAT, serial_number, PROPERTY_SENSOR_MANUFACTURER);
        path[MAX_STRING_LENGTH - 1] = 0;
        fd = open(path, O_RDONLY);
        if (fd < 0) {
                ALOGE("%s line: %d Open %s error: %s", __FUNCTION__, __LINE__, path, strerror(errno));
                return NULL;
        }

        ret = read(fd, valueBuffer, MAX_STRING_LENGTH - 1);
        if (ret < 0) {
                ALOGE("%s line: %d read from %s error: %s", __FUNCTION__, __LINE__, path, strerror(errno));
                close(fd);
                return NULL;
        }
        valueBuffer[ret] = 0;
        close(fd);
        vendor = valueBuffer;

        struct sensor_device_t device;
        sensor_additional_information_t information;
        ish_private_data_t data;

        if (!enumerator->initializeSensorDevice(device, name.c_str(), vendor.c_str(), NULL)) {
                ALOGE("%s line: %d initialize sensor device error!", __FUNCTION__, __LINE__);
                return NULL;
        }

        information.expose_to_android = true;
        information.expose_to_csdk = true;
        information.axisMap[AXIS_X] = AXIS_X;
        information.axisMap[AXIS_Y] = AXIS_Y;
        information.axisMap[AXIS_Z] = AXIS_Z;
        information.scale_android.clear();
        information.scale_csdk.clear();
        data.sensitivity = SENSITIVITY_DEFAULT;
        memset(data.luid_string, 0, LUID_STRING_LENGTH + 1);
        strncpy(data.luid_string, luidString, LUID_STRING_LENGTH);
        initializePrivateData(data, false);
        sensor = new ISHCustomSensor(device, information, data);

        return sensor;
}

int ISHService::getNotifyFd()
{
        return notifyPipeFd[0];
}

void ISHService::handleNotify(int notification)
{
        ALOGD_IF(DEBUG, "%s line: %d notification: %d", __FUNCTION__, __LINE__, notification);
        if (notification != NOTIFY_FIRMWARE_RESET) {
                ALOGE("%s line: %d unknown notification: %d", __FUNCTION__, __LINE__, notification);
        }

        if (!waitingForISHDriverReady())
                ALOGE("%s line: %d ISH firmware reset failed!", __FUNCTION__, __LINE__);

        std::map<unsigned int, Sensor *>::iterator it;
        for (it = mISHSensorSet.begin(); it != mISHSensorSet.end(); it++) {
                ISHSensor *sensor = reinterpret_cast<ISHSensor *>(it->second);
                sensor->restartStreaming();
                // Todo: check if the serial number changed if user modified PDT.
        }
}
