/*
 * Copyright (c) 2015 Intel Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "Log.h"
#include <unistd.h>
#include <errno.h>
#include <string.h>
#include "Session.h"

Session::Session(int eventsFd)
{
        this->eventsFd = eventsFd;
        ctrlFd = -1;
        category = ANDROID_STANDARD;
}

Session::~Session()
{
        for (std::map<int, Sensor *>::iterator it = activateSensors.begin(); it != activateSensors.end(); it++) {
                Sensor *sensor = it->second;
                sensor->removeSession(this);
        }
}

void Session::setCategory(session_category_t category)
{
        this->category = category;
}

void Session::setControlFd(int ctrlFd)
{
        this->ctrlFd = ctrlFd;
}

void Session::activateSensor(int handle, Sensor * sensor, bool activate)
{
        if (!activate) {
                if (activateSensors.count(handle) == 0) {
                        ALOGW("%s line: %d Sensor(handle) %d not activated!", __FUNCTION__, __LINE__, handle);
                        return;
                }
                int count = activateSensors.erase(handle);
                if (count != 1)
                        ALOGE("%s line: %d  Session does not exist! count: %d", __FUNCTION__, __LINE__, count);
                return;
        }

        if (activateSensors.count(handle) > 0) {
                ALOGE("%s line: %d Sensor(handle) %d already activated!", __FUNCTION__, __LINE__, handle);
                return;
        }

        if (category == AWARE_SERVICE) {
                if (activateSensors.size() > 0) {
                        ALOGE("%s line: %d Only one sensor can be activated in one session!", __FUNCTION__, __LINE__);
                        return;
                }
        }

        activateSensors[handle] = sensor;
}

bool Session::sensorIsActivated(int handle)
{
        if (activateSensors.count(handle) > 0) {
                return true;
        }

        return false;
}

void Session::dispatchEvents(size_t size, void * buf)
{
        ssize_t ret, written = 0;
        if (buf == NULL) {
                ALOGE("%s line: %d buf is NULL!", __FUNCTION__, __LINE__);
                return;
        }

        while (written < size) {
                ret = write(eventsFd, (reinterpret_cast<char *>(buf)) + written, size - written);
                if (ret < 0) {
                        ALOGE("%s line: %d Write to fd %d error: %s", __FUNCTION__, __LINE__, eventsFd, strerror(errno));
                        return;
                }
                written += ret;
        }
}
