/*
 * Copyright (C) 2007 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#ifndef _MUTEX_H_
#define _MUTEX_H_

#include <pthread.h>

class Mutex {
public:
        Mutex();
        ~Mutex();
        
        // lock or unlock the mutex
        int lock();
        int unlock();

        // lock if possible; returns 0 on success, error otherwise
        int tryLock();

        // Manages the mutex automatically. It'll be locked when Autolock is
        // constructed and released when Autolock goes out of scope.
        class Autolock {
        public:
                inline Autolock(Mutex& mutex) : mLock(mutex)  { mLock.lock(); }
                inline Autolock(Mutex* mutex) : mLock(*mutex) { mLock.lock(); }
                inline ~Autolock() { mLock.unlock(); }
        private:
                Mutex& mLock;
        };

private:
        // A mutex cannot be copied
        Mutex(const Mutex&);
        Mutex& operator = (const Mutex&);

        pthread_mutex_t mMutex;
};


inline Mutex::Mutex() {
        pthread_mutex_init(&mMutex, NULL);
}

inline Mutex::~Mutex() {
        pthread_mutex_destroy(&mMutex);
}

inline int Mutex::lock() {
        return pthread_mutex_lock(&mMutex);
}
inline int Mutex::unlock() {
        return pthread_mutex_unlock(&mMutex);
}
inline int Mutex::tryLock() {
        return pthread_mutex_trylock(&mMutex);
}

// ---------------------------------------------------------------------------

/*
 * Automatic mutex.  Declare one of these at the top of a function.
 * When the function returns, it will go out of scope, and release the
 * mutex.
 */

typedef Mutex::Autolock AutoMutex;

#endif // _MUTEX_H_
