#
# This file is part of Python Terra
# Copyright (C) 2007-2009 Instituto Nokia de Tecnologia
# Contact: Renato Chencarek <renato.chencarek@openbossa.org>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#

import sys
import os
import os.path
import shutil
from ConfigParser import ConfigParser
from zipfile import ZipFile

from terra_config import TerraConfig
from plugin_config import PluginConfig, PluginConfigManager
from cli_utils import die, process_cmdline


class FakeFile(object):
    def __init__(self, content):
        self.linelist = content.split("\n")

    def readline(self, length=None):
        if not self.linelist:
            return ""

        line = self.linelist.pop(0)
        if length is not None and 0 <= length <= len(line):
            self.linelist.insert(0, line[length:])
            line = line[:length]
        else:
            line = "%s\n" % line

        return line


def _fix_path(path):
    if path.endswith("/"):
        return path[:-1]
    return path


def _is_valid_collection(path):
    return os.path.exists(path) and \
        (os.path.isdir(path) or (os.path.isfile(path) and
                                 path.endswith(".zip")))


def _validate_collection(path):
    if _is_valid_collection(path):
        return
    die("collection must be a .zip file or directory")


def _process_cmdline():
    parser, opts, args = process_cmdline()
    cfg = TerraConfig(opts.config_file)

    return cfg, args, parser


def _read_plugin(cfg, plg):
    modname = cfg.get(plg, "modname")
    if "." not in modname and \
       "/" not in modname:
        raise ValueError("plugin %r doesn't have a namespace for its module %r "
                         "(e.g. plugin/module or plugin.module)" %
                         (plg, modname))

    tmp = cfg.get(plg, "filter_map").replace(" ", "").split("\n")
    filter_map = [tuple(i.split("-")) for i in tmp]

    rank = int(cfg.get(plg, "rank"))
    enabled = cfg.get(plg, "enabled") not in ("False", "false")

    return [plg, modname, filter_map, rank, enabled]


def _load_plugins_info(path):
    plugins = []
    filename = "plugins.info"

    try:
        if path.endswith(".zip"):
            prefix = os.path.splitext(os.path.basename(path))[0]
            pinfo = "%s/%s" % (prefix, filename)
            fobj = FakeFile(ZipFile(path).read(pinfo))
        else:
            # path should be a directory here
            pinfo = "%s/%s" % (path, filename)
            fobj = open(pinfo)

        cfg = ConfigParser()
        cfg.optionxform = str
        cfg.readfp(fobj)

        for plg in cfg.sections():
            plugins.append(_read_plugin(cfg, plg))
    except Exception, e:
        die("couldn't read plugins information from %s: %s" % (pinfo, e))

    return plugins


def _addto_config_file(path, plugins, plugins_cfg):
    try:
        pcm = PluginConfigManager(plugins_cfg)
        pcm.load()

        moddir = os.path.basename(path)
        for name, modname, filter_map, rank, enabled in plugins:
            cfg = PluginConfig(name, moddir, modname, filter_map, rank, enabled)
            pcm.add(cfg)

        pcm.save()
    except Exception, e:
        die("couldn't update config file %r: %s" % (plugins_cfg, e))


def _delfrom_config_file(plugins, plugins_cfg):
    try:
        pcm = PluginConfigManager(plugins_cfg)
        pcm.load()

        for info in plugins:
            pcm.remove(info[0])

        pcm.save()
    except Exception, e:
        die("couldn't update config file %r: %s" % (plugins_cfg, e))


def _copy_collection(path, plugins_dir):
    try:
        if os.path.isfile(path):
            shutil.copy2(path, plugins_dir)
        else:
            dst = os.path.join(plugins_dir, os.path.basename(path))
            shutil.copytree(path, dst)
    except Exception, e:
        die("couldn't copy collection to default plugins directory %r: %s" %
             (plugins_dir, e))


def _remove_collection(path):
    try:
        if os.path.isfile(path):
            os.remove(path)
        else:
            shutil.rmtree(path)
    except Exception, e:
        die("couldn't remove collection %r: %s" % (path, e))


def install_collection():
    cfg, args, parser = _process_cmdline()
    if len(args) != 1:
        parser.error("expected one collection but received %r" % args)

    path = _fix_path(args[0])
    _validate_collection(path)

    plugins = _load_plugins_info(path)
    _copy_collection(path, cfg.plugins_dir)
    _addto_config_file(path, plugins, cfg.plugins_cfg)


def uninstall_collection():
    cfg, args, parser = _process_cmdline()
    if len(args) != 1:
        parser.error("expected one collection but received %r" % args)

    path = os.path.join(cfg.plugins_dir, os.path.basename(args[0]))
    _validate_collection(path)

    plugins = _load_plugins_info(path)
    _delfrom_config_file(plugins, cfg.plugins_cfg)
    _remove_collection(path)


def rescan_collections():
    cfg, args, parser = _process_cmdline()
    if len(args) != 0:
        parser.error("unexpected arguments %r" % args)

    if os.path.exists(cfg.plugins_cfg):
        os.remove(cfg.plugins_cfg)

    for p in os.listdir(cfg.plugins_dir):
        path = os.path.join(cfg.plugins_dir, p)
        if not _is_valid_collection(path):
            sys.stderr.write("rescan_collections: skipping invalid "
                             "collection %r ...\n" % path)
            continue

        plugins = _load_plugins_info(path)
        _addto_config_file(path, plugins, cfg.plugins_cfg)
