#
# This file is part of Python Terra
# Copyright (C) 2007-2009 Instituto Nokia de Tecnologia
# Contact: Renato Chencarek <renato.chencarek@openbossa.org>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#

import os
import shutil
import logging

from singleton import Singleton
from terra_config import TerraConfig
from plugin_config import PluginConfigManager, get_user_plugins_cfg
from plugin_manager import PluginManager
from model import ModelStatus


log = logging.getLogger("terra.core.manager")


class Manager(Singleton):
    """Terra central manager.

    This is responsible for dealing with:

     - Terra central configuration
     - Terra plugin system
     - Terra status notifiers

    @note: depending on the application the manager can be used to store
    references to global information needed by different plugins.
    """
    def __init__(self, config_file):
        Singleton.__init__(self)
        self._load_config(config_file)
        self._setup_plugin_manager()
        self._status_notifiers = {}     # maps filters to notifiers

    def _load_config(self, config_file):
        self.terra_config = TerraConfig(config_file)
        log.info("loaded terra config %r", self.terra_config.config_file)

    def _setup_plugin_manager(self):
        user_plugins_cfg = get_user_plugins_cfg(self.terra_config)

        if not os.path.exists(user_plugins_cfg):
            shutil.copy2(self.terra_config.plugins_cfg, user_plugins_cfg)

        try:
            pcm = PluginConfigManager(user_plugins_cfg)
            pcm.load()
            pcm.sync(self.terra_config.plugins_cfg)
        except Exception, e:
            log.error("couldn't load and sync plugins config file: %s", e)
            raise

        self.plugin_manager = PluginManager(self.terra_config.plugins_dir, pcm)
        log.info("plugin manager up and running")

    def add_status_notifier(self, notifier):
        """Register a status notifier.

        @param notifier: instance of ModelStatus
        """
        if not isinstance(notifier, ModelStatus):
            raise TypeError("notifier should be instance of ModelStatus")

        self._status_notifiers[notifier.terra_type] = notifier

    def remove_status_notifier(self, notifier):
        """Remove a status notifier.

        @param notifier: instance of ModelStatus
        """
        if not isinstance(notifier, ModelStatus):
            raise TypeError("notifier should be instance of ModelStatus")

        try:
            del self._status_notifiers[notifier.terra_type]
        except:
            pass

    def get_status_notifier(self, suffix):
        """Get a status notifier.

        @param suffix: suffix of notifier's terra_type (i.e. Battery)
        """
        if not isinstance(suffix, basestring):
            raise TypeError("suffix should be a string")

        if suffix.startswith("Model/Status/"):
            filter = suffix
        else:
            filter = "Model/Status/%s" % suffix

        if filter not in self._status_notifiers:
            return None
        return self._status_notifiers[filter]

    def get_classes(self, filter):
        """Get classes implementing filter.

        @param filter: filter string
        @return: list of classes implementing C{filter}
        @raise ValueError: when no class was found with this filter.
        """
        log.debug("looking for classes with filter %r", filter)
        try:
            classes = self.plugin_manager.load(filter)
        except Exception, e:
            log.debug_error("[%s] while loading classes for filter %r",
                            e, filter, exc_info=True)
            raise

        if not classes:
            raise ValueError("No plugin found with filter %r" % filter)

        return classes

    def get_classes_by_regexp(self, regexp, unique=False):
        """Get classes with filters that match a specific regular expression.

        @param regexp: regular expression string
        @param unique: True if only one class per filter should be returned and
            False otherwise
        @return: list of classes with filters matching C{regexp}
        @raise ValueError: when no class was found with this filter.
        """
        log.debug("looking for classes with filter matching regexp %r", regexp)
        try:
            classes = self.plugin_manager.load_by_regexp(regexp, unique=unique)
        except Exception, e:
            log.debug_error("[%s] while loading classes by regexp %r",
                            e, regexp, exc_info=True)
            raise

        if not classes:
            raise ValueError("No plugin found with filter %r" % filter)

        return classes

    def get_class(self, filter):
        """Get first class found that implements a specific filter.

        @param filter: filter string
        @return: first class found that implements C{filter}
        @raise ValueError: when no class was found with this filter.
        """
        log.debug("looking for first class with filter %r", filter)
        classes = self.get_classes(filter)
        if classes:
            return classes[0]
