#!/usr/bin/env python
#
# This file is part of Python Download Manager
# Copyright (C) 2007-2009 Instituto Nokia de Tecnologia
# Author: Kenneth Christiansen <kenneth.christiansen@openbossa.org>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#

import pygtk
pygtk.require('2.0')
import gtk
import gtk.glade
import gobject
import os
import dbus
import dbus.glib
from downloadmanager.client import DownloadManager, DownloadItem
from downloadmanager.constants import DownloadState

(
    STATE,
    FILE,
    RATE,
    RECEIVED,
    TOTAL,
    PROGRESS,
    COMPLETED,
    LOCATION,
    URI,
    TIMEOUTID,
    OBJECT
) = xrange(11)

TIMEOUT = 1000

class PyGet:
    def __init__(self):
        self.status_icons = {}
        self.init_icons()

        home_dir = os.path.expanduser('~')
        self.config_dir = os.path.join(home_dir, '.gnome2', 'pyget')

        wtree = gtk.glade.XML('pyget.glade')
        self.window = wtree.get_widget('pyget_window')
        self.about = wtree.get_widget('aboutdialog')

        # Add download dialog
        self.add_download_dialog = wtree.get_widget('add_download_dialog')
        self.add_download_dialog.add_buttons(gtk.STOCK_CANCEL,
                                             gtk.RESPONSE_CANCEL,
                                             gtk.STOCK_OK,
                                             gtk.RESPONSE_OK)
        self.add_download_dialog.set_default_response(gtk.RESPONSE_OK)
        self.uri_entry = wtree.get_widget('uri_entry')
        self.downdir_chooser = wtree.get_widget('downdir_chooser')

        # Info dialog
        self.infodialog = wtree.get_widget('infodialog')
        self.infodialog.add_buttons(gtk.STOCK_CLOSE, gtk.RESPONSE_CLOSE)
        self.uri_label = wtree.get_widget('uri_label')
        self.local_label = wtree.get_widget('local_label')

        # Widgets
        self.remove_button = wtree.get_widget('remove_button')
        self.play_button = wtree.get_widget('play_button')
        self.pause_button = wtree.get_widget('pause_button')
        self.remove_item = wtree.get_widget('remove_item')
        self.play_item = wtree.get_widget('play_item')
        self.pause_item = wtree.get_widget('pause_item')
        self.play_item2 = wtree.get_widget('play_item2')
        self.pause_item2 = wtree.get_widget('pause_item2')
        self.remove_item2 = wtree.get_widget('remove_item')
        self.downloads_menu = wtree.get_widget('downloads_menu')

        # Downloads treeview
        self.downloads_treeview = wtree.get_widget('downloads_treeview')
        self.downloads_model = gtk.ListStore(int, str, float, float,
                                             float, float, bool,
                                             str, str, int,
                                             gobject.TYPE_PYOBJECT)
        self.downloads_treeview.set_model(self.downloads_model)

        # Columns
        renderer = gtk.CellRendererPixbuf()
        column = gtk.TreeViewColumn('', renderer)

        def _draw_status_icon(column, renderer, model, iter):
            renderer.set_property('pixbuf',
                                  self.status_icons[model[iter][STATE]])
            return

        column.set_cell_data_func(renderer, _draw_status_icon)

        self.downloads_treeview.append_column(column)

        renderer = gtk.CellRendererText()
        column = gtk.TreeViewColumn('Download', renderer, text=FILE)
        column.set_expand(True)

        self.downloads_treeview.append_column(column)

        renderer = gtk.CellRendererText()
        column = gtk.TreeViewColumn('Rate', renderer)
        column.set_cell_data_func(renderer, self.show_formated_value, RATE)
        column.set_expand(False)

        self.downloads_treeview.append_column(column)

        renderer = gtk.CellRendererText()
        column = gtk.TreeViewColumn('Received', renderer)
        column.set_cell_data_func(renderer, self.show_formated_value, RECEIVED)
        column.set_expand(False)

        self.downloads_treeview.append_column(column)

        renderer = gtk.CellRendererText()
        column = gtk.TreeViewColumn('Total', renderer)
        column.set_cell_data_func(renderer, self.show_formated_value, TOTAL)
        column.set_expand(False)

        self.downloads_treeview.append_column(column)

        renderer = gtk.CellRendererProgress()
        column = gtk.TreeViewColumn('Progress', renderer, value=PROGRESS)
        column.set_expand(False)

        self.downloads_treeview.append_column(column)

        # Drag-n-Drop (in fact, only drop)
        self.downloads_treeview.drag_dest_set(gtk.DEST_DEFAULT_MOTION |
                                              gtk.DEST_DEFAULT_HIGHLIGHT |
                                              gtk.DEST_DEFAULT_DROP,
                                              [('text/plain', 0, 1),
                                               ('text/x-uri', 0, 1)],
                                              gtk.gdk.ACTION_DEFAULT |
                                              gtk.gdk.ACTION_COPY)
        self.downloads_treeview.connect('drag-data-received',
                                        self.place_data_received)


        # TrayIcon
        self.trayicon = gtk.status_icon_new_from_icon_name('gtk-go-down')
        self.trayicon.set_tooltip('PyGet')
        #self.trayicon.connect('popup_menu', self.trayaction_menu)
        self.trayicon.connect('activate', self.trayaction_activate)

        wtree.signal_autoconnect(self)
        self.window.show_all()

        self.manager = None
        self.downloads = {}

        self.load_state()


    # Callbacks ---------------------------------------------------------------
    def on_add_cb(self, widget):
        self.uri_entry.set_text('')
        self.add_download_dialog.run()

    def on_add_download_dialog_response(self, widget, response):
        if response == gtk.RESPONSE_OK:
            uri = self.uri_entry.get_text().strip()
            if uri == '':
                return

            if self.manager is None:
                try:
                    self.manager = DownloadManager()
                except dbus.exceptions.DBusException:
                    self.message('DownloadManager is not running!')

            location = self.downdir_chooser.get_current_folder()
            self.add_download(uri, location)

        self.add_download_dialog.hide()

    def on_add_download_dialog_close(self, widget, data=None):
        self.add_download_dialog.hide()
        return True

    def on_remove_cb(self, widget):
        path, column = self.downloads_treeview.get_cursor()
        down_item = self.downloads_model[path][OBJECT]
        down_item.cancel()

        it = self.downloads_model.get_iter(path)
        self.downloads_model.remove(it)

        path, column = self.downloads_treeview.get_cursor()
        has_selected = path is not None
        self.remove_button.set_sensitive(has_selected)
        self.remove_item.set_sensitive(has_selected)

    def on_play_cb(self, widget):
        path, column = self.downloads_treeview.get_cursor()
        down_item = self.downloads_model[path][OBJECT]
        down_item.start(True)

    def on_pause_cb(self, widget):
        path, column = self.downloads_treeview.get_cursor()
        down_item = self.downloads_model[path][OBJECT]
        down_item.pause()

    def on_downloads_treeview_cursor_changed(self, widget):
        self.set_ui_state()
        self.remove_button.set_sensitive(True)
        self.remove_item.set_sensitive(True)

    def on_downloads_treeview_button_release_event(self, treeview, event):
        if event.button != 3:
            return False

        try:
            path, col, cellx, celly = treeview.get_path_at_pos(int(event.x),
                                                               int(event.y))
            self.downloads_menu.popup(None, None, None,
                                      event.button, event.time)
        except:
            pass

        return True

    def on_info_cb(self, widget, path=None, column=None):
        if path is None:
            path, column = self.downloads_treeview.get_cursor()

        row = self.downloads_model[path]
        self.uri_label.set_text(row[URI])
        self.local_label.set_text(os.path.join(row[LOCATION], row[FILE]))
        self.infodialog.show()

    def on_infodialog_close(self, widget, data=None):
        self.infodialog.hide()
        return True

    def on_about_cb(self, widget):
        self.about.show()

    def on_aboutdialog_close(self, widget, data=None):
        self.about.hide()
        return True

    def trayaction_activate(self, status_icon):
        if self.window.get_property('visible'):
            self.window.hide()
        else:
            self.window.present()


    def on_quit_cb(self, widget, event=None):
        self.save_state()
        gtk.main_quit()

    # Download callbacks ------------------------------------------------------
    def download_state_changed_cb(self, download_item, state, status=-2):
        row = self.downloads[download_item]
        row[STATE] = state
        self.set_ui_state()

        if state == DownloadState.IN_PROGRESS:
            row[TIMEOUTID] = gobject.timeout_add(TIMEOUT,
                                                 self.download_progress_cb,
                                                 download_item)
        elif state == DownloadState.COMPLETED:
            row[PROGRESS] = 100.0
            self.dispose_download(download_item)

        elif state == DownloadState.CANCELLED:
            self.dispose_download(download_item)

        elif state == DownloadState.EXCEPTION:
            self.message('Download error')
            self.dispose_download(download_item)

    def download_progress_cb(self, download_item):
        bytes_written, size = download_item.get_progress()
        row = self.downloads[download_item]

        if size == None:
            row[TOTAL] = -1.0
            row[RATE] = 0.0
            row[PROGRESS] = 0.0
            #set_property('pulse', True)

        else:
            percent = bytes_written * 100.0 / size
            row[TOTAL] = size
            row[RATE] = bytes_written - row[RECEIVED]
            row[PROGRESS] = percent

        row[RECEIVED] = bytes_written

        return bytes_written != size

    # Auxiliary Methods -------------------------------------------------------
    def init_icons(self):
        icon_theme = gtk.icon_theme_get_default()

        self.status_icons[DownloadState.NOT_STARTED] = None

        icon_info = icon_theme.lookup_icon('gtk-go-down', gtk.ICON_SIZE_MENU,
                                           gtk.ICON_LOOKUP_USE_BUILTIN)
        self.status_icons[DownloadState.IN_PROGRESS] = icon_info.load_icon()

        icon_info = icon_theme.lookup_icon('gtk-media-pause', gtk.ICON_SIZE_MENU,
                                           gtk.ICON_LOOKUP_USE_BUILTIN)
        self.status_icons[DownloadState.PAUSED] = icon_info.load_icon()

        icon_info = icon_theme.lookup_icon('gtk-stop', gtk.ICON_SIZE_MENU,
                                           gtk.ICON_LOOKUP_USE_BUILTIN)
        self.status_icons[DownloadState.CANCELLED] = icon_info.load_icon()

        icon_info = icon_theme.lookup_icon('gtk-stop', gtk.ICON_SIZE_MENU,
                                           gtk.ICON_LOOKUP_USE_BUILTIN)
        self.status_icons[DownloadState.EXCEPTION] = icon_info.load_icon()

        icon_info = icon_theme.lookup_icon('gtk-yes', gtk.ICON_SIZE_DIALOG,
                                           gtk.ICON_LOOKUP_USE_BUILTIN)
        self.status_icons[DownloadState.COMPLETED] = icon_info.load_icon()

    def add_download(self, uri, location):
        if self.manager is None:
            return

        try:
            if not os.path.exists(location):
                os.mkdir(location)
        except OSError, oe:
            print oe
            print dir(oe)
            self.message(oe.message)

        filename = os.path.basename(uri)
        filepath = os.path.join(location, filename)

        try:
            file_in = open(filepath, 'r')
            is_file = True
        except IOError, e:
            is_file = False

        if not is_file:
            down_item = self.manager.add(uri, filepath)
            down_it = self.downloads_model.append([DownloadState.NOT_STARTED,
                                                   filename, 0.0,
                                                   0.0, 0.0, 0.0,
                                                   False, location,
                                                   uri, -1, down_item])
            self.downloads[down_item] = self.downloads_model[down_it]
            down_item.on_state_changed_add(self.download_state_changed_cb)
            down_item.start(True)

        else:
            pass

    def save_state(self):
        if not os.path.exists(self.config_dir):
            os.mkdir(self.config_dir)

        print self.config_dir

        fd = open(os.path.join(self.config_dir, 'last_dir'), 'w')
        fd.write(self.downdir_chooser.get_current_folder())
        fd.close()

        fd = open(os.path.join(self.config_dir, 'downloads'), 'w')
        for row in self.downloads_model:
            fd.write('%d ; %s ; %s\n' % (row[STATE], row[URI], row[LOCATION]))
        fd.close()

    def load_state(self):
        if not os.path.exists(self.config_dir):
            return

        fd = open(os.path.join(self.config_dir, 'last_dir'), 'r')
        down_dir = fd.read().strip()
        fd.close()

        try:
            self.downdir_chooser.set_current_folder(down_dir)
        except:
            home_dir = os.path.expanduser('~')
            self.downdir_chooser.set_current_folder(home_dir)

        fd = open(os.path.join(self.config_dir, 'downloads'), 'r')
        for line in fd:
            state, uri, location = line.strip().split(' ; ')
            state = int(state)
            filename = os.path.basename(uri)
            filepath = os.path.join(location, filename)

            if state != DownloadState.COMPLETED:
                self.add_download(uri, location)

            elif os.path.exists(filepath):
                received = float(os.path.getsize(filepath))

                if state == DownloadState.COMPLETED:
                    progress = 100.0
                    total = received
                else:
                    progress = 0.0
                    total = 0.0

                down_it = self.downloads_model.append([state, filename,
                                                       0.0, received, total,
                                                       progress, False,
                                                       location, uri, -1,
                                                       None])

        fd.close()

    def place_data_received(self, widget, ct, x, y, selection, info, timestamp):
        self.uri_entry.set_text(selection.data)
        self.add_download_dialog.run()

    def dispose_download(self, download_item):
        timeout_id = self.downloads[download_item][TIMEOUTID]

        if timeout_id > -1:
            gobject.source_remove(timeout_id)

        self.download_progress_cb(download_item)

    def set_ui_state(self):
        path, column = self.downloads_treeview.get_cursor()
        has_selected = path is not None
        if has_selected:
            state = self.downloads_model[path][STATE]
            play = (state == DownloadState.PAUSED)
            pause = (state == DownloadState.IN_PROGRESS)
        else:
            play = False
            pause = False

        self.play_button.set_sensitive(play)
        self.play_item.set_sensitive(play)
        self.play_item2.set_sensitive(play)
        self.pause_button.set_sensitive(pause)
        self.pause_item.set_sensitive(pause)
        self.pause_item2.set_sensitive(pause)

    def show_formated_value(self, column, cell, model, iter, colnum):
        value = model[iter][colnum]
        order = ['b', 'kb', 'Mb', 'Gb', 'Tb']

        i = 0
        while 0 <= value > 1000.0:
            value = value / 1024.0
            i += 1

        if value == -1:
            text = '???'
        else:
            text = '%.2f %s' % (value, order[i])
            if colnum == RATE:
                text += '/s'

        cell.set_property('text', text)

    def message(self, message, msg_type=gtk.MESSAGE_ERROR):
        msg = gtk.MessageDialog(self.window,
                                gtk.DIALOG_DESTROY_WITH_PARENT,
                                msg_type, gtk.BUTTONS_CLOSE,
                                message)
        msg.set_icon_name('gtk-go-down')
        msg.run()
        msg.destroy()


def main():
    app = PyGet()
    gtk.main()


if __name__ == '__main__':
    main()
