/******************************************************************************
 *
 * Copyright(c) 2005 - 2013 Intel Corporation.
 * All rights reserved.
 *
 * LICENSE PLACE HOLDER
 *
 *****************************************************************************/
#include <linux/kfifo.h>
#include <linux/kthread.h>

#include "iwl-em-idi-rx.h"
#include "iwl-emulation.h"
#include "iwl-amfh.h"
#include "iwl-idi.h"

/*
 * IWL IDI RX Emulation Logger
 */
#define IDI_RX_LOG_PREFIX	"[IDI_RX]"

#ifdef __IWL_IDI_RX_LOG_ENABLED__
#define IDI_RX_TRACE_ENTER \
	IWL_EM_TRACE_ENTER(__IDI_RX_LOG_LEVEL_TRACE__, IDI_RX_LOG_PREFIX)
#define IDI_RX_TRACE_EXIT \
	IWL_EM_TRACE_EXIT(__IDI_RX_LOG_LEVEL_TRACE__, IDI_RX_LOG_PREFIX)
#define IDI_RX_TRACE_EXIT_RET_STR(_ret) \
	IWL_EM_LOG(__IDI_RX_LOG_LEVEL_TRACE__, IDI_RX_LOG_PREFIX, "<<< "_ret)
#define IDI_RX_TRACE_EXIT_RET(_ret) \
	IWL_EM_TRACE_EXIT_RET(__IDI_RX_LOG_LEVEL_TRACE__, IDI_RX_LOG_PREFIX,\
			      _ret)
#define IWL_IDI_RX_LOG(fmt, args ...) \
	IWL_EM_LOG(__IDI_RX_LOG_LEVEL_DEBUG__, IDI_RX_LOG_PREFIX, fmt, ## args)
#define IWL_IDI_RX_LOG_HEX_DUMP(msg, p, len) \
	IWL_EM_LOG_HEX_DUMP(msg, p, len)
#else

#define IDI_RX_TRACE_ENTER
#define IDI_RX_TRACE_EXIT
#define IDI_RX_TRACE_EXIT_RET_STR(_ret)
#define IDI_RX_TRACE_EXIT_RET(_ret)
#define IWL_IDI_RX_LOG(fmt, args ...)
#define IWL_IDI_RX_LOG_HEX_DUMP(msg, p, len)

#endif
/* Always print error messages */
#define IWL_IDI_RX_LOG_ERR(fmt, args ...) \
	IWL_EM_LOG_ERR(fmt, ## args)

/* IDI RX Workqueue definitons */
#define IWL_IDI_RX_WQ_FLAGS		(WQ_HIGHPRI |\
					WQ_UNBOUND |\
					WQ_NON_REENTRANT)
#define IWL_IDI_RX_MAX_ACTIVE_WORKERS	1
#define IDI_RX_READ_TIMEOUT 2000

/*
 * RXB Size MAX:
 * 8K max per AMPDU
 * 16 Bursts max*/
#define IWL_IDI_RX_RXB_SIZE		(8 * 16 * 1024)

/* Forward Declarations */
static int iwl_idi_rx_em_work(void *data);

/*
 * The IDI and yDMA RX flow struct.
 */
struct iwl_idi_rx_em_t {

	/* Thread for the IDI RX work */
	struct task_struct *idi_rx_work_thread;

	/* Waitqueue for the IDI RX thread */
	wait_queue_head_t idi_rx_wq;

	/* A Flag to signal if there is an active RX transaction */
	bool idi_rx_active;

	/*
	 * The RX RB buffer fifo.
	 * The IDI_RX will pull the burst from this buffer,the AMFH pushes
	 * to this buffer.
	 */
	struct kfifo rx_rb_pipe;
};

/* Global IDI RX struct */
static struct iwl_idi_rx_em_t iwl_idi_rx_em_global;
static struct iwl_idi_rx_em_t *iwl_idi_rx_em = &iwl_idi_rx_em_global;

/**
 * Initialize the IDI RX emulation block.
 *
 * Returns 0 on success,  negative value describing the error otherwise.
 */
int iwl_idi_rx_em_init(void)
{
	int ret;

	IDI_RX_TRACE_ENTER;

	/* Init the IDI RX struct */
	memset(iwl_idi_rx_em , 0, sizeof(struct iwl_idi_rx_em_t));

	/* Allocate the RX RB buffer fifo */
	ret = kfifo_alloc(&iwl_idi_rx_em->rx_rb_pipe,
			  IWL_IDI_RX_RXB_SIZE,
			  GFP_KERNEL);
	if (ret) {
		IWL_IDI_RX_LOG_ERR("Cannot allocate RX RB buffer");
		goto out;
	}
	IWL_IDI_RX_LOG("IDI RX RB Initialized");

	/* Waitqueue init */
	init_waitqueue_head(&(iwl_idi_rx_em->idi_rx_wq));
	IWL_IDI_RX_LOG("IDI RX Waitqueue Initialized");

	IWL_IDI_RX_LOG("IDI RX Initialized");
out:
	IDI_RX_TRACE_EXIT;
	return ret;
}

/*
 * Free the IDI RX emulation block.
 * The API should be called only after the IDI RX and
 * AMFH were gracefully stopped.
 */
void iwl_idi_rx_em_free(void)
{
	IDI_RX_TRACE_ENTER;

	/* Free the RX RB Buffer fifo*/
	kfifo_free(&iwl_idi_rx_em->rx_rb_pipe);

	IWL_IDI_RX_LOG("IDI_RX Freed");
	IDI_RX_TRACE_EXIT;
}

/*
 * Start the IDI Emulation
 *
 * Returns 0 on success, negative value describing the error otherwise.
 */
int iwl_idi_rx_em_start(void)
{
	int ret = 0;
	IDI_RX_TRACE_ENTER;

	/* Reset the kfifo */
	kfifo_reset(&iwl_idi_rx_em->rx_rb_pipe);

	/* Start IDI_RX HP Emulator */
	iwl_idi_rx_em->idi_rx_work_thread =
				kthread_create(iwl_idi_rx_em_work,
					       NULL,
					       "IDI_RX_CH_WORK");
	if (IS_ERR(iwl_idi_rx_em->idi_rx_work_thread)) {
		IWL_IDI_RX_LOG_ERR("Cannot create thread for IDI RX");
		return -ENOMEM;
		goto out;
	}
	kthread_bind(iwl_idi_rx_em->idi_rx_work_thread, IWL_IDI_RX_WORK_CPU);
	wake_up_process(iwl_idi_rx_em->idi_rx_work_thread);
	IWL_IDI_RX_LOG("IDI_RX thread - STARTED");

	IWL_IDI_RX_LOG("IDI_RX STARTED");
out:
	IDI_RX_TRACE_EXIT;
	return ret;
}

/*
 * Stop the IDI Emulation
 *
 * Returns 0 on success, negative value describing the error otherwise.
 */
int iwl_idi_rx_em_stop(void)
{
	IDI_RX_TRACE_ENTER;

	/* Stop the IDI RX HP worker thread */
	if (iwl_idi_rx_em->idi_rx_work_thread)
		kthread_stop(iwl_idi_rx_em->idi_rx_work_thread);
	iwl_idi_rx_em->idi_rx_work_thread = NULL;

	IWL_IDI_RX_LOG("IDI_RX STOPPED");
	IDI_RX_TRACE_EXIT;
	return 0;
}

/*
 * Check if there is enough space in the RXB to push the required size.
 *
 * @ size_required - The size we need to test for available space for.
 * Returns TRUE if there is available space for this size, false otherwise.
 */
bool iwl_idi_rx_em_available_space(u32 size_required)
{
	return kfifo_avail(&iwl_idi_rx_em->rx_rb_pipe) > size_required;
}

/*
 * Alert the IDI RX module that there has been a change in it's parameters
 * and that there maybe work that needs to be done by the module due
 * to the change.
 */
void iwl_idi_rx_em_alert_change(void)
{
	wake_up_interruptible(&(iwl_idi_rx_em->idi_rx_wq));
}

/*
 * Pushes data to the IDI RXB.
 * Pushes the given data from the given target in the given size.
 *
 * Returns 0 on success, negative value describing the error otherwise.
 * NOTE: only one thread can push the data at any given time.
 */
int iwl_idi_rx_em_push_to_rxb(const void *source, u32 size)
{

	IDI_RX_TRACE_ENTER;

	/* Check that there is enough room for pushing */
	if (!iwl_idi_rx_em_available_space(size)) {
		WARN_ON(1);
		return -ENOMEM;
	}

	/* Insert the given buffer to the fifo */
	kfifo_in(&iwl_idi_rx_em->rx_rb_pipe, source, size);

	/* Wake up the RX worker
	 * Stage 1 - Wake on every data push */
	iwl_idi_rx_em_alert_change();

	IDI_RX_TRACE_EXIT;
	return 0;
}

/* The condition on which the reading operation of the RXB pipe can sleep on */
#define IDI_RX_RXB_DATA_COND (kfifo_len(&iwl_idi_rx_em->rx_rb_pipe))

/* The condition on which the testing of dma enabled operation can sleep on */
#define IDI_RX_DMA_ENABLED_COND (atomic_read(dma_enabled_rx) \
							== IWL_IDI_ENABLED)

/* The condition on which the testing of SG ll is not null operation
 * can sleep on */
#define IDI_RX_DMA_SG_LL_COND (NULL != iwl_idi_get_sg_ll(IWL_IDI_RX_CHANNEL))

/* The condition on which the RX engine can start it's work */
#define IDI_RX_ENGINE_WORK (IDI_RX_DMA_SG_LL_COND && \
			     IDI_RX_DMA_ENABLED_COND  && \
			     !kthread_should_stop())

/* Checks and exists if the kthread was called to stop */
#define IDI_RX_CHECK_ENGINE_SHOULD_RUN \
	do { \
			if (kthread_should_stop()) { \
				IWL_IDI_RX_LOG("IDI RX request to stop"); \
				goto exit; \
			} \
	} while (0);

/*
 * Read the given size of BYTES to the given target.
 * This method can sleep if the RXB pipe is empty.
 *
 * If the thread needs to be stopped then the function will exit without
 * Reading the required size and garbage will remain on the pipe.
 */
static void iwl_idi_rx_em_rxb_read(void *target, u32 size,
				   atomic_t *dma_enabled_rx)
{
	u32 read_left = size, size_read;

	/* Validate size */
	if (WARN_ON(!size))
		return;

	IWL_IDI_RX_LOG("Setting read for RXB to target 0x%x size %d",
		       (unsigned int)target,
		       size);

	/* Try to read, sleep in case the pipe is empty */
	while (!kthread_should_stop() && read_left) {

		/* Sleep until there is data on the pipe */
		wait_event_interruptible(iwl_idi_rx_em->idi_rx_wq,
					 (IDI_RX_RXB_DATA_COND ||
					  !IDI_RX_ENGINE_WORK));

		/* Check that the Engine doesn't need to stop */
		if (!IDI_RX_ENGINE_WORK) {
			IWL_IDI_RX_LOG("RX engine STOP called on RXB");
			return;
		}

		/* Read from pipe to target */
		size_read = kfifo_out(&iwl_idi_rx_em->rx_rb_pipe,
				       target,
				       read_left);
		read_left -= size_read;
		target = (u8 *)target + size_read;

		/* Alert the AMFH that there is available room to push
		 * Stage 1 - Alert on every read */
		iwl_amfh_resume();
	}

	IWL_IDI_RX_LOG("Reading Complete on RXB of %d bytes", size);
}

/*
 * The thread that does the IDI RX work.
 *
 * This method sleeps until there is a DBB SG list set and the DMA is enabled.
 * After these conditions are met the function will write data from the RXB to
 * the desired target with the desired size specified in the DBB RX SG list.
 * In case the RXB pipe is empty the thread will sleep until it has more data
 * to read.
 */
static int iwl_idi_rx_em_work(void *data)
{
	struct idi_sg_desc *rx_sg_ll;
	atomic_t *dma_enabled_rx = iwl_idi_get_dma_state(IWL_IDI_RX_CHANNEL);
	bool call_inter;

	IDI_RX_TRACE_ENTER;

	while (!kthread_should_stop()) {

		/* Wait until the RX SG LL is loaded on the DBB side */
		wait_event_interruptible(
				iwl_idi_rx_em->idi_rx_wq,
				IDI_RX_ENGINE_WORK || kthread_should_stop());
		IDI_RX_CHECK_ENGINE_SHOULD_RUN;

		/* Get the DBB SG list */
		rx_sg_ll = iwl_idi_get_sg_ll(IWL_IDI_RX_CHANNEL);

		/* Go over the list and copy the data according to
		 * list fields */
		while (NULL != rx_sg_ll) {

			/* Signal RX transaction start */
			iwl_idi_rx_em->idi_rx_active = true;

			/* In case there was an external request to stop */
			if (!IDI_RX_ENGINE_WORK)
				goto exit;

			/*
			 * Read from the RXB pipe and write to the given target
			 * address from the DBB SG list in the given size.
			 * Will sleep if the pipe is empty.
			 */
			iwl_idi_rx_em_rxb_read(
					iwl_idi_calc_dbb_addr(rx_sg_ll),
					iwl_idi_calc_dbb_size(rx_sg_ll),
					dma_enabled_rx);
			IDI_RX_CHECK_ENGINE_SHOULD_RUN;

			/* Test if we need to call and interrupt */
			call_inter = !!iwl_idi_calc_dbb_I_bit(rx_sg_ll);

			/* Advance the sg list descriptor */
			rx_sg_ll = iwl_idi_calc_dbb_next(rx_sg_ll);

			/* Disable the DMA on this channel and clear the
			 * DBB SG list if processing is done */
			if (NULL == rx_sg_ll) {
				atomic_set(dma_enabled_rx, IWL_IDI_DISABLED);
				iwl_idi_set_sg_ll(IWL_IDI_RX_CHANNEL, NULL);
			}

			/* Signal RX transaction start */
			iwl_idi_rx_em->idi_rx_active = false;

			/* Call RX interrupt if required */
			if (call_inter && !kthread_should_stop())
				iwl_idi_call_inter(IWL_IDI_RX_CHANNEL);
		}
	}
exit:
	iwl_idi_rx_em->idi_rx_active = false;
	IDI_RX_TRACE_EXIT;
	return 0;
}

/*
 * Waits until the RX transaction has ended,
 * wil return only when the RX engine is Idle.
 */
void iwl_idi_rx_em_wait_for_rx_stop(void)
{
	while (iwl_idi_rx_em->idi_rx_active) {
		iwl_idi_rx_em_alert_change();
		schedule();
	}
}

