#!/usr/bin/python
#
# Copyright (C) 2012 Canonical, Ltd
# Author: Matthew Fischer <matthew.fischer@canonical.com>

import sys
import os
import unittest
sys.path.append(os.path.abspath(os.path.dirname(os.path.abspath(__file__)) + '/..'))
import nudev

class DeviceTests(unittest.TestCase):

        # generic tests for the device class
        @unittest.skipUnless(nudev.Device('/sys/class/input/event3').name() is not None, "no input device event3")
        def test_device_generic(self):
            d = nudev.Device('/sys/class/input/event3')
            self.failUnless(d.dev_path() == '/dev/input/event3')

            # XXX - we're not reading the capabilities here because
            # the device we chose may not exist, we could look at /sys
            # to find one.

            # we cannot assume event3 is on this system, so just make
            # sure we don't get a None return
            self.failUnless(d.name() is not None)

            # make sure the masks were created
            self.failUnless(d._evmask is not None)
            self.failUnless(d._absmask is not None)
            self.failUnless(d._relmask is not None)
            self.failUnless(d._keymask is not None)
            self.failUnless(d._ledmask is not None)

            # not a mouse or keyboard and no leds
            self.failUnless(d.is_mouse() is False)
            self.failUnless(d.is_keyboard() is False)
            self.failUnless(d.has_leds() is False)

        def test_device_keyboard(self):
            d = nudev.Device('test_keyboard')
            # fail because we have no caps at all
            self.failUnless(d.is_keyboard() is False)

            # set EV_KEY, we still need more
            d._evmask[nudev.Bits.EV_KEY] = True
            self.failUnless(d.is_keyboard() is False)

            # set some keys, this fails the final test still
            d._keymask[nudev.Bits.KEY_1] = True
            d._keymask[nudev.Bits.KEY_2] = True
            d._keymask[nudev.Bits.KEY_3] = True
            d._keymask[nudev.Bits.KEY_Q] = True
            d._keymask[nudev.Bits.KEY_W] = True
            d._keymask[nudev.Bits.KEY_E] = True
            self.failUnless(d.is_keyboard() is False)

            # this will pass the final check
            for i in range(1,32):
                d._keymask[i] = True
            self.failUnless(d.is_keyboard() is True)

            # sanity check to make sure we also didn't make a mouse
            self.failUnless(d.is_mouse() is False)

        def test_mutliple_keyboards_no_leds(self):
            # with no led caps set, this will return the last one in the
            # sorted list based on the basename
            devices = []
            d1 = nudev.Device('/foo/bar/event2')
            devices.append(d1)
            d2 = nudev.Device('/foo/bar/event1')
            devices.append(d2)
            d3 = nudev.Device('/blah/blah/event3')
            devices.append(d3)

            keyboard = nudev.find_keyboard_candidate(devices)
            self.failUnless(keyboard == d3)

        def test_mutliple_keyboards_with_leds(self):
            devices = []
            d1 = nudev.Device('/foo/bar/event2')
            d2 = nudev.Device('/foo/bar/event1')
            d2._ledmask[0] = True
            self.failUnless(d2.has_leds() == True)
            d3 = nudev.Device('/blah/blah/event3')
            devices.append(d1)
            devices.append(d2)
            devices.append(d3)

            keyboard = nudev.find_keyboard_candidate(devices)
            self.failUnless(keyboard == d2)

            # if > 1 have LEDs, fallback to the basename method
            d1._ledmask[0] = True
            self.failUnless(d1.has_leds() == True)
            keyboard = nudev.find_keyboard_candidate(devices)
            self.failUnless(keyboard == d1)

        def test_device_mouse(self):
            d = nudev.Device('test_mouse')
            # fail because we have no caps at all
            self.failUnless(d.is_mouse() is False)

            # we have buttons, but we need more
            d._keymask[nudev.Bits.BTN_MOUSE] = True
            self.failUnless(d.is_mouse() is False)

            # create an ABS mouse
            d._evmask[nudev.Bits.EV_ABS] = True
            d._absmask[nudev.Bits.ABS_X] = True
            d._absmask[nudev.Bits.ABS_Y] = True
            self.failUnless(d.is_mouse() is True)

            # switch to REL mouse (which is most mice I think)
            d._evmask[nudev.Bits.EV_ABS] = False
            d._absmask[nudev.Bits.ABS_X] = False
            d._absmask[nudev.Bits.ABS_Y] = False
            d._evmask[nudev.Bits.EV_REL] = True
            d._relmask[nudev.Bits.ABS_X] = True
            d._relmask[nudev.Bits.ABS_Y] = True
            self.failUnless(d.is_mouse() is True)

            # we had a mouse, now make it a touchpad, which looks
            # similar but is not the same
            d._keymask[nudev.Bits.BTN_TOOL_FINGER] = True
            self.failUnless(d.is_mouse() is False)

            # sanity check to make sure we also didn't make a keyboard
            self.failUnless(d.is_keyboard() is False)


def main():
    unittest.main()

if __name__ == '__main__':
    main()
