/*******************************************************************************
**3456789 123456789 123456789 123456789 123456789 123456789 123456789 123456789
**      10        20        30        40        50        60        70        80
**
** notify-osd
**
** test-split-button.c - unit-test for split-button class used by snap-decisions
**
** Copyright 2012 Canonical Ltd.
**
** Authors:
**    Mirco "MacSlow" Mueller <mirco.mueller@canonical.com>
**
** This program is free software: you can redistribute it and/or modify it
** under the terms of the GNU General Public License version 3, as published
** by the Free Software Foundation.
**
** This program is distributed in the hope that it will be useful, but
** WITHOUT ANY WARRANTY; without even the implied warranties of
** MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
** PURPOSE.  See the GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License along
** with this program.  If not, see <http://www.gnu.org/licenses/>.
**
*******************************************************************************/

#include <glib.h>
#include <cairo.h>

#include "config.h"
#include "split-button.h"
#include "settings.h"

static void
test_split_button_new ()
{
	SplitButton* split_button = NULL;
	gchar*       labels[]     = {"ok_id",        "Ok",
                                 "decline_id",   "Decline",
                                 "not_there_id", "I'm not there",
                                 "busy_id",      "Really busy"};
	Settings*    settings     = settings_new ();

	split_button = split_button_new (10,
	                                 10,
	                                 settings->button,
	                                 settings->split_button,
	                                 &labels[2],
	                                 6);
	g_assert (split_button != NULL);
	g_assert (IS_SPLIT_BUTTON (split_button));
	g_object_unref (split_button);
	settings_del (settings);
	split_button = NULL;
}

static void
test_split_button_del ()
{
	SplitButton* split_button = NULL;
	gchar*       labels[]     = {"ok_id",        "Ok",
                                 "decline_id",   "Decline",
                                 "not_there_id", "I'm not there",
                                 "busy_id",      "Really busy"};
	Settings*    settings     = settings_new ();

	split_button = split_button_new (10,
	                                 10,
	                                 settings->button,
	                                 settings->split_button,
	                                 &labels[2],
	                                 6);
	g_assert (split_button != NULL);
	g_assert (IS_SPLIT_BUTTON (split_button));

	split_button_del (split_button);
	g_assert (!IS_SPLIT_BUTTON (split_button));
	split_button = NULL;
	settings_del (settings);
}

static void
test_split_button_getset_position ()
{
	SplitButton* split_button = NULL;
	gchar*       labels[]     = {"ok_id",        "Ok",
                                 "decline_id",   "Decline",
                                 "not_there_id", "I'm not there",
                                 "busy_id",      "Really busy"};
	Settings*    settings     = settings_new ();

	split_button = split_button_new (20,
	                                 35,
	                                 settings->button,
	                                 settings->split_button,
	                                 &labels[2],
	                                 6);

	g_assert (split_button != NULL);
	g_assert (IS_SPLIT_BUTTON (split_button));

	gint x = 0;
	gint y = 0;

	split_button_get_position (split_button, &x, &y);
	g_assert_cmpint (x, ==, 20);
	g_assert_cmpint (y, ==, 35);
	split_button_set_position (split_button, 42, 69);
	split_button_get_position (split_button, &x, &y);
	g_assert_cmpint (x, ==, 42);
	g_assert_cmpint (y, ==, 69);

	split_button_del (split_button);
	settings_del (settings);
}

static void
test_split_button_getset_form ()
{
	SplitButton* split_button = NULL;
	gchar*       labels[]     = {"ok_id",        "Ok",
                                 "decline_id",   "Decline",
                                 "not_there_id", "I'm not there",
                                 "busy_id",      "Really busy"};
	Settings*    settings     = settings_new ();

	split_button = split_button_new (10,
	                                 10,
	                                 settings->button,
	                                 settings->split_button,
	                                 &labels[2], 6);
	g_assert (split_button != NULL);
	g_assert (IS_SPLIT_BUTTON (split_button));

	g_assert_cmpint (split_button_get_form (split_button),
	                 ==,
	                 SPLIT_BUTTON_FORM_COLLAPSED);
	split_button_set_form (split_button, SPLIT_BUTTON_FORM_EXPANDED);
	g_assert_cmpint (split_button_get_form (split_button),
	                 ==,
	                 SPLIT_BUTTON_FORM_EXPANDED);
	split_button_set_form (split_button, SPLIT_BUTTON_FORM_COLLAPSED);
	g_assert_cmpint (split_button_get_form (split_button),
	                 ==,
	                 SPLIT_BUTTON_FORM_COLLAPSED);

	split_button_del (split_button);
	settings_del (settings);
}

static void
test_split_button_getset_state ()
{
	SplitButton* split_button = NULL;
	gchar*       labels[]     = {"ok_id",        "Ok",
                                 "decline_id",   "Decline",
                                 "not_there_id", "I'm not there",
                                 "busy_id",      "Really busy"};
	Settings*    settings     = settings_new ();

	split_button = split_button_new (10,
	                                 10,
	                                 settings->button,
	                                 settings->split_button,
	                                 &labels[2], 6);
	g_assert (split_button != NULL);
	g_assert (IS_SPLIT_BUTTON (split_button));

	g_assert_cmpint (split_button_get_state (split_button),
	                 ==,
	                 SPLIT_BUTTON_STATE_NORMAL);
	split_button_set_state (split_button, SPLIT_BUTTON_STATE_HOVER);
	g_assert_cmpint (split_button_get_state (split_button),
	                 ==,
	                 SPLIT_BUTTON_STATE_HOVER);
	split_button_set_state (split_button, SPLIT_BUTTON_STATE_PRESSED);
	g_assert_cmpint (split_button_get_state (split_button),
	                 ==,
	                 SPLIT_BUTTON_STATE_PRESSED);
	split_button_set_state (split_button, SPLIT_BUTTON_STATE_NORMAL);
	g_assert_cmpint (split_button_get_state (split_button),
	                 ==,
	                 SPLIT_BUTTON_STATE_NORMAL);
	
	split_button_del (split_button);
	settings_del (settings);
}

static void
test_split_button_getset_hit_label ()
{
	SplitButton* split_button = NULL;
	gchar*       labels[]     = {"ok_id",        "Ok",
                                 "decline_id",   "Decline",
                                 "not_there_id", "I'm not there",
                                 "busy_id",      "Really busy"};
	Settings*    settings     = settings_new ();

	split_button = split_button_new (10,
	                                 10,
	                                 settings->button,
	                                 settings->split_button,
	                                 &labels[2], 6);
	g_assert (split_button != NULL);
	g_assert (IS_SPLIT_BUTTON (split_button));

	g_assert_cmpint (split_button_get_hit_label (split_button), ==, -1);
	split_button_set_hit_label (split_button, 0);
	g_assert_cmpint (split_button_get_hit_label (split_button), ==, 0);
	split_button_set_hit_label (split_button, 1);
	g_assert_cmpint (split_button_get_hit_label (split_button), ==, 1);
	split_button_set_hit_label (split_button, 2);
	g_assert_cmpint (split_button_get_hit_label (split_button), ==, 2);
	split_button_set_hit_label (split_button, 3);
	g_assert_cmpint (split_button_get_hit_label (split_button), ==, 2);
	split_button_set_hit_label (split_button, -1);
	g_assert_cmpint (split_button_get_hit_label (split_button), ==, -1);

	split_button_del (split_button);
	settings_del (settings);
}

static void
test_split_button_get_expanded_height ()
{
	SplitButton* split_button = NULL;
	gchar*       labels[]     = {"ok_id",        "Ok",
                                 "decline_id",   "Decline",
                                 "not_there_id", "I'm not there",
                                 "busy_id",      "Really busy"};
	Settings*    settings     = settings_new ();

	split_button = split_button_new (10,
	                                 10,
	                                 settings->button,
	                                 settings->split_button,
	                                 &labels[2], 6);
	g_assert (split_button != NULL);
	g_assert (IS_SPLIT_BUTTON (split_button));

	g_assert_cmpuint (split_button_get_expanded_height (split_button), >, 0);

	split_button_del (split_button);
	settings_del (settings);
}

static void
test_split_button_is_hit ()
{
	SplitButton* split_button = NULL;
	gchar*       labels[]     = {"ok_id",        "Ok",
                                 "decline_id",   "Decline",
                                 "not_there_id", "I'm not there",
                                 "busy_id",      "Really busy",
                                 "leave_me_id",  "Leave me alone"};
	Settings*    settings     = settings_new ();

	split_button = split_button_new (10,
	                                 10,
	                                 settings->button,
	                                 settings->split_button,
	                                 &labels[2], 8);
	g_assert (split_button != NULL);
	g_assert (IS_SPLIT_BUTTON (split_button));

	gboolean result = FALSE;

	split_button_set_form (split_button, SPLIT_BUTTON_FORM_COLLAPSED);
	result = split_button_is_hit (split_button, 11, 5);
	g_assert (!result);
	split_button_set_form (split_button, SPLIT_BUTTON_FORM_EXPANDED);
	result = split_button_is_hit (split_button,
	                              11,
	                              10 + split_button_get_expanded_height (split_button) - 1);
	g_assert (result);

	split_button_set_form (split_button, SPLIT_BUTTON_FORM_COLLAPSED);
	result = split_button_is_hit (split_button, 11, 11);
	g_assert (result);
	split_button_set_form (split_button, SPLIT_BUTTON_FORM_EXPANDED);
	result = split_button_is_hit (split_button, 11, 11);
	g_assert (result);

	split_button_set_form (split_button, SPLIT_BUTTON_FORM_COLLAPSED);
	result = split_button_is_hit (split_button,
	                              11,
	                              10 + split_button_get_expanded_height (split_button) + 1);
	g_assert (!result);
	split_button_set_form (split_button, SPLIT_BUTTON_FORM_EXPANDED);
	result = split_button_is_hit (split_button,
	                              11,
	                              10 + split_button_get_expanded_height (split_button) + 1);
	g_assert (!result);

	split_button_del (split_button);
	settings_del (settings);
}

static void
test_split_button_is_arrow_hit ()
{
	SplitButton* split_button = NULL;
	gchar*       labels[]     = {"ok_id",        "Ok",
                                 "decline_id",   "Decline",
                                 "not_there_id", "I'm not there",
                                 "busy_id",      "Really busy",
                                 "leave_me_id",  "Leave me alone"};
	Settings*    settings     = settings_new ();

	split_button = split_button_new (10,
	                                 10,
	                                 settings->button,
	                                 settings->split_button,
	                                 &labels[2], 8);
	g_assert (split_button != NULL);
	g_assert (IS_SPLIT_BUTTON (split_button));

	gboolean result = FALSE;

	split_button_set_form (split_button, SPLIT_BUTTON_FORM_COLLAPSED);
	result = split_button_is_arrow_hit (split_button, 11, 5);
	g_assert (!result);
	result = split_button_is_arrow_hit (split_button, 13, 13);
	g_assert (result);

	split_button_set_form (split_button, SPLIT_BUTTON_FORM_EXPANDED);
	result = split_button_is_arrow_hit (split_button,
	                                    11,
	                                    10 + settings->split_button->expanded_height/2 - 1);
	g_assert (result);
	result = split_button_is_arrow_hit (split_button,
	                                    11,
	                                    10 + settings->split_button->expanded_height * 2);
	g_assert (!result);

	split_button_del (split_button);
	settings_del (settings);
}

static void
test_split_button_get_num_actions ()
{
	SplitButton* split_button = NULL;
	gchar*       labels[5]    = {"Accept", "Decline", "No", "Nein", "Non"};
	guint        num_actions  = 0;
	Settings*    settings     = settings_new ();

	split_button = split_button_new (10,
	                                 10,
	                                 settings->button,
	                                 settings->split_button,
	                                 &labels[1], 4);
	g_assert (split_button != NULL);
	g_assert (IS_SPLIT_BUTTON (split_button));

	num_actions = split_button_get_num_actions (split_button);
	g_assert_cmpuint (num_actions, ==, 4);

	split_button_del (split_button);
	settings_del (settings);
}

static void
test_split_button_getset_actions ()
{
	SplitButton* split_button = NULL;
	gchar*       labels5[]     = {"ok_id",        "Ok",
                                  "decline_id",   "Decline",
                                  "not_there_id", "I'm not there",
                                  "busy_id",      "Really busy",
                                  "leave_me_id",  "Leave me alone"};

	gchar*       labels4[]     = {"do_it_id",  "Do it",
                                  "cancel_id", "Cancel",
                                  "afk_id",    "Away from keyboard",
                                  "errand_id", "On an errand"};
	guint        num_actions  = 0;
	Settings*    settings     = settings_new ();

	split_button = split_button_new (0,
	                                 0,
	                                 settings->button,
	                                 settings->split_button,
	                                 &labels5[2], 8);
	g_assert (split_button != NULL);
	g_assert (IS_SPLIT_BUTTON (split_button));

	num_actions = split_button_get_num_actions (split_button);
	g_assert_cmpuint (num_actions, ==, 8);

	g_assert_cmpstr (split_button_get_action_by_index (split_button, 0),
	                 ==,
	                 "decline_id");
	g_assert_cmpstr (split_button_get_action_by_index (split_button, 1),
	                 ==,
	                 "Decline");
	g_assert_cmpstr (split_button_get_action_by_index (split_button, 2),
	                 ==,
	                 "not_there_id");
	g_assert_cmpstr (split_button_get_action_by_index (split_button, 3),
	                 ==,
	                 "I'm not there");

	split_button_set_actions (split_button, &labels4[2], 6);

	g_assert_cmpstr (split_button_get_action_by_index (split_button, 0),
	                 ==,
	                 "cancel_id");
	g_assert_cmpstr (split_button_get_action_by_index (split_button, 1),
	                 ==,
	                 "Cancel");
	g_assert_cmpstr (split_button_get_action_by_index (split_button, 2),
	                 ==,
	                 "afk_id");

	split_button_del (split_button);
	settings_del (settings);
}

static void
test_split_button_paint ()
{
	SplitButton* split_button = NULL;
	gchar*       labels[]     = {"do_it_id",  "Do it",
                                 "cancel_id", "Cancel",
                                 "afk_id",    "Away",
                                 "errand_id", "Errand"};
	Settings*    settings     = settings_new ();

	split_button = split_button_new (10,
	                                 10,
	                                 settings->button,
	                                 settings->split_button,
	                                 &labels[2], 6);
	g_assert (split_button != NULL);
	g_assert (IS_SPLIT_BUTTON (split_button));

	cairo_surface_t* surface = NULL;
	cairo_t*         cr      = NULL;

	surface = cairo_image_surface_create (CAIRO_FORMAT_ARGB32, 140, 220);
	cr = cairo_create (surface);
	cairo_scale (cr, 1.0, 1.0);
	cairo_set_operator (cr, CAIRO_OPERATOR_CLEAR);
	cairo_paint (cr);
	cairo_set_operator (cr, CAIRO_OPERATOR_OVER);

	split_button_set_form (split_button, SPLIT_BUTTON_FORM_COLLAPSED);
	split_button_set_state (split_button, SPLIT_BUTTON_STATE_NORMAL);
	split_button_set_position (split_button, 10, 10);
	split_button_paint (split_button, cr);
	split_button_set_state (split_button, SPLIT_BUTTON_STATE_HOVER);
	split_button_set_position (split_button, 10, 40);
	split_button_paint (split_button, cr);
	split_button_set_state (split_button, SPLIT_BUTTON_STATE_PRESSED);
	split_button_set_position (split_button, 10, 70);
	split_button_paint (split_button, cr);

	split_button_set_form (split_button, SPLIT_BUTTON_FORM_EXPANDED);
	split_button_set_state (split_button, SPLIT_BUTTON_STATE_NORMAL);
	split_button_set_hit_label (split_button, 0);
	split_button_set_position (split_button, 70, 10);
	split_button_paint (split_button, cr);
	split_button_set_state (split_button, SPLIT_BUTTON_STATE_HOVER);
	split_button_set_hit_label (split_button, 1);
	split_button_set_position (split_button, 70, 80);
	split_button_paint (split_button, cr);
	split_button_set_state (split_button, SPLIT_BUTTON_STATE_PRESSED);
	split_button_set_hit_label (split_button, 2);
	split_button_set_position (split_button, 70, 150);
	split_button_paint (split_button, cr);

	cairo_destroy (cr);
	cairo_surface_write_to_png (surface, "/tmp/test-split-button-paint.png");
	cairo_surface_destroy (surface);

	split_button_del (split_button);
	settings_del (settings);
}

GTestSuite *
test_split_button_create_test_suite (void)
{
	GTestSuite *ts = NULL;

	ts = g_test_create_suite ("split-button");

#define TC(x) g_test_create_case(#x, 0, NULL, NULL, x, NULL)

	g_test_suite_add (ts, TC(test_split_button_new));
	g_test_suite_add (ts, TC(test_split_button_del));
	g_test_suite_add (ts, TC(test_split_button_getset_position));
	g_test_suite_add (ts, TC(test_split_button_get_expanded_height));
	g_test_suite_add (ts, TC(test_split_button_is_hit));
	g_test_suite_add (ts, TC(test_split_button_is_arrow_hit));
	g_test_suite_add (ts, TC(test_split_button_getset_form));	
	g_test_suite_add (ts, TC(test_split_button_getset_state));
	g_test_suite_add (ts, TC(test_split_button_getset_hit_label));
	g_test_suite_add (ts, TC(test_split_button_get_num_actions));
	g_test_suite_add (ts, TC(test_split_button_getset_actions));
	g_test_suite_add (ts, TC(test_split_button_paint));

	return ts;
}
