/*******************************************************************************
**3456789 123456789 123456789 123456789 123456789 123456789 123456789 123456789
**      10        20        30        40        50        60        70        80
**
** notify-osd
**
** test-json-parser.c - unit-tests for json-parser class
**
** Copyright 2012 Canonical Ltd.
**
** Authors:
**    Mirco "MacSlow" Mueller <mirco.mueller@canonical.com>
**
** This program is free software: you can redistribute it and/or modify it
** under the terms of the GNU General Public License version 3, as published
** by the Free Software Foundation.
**
** This program is distributed in the hope that it will be useful, but
** WITHOUT ANY WARRANTY; without even the implied warranties of
** MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
** PURPOSE.  See the GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License along
** with this program.  If not, see <http://www.gnu.org/licenses/>.
**
*******************************************************************************/

#include <glib.h>
#include <gdk/gdk.h>

#include "config.h"
#include "json-parser.h"

static void
test_parser_new ()
{
	Parser* parser = NULL;

	parser = parser_new ();
	g_assert (parser != NULL);
	g_assert (IS_PARSER(parser));
	g_object_unref (parser);
}

static void
test_parser_open ()
{
	Parser*  parser = NULL;
	gboolean result = FALSE;

	parser = parser_new ();
	g_assert (parser != NULL);
	g_assert (IS_PARSER(parser));

	/* failure test */
	result = parser_open (parser, "./asdfyxcvqewr.json");
	g_assert (!result);

	/* success test */
	result = parser_open (parser, "./test-json-parser.json");
	g_assert (result);

	g_object_unref (parser);
}

static void
test_parser_read_int ()
{
	Parser*  parser = NULL;
	gboolean result = FALSE;
	gint     value  = 0;

	parser = parser_new ();
	g_assert (parser != NULL);

	result = parser_open (parser, "./test-json-parser.json");
	g_assert (result);

	value = parser_read_int (parser, "parser-test", "single-int-read-test");
	g_assert_cmpint (value, ==, 42);

	g_object_unref (parser);
}

static void
test_parser_read_ints ()
{
	Parser*  parser = NULL;
	gboolean result = FALSE;
	gint*    values = NULL;
	guint    size   = 0;

	parser = parser_new ();
	g_assert (parser != NULL);

	result = parser_open (parser, "./test-json-parser.json");
	g_assert (result);

	/* failure tests */
	values = parser_read_ints (parser,
	                           "parser-test",
	                           "array-int-read-test",
	                           NULL);
	g_assert (!values);

	values = parser_read_ints (NULL,
	                           "parser-test",
	                           "array-int-read-test",
	                           &size);
	g_assert (!values);

	/* success tests */
	values = parser_read_ints (parser,
	                           "parser-test",
	                           "array-int-read-test",
	                           &size);
	g_assert_cmpint (size, ==, 3);
	g_assert_cmpint (values[0], ==, 42);
	g_assert_cmpint (values[1], ==, 69);
	g_assert_cmpint (values[2], ==, 4711);
	g_free (values);

	g_object_unref (parser);
}

static void
test_parser_read_double ()
{
	Parser*  parser = NULL;
	gboolean result = FALSE;
	gdouble  value  = 0.0;

	parser = parser_new ();
	g_assert (parser != NULL);

	result = parser_open (parser, "./test-json-parser.json");
	g_assert (result);

	value = parser_read_double (parser,
	                            "parser-test",
	                            "single-double-read-test");
	g_assert_cmpfloat (value, ==, 90.123);

	g_object_unref (parser);	
}

static void
test_parser_read_doubles ()
{
	Parser*  parser = NULL;
	gboolean result = FALSE;
	gdouble* values = NULL;
	guint    size   = 0;

	parser = parser_new ();
	g_assert (parser != NULL);

	result = parser_open (parser, "./test-json-parser.json");
	g_assert (result);

	/* failure tests */
	values = parser_read_doubles (parser,
	                              "parser-test",
	                              "array-double-read-test",
	                              NULL);
	g_assert (!values);

	values = parser_read_doubles (NULL,
	                              "parser-test",
	                              "array-double-read-test",
	                              &size);
	g_assert (!values);

	/* success tests */
	values = parser_read_doubles (parser,
	                              "parser-test",
	                              "array-double-read-test",
	                              &size);
	g_assert_cmpint (size, ==, 3);
	g_assert_cmpfloat (values[0], ==, 4711.0815);
	g_assert_cmpfloat (values[1], ==, 2.71828183);
	g_assert_cmpfloat (values[2], ==, 3.14159265);
	g_free (values);

	g_object_unref (parser);
}

static void
test_parser_read_color ()
{
	Parser*  parser = NULL;
	gboolean result = FALSE;
	GdkColor color  = {0, 0, 0, 0};

	parser = parser_new ();
	g_assert (parser != NULL);

	result = parser_open (parser, "./test-json-parser.json");
	g_assert (result);

	/* failure test */
	result = parser_read_color (parser,
	                            "parser-test",
	                            "single-color-read-test",
	                            NULL);
	g_assert (!result);

	/* success test */
	result = parser_read_color (parser,
	                            "parser-test",
	                            "single-color-read-test",
	                            &color);
	g_assert (result);
	g_assert_cmpint (color.red, ==, 0xFFFF);
	g_assert_cmpint (color.green, ==, 0xAAAA);
	g_assert_cmpint (color.blue, ==, 0xBBBB);

	g_object_unref (parser);	
}

static void
test_parser_read_colors ()
{
	Parser*   parser = NULL;
	gboolean  result = FALSE;
	guint     size   = 0;
	GdkColor* colors = NULL;

	parser = parser_new ();
	g_assert (parser != NULL);

	result = parser_open (parser, "./test-json-parser.json");
	g_assert (result);

	/* failure test */
	colors = parser_read_colors (parser,
	                             "parser-test",
	                             "array-color-read-test",
	                             NULL);
	g_assert (colors == NULL);

	/* success test */
	colors = parser_read_colors (parser,
	                             "parser-test",
	                             "array-color-read-test",
	                             &size);
	g_assert (colors);
	g_assert_cmpint (size, ==, 3);
	g_assert_cmpint (colors[0].red, ==, 0xCAFE);
	g_assert_cmpint (colors[0].green, ==, 0xAFFE);
	g_assert_cmpint (colors[0].blue, ==, 0xBEEF);
	g_assert_cmpint (colors[1].red, ==, 0xDEAD);
	g_assert_cmpint (colors[1].green, ==, 0xFEED);
	g_assert_cmpint (colors[1].blue, ==, 0xF00D);
	g_assert_cmpint (colors[2].red, ==, 0xDEAD);
	g_assert_cmpint (colors[2].green, ==, 0xF00D);
	g_assert_cmpint (colors[2].blue, ==, 0xFACE);
	g_free (colors);

	g_object_unref (parser);	
}

static void
test_parser_read_font_style ()
{
	Parser*   parser = NULL;
	gboolean  result = FALSE;

	parser = parser_new ();
	g_assert (parser != NULL);

	result = parser_open (parser, "./test-json-parser.json");
	g_assert (result);

	FontStyle style = FONT_STYLE_NONE;

	style = parser_read_font_style (parser,
	                                "parser-test",
	                                "single-fontstyle-read-test-1");
	g_assert_cmpint (style, ==, FONT_STYLE_NORMAL);
	style = parser_read_font_style (parser,
	                                "parser-test",
	                                "single-fontstyle-read-test-2");
	g_assert_cmpint (style, ==, FONT_STYLE_OBLIQUE);
	style = parser_read_font_style (parser,
	                                "parser-test",
	                                "single-fontstyle-read-test-3");
	g_assert_cmpint (style, ==, FONT_STYLE_ITALIC);

	g_object_unref (parser);
}

static void
test_parser_read_font_styles ()
{
	Parser*    parser = NULL;
	gboolean   result = FALSE;
	guint      size   = 0;
	FontStyle* styles = NULL;

	parser = parser_new ();
	g_assert (parser != NULL);

	result = parser_open (parser, "./test-json-parser.json");
	g_assert (result);

	/* failure test */
	styles = parser_read_font_styles (parser,
	                                  "parser-test",
	                                  "array-fontstyle-read-test",
	                                  NULL);
	g_assert (styles == NULL);

	/* success test */
	styles = parser_read_font_styles (parser,
	                                  "parser-test",
	                                  "array-fontstyle-read-test",
	                                  &size);
	g_assert (styles);
	g_assert_cmpint (size, ==, 4);
	g_assert_cmpint (styles[0], ==, FONT_STYLE_OBLIQUE);
	g_assert_cmpint (styles[1], ==, FONT_STYLE_NORMAL);
	g_assert_cmpint (styles[2], ==, FONT_STYLE_ITALIC);
	g_assert_cmpint (styles[3], ==, FONT_STYLE_OBLIQUE);
	g_free (styles);

	g_object_unref (parser);
}

static void
test_parser_read_font_weight ()
{
	Parser*   parser = NULL;
	gboolean  result = FALSE;

	parser = parser_new ();
	g_assert (parser != NULL);

	result = parser_open (parser, "./test-json-parser.json");
	g_assert (result);

	FontWeight weight = FONT_WEIGHT_NONE;

	weight = parser_read_font_weight (parser,
	                                  "parser-test",
	                                  "single-fontweight-read-test-1");
	g_assert_cmpint (weight, ==, FONT_WEIGHT_LIGHT);
	weight = parser_read_font_weight (parser,
	                                  "parser-test",
	                                  "single-fontweight-read-test-2");
	g_assert_cmpint (weight, ==, FONT_WEIGHT_NORMAL);
	weight = parser_read_font_weight (parser,
	                                  "parser-test",
	                                  "single-fontweight-read-test-3");
	g_assert_cmpint (weight, ==, FONT_WEIGHT_BOLD);

	g_object_unref (parser);
}

static void
test_parser_read_font_weights ()
{
	Parser*     parser = NULL;
	gboolean    result = FALSE;
	guint       size   = 0;
	FontWeight* weights = NULL;

	parser = parser_new ();
	g_assert (parser != NULL);

	result = parser_open (parser, "./test-json-parser.json");
	g_assert (result);

	/* failure test */
	weights = parser_read_font_weights (parser,
	                                    "parser-test",
	                                    "array-fontweight-read-test",
	                                    NULL);
	g_assert (weights == NULL);

	/* success test */
	weights = parser_read_font_weights (parser,
	                                    "parser-test",
	                                    "array-fontweight-read-test",
	                                    &size);
	g_assert (weights);
	g_assert_cmpint (size, ==, 5);
	g_assert_cmpint (weights[0], ==, FONT_WEIGHT_BOLD);
	g_assert_cmpint (weights[1], ==, FONT_WEIGHT_LIGHT);
	g_assert_cmpint (weights[2], ==, FONT_WEIGHT_NORMAL);
	g_assert_cmpint (weights[3], ==, FONT_WEIGHT_BOLD);
	g_assert_cmpint (weights[4], ==, FONT_WEIGHT_NORMAL);
	g_free (weights);

	g_object_unref (parser);
}

static void
test_parser_read_blend_mode ()
{
	Parser*   parser = NULL;
	gboolean  result = FALSE;

	parser = parser_new ();
	g_assert (parser != NULL);

	result = parser_open (parser, "./test-json-parser.json");
	g_assert (result);

	BlendMode mode = BLEND_MODE_NONE;

	mode = parser_read_blend_mode (parser,
	                               "parser-test",
	                               "single-blendmode-read-test-1");
	g_assert_cmpint (mode, ==, BLEND_MODE_SOURCE);
	mode = parser_read_blend_mode (parser,
	                               "parser-test",
	                               "single-blendmode-read-test-2");
	g_assert_cmpint (mode, ==, BLEND_MODE_OVER);

	g_object_unref (parser);
}

static void
test_parser_read_blend_modes ()
{
	Parser*     parser = NULL;
	gboolean    result = FALSE;
	guint       size   = 0;
	BlendMode*  modes  = NULL;

	parser = parser_new ();
	g_assert (parser != NULL);

	result = parser_open (parser, "./test-json-parser.json");
	g_assert (result);

	/* failure test */
	modes = parser_read_blend_modes (parser,
	                                 "parser-test",
	                                 "array-blendmode-read-test",
	                                 NULL);
	g_assert (modes == NULL);

	/* success test */
	modes = parser_read_blend_modes (parser,
	                                 "parser-test",
	                                 "array-blendmode-read-test",
	                                 &size);
	g_assert (modes);
	g_assert_cmpint (size, ==, 4);
	g_assert_cmpint (modes[0], ==, BLEND_MODE_OVER);
	g_assert_cmpint (modes[1], ==, BLEND_MODE_SOURCE);
	g_assert_cmpint (modes[2], ==, BLEND_MODE_SOURCE);
	g_assert_cmpint (modes[3], ==, BLEND_MODE_OVER);
	g_free (modes);

	g_object_unref (parser);
}

static void
test_parser_read_string ()
{
	Parser*   parser = NULL;
	gboolean  result = FALSE;

	parser = parser_new ();
	g_assert (parser != NULL);

	result = parser_open (parser, "./test-json-parser.json");
	g_assert (result);

	gchar* string = NULL;

	string = parser_read_string (parser,
	                             "parser-test",
	                             "single-string-read-test");
	g_assert_cmpstr (string, ==, "foo bar");

	g_object_unref (parser);
}

static void
test_parser_read_strings ()
{
	Parser*  parser  = NULL;
	gboolean result  = FALSE;
	guint    size    = 0;
	gchar**  strings = NULL;

	parser = parser_new ();
	g_assert (parser != NULL);

	result = parser_open (parser, "./test-json-parser.json");
	g_assert (result);

	/* failure test */
	strings = parser_read_strings (parser,
	                               "parser-test",
	                               "array-string-read-test",
	                               NULL);
	g_assert (strings == NULL);

	/* success test */
	strings = parser_read_strings (parser,
	                               "parser-test",
	                               "array-string-read-test",
	                               &size);
	g_assert (strings);
	g_assert_cmpint (size, ==, 3);
	g_assert_cmpstr (strings[0], ==, "laber");
	g_assert_cmpstr (strings[1], ==, "fasel");
	g_assert_cmpstr (strings[2], ==, "suelz");

	/* clean up */
	gint i = 0;
	for (; i < size; i++)
		g_free (strings[i]);
	g_free (strings);
	strings = NULL;

	g_object_unref (parser);
}

GTestSuite *
test_json_parser_create_test_suite (void)
{
	GTestSuite *ts = NULL;
	
	ts = g_test_create_suite ("json-parser");

#define TC(x) g_test_create_case(#x, 0, NULL, NULL, x, NULL)

	g_test_suite_add(ts, TC(test_parser_new));
	g_test_suite_add(ts, TC(test_parser_open));
	g_test_suite_add(ts, TC(test_parser_read_int));
	g_test_suite_add(ts, TC(test_parser_read_ints));
	g_test_suite_add(ts, TC(test_parser_read_double));
	g_test_suite_add(ts, TC(test_parser_read_doubles));
	g_test_suite_add(ts, TC(test_parser_read_color));
	g_test_suite_add(ts, TC(test_parser_read_colors));
	g_test_suite_add(ts, TC(test_parser_read_font_style));
	g_test_suite_add(ts, TC(test_parser_read_font_styles));
	g_test_suite_add(ts, TC(test_parser_read_font_weight));
	g_test_suite_add(ts, TC(test_parser_read_font_weights));
	g_test_suite_add(ts, TC(test_parser_read_blend_mode));
	g_test_suite_add(ts, TC(test_parser_read_blend_modes));
	g_test_suite_add(ts, TC(test_parser_read_string));
	g_test_suite_add(ts, TC(test_parser_read_strings));

	return ts;
}
