/* -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */

#include <string.h>
#include <nm-settings.h>
#include "nmn-ethernet-handler.h"
#include "nmn-ethernet-item.h"
#include "utils.h"

G_DEFINE_TYPE (NmnEthernetHandler, nmn_ethernet_handler, NMN_TYPE_DEVICE_HANDLER)

#define GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj), NMN_TYPE_ETHERNET_HANDLER, NmnEthernetHandlerPrivate))

typedef struct {
    gulong carrier_changed_id;
    gulong ethernet_toggled_id;

    gboolean disposed;
} NmnEthernetHandlerPrivate;

NmnDeviceHandler *
nmn_ethernet_handler_new (NmnNMData *nm_data,
                          NMDeviceEthernet *device)
{
    g_return_val_if_fail (NMN_IS_NM_DATA (nm_data), NULL);
    g_return_val_if_fail (NM_IS_DEVICE_ETHERNET (device), NULL);

    return NMN_DEVICE_HANDLER (g_object_new (NMN_TYPE_ETHERNET_HANDLER,
                                             NMN_DEVICE_HANDLER_NM_DATA, nm_data,
                                             NMN_DEVICE_HANDLER_DEVICE, device,
                                             NULL));
}

static void
connection_added (NmnDeviceHandler *handler,
                  NMExportedConnection *exported)
{
    NMConnection *wrapped;
    NMDeviceEthernet *device;
    GtkWidget *item;

    device = NM_DEVICE_ETHERNET (nmn_device_handler_get_device (handler));
    if (!nm_device_ethernet_get_carrier (device))
        return;

    wrapped = nm_exported_connection_get_connection (exported);
    if (utils_connection_valid_for_device (wrapped, NM_DEVICE (device), NULL)) {
        item = nmn_ethernet_item_new (nmn_device_handler_get_nm_data (handler), device);
        g_object_set (item, NMN_NETWORK_ITEM_CONNECTION, exported, NULL);
        nmn_device_handler_add_item (handler, NMN_ITEM (item));
    }
}

static void
state_changed (NmnDeviceHandler *handler, gboolean active)
{
    if (active)
        nmn_device_handler_add_items (handler);
    else
        nmn_device_handler_remove_items (handler);
}

static void
carrier_changed (NMDeviceEthernet *device,
                 GParamSpec *pspec,
                 gpointer user_data)
{
    state_changed (NMN_DEVICE_HANDLER (user_data), nm_device_ethernet_get_carrier (device));
}

static void
ethernet_toggled (NmnNMData *nm_data,
                  gboolean active,
                  gpointer user_data)
{
    NmnDeviceHandler *handler = NMN_DEVICE_HANDLER (user_data);

    state_changed (handler, active);

    /* When ethernet is enabled, it means we really want it enabled */
    if (active) {
        GSList *list;
        GSList *iter;

        list = nmn_device_handler_get_items (handler);
        for (iter = list; iter; iter = iter->next) {
            NmnItem *item = NMN_ITEM (iter->data);

            if (nmn_item_get_status (item) == NMN_ITEM_STATUS_DISCONNECTED)
                nmn_item_connect_request (item);
        }
    }
}

static GObject*
constructor (GType type,
             guint n_construct_params,
             GObjectConstructParam *construct_params)
{
    GObject *object;
    NmnEthernetHandlerPrivate *priv;
    NMDeviceEthernet *device;
    NmnNMData *nm_data;

    object = G_OBJECT_CLASS (nmn_ethernet_handler_parent_class)->constructor
        (type, n_construct_params, construct_params);

    if (!object)
        return NULL;

    priv = GET_PRIVATE (object);

    device = NM_DEVICE_ETHERNET (nmn_device_handler_get_device (NMN_DEVICE_HANDLER (object)));
    priv->carrier_changed_id = g_signal_connect (device, "notify::" NM_DEVICE_ETHERNET_CARRIER,
                                                 G_CALLBACK (carrier_changed), object);

    nm_data = nmn_device_handler_get_nm_data (NMN_DEVICE_HANDLER (object));
    priv->ethernet_toggled_id = g_signal_connect (nm_data, "ethernet-toggled",
                                                  G_CALLBACK (ethernet_toggled), object);

    return object;
}

static void
nmn_ethernet_handler_init (NmnEthernetHandler *handler)
{
}

static void
dispose (GObject *object)
{
    NmnEthernetHandlerPrivate *priv = GET_PRIVATE (object);
    NMDevice *device;
    NmnNMData *nm_data;

    if (priv->disposed)
        return;

    device = nmn_device_handler_get_device (NMN_DEVICE_HANDLER (object));
    g_signal_handler_disconnect (device, priv->carrier_changed_id);

    nm_data = nmn_device_handler_get_nm_data (NMN_DEVICE_HANDLER (object));
    g_signal_handler_disconnect (nm_data, priv->ethernet_toggled_id);

    priv->disposed = TRUE;

    G_OBJECT_CLASS (nmn_ethernet_handler_parent_class)->dispose (object);
}

static void
nmn_ethernet_handler_class_init (NmnEthernetHandlerClass *class)
{
    GObjectClass *object_class = G_OBJECT_CLASS (class);
    NmnDeviceHandlerClass *handler_class = NMN_DEVICE_HANDLER_CLASS (class);

    g_type_class_add_private (object_class, sizeof (NmnEthernetHandlerPrivate));

    object_class->constructor = constructor;
    object_class->dispose = dispose;

    handler_class->connection_added = connection_added;
}
