/* -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */

#include "nmn-list.h"

#define NMN_DRAG_TARGET "NMN_DRAG_TARGET"
static const GtkTargetEntry nmn_list_targets [] = {
    { NMN_DRAG_TARGET, GTK_TARGET_SAME_APP, 0 },
};

G_DEFINE_TYPE (NmnList, nmn_list, GTK_TYPE_VBOX)

#define NMN_LIST_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj), NMN_TYPE_LIST, NmnListPrivate))

typedef struct {
    GtkWidget *dnd_window;
} NmnListPrivate;

GtkWidget *
nmn_list_new (void)
{
    return GTK_WIDGET (g_object_new (NMN_TYPE_LIST, NULL));
}

typedef struct {
    GtkWidget *needle;
    int counter;
    int found;
} ContainerPosInfo;

static void
get_position_cb (GtkWidget *widget, gpointer data)
{
    ContainerPosInfo *info = (ContainerPosInfo *) data;

    if (widget == info->needle)
        info->found = info->counter;

    info->counter++;
}

static int
container_get_position (GtkContainer *container, GtkWidget *child)
{
    ContainerPosInfo info;

    info.counter = 0;
    info.found = -1;
    info.needle = child;

    gtk_container_foreach (container, get_position_cb, &info);

    return info.found;
}

static void
drag_begin (GtkWidget *widget, GdkDragContext *drag_context, gpointer user_data)
{
    NmnListPrivate *priv = NMN_LIST_GET_PRIVATE (user_data);

    g_object_set_data (G_OBJECT (widget), "original-location",
                       GINT_TO_POINTER (container_get_position (GTK_CONTAINER (user_data), widget)));

    g_object_ref (widget);
    gtk_container_remove (GTK_CONTAINER (user_data), widget);

    priv->dnd_window = gtk_window_new (GTK_WINDOW_POPUP);
    gtk_container_add (GTK_CONTAINER (priv->dnd_window), widget);
    g_object_unref (widget);

    gtk_drag_set_icon_widget (drag_context, priv->dnd_window, -2, -2);
}

static void
drag_data_get (GtkWidget *widget,
               GdkDragContext *context,
               GtkSelectionData *data,
               guint info,
               guint time)
{
    if (data->target == gdk_atom_intern_static_string (NMN_DRAG_TARGET)) {
        gtk_selection_data_set (data,
                                data->target,
                                8,
                                (void*) &widget,
                                sizeof (gpointer));
    }
}

static gboolean
drag_failed (GtkWidget *widget,
             GdkDragContext *drag_context,
             GtkDragResult result,
             gpointer user_data)
{
    if (result != GTK_DRAG_RESULT_SUCCESS) {
        NmnListPrivate *priv = NMN_LIST_GET_PRIVATE (user_data);

        g_debug ("drag failed");
        g_object_ref (widget);
        gtk_container_remove (GTK_CONTAINER (priv->dnd_window), widget);
        gtk_box_pack_start (GTK_BOX (user_data), widget, FALSE, FALSE, 0);
        gtk_box_reorder_child (GTK_BOX (user_data), widget,
                               GPOINTER_TO_INT (g_object_get_data (G_OBJECT (widget), "original-location")));

        g_object_unref (widget);

        return TRUE;
    }

    return FALSE;
}

static void
drag_end (GtkWidget *widget, GdkDragContext *drag_context, gpointer user_data)
{
    NmnListPrivate *priv = NMN_LIST_GET_PRIVATE (user_data);

    g_debug ("drag end");
    GTK_BIN (priv->dnd_window)->child = NULL;
    gtk_widget_destroy (priv->dnd_window);
}

static gint
compare_items (gconstpointer a,
               gconstpointer b)
{
    guint aa;
    guint bb;

    aa = nmn_item_get_priority (NMN_ITEM (a));
    bb = nmn_item_get_priority (NMN_ITEM (b));

    if (aa < bb)
        return 1;

    if (aa == bb)
        return 0;

    return -1;
}

static void
nmn_list_sort (NmnList *list)
{
    GList *items;
    GList *iter;
    int i;

    items = gtk_container_get_children (GTK_CONTAINER (list));
    items = g_list_sort (items, compare_items);
    i = 0;

    for (iter = items; iter; iter = iter->next)
        gtk_box_reorder_child (GTK_BOX (list), GTK_WIDGET (iter->data), i++);
}

void
nmn_list_add_item (NmnList *list,
                   NmnItem *item)
{
    g_return_if_fail (NMN_IS_LIST (list));
    g_return_if_fail (NMN_IS_ITEM (item));

    gtk_drag_source_set (GTK_WIDGET (item), GDK_BUTTON1_MASK, 
                         nmn_list_targets, G_N_ELEMENTS (nmn_list_targets),
                         GDK_ACTION_MOVE);

    g_signal_connect (item, "drag-begin",
                      G_CALLBACK (drag_begin),
                      list);

    g_signal_connect (item, "drag-data-get",
                      G_CALLBACK (drag_data_get),
                      list);

    g_signal_connect (item, "drag-failed",
                      G_CALLBACK (drag_failed),
                      list);

    g_signal_connect (item, "drag-end",
                      G_CALLBACK (drag_end),
                      list);

    g_signal_connect_swapped (item, "priority-changed",
                              G_CALLBACK (nmn_list_sort),
                              list);

    gtk_box_pack_start (GTK_BOX (list), GTK_WIDGET (item), FALSE, FALSE, 0);
    nmn_list_sort (list);
    gtk_widget_show (GTK_WIDGET (item));
}


typedef struct {
    GtkWidget *match;
    gint match_position;
    gint y;
    gint counter;
} FindActiveInfo;

static void
find_active_item_cb (GtkWidget *widget, gpointer data)
{
    FindActiveInfo *info = (FindActiveInfo *) data;

    if (widget->allocation.y <= info->y && widget->allocation.y + widget->allocation.height > info->y) {
        info->match = widget;
        info->match_position = info->counter;
    }

    info->counter++;
}

static gboolean
find_active_item (GtkContainer *container, gint y, GtkWidget **active_widget, gint *active_position)
{
    FindActiveInfo info;

    info.match = NULL;
    info.match_position = 0;
    info.counter = 0;
    info.y = y;
    gtk_container_foreach (container, find_active_item_cb, &info);

    if (info.match) {
        *active_widget = info.match;
        *active_position = info.match_position;
    }

    return info.match != NULL;
}

static gboolean
nmn_list_drag_drop (GtkWidget *widget,
                    GdkDragContext *context,
                    gint x,
                    gint y,
                    guint time)
{
    GdkAtom target, item_target;

    g_debug ("drag_drop1");
    target = gtk_drag_dest_find_target (widget, context, NULL);
    item_target = gdk_atom_intern_static_string (NMN_DRAG_TARGET);

    if (target == item_target) {
        gtk_drag_get_data (widget, context, target, time);
        g_debug ("drag_drop2");
        return TRUE;
    }

    g_debug ("drag_drop3");
    return FALSE;
}

static void
nmn_list_drag_data_received (GtkWidget *widget,
                             GdkDragContext *context,
                             gint x,
                             gint y,
                             GtkSelectionData *data,
                             guint info,
                             guint time)
{
    GtkWidget *source_widget;
    NmnListPrivate *priv = NMN_LIST_GET_PRIVATE (widget);

    g_debug ("data_received1");
    source_widget = gtk_drag_get_source_widget (context);
    if (source_widget && data->target == gdk_atom_intern_static_string (NMN_DRAG_TARGET)) {
        GtkWidget *active_widget;
        gint active_position = -1;

        g_debug ("data_received2");
        find_active_item (GTK_CONTAINER (widget), y, &active_widget, &active_position);

        g_object_ref (source_widget);
        g_print ("child %p, source_widget: %p target: %p\n", GTK_BIN (priv->dnd_window)->child, source_widget, (void*) data->data);
        gtk_container_remove (GTK_CONTAINER (priv->dnd_window), source_widget);
        gtk_box_pack_start (GTK_BOX (widget), source_widget, FALSE, FALSE, 0);
        gtk_box_reorder_child (GTK_BOX (widget), source_widget, active_position);
        g_object_unref (source_widget);

        gtk_drag_finish (context, TRUE, FALSE, time);
    } else {
        g_debug ("data_received3");
        gtk_drag_finish (context, FALSE, FALSE, time);
    }

    g_debug ("data_received4");
}

static void
nmn_list_init (NmnList *list)
{
    gtk_drag_dest_set (GTK_WIDGET (list),
                       GTK_DEST_DEFAULT_ALL,
                       nmn_list_targets,
                       G_N_ELEMENTS (nmn_list_targets),
                       GDK_ACTION_MOVE);

    gtk_box_set_homogeneous (GTK_BOX (list), FALSE);
    gtk_box_set_spacing (GTK_BOX (list), 12);
}

static void
nmn_list_class_init (NmnListClass *class)
{
    GObjectClass *object_class = G_OBJECT_CLASS (class);
    GtkWidgetClass *widget_class = GTK_WIDGET_CLASS (class);

    g_type_class_add_private (object_class, sizeof (NmnListPrivate));

    widget_class->drag_drop = nmn_list_drag_drop;
    widget_class->drag_data_received = nmn_list_drag_data_received;
}
