/* mozhelper: A GObject wrapper for the Mozilla Mozhelper API
 *
 * Copyright (C) 2009  Intel Corporation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <glib-object.h>
#include <dbus/dbus-glib.h>
#include <new>
#include <nsINavHistoryService.h>
#include <nsIFaviconService.h>
#include <nsINavBookmarksService.h>
#include <nsIAutoCompleteSearch.h>
#include <nsIAutoCompleteResult.h>
#include <nsIIOService.h>
#include <nsIGlobalHistory2.h>
#include <nsIBrowserHistory.h>
#include <nsToolkitCompsCID.h>
#include <nsNetCID.h>
#include <nsServiceManagerUtils.h>
#include <nsCOMPtr.h>
#include <nsStringAPI.h>
#include <nsMemory.h>
#include <nsNetUtil.h>
#include <nsDocShellCID.h>

#include "mozhelper-history.h"
#include "mozhelper-error-private.h"
#include "mozhelper-service.h"
#include "mozhelper-marshal.h"

/* D-Bus method implementations */

static gboolean mozhelper_history_add_uri (MozhelperHistory *self,
                                        const gchar *uri,
                                        gboolean redirect,
                                        gboolean top_level,
                                        const gchar *referrer,
                                        GError **error);

static gboolean mozhelper_history_is_visited (MozhelperHistory *self,
                                           const gchar *uri,
                                           gboolean *is_visited,
                                           GError **error);

static gboolean mozhelper_history_set_page_title (MozhelperHistory *self,
                                               const gchar *uri,
                                               const gchar *title,
                                               GError **error);

static gboolean mozhelper_history_start_ac_search (MozhelperHistory *self,
                                                const gchar *search_str,
                                                guint32 *search_id,
                                                GError **error);

static gboolean mozhelper_history_stop_ac_search (MozhelperHistory *self,
                                               guint32 search_id,
                                               GError **error);

static gboolean mozhelper_history_set_favicon_url (MozhelperHistory *self,
                                                const gchar *page_uri_str,
                                                const gchar *favicon_uri_str,
                                                GError **error);

static gboolean mozhelper_history_set_default_favicon_url
                                               (MozhelperHistory *self,
                                                const gchar *page_uri_str,
                                                GError **error);

static gboolean mozhelper_history_get_favicon (MozhelperHistory *self,
                                            const gchar *page_uri,
                                            gboolean download,
                                            DBusGMethodInvocation *context);

static gboolean mozhelper_history_get_favorites (MozhelperHistory *self,
                                           GError **error);

static gboolean mozhelper_history_remove_favorite (MozhelperHistory *self,
                                             const gchar *uri,
                                             GError **error);

static gboolean mozhelper_history_clear_history (MozhelperHistory *self,
                                           GError **error);

static gboolean mozhelper_history_get_is_page_pinned (MozhelperHistory *self,
                                                const gchar *page_uri,
                                                gboolean *is_page_pinned,
                                                GError **error);

static gboolean mozhelper_history_get_pinned_pages (MozhelperHistory *self,
                                              GError **error);

static gboolean mozhelper_history_pin_page (MozhelperHistory *self,
                                      const gchar *uri_str,
                                      const gchar *page_title,
                                      GError **error);

static gboolean mozhelper_history_unpin_page (MozhelperHistory *self,
                                        const gchar *uri_str,
                                        GError **error);

static gboolean mozhelper_history_unpin_all_pages (MozhelperHistory *self,
                                             GError **error);

#include "mozhelper-history-glue.h"

/* End D-Bus method implementations */

static void mozhelper_history_finalize (GObject *object);

G_DEFINE_TYPE (MozhelperHistory, mozhelper_history, G_TYPE_OBJECT);

#define MOZHELPER_HISTORY_GET_PRIVATE(obj) \
  (G_TYPE_INSTANCE_GET_PRIVATE ((obj), MOZHELPER_TYPE_HISTORY, \
                                MozhelperHistoryPrivate))

#define MOZHELPER_HISTORY_N_FAVORITES 36

class MozhelperHistoryObserver : public nsIAutoCompleteObserver,
                              public nsINavHistoryObserver
{
public:
  NS_DECL_ISUPPORTS
  NS_DECL_NSIAUTOCOMPLETEOBSERVER
  NS_DECL_NSINAVHISTORYOBSERVER

  MozhelperHistory *history;
};

class MozhelperBookmarksObserver : public nsINavBookmarkObserver
{
public:
  NS_DECL_ISUPPORTS
  NS_DECL_NSINAVBOOKMARKOBSERVER

  MozhelperHistory *history;
};

struct _MozhelperHistoryPrivate
{
  guint32 ac_search_id;
  gboolean in_ac_search;

  /* Number of history items that we have reported back from the
     result set so far */
  guint n_results_reported;

  gboolean history_observer_added;
  MozhelperHistoryObserver observer;

  gboolean bookmarks_observer_added;
  MozhelperBookmarksObserver bookmarks_observer;

  GSList *favicon_closures;

  /* Idle handler for fetching the favorites */
  guint favorites_idle_handler;
  /* and for fetching the pinned pages */
  guint pinned_pages_idle_handler;
};

struct MozhelperHistoryFaviconClosure
{
  MozhelperHistoryFaviconClosure (nsIURI *_page_uri, nsIURI *_favicon_uri,
                               DBusGMethodInvocation *_context)
    : page_uri (_page_uri),
      favicon_uri (_favicon_uri),
      context (_context)
  {
  }

  nsCOMPtr<nsIURI> page_uri, favicon_uri;
  DBusGMethodInvocation *context;
};

enum
  {
    AC_RESULT_RECEIVED_SIGNAL,
    LINK_VISITED_SIGNAL,
    FAVORITES_RECEIVED_SIGNAL,
    PINNED_PAGE_SIGNAL,
    UNPINNED_PAGE_SIGNAL,

    LAST_SIGNAL
  };

static guint history_signals[LAST_SIGNAL] = { 0, };

NS_IMPL_QUERY_INTERFACE2 (MozhelperHistoryObserver,
                          nsIAutoCompleteObserver,
                          nsINavHistoryObserver)

/* We don't want to implement reference counting for
   MozhelperHistoryObserver because it is entirely owned by the
   MozhelperHistory instance */
NS_IMETHODIMP_(nsrefcnt)
MozhelperHistoryObserver::AddRef ()
{
  return 1;
}

NS_IMETHODIMP_(nsrefcnt)
MozhelperHistoryObserver::Release ()
{
  return 1;
}

static void
mozhelper_history_favicon_closure_destroy (MozhelperHistoryFaviconClosure *data)
{
  if (data->context)
    {
      GError *error = NULL;
      mozhelper_error_set_from_nsresult (NS_ERROR_NOT_AVAILABLE, &error);
      dbus_g_method_return_error (data->context, error);
      g_error_free (error);
    }

  data->~MozhelperHistoryFaviconClosure ();
  g_slice_free (MozhelperHistoryFaviconClosure, data);
}

NS_IMETHODIMP
MozhelperHistoryObserver::OnSearchResult (nsIAutoCompleteSearch *search,
                                       nsIAutoCompleteResult *result)
{
  nsresult rv;
  MozhelperHistoryPrivate *priv = history->priv;
  PRUint16 search_result;
  PRUint32 match_count;

  rv = result->GetSearchResult (&search_result);

  if (NS_FAILED (rv))
    {
      g_warning ("GetSearchResult failed: 0x%x\n", rv);
      return rv;
    }

  rv = result->GetMatchCount (&match_count);

  if (NS_FAILED (rv))
    {
      g_warning ("GetMatchCount failed: 0x%x\n", rv);
      return rv;
    }

  if (search_result == nsIAutoCompleteResult::RESULT_SUCCESS
      || search_result == nsIAutoCompleteResult::RESULT_SUCCESS_ONGOING)
    while (priv->n_results_reported < match_count)
      {
        nsAutoString value, comment, image;

        if (NS_SUCCEEDED (result->GetValueAt (priv->n_results_reported, value))
            && NS_SUCCEEDED (result->GetCommentAt (priv->n_results_reported,
                                                   comment)))
          {
            NS_ConvertUTF16toUTF8 value_utf8 (value);
            NS_ConvertUTF16toUTF8 comment_utf8 (comment);

            g_signal_emit (history,
                           history_signals[AC_RESULT_RECEIVED_SIGNAL], 0,
                           priv->ac_search_id,
                           value_utf8.get (),
                           comment_utf8.get ());
          }

        priv->n_results_reported++;
      }

  return NS_OK;
}

NS_IMETHODIMP
MozhelperHistoryObserver::OnBeginUpdateBatch ()
{
  return NS_ERROR_NOT_IMPLEMENTED;
}

NS_IMETHODIMP
MozhelperHistoryObserver::OnEndUpdateBatch ()
{
  return NS_ERROR_NOT_IMPLEMENTED;
}

NS_IMETHODIMP
MozhelperHistoryObserver::OnVisit (nsIURI *uri,
                                PRInt64 visit_id,
                                PRTime when,
                                PRInt64 session_id,
                                PRInt64 referring_id,
                                PRUint32 transition_type,
                                PRUint32 *added NS_OUTPARAM)
{
  gint visit_time = when / G_USEC_PER_SEC;
  nsCAutoString uri_spec;
  nsresult rv;

  rv = uri->GetSpec (uri_spec);
  if (NS_FAILED (rv))
    return rv;

  // Signal that a link has been visited
  g_signal_emit (history, history_signals[LINK_VISITED_SIGNAL], 0,
                 uri_spec.get (), visit_time);

  return NS_OK;
}

NS_IMETHODIMP
MozhelperHistoryObserver::OnTitleChanged (nsIURI *uri,
                                       const nsAString &page_title)
{
  return NS_ERROR_NOT_IMPLEMENTED;
}

NS_IMETHODIMP
MozhelperHistoryObserver::OnDeleteURI (nsIURI *uri)
{
  return NS_ERROR_NOT_IMPLEMENTED;
}

NS_IMETHODIMP
MozhelperHistoryObserver::OnClearHistory ()
{
  return NS_ERROR_NOT_IMPLEMENTED;
}

NS_IMETHODIMP
MozhelperHistoryObserver::OnPageChanged (nsIURI *uri,
                                      PRUint32 what_changed,
                                      const nsAString &value)
{
  MozhelperHistoryPrivate *priv = history->priv;
  nsresult rv;

  if (what_changed == ATTRIBUTE_FAVICON)
    {
      GSList *next;

      nsCOMPtr<nsIFaviconService> favicon_service
        = do_GetService (NS_FAVICONSERVICE_CONTRACTID, &rv);

      if (NS_FAILED (rv))
        return rv;

      /* Look for a matching URI in the list of closures we are
         waiting for */
      for (GSList *l = priv->favicon_closures; l; l = next)
        {
          MozhelperHistoryFaviconClosure *data
            = (MozhelperHistoryFaviconClosure *) l->data;
          PRBool uri_equals;

          next = l->next;

          rv = data->page_uri->Equals (uri, &uri_equals);

          if (NS_SUCCEEDED (rv) && uri_equals)
            {
              nsCAutoString mime_type;
              PRUint32 data_len;
              PRUint8 *icon_data;

              rv = favicon_service->GetFaviconData (data->favicon_uri,
                                                    mime_type,
                                                    &data_len, &icon_data);

              if (NS_FAILED (rv))
                {
                  GError *error = NULL;
                  mozhelper_error_set_from_nsresult (rv, &error);
                  dbus_g_method_return_error (data->context, error);
                  g_error_free (error);
                }
              else
                {
                  /* Copy the data into a GArray to pass to D-BUS */
                  GArray *data_array =
                    g_array_sized_new (FALSE, FALSE, sizeof (guint8), data_len);

                  g_array_append_vals (data_array, icon_data, data_len);

                  dbus_g_method_return (data->context,
                                        mime_type.get (),
                                        data_array);

                  g_array_free (data_array, TRUE);
                  nsMemory::Free (icon_data);
                }

              data->context = NULL;

              priv->favicon_closures
                = g_slist_remove (priv->favicon_closures, data);
              mozhelper_history_favicon_closure_destroy (data);
            }
        }
    }

  return NS_OK;
}

NS_IMETHODIMP
MozhelperHistoryObserver::OnPageExpired (nsIURI *uri,
                                      PRTime visit_time,
                                      PRBool whole_entry)
{
  return NS_ERROR_NOT_IMPLEMENTED;
}

NS_IMPL_QUERY_INTERFACE1 (MozhelperBookmarksObserver,
                          nsINavBookmarkObserver)

/* We don't want to implement reference counting for
   MozhelperBookmarksObserver because it is entirely owned by the
   MozhelperHistory instance */
NS_IMETHODIMP_(nsrefcnt)
MozhelperBookmarksObserver::AddRef ()
{
  return 1;
}

NS_IMETHODIMP_(nsrefcnt)
MozhelperBookmarksObserver::Release ()
{
  return 1;
}

NS_IMETHODIMP
MozhelperBookmarksObserver::OnBeginUpdateBatch ()
{
  return NS_ERROR_NOT_IMPLEMENTED;
}

NS_IMETHODIMP
MozhelperBookmarksObserver::OnEndUpdateBatch ()
{
  return NS_ERROR_NOT_IMPLEMENTED;
}

static nsresult
mozhelper_history_emit_pinned_page_for_item (MozhelperHistory *self,
                                       nsCOMPtr<nsINavBookmarksService>
                                       bookmark_service,
                                       nsCOMPtr<nsINavHistoryService>
                                       history_service,
                                       PRInt64 bookmark_id,
                                       gboolean more_pending)
{
  nsresult rv = NS_OK;
  nsCAutoString title, uri_str;
  PRUint16 item_type;
  nsCOMPtr<nsIURI> uri;

  rv = bookmark_service->GetItemType (bookmark_id, &item_type);
  if (NS_FAILED (rv))
    goto rv_error;

  if (item_type == nsINavBookmarksService::TYPE_BOOKMARK)
    {
      nsCOMPtr<nsINavHistoryQuery> query;
      nsCOMPtr<nsINavHistoryQueryOptions> query_options;
      nsCOMPtr<nsINavHistoryResult> query_results;
      nsCOMPtr<nsINavHistoryContainerResultNode> root_result;
      nsCOMPtr<nsINavHistoryResultNode> result_node;
      PRTime visit_time = 0;

      rv = bookmark_service->GetItemTitle (bookmark_id, title);
      if (NS_FAILED (rv))
        goto rv_error;
      rv = bookmark_service->GetBookmarkURI (bookmark_id,
                                             getter_AddRefs (uri));
      if (NS_FAILED (rv))
        goto rv_error;

      /* Query for the last visit time to this URI */
      rv = history_service->GetNewQuery (getter_AddRefs (query));
      if (NS_FAILED (rv))
        goto rv_error;
      rv = history_service->GetNewQueryOptions (getter_AddRefs (query_options));
      if (NS_FAILED (rv))
        goto rv_error;

      rv = query->SetUri (uri);
      if (NS_FAILED (rv))
        goto rv_error;
      rv = query_options->SetResultType (nsINavHistoryQueryOptions
                                         ::RESULTS_AS_URI);
      if (NS_FAILED (rv))
        goto rv_error;

      rv = history_service->ExecuteQuery (query, query_options,
                                          getter_AddRefs (query_results));
      if (NS_FAILED (rv))
        goto rv_error;

      rv = query_results->GetRoot (getter_AddRefs (root_result));
      if (NS_FAILED (rv))
        goto rv_error;

      rv = root_result->SetContainerOpen (PR_TRUE);
      if (NS_FAILED (rv))
        goto rv_error;

      rv = root_result->GetChild (0, getter_AddRefs (result_node));
      if (NS_SUCCEEDED (rv))
        {
          rv = result_node->GetTime (&visit_time);
          if (NS_FAILED (rv))
            goto rv_error;
        }
      /* If there was no entry then it's not an error and we can just
         leave the visit time as zero */
      else if (rv != NS_ERROR_INVALID_ARG)
        goto rv_error;

      rv = uri->GetSpec (uri_str);
      if (NS_FAILED (rv))
        goto rv_error;

      g_signal_emit (self, history_signals[PINNED_PAGE_SIGNAL], 0,
                     title.get (), uri_str.get (),
                     (gint) (visit_time / G_USEC_PER_SEC),
                     more_pending);
    }

 rv_error:
  return rv;
}

NS_IMETHODIMP
MozhelperBookmarksObserver::OnItemAdded (PRInt64 item_id,
                                   PRInt64 folder,
                                   PRInt32 index)
{
  nsresult rv;
  nsCOMPtr<nsINavBookmarksService> bookmark_service;
  nsCOMPtr<nsINavHistoryService> history_service;

  bookmark_service = do_GetService (NS_NAVBOOKMARKSSERVICE_CONTRACTID, &rv);
  if (NS_FAILED (rv))
    return rv;
  history_service = do_GetService (NS_NAVHISTORYSERVICE_CONTRACTID, &rv);
  if (NS_FAILED (rv))
    return rv;

  mozhelper_history_emit_pinned_page_for_item (history,
                                         bookmark_service,
                                         history_service,
                                         item_id,
                                         FALSE);
  if (NS_FAILED (rv))
    return rv;

  return NS_OK;
}

NS_IMETHODIMP
MozhelperBookmarksObserver::OnItemRemoved (PRInt64 item_id,
                                     PRInt64 folder,
                                     PRInt32 index)
{
  return NS_ERROR_NOT_IMPLEMENTED;
}

NS_IMETHODIMP
MozhelperBookmarksObserver::OnItemChanged (PRInt64 bookmark_id,
                                     const nsACString &property,
                                     PRBool is_annotation_property,
                                     const nsACString &value)
{
  return NS_ERROR_NOT_IMPLEMENTED;
}

NS_IMETHODIMP
MozhelperBookmarksObserver::OnItemVisited (PRInt64 bookmark_id,
                                     PRInt64 visit_id,
                                     PRTime time)
{
  return NS_ERROR_NOT_IMPLEMENTED;
}

NS_IMETHODIMP
MozhelperBookmarksObserver::OnItemMoved (PRInt64 item_id,
                                   PRInt64 old_parent,
                                   PRInt32 old_index,
                                   PRInt64 new_parent,
                                   PRInt32 new_index)
{
  return NS_ERROR_NOT_IMPLEMENTED;
}

static void
mozhelper_history_class_init (MozhelperHistoryClass *klass)
{
  GObjectClass *gobject_class = (GObjectClass *) klass;

  gobject_class->finalize = mozhelper_history_finalize;

  history_signals[AC_RESULT_RECEIVED_SIGNAL] =
    g_signal_new ("ac-result-received",
                  G_TYPE_FROM_CLASS (gobject_class),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (MozhelperHistoryClass, ac_result_received),
                  NULL, NULL,
                  mozhelper_marshal_VOID__UINT_STRING_STRING,
                  G_TYPE_NONE, 3,
                  G_TYPE_UINT,
                  G_TYPE_STRING,
                  G_TYPE_STRING);

  history_signals[LINK_VISITED_SIGNAL] =
    g_signal_new ("link-visited",
                  G_TYPE_FROM_CLASS (gobject_class),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (MozhelperHistoryClass, link_visited),
                  NULL, NULL,
                  mozhelper_marshal_VOID__STRING_INT,
                  G_TYPE_NONE, 2, G_TYPE_STRING, G_TYPE_INT);

  history_signals[FAVORITES_RECEIVED_SIGNAL] =
    g_signal_new ("favorites-received",
                  G_TYPE_FROM_CLASS (gobject_class),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (MozhelperHistoryClass, favorites_received),
                  NULL, NULL,
                  mozhelper_marshal_VOID__BOXED_BOXED,
                  G_TYPE_NONE, 2, G_TYPE_STRV, G_TYPE_STRV);

  history_signals[PINNED_PAGE_SIGNAL] =
    g_signal_new ("pinned-page",
                  G_TYPE_FROM_CLASS (gobject_class),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (MozhelperHistoryClass, pinned_page),
                  NULL, NULL,
                  mozhelper_marshal_VOID__STRING_STRING_INT_BOOL,
                  G_TYPE_NONE, 4,
                  G_TYPE_STRING, G_TYPE_STRING, G_TYPE_INT, G_TYPE_BOOLEAN);

  history_signals[UNPINNED_PAGE_SIGNAL] =
    g_signal_new ("unpinned-page",
                  G_TYPE_FROM_CLASS (gobject_class),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (MozhelperHistoryClass, unpinned_page),
                  NULL, NULL,
                  g_cclosure_marshal_VOID__STRING,
                  G_TYPE_NONE, 1, G_TYPE_STRING);

  g_type_class_add_private (klass, sizeof (MozhelperHistoryPrivate));

  dbus_g_object_type_install_info (MOZHELPER_TYPE_HISTORY,
                                   &dbus_glib_mozhelper_history_object_info);
}

static void
mozhelper_history_init (MozhelperHistory *self)
{
  MozhelperHistoryPrivate *priv;
  DBusGConnection *connection;
  nsresult rv;
  GError *error = NULL;

  priv = self->priv = MOZHELPER_HISTORY_GET_PRIVATE (self);

  priv->ac_search_id = 1;

  /* 'placement new' to call the constructor for
     MozhelperHistoryPrivate */
  new (reinterpret_cast<void *> (priv)) MozhelperHistoryPrivate;

  if ((connection = dbus_g_bus_get (DBUS_BUS_SESSION, &error)) == NULL)
    {
      g_warning ("Error connecting to session bus: %s", error->message);
      g_error_free (error);
    }
  else
    {
      dbus_g_connection_register_g_object (connection,
                                           MOZHELPER_SERVICE_HISTORY_PATH,
                                           G_OBJECT (self));
      dbus_g_connection_unref (connection);
    }

  priv->observer.history = self;
  priv->bookmarks_observer.history = self;

  /* Register the observer for bookmark notifications */
  nsCOMPtr<nsINavBookmarksService> bookmarks_service
    = do_GetService (NS_NAVBOOKMARKSSERVICE_CONTRACTID, &rv);
  if (NS_SUCCEEDED (rv))
    {
      rv = bookmarks_service->AddObserver (&priv->bookmarks_observer,
                                           PR_FALSE);

      if (NS_SUCCEEDED (rv))
        priv->bookmarks_observer_added = TRUE;
    }

  /* Register for history notifications */
  nsCOMPtr<nsINavHistoryService> history_service
    = do_GetService (NS_NAVHISTORYSERVICE_CONTRACTID, &rv);
  if (NS_SUCCEEDED (rv))
    {
      rv = history_service->AddObserver (&priv->observer, PR_FALSE);

      if (NS_SUCCEEDED (rv))
        priv->history_observer_added = TRUE;
    }
}

static void
mozhelper_history_finalize (GObject *object)
{
  MozhelperHistory *self = (MozhelperHistory *) object;
  MozhelperHistoryPrivate *priv = self->priv;
  nsresult rv;

  mozhelper_history_stop_ac_search (self, priv->ac_search_id, NULL);

  if (priv->history_observer_added)
    {
      /* Unregister the observer for history notifications as the observer
         object is going to be destroyed */
      nsCOMPtr<nsINavHistoryService> history_service
        = do_GetService (NS_NAVHISTORYSERVICE_CONTRACTID, &rv);
      if (NS_SUCCEEDED (rv))
        history_service->RemoveObserver (&priv->observer);
    }
  if (priv->bookmarks_observer_added)
    {
      /* Same for bookmark notifications */
      nsCOMPtr<nsINavBookmarksService> bookmarks_service
        = do_GetService (NS_NAVBOOKMARKSSERVICE_CONTRACTID, &rv);
      if (NS_SUCCEEDED (rv))
        bookmarks_service->RemoveObserver (&priv->bookmarks_observer);
    }

  g_slist_foreach (priv->favicon_closures,
                   (GFunc) mozhelper_history_favicon_closure_destroy, NULL);
  g_slist_free (priv->favicon_closures);
  priv->favicon_closures = NULL;

  if (priv->favorites_idle_handler)
    {
      g_source_remove (priv->favorites_idle_handler);
      priv->favorites_idle_handler = 0;
    }

  if (priv->pinned_pages_idle_handler)
    {
      g_source_remove (priv->pinned_pages_idle_handler);
      priv->pinned_pages_idle_handler = 0;
    }

  /* Explicitly call the destructor for the private data (so that it
     destructs without trying to free the memory) */
  priv->~MozhelperHistoryPrivate ();

  G_OBJECT_CLASS (mozhelper_history_parent_class)->finalize (object);
}

MozhelperHistory *
mozhelper_history_new (void)
{
  MozhelperHistory *self = (MozhelperHistory *) g_object_new (MOZHELPER_TYPE_HISTORY,
                                                        NULL);

  return self;
}

static gboolean
mozhelper_history_add_uri (MozhelperHistory *self,
                        const gchar *uri,
                        gboolean redirect,
                        gboolean top_level,
                        const gchar *referrer,
                        GError **error)
{
  nsresult rv;

  g_return_val_if_fail (MOZHELPER_IS_HISTORY (self), FALSE);

  nsCOMPtr<nsIGlobalHistory2> global_history2
    = do_GetService (NS_NAVHISTORYSERVICE_CONTRACTID, &rv);

  if (NS_FAILED (rv))
    {
      mozhelper_error_set_from_nsresult (rv, error);
      return FALSE;
    }

  nsCOMPtr<nsIURI> nsuri;
  rv = NS_NewURI (getter_AddRefs (nsuri), uri);
  if (NS_FAILED (rv))
    {
      mozhelper_error_set_from_nsresult (rv, error);
      return FALSE;
    }

  nsCOMPtr<nsIURI> referrer_nsuri;
  if (redirect)
    {
      rv = NS_NewURI (getter_AddRefs (referrer_nsuri), uri);
      if (NS_FAILED (rv))
        {
          mozhelper_error_set_from_nsresult (rv, error);
          return FALSE;
        }
    }

  rv = global_history2->AddURI (nsuri, redirect, top_level,
                                referrer_nsuri);

  if (NS_FAILED (rv))
    {
      mozhelper_error_set_from_nsresult (rv, error);
      return FALSE;
    }

  return TRUE;
}

static gboolean
mozhelper_history_is_visited (MozhelperHistory *self,
                           const gchar *uri,
                           gboolean *is_visited,
                           GError **error)
{
  nsresult rv;

  g_return_val_if_fail (MOZHELPER_IS_HISTORY (self), FALSE);

  nsCOMPtr<nsIGlobalHistory2> global_history2
    = do_GetService (NS_NAVHISTORYSERVICE_CONTRACTID, &rv);

  if (NS_FAILED (rv))
    {
      mozhelper_error_set_from_nsresult (rv, error);
      return FALSE;
    }

  nsCOMPtr<nsIURI> nsuri;
  rv = NS_NewURI (getter_AddRefs (nsuri), uri);
  if (NS_FAILED (rv))
    {
      mozhelper_error_set_from_nsresult (rv, error);
      return FALSE;
    }

  if (is_visited)
    {
      PRBool result;
      rv = global_history2->IsVisited (nsuri, &result);

      if (NS_FAILED (rv))
        {
          mozhelper_error_set_from_nsresult (rv, error);
          return FALSE;
        }

      *is_visited = result;
    }

  return TRUE;
}

static gboolean
mozhelper_history_set_page_title (MozhelperHistory *self,
                               const gchar *uri,
                               const gchar *title,
                               GError **error)
{
  nsresult rv;

  nsCOMPtr<nsIGlobalHistory2> global_history2
    = do_GetService (NS_NAVHISTORYSERVICE_CONTRACTID, &rv);

  if (NS_FAILED (rv))
    {
      mozhelper_error_set_from_nsresult (rv, error);
      return FALSE;
    }

  nsCOMPtr<nsIURI> nsuri;
  rv = NS_NewURI (getter_AddRefs (nsuri), uri);
  if (NS_FAILED (rv))
    {
      mozhelper_error_set_from_nsresult (rv, error);
      return FALSE;
    }

  nsDependentCString title_cstring (title);
  NS_ConvertUTF8toUTF16 title_string (title_cstring);

  /* The redirect parameter is deprecated so we always set it to false */
  rv = global_history2->SetPageTitle (nsuri, title_string);

  if (NS_FAILED (rv))
    {
      mozhelper_error_set_from_nsresult (rv, error);
      return FALSE;
    }

  return TRUE;
}

static gboolean
mozhelper_history_start_ac_search (MozhelperHistory *self,
                                const gchar *search_str,
                                guint32 *search_id,
                                GError **error)
{
  MozhelperHistoryPrivate *priv = self->priv;
  nsresult rv;

  nsCOMPtr<nsIAutoCompleteSearch> auto_complete
    = do_GetService ("@mozilla.org/autocomplete/search;1?name=history", &rv);

  if (NS_FAILED (rv))
    {
      mozhelper_error_set_from_nsresult (rv, error);
      return FALSE;
    }

  /* Stop any existing searches */
  rv = auto_complete->StopSearch ();
  if (NS_FAILED (rv))
    {
      mozhelper_error_set_from_nsresult (rv, error);
      return FALSE;
    }

  priv->n_results_reported = 0;

  rv = auto_complete->StartSearch (NS_ConvertUTF8toUTF16 (search_str),
                                   EmptyString (),
                                   NULL,
                                   &priv->observer);
  if (NS_FAILED (rv))
    {
      mozhelper_error_set_from_nsresult (rv, error);
      return FALSE;
    }

  priv->in_ac_search = TRUE;

  *search_id = priv->ac_search_id;

  return TRUE;
}

static gboolean
mozhelper_history_stop_ac_search (MozhelperHistory *self,
                               guint32 search_id,
                               GError **error)
{
  nsresult rv;

  g_return_val_if_fail (MOZHELPER_IS_HISTORY (self), FALSE);

  MozhelperHistoryPrivate *priv = self->priv;

  if (priv->in_ac_search && priv->ac_search_id == search_id)
    {
      nsCOMPtr<nsIAutoCompleteSearch> auto_complete
        = do_GetService ("@mozilla.org/autocomplete/search;1?name=history", &rv);

      if (NS_FAILED (rv))
        {
          mozhelper_error_set_from_nsresult (rv, error);
          return FALSE;
        }

      rv = auto_complete->StopSearch ();
      if (NS_FAILED (rv))
        {
          mozhelper_error_set_from_nsresult (rv, error);
          return FALSE;
        }

      priv->in_ac_search = FALSE;

      /* Increase the search id so that we know to ignore any results
         for previous queries. In the unlikely event of wrap-around,
         we don't want to reuse search id 0 */
      if (++priv->ac_search_id == 0)
        priv->ac_search_id = 1;
    }

  return TRUE;
}

static gboolean
mozhelper_history_set_favicon_url (MozhelperHistory *self,
                                const gchar *page_uri_str,
                                const gchar *favicon_uri_str,
                                GError **error)
{
  nsresult rv;

  g_return_val_if_fail (MOZHELPER_IS_HISTORY (self), FALSE);

  nsCOMPtr<nsIURI> favicon_uri, page_uri;

  nsCOMPtr<nsIFaviconService> favicon_service
    = do_GetService (NS_FAVICONSERVICE_CONTRACTID, &rv);

  if (NS_FAILED (rv))
    {
      mozhelper_error_set_from_nsresult (rv, error);
      return FALSE;
    }

  rv = NS_NewURI (getter_AddRefs (page_uri), page_uri_str);
  if (NS_FAILED (rv))
    {
      mozhelper_error_set_from_nsresult (rv, error);
      return FALSE;
    }
  rv = NS_NewURI (getter_AddRefs (favicon_uri), favicon_uri_str);
  if (NS_FAILED (rv))
    {
      mozhelper_error_set_from_nsresult (rv, error);
      return FALSE;
    }

  rv = favicon_service->SetFaviconUrlForPage (page_uri, favicon_uri);

  if (NS_FAILED (rv))
    {
      mozhelper_error_set_from_nsresult (rv, error);
      return FALSE;
    }

  return TRUE;
}

static gboolean
mozhelper_history_set_default_favicon_url (MozhelperHistory *self,
                                        const gchar *page_uri_str,
                                        GError **error)
{
  nsresult rv;

  g_return_val_if_fail (MOZHELPER_IS_HISTORY (self), FALSE);

  nsCOMPtr<nsIURI> favicon_uri, page_uri;
  nsCAutoString uri_path;

  nsCOMPtr<nsIFaviconService> favicon_service
    = do_GetService (NS_FAVICONSERVICE_CONTRACTID, &rv);

  if (NS_FAILED (rv))
    {
      mozhelper_error_set_from_nsresult (rv, error);
      return FALSE;
    }

  rv = NS_NewURI (getter_AddRefs (page_uri), page_uri_str);
  if (NS_FAILED (rv))
    {
      mozhelper_error_set_from_nsresult (rv, error);
      return FALSE;
    }

  /* Make a new URL with the pre_path of the page URI + "/favicon.uri" */
  rv = page_uri->GetPrePath (uri_path);
  if (NS_FAILED (rv))
    {
      mozhelper_error_set_from_nsresult (rv, error);
      return FALSE;
    }

  uri_path.AppendLiteral ("/favicon.ico");

  rv = NS_NewURI (getter_AddRefs (favicon_uri), uri_path);
  if (NS_FAILED (rv))
    {
      mozhelper_error_set_from_nsresult (rv, error);
      return FALSE;
    }

  rv = favicon_service->SetFaviconUrlForPage (page_uri, favicon_uri);

  if (NS_FAILED (rv))
    {
      mozhelper_error_set_from_nsresult (rv, error);
      return FALSE;
    }

  return TRUE;
}

static gboolean
mozhelper_history_get_favicon (MozhelperHistory *self,
                            const gchar *page_uri_str,
                            gboolean download,
                            DBusGMethodInvocation *context)
{
  nsresult rv;
  GError *error = NULL;

  g_return_val_if_fail (MOZHELPER_IS_HISTORY (self), FALSE);

  MozhelperHistoryPrivate *priv = self->priv;
  nsCOMPtr<nsIURI> favicon_uri, page_uri;

  nsCOMPtr<nsIFaviconService> favicon_service
    = do_GetService (NS_FAVICONSERVICE_CONTRACTID, &rv);

  if (NS_FAILED (rv))
    goto rv_error;

  rv = NS_NewURI (getter_AddRefs (page_uri), page_uri_str);
  if (NS_FAILED (rv))
    goto rv_error;

  /* Get the URL for the favicon of this page or bail out if it hasn't
     been set yet */
  rv = favicon_service->GetFaviconForPage (page_uri,
                                           getter_AddRefs (favicon_uri));
  if (NS_FAILED (rv))
    goto rv_error;

  /* Check if mozhelper already has the data for the icon cached */
  {
    nsCAutoString mime_type;
    PRUint32 data_len;
    PRUint8 *data = nsnull;

    rv = favicon_service->GetFaviconData (favicon_uri, mime_type,
                                          &data_len, &data);

    /* The documentation says that GetFaviconData will
       return NS_ERROR_NOT_AVAILABLE is there is no data, but it
       actually appears to return an empty array instead */
    if (NS_SUCCEEDED (rv) && data_len == 0)
      {
        if (data != nsnull)
          nsMemory::Free (data);
        rv = NS_ERROR_NOT_AVAILABLE;
      }

    /* If it isn't available then start downloading it
       asynchronously */
    if (rv == NS_ERROR_NOT_AVAILABLE && download)
      {
        MozhelperHistoryFaviconClosure *closure
          = g_slice_new (MozhelperHistoryFaviconClosure);
        new (reinterpret_cast<void *> (closure))
          MozhelperHistoryFaviconClosure (page_uri, favicon_uri, context);
        priv->favicon_closures = g_slist_prepend (priv->favicon_closures, closure);

        rv = favicon_service->SetAndLoadFaviconForPage (page_uri, favicon_uri,
                                                        PR_FALSE);
        if (NS_FAILED (rv))
          {
            priv->favicon_closures
              = g_slist_remove (priv->favicon_closures, closure);
            closure->context = NULL;
            mozhelper_history_favicon_closure_destroy (closure);
            goto rv_error;
          }
      }
    else if (NS_FAILED (rv))
      goto rv_error;
    else
      {
        /* Copy the data into a GArray to pass to D-BUS */
        GArray *data_array =
          g_array_sized_new (FALSE, FALSE, sizeof (guint8), data_len);

        g_array_append_vals (data_array, data, data_len);

        dbus_g_method_return (context,
                              mime_type.get (),
                              data_array);

        g_array_free (data_array, TRUE);
        nsMemory::Free (data);
      }
  }

  return TRUE;

 rv_error:
  mozhelper_error_set_from_nsresult (rv, &error);
  dbus_g_method_return_error (context, error);
  g_error_free (error);
  return TRUE;
}

static gboolean
mozhelper_history_favorites_cb (gpointer data)
{
  nsresult rv = NS_OK;
  MozhelperHistory *self = (MozhelperHistory *) data;
  MozhelperHistoryPrivate *priv = self->priv;
  gchar **urls, **titles;
  PRUint32 root_child_count;

  priv->favorites_idle_handler = 0;

  nsCOMPtr<nsINavHistoryService> history_service;
  nsCOMPtr<nsINavHistoryQuery> query;
  nsCOMPtr<nsINavHistoryQueryOptions> query_options;
  nsCOMPtr<nsINavHistoryResult> result;
  nsCOMPtr<nsINavHistoryContainerResultNode> root_node;

  history_service = do_GetService (NS_NAVHISTORYSERVICE_CONTRACTID, &rv);
  if (NS_FAILED (rv))
    goto rv_error;

  rv = history_service->GetNewQuery (getter_AddRefs (query));
  if (NS_FAILED (rv))
    goto rv_error;

  rv = history_service->GetNewQueryOptions (getter_AddRefs (query_options));
  if (NS_FAILED (rv))
    goto rv_error;

  query_options->SetSortingMode (nsINavHistoryQueryOptions
                                 ::SORT_BY_VISITCOUNT_DESCENDING);

  if (NS_FAILED (rv))
    goto rv_error;

  query_options->SetResultType (nsINavHistoryQueryOptions
                                ::RESULTS_AS_SITE_QUERY);
  if (NS_FAILED (rv))
    goto rv_error;

  query_options->SetMaxResults (MOZHELPER_HISTORY_N_FAVORITES);
  if (NS_FAILED (rv))
    goto rv_error;

  query_options->SetExcludeQueries(TRUE);

  query_options->SetQueryType (nsINavHistoryQueryOptions
                               ::QUERY_TYPE_HISTORY);
  if (NS_FAILED (rv))
    goto rv_error;
  query_options->SetExpandQueries (PR_TRUE);
  if (NS_FAILED (rv))
    goto rv_error;

  rv = history_service->ExecuteQuery (query, query_options,
                                      getter_AddRefs (result));
  if (NS_FAILED (rv))
    goto rv_error;

  rv = result->GetRoot (getter_AddRefs (root_node));
  if (NS_FAILED (rv))
    goto rv_error;

  rv = root_node->SetContainerOpen (PR_TRUE);
  if (NS_FAILED (rv))
    goto rv_error;

  urls = (gchar **) g_malloc0 (sizeof (gchar *)
                               * (MOZHELPER_HISTORY_N_FAVORITES + 1));
  titles = (gchar **) g_malloc0 (sizeof (gchar *)
                                 * (MOZHELPER_HISTORY_N_FAVORITES + 1));

  rv = root_node->GetChildCount (&root_child_count);
  if (NS_SUCCEEDED (rv))
    {
      PRUint32 outer;

      if (root_child_count > MOZHELPER_HISTORY_N_FAVORITES)
        root_child_count = MOZHELPER_HISTORY_N_FAVORITES;

      for (outer = 0; outer < root_child_count; outer++)
        {
          nsCOMPtr<nsINavHistoryResultNode> node, best_node;
          nsCOMPtr<nsINavHistoryContainerResultNode> container_node;
          PRUint32 best_access_count = 0, inner, child_count;
          nsCAutoString url, title;

          rv = root_node->GetChild (outer, getter_AddRefs (node));
          if (NS_FAILED (rv))
            break;

          container_node = do_QueryInterface (node, &rv);
          if (NS_FAILED (rv))
            break;

          /* Find the child of this node that has the highest visit
             count */
          rv = container_node->SetContainerOpen (PR_TRUE);
          if (NS_FAILED (rv))
            break;
          rv = container_node->GetChildCount (&child_count);
          if (NS_FAILED (rv))
            break;
          for (inner = 0; inner < child_count; inner++)
            {
              nsCOMPtr<nsINavHistoryResultNode> inner_node;
              PRUint32 inner_access_count;

              rv = container_node->GetChild (inner,
                                             getter_AddRefs (inner_node));
              if (NS_FAILED (rv))
                break;

              rv = inner_node->GetAccessCount (&inner_access_count);
              if (NS_FAILED (rv))
                break;

              if (inner_access_count > best_access_count)
                {
                  best_access_count = inner_access_count;
                  best_node = inner_node;
                }
            }
          container_node->SetContainerOpen (PR_FALSE);

          if (inner < child_count)
            break;
          if (best_access_count == 0)
            {
              rv = NS_ERROR_OUT_OF_MEMORY;
              break;
            }

          rv = best_node->GetUri (url);
          if (NS_FAILED (rv))
            break;
          rv = best_node->GetTitle (title);
          if (NS_FAILED (rv))
            break;

          urls[outer] = g_strdup (url.get ());
          titles[outer] = g_strdup (title.get ());
        }

      if (outer == root_child_count)
        g_signal_emit (self, history_signals[FAVORITES_RECEIVED_SIGNAL], 0,
                       urls, titles);
    }

  g_strfreev (titles);
  g_strfreev (urls);

  root_node->SetContainerOpen (PR_FALSE);

 rv_error:
  if (NS_FAILED (rv))
    {
      GError *error = NULL;
      mozhelper_error_set_from_nsresult (rv, &error);
      g_warning ("%s", error->message);
      g_error_free (error);
    }
  return FALSE;
}

static gboolean
mozhelper_history_get_favorites (MozhelperHistory *self,
                           GError **error)
{
  g_return_val_if_fail (MOZHELPER_IS_HISTORY (self), FALSE);

  MozhelperHistoryPrivate *priv = self->priv;

  /* Queue a new favorites query if we haven't already got one */
  if (priv->favorites_idle_handler == 0)
    priv->favorites_idle_handler
      = g_idle_add (mozhelper_history_favorites_cb, self);

  return TRUE;
}

static gboolean
mozhelper_history_remove_favorite (MozhelperHistory *self,
                             const gchar *uri_str,
                             GError **error)
{
  g_return_val_if_fail (MOZHELPER_IS_HISTORY (self), FALSE);

  nsresult rv;

  {
    nsCOMPtr<nsIURI> uri;
    rv = NS_NewURI (getter_AddRefs (uri), uri_str);
    if (NS_FAILED (rv))
      goto rv_error;

    nsCOMPtr<nsIBrowserHistory> browser_history
      = do_GetService (NS_GLOBALHISTORY2_CONTRACTID, &rv);
    if (NS_FAILED (rv))
      goto rv_error;

    nsCAutoString host;
    rv = uri->GetHost(host);
    if (NS_FAILED (rv))
      goto rv_error;

    // The favorites are grouped by host so to remove this favorite we
    // need to remove all history entries for the host of the URI
    rv = browser_history->RemovePagesFromHost (host, PR_FALSE);
    if (NS_FAILED (rv))
      goto rv_error;
  }

  return TRUE;

 rv_error:
  mozhelper_error_set_from_nsresult (rv, error);
  return FALSE;
}

static gboolean
mozhelper_history_clear_history (MozhelperHistory *self,
                           GError **error)
{
  g_return_val_if_fail (MOZHELPER_IS_HISTORY (self), FALSE);

  nsresult rv;

  nsCOMPtr<nsIBrowserHistory> browser_history
    = do_GetService (NS_GLOBALHISTORY2_CONTRACTID, &rv);

  if (NS_FAILED (rv))
    goto rv_error;

  rv = browser_history->RemoveAllPages ();
  if (NS_FAILED (rv))
    goto rv_error;

  return TRUE;

 rv_error:
  mozhelper_error_set_from_nsresult (rv, error);
  return FALSE;
}

static gboolean
mozhelper_history_get_is_page_pinned (MozhelperHistory *self,
                                const gchar *page_uri,
                                gboolean *is_page_pinned,
                                GError **error)
{
  g_return_val_if_fail (MOZHELPER_IS_HISTORY (self), FALSE);

  nsCOMPtr<nsINavBookmarksService> bookmark_service;
  nsCOMPtr<nsIURI> uri;
  PRBool is_page_pinned_pr;
  nsresult rv;

  bookmark_service = do_GetService (NS_NAVBOOKMARKSSERVICE_CONTRACTID, &rv);
  if (NS_FAILED (rv))
    goto rv_error;

  rv = NS_NewURI (getter_AddRefs (uri), page_uri);
  if (NS_FAILED (rv))
    goto rv_error;

  rv = bookmark_service->IsBookmarked (uri, &is_page_pinned_pr);
  if (NS_FAILED (rv))
    goto rv_error;

  *is_page_pinned = is_page_pinned_pr;

  return TRUE;

 rv_error:
  mozhelper_error_set_from_nsresult (rv, error);
  return FALSE;
}

static gboolean
mozhelper_history_pinned_pages_cb (gpointer data)
{
  nsresult rv = NS_OK;
  MozhelperHistory *self = (MozhelperHistory *) data;
  MozhelperHistoryPrivate *priv = self->priv;
  nsCOMPtr<nsINavBookmarksService> bookmark_service;
  nsCOMPtr<nsINavHistoryService> history_service;
  PRInt64 bookmarks_menu_folder;
  PRInt64 bookmark_id;
  PRInt32 index = 0;

  priv->pinned_pages_idle_handler = 0;

  bookmark_service = do_GetService (NS_NAVBOOKMARKSSERVICE_CONTRACTID, &rv);
  if (NS_FAILED (rv))
    goto rv_error;
  history_service = do_GetService (NS_NAVHISTORYSERVICE_CONTRACTID, &rv);
  if (NS_FAILED (rv))
    goto rv_error;

  rv = bookmark_service->GetBookmarksMenuFolder (&bookmarks_menu_folder);
  if (NS_FAILED (rv))
    goto rv_error;

  rv = bookmark_service->GetIdForItemAt (bookmarks_menu_folder, index++,
                                         &bookmark_id);
  if (NS_FAILED (rv))
    goto rv_error;

  while (bookmark_id != -1)
    {
      nsCOMPtr<nsIURI> uri;
      PRInt64 next_bookmark_id;

      rv = bookmark_service->GetIdForItemAt (bookmarks_menu_folder, index++,
                                             &next_bookmark_id);
      if (NS_FAILED (rv))
        goto rv_error;

      rv = mozhelper_history_emit_pinned_page_for_item (self,
                                                  bookmark_service,
                                                  history_service,
                                                  bookmark_id,
                                                  next_bookmark_id != -1);
      if (NS_FAILED (rv))
        goto rv_error;

      bookmark_id = next_bookmark_id;
    }

 rv_error:
  if (NS_FAILED (rv))
    {
      GError *error = NULL;
      mozhelper_error_set_from_nsresult (rv, &error);
      g_warning ("%s", error->message);
      g_error_free (error);
    }
  return FALSE;
}

static gboolean
mozhelper_history_get_pinned_pages (MozhelperHistory *self,
                              GError **error)
{
  g_return_val_if_fail (MOZHELPER_IS_HISTORY (self), FALSE);

  MozhelperHistoryPrivate *priv = self->priv;

  /* Queue a new pinned pages query if we haven't already got one */
  if (priv->pinned_pages_idle_handler == 0)
    priv->pinned_pages_idle_handler
      = g_idle_add (mozhelper_history_pinned_pages_cb, self);

  return TRUE;
}

static gboolean
mozhelper_history_pin_page (MozhelperHistory *self,
                      const gchar *uri_str,
                      const gchar *page_title,
                      GError **error)
{
  g_return_val_if_fail (MOZHELPER_IS_HISTORY (self), FALSE);

  nsCOMPtr<nsINavBookmarksService> bookmark_service;
  nsCOMPtr<nsIURI> uri;
  PRBool is_pinned;
  nsresult rv;

  bookmark_service = do_GetService (NS_NAVBOOKMARKSSERVICE_CONTRACTID, &rv);
  if (NS_FAILED (rv))
    goto rv_error;

  rv = NS_NewURI (getter_AddRefs (uri), uri_str);
  if (NS_FAILED (rv))
    goto rv_error;

  /* Don't do anything if the page is already pinned */
  rv = bookmark_service->IsBookmarked (uri, &is_pinned);
  if (NS_FAILED (rv))
    goto rv_error;

  if (!is_pinned)
    {
      PRInt64 bookmarks_menu_folder;
      PRInt64 bookmark_id;

      rv = bookmark_service->GetBookmarksMenuFolder (&bookmarks_menu_folder);
      if (NS_FAILED (rv))
        goto rv_error;

      rv = bookmark_service
        ->InsertBookmark (bookmarks_menu_folder,
                          uri, nsINavBookmarksService::DEFAULT_INDEX,
                          nsDependentCString (page_title),
                          &bookmark_id);
      if (NS_FAILED (rv))
        goto rv_error;
    }

  return TRUE;

 rv_error:
  mozhelper_error_set_from_nsresult (rv, error);
  return FALSE;
}

static gboolean
mozhelper_history_unpin_page (MozhelperHistory *self,
                        const gchar *uri_str,
                        GError **error)
{
  g_return_val_if_fail (MOZHELPER_IS_HISTORY (self), FALSE);

  nsCOMPtr<nsINavBookmarksService> bookmark_service;
  nsCOMPtr<nsIURI> uri;
  nsAutoTArray<PRInt64, 8> bookmark_ids;
  nsresult rv;
  unsigned int i;

  bookmark_service = do_GetService (NS_NAVBOOKMARKSSERVICE_CONTRACTID, &rv);
  if (NS_FAILED (rv))
    goto rv_error;

  rv = NS_NewURI (getter_AddRefs (uri), uri_str);
  if (NS_FAILED (rv))
    goto rv_error;

  rv = bookmark_service->GetBookmarkIdsForURITArray (uri, &bookmark_ids);
  if (NS_FAILED (rv))
    goto rv_error;

  for (i = 0; i < bookmark_ids.Length (); i++)
    {
      rv = bookmark_service->RemoveItem (bookmark_ids[i]);
      if (NS_FAILED (rv))
        goto rv_error;
    }

  return TRUE;

 rv_error:
  mozhelper_error_set_from_nsresult (rv, error);
  return FALSE;
}

static gboolean
mozhelper_history_unpin_all_pages (MozhelperHistory *self,
                             GError **error)
{
  g_return_val_if_fail (MOZHELPER_IS_HISTORY (self), FALSE);

  nsCOMPtr<nsINavBookmarksService> bookmark_service;
  PRInt64 bookmarks_menu_folder;
  nsresult rv;

  bookmark_service = do_GetService (NS_NAVBOOKMARKSSERVICE_CONTRACTID, &rv);
  if (NS_FAILED (rv))
    goto rv_error;

  rv = bookmark_service->GetBookmarksMenuFolder (&bookmarks_menu_folder);
  if (NS_FAILED (rv))
    goto rv_error;

  rv = bookmark_service->RemoveFolderChildren (bookmarks_menu_folder);
  if (NS_FAILED (rv))
    goto rv_error;

  return TRUE;

 rv_error:
  mozhelper_error_set_from_nsresult (rv, error);
  return FALSE;
}
