/*
 * Moblin-Web-Browser: The web browser for Moblin
 * Copyright (c) 2009, Intel Corporation.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 2.1, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public
 * License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

function SettingsBranchObserver()
{
  this.branches = [];
}

SettingsBranchObserver.prototype =
{
  register : function(branch)
  {
    // Check if we've already registered for this branch
    for (var i in this.branches)
      if (this.branches[i].root == branch.root)
        return;

    // Otherwise add a new observer
    branch.QueryInterface(Components.interfaces.nsIPrefBranch2);
    branch.addObserver("", this, true);
    this.branches.push(branch);
  },

  observe : function(branch, topic, pref)
  {
    branch.QueryInterface(Components.interfaces.nsIPrefBranch);
    var fullPref = branch.root + pref;

    updateBooleanSetting(branch, pref);

    // If the list of search engines has changed then rebuild the
    // checkboxes
    if (fullPref == "mwb.search_engine.names")
      updateSearchEngines();
    // same for the selected search engine
    else if (fullPref == "mwb.search_engine.selected")
      updateSelectedSearchEngine();

    // If the history expiration has changed then update the combo box
    if (fullPref == "browser.history_expire_days")
      updateHistoryAge();
  },

  QueryInterface : function(aIID)
  {
    if (!aIID.equals(Components.interfaces.nsIObserver)
        && !aIID.equals(Components.interfaces.nsISupports)
        && !aIID.equals(Components.interfaces.nsISupportsWeakReference))
      throw Components.results.NS_ERROR_NO_INTERFACE;
    return this;
  }
};

function updateBooleanSetting(branch, pref)
{
  branch.QueryInterface(Components.interfaces.nsIPrefBranch);
  var fullPref = branch.root + pref;

  // Look for the checkbox for this setting
  for (var checkboxId in BOOLEAN_SETTINGS)
    {
      if (BOOLEAN_SETTINGS[checkboxId] == fullPref)
        {
          // Update the checkbox
          var value = branch.getBoolPref(pref);
          document.getElementById(checkboxId).checked = value;
          break;
        }
    }
}

function getSearchEngineList()
{
  var rootBranch = prefsService.getBranch("");
  var engineNames = rootBranch.getCharPref("mwb.search_engine.names");
  engineNames = engineNames.split(";");
  // Sort the search engines by name
  engineNames.sort();
  return engineNames;
}

function updateSelectedSearchEngine()
{
  // Get the new list of search engines
  var engineNames = getSearchEngineList();
  // Get the selected engine
  var mwbPrefs = prefsService.getBranch("mwb.");
  var selectedEngine = mwbPrefs.getCharPref("search_engine.selected");

  // Update the checked state of all the radio controls depending on
  // whether they are the selected engine
  for (var engineNum in engineNames)
    {
      var engineId = "search_engine_" + engineNum;
      var input = document.getElementById(engineId);
      input.checked = engineNames[engineNum] == selectedEngine;
    }
}

function changeSearchEngine(event)
{
  var match;

  if (event.target.checked &&
      (match = /^search_engine_([0-9]+)$/.exec(event.target.id)))
    {
      var engineNames = getSearchEngineList();
      // Set the selected_search_engine
      var mwbPrefs = prefsService.getBranch("mwb.");
      mwbPrefs.setCharPref("search_engine.selected", engineNames[match[1]]);
    }
}

function updateSearchEngines()
{
  var p = document.getElementById("search_engine_buttons");

  // Get rid of the existing list of search engines
  p.innerHTML = "";

  // Get the new list of search engines
  var engineNames = getSearchEngineList();
  // Get the selected engine
  var mwbPrefs = prefsService.getBranch("mwb.");
  var selectedEngine = mwbPrefs.getCharPref("search_engine.selected");

  // Add a radio button for each entry
  for (var engineNum in engineNames)
    {
      if (engineNum > 0)
        p.appendChild(document.createElement("br"));

      var engineId = "search_engine_" + engineNum;
      var label = document.createElement("label");
      label.setAttribute("for", engineId);
      var input = document.createElement("input");
      input.setAttribute("id", engineId);
      input.setAttribute("type", "radio");
      input.setAttribute("name", "search_engine");
      label.appendChild(input);
      label.appendChild(document.createTextNode(engineNames[engineNum]));
      p.appendChild(label);

      input.onclick = changeSearchEngine;
    }

  updateSelectedSearchEngine();
}

function initBooleanSettings()
{
  // Add an observer for each boolean setting and fetch the initial value
  for (var checkboxId in BOOLEAN_SETTINGS)
    {
      var pref = BOOLEAN_SETTINGS[checkboxId];
      var lastDot = pref.lastIndexOf(".");
      var branch = prefsService.getBranch(pref.substr(0, lastDot + 1));
      // Get the preference name relative to the branch
      pref = pref.substr(lastDot + 1);
      // Update the initial value immediately
      updateBooleanSetting(branch, pref);
      // Add an observer for the branch if we haven't already
      observer.register(branch);

      // Listen for changes to the checkbox state. Using 'with' to form
      // a closure containing the branch and pref vars
      with ({ branch : branch, pref : pref })
        {
          document.getElementById(checkboxId).onclick = function(event)
            {
              // Update the corresponding preference
              branch.setBoolPref(pref, event.target.checked);
            };
        }
    }
}

function initHistorySettings()
{
  // Observe changes to the browser history expiration pref
  var branch = prefsService.getBranch("browser.history_");
  observer.register(branch);

  document.getElementById("history_age").onchange = function(event)
  {
    branch.setIntPref("expire_days_min", event.target.value);
    branch.setIntPref("expire_days", event.target.value);
  }
}

function clearHistory()
{
  var nsIPromptService = Components.interfaces.nsIPromptService;
  var promptService =
    Components.classes["@mozilla.org/embedcomp/prompt-service;1"]
    .getService(nsIPromptService);

  var choice = promptService.confirmEx(window,
                                       "",
                                       _("Do you want to reset your browser? " +
                                         "This will remove your web history, " +
                                         "pinned pages and personal settings. "),
                                       nsIPromptService.BUTTON_POS_0 *
                                       nsIPromptService.BUTTON_TITLE_IS_STRING +
                                       nsIPromptService.BUTTON_POS_1 *
                                       nsIPromptService.BUTTON_TITLE_IS_STRING +
                                       nsIPromptService.BUTTON_POS_0_DEFAULT,
                                       _("Yes, delete"),
                                       _("No, save"),
                                       null,
                                       null,
                                       {});

  if (choice == 0)
    {
      var pagesService =
        Components.classes["@moblin.org/mwb-pages-service;1"]
        .getService(Components.interfaces.MwbIPagesService);
      pagesService.clearHistory();
    }
}

function updateHistoryAge()
{
  var combobox = document.getElementById("history_age");
  var rootBranch = prefsService.getBranch("");
  var expireDays = rootBranch.getIntPref("browser.history_expire_days");
  var bestDistance;
  var bestElem;

  // Find the entry that most closely matches the pref value
  for (var elem = combobox.firstElementChild;
       elem;
       elem = elem.nextElementSibling)
    if (elem.nodeName.toLowerCase() == "option")
      {
        var elemAge = elem.getAttribute("value");
        if (bestDistance === undefined ||
            Math.abs(elemAge - expireDays) < bestDistance)
          {
            bestDistance = Math.abs(elemAge - expireDays);
            bestElem = elem;
          }
        elem.removeAttribute("selected");
      }
  bestElem.setAttribute("selected", "selected");
}

const BOOLEAN_SETTINGS = { "complete_domains" : "mwb.complete_domains",
                           "search_in_automagic" : "mwb.search_in_automagic",
                           "remember_passwords" : "signon.rememberSignons",
                           "remember_forms" : "signon.autofillForms" };

var prefsService = (Components.classes["@mozilla.org/preferences-service;1"]
                    .getService(Components.interfaces.nsIPrefService));
var observer;

// Run the initialisation once the page has finished loading
document.addEventListener("DOMContentLoaded", function(event)
{
  observer = new SettingsBranchObserver();

  initBooleanSettings();
  initHistorySettings();
  updateSearchEngines();
  updateHistoryAge();

  var clearHistoryButton = document.getElementById("clear_history");
  clearHistoryButton.addEventListener("click", clearHistory, false);

  // Unregister the load event so we don't initialize more than once
  event.currentTarget.removeEventListener("DOMContentLoaded",
                                          arguments.callee,
                                          false);
}, false);
