/*
 * Copyright (C) 2009 Canonical, Ltd.
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License
 * version 3.0 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3.0 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library. If not, see
 * <http://www.gnu.org/licenses/>.
 *
 * Authored by
 *             Mikkel Kamstrup Erlandsen <mikkel.kamstrup@canonical.com>
 *             Jason Smith <jason.smith@canonical.com>
 */

/**
 * SECTION:zeitgeist-event 
 * @short_description: #ZetigeistEvent objects abstract events returned from Zeitgeist queries
 * @include: zeitgeist.h
 *
 * The #ZeitgeistEvent class is one of the primary elements for communicating
 * with the Zeitgeist daemon. #ZeitgeistEvent<!-- -->s serve two purposes.
 * Unsurprisingly they represent events that have happened, but they also
 * can act as <emphasis>templates</emphasis>. See also #ZeitgeistSubject.
 * 
 * An event in the Zeitgeist world is characterized by two main properties.
 * &quot;What happened&quot; also called the <emphasis>interpretation</emphasis>,
 * and &quot;How did it happen&quot; also called the
 * <emphasis>manifestation</emphasis>. Besides these properties and event
 * also has an <emphasis>actor</emphasis> which identifies the party responsible
 * for triggering the event which in most cases will be an application.
 * Lastly there is an event <emphasis>timestamp</emphasis> and 
 * <emphasis>event id</emphasis>. The timestamp is calculated as the number
 * of milliseconds since the Unix epoch and the event id is a serial number
 * assigned to the event by the Zeitgeist engine when it's logged. These
 * five properties are collectively known as the
 * <emphasis>event metadata</emphasis>.
 *
 * An event must also describe what it happened to. This is called the event
 * <emphasis>subjects</emphasis>. Most events have one subject, but they are
 * allowed to have zero or many too. The metadata of the subjects are 
 * recorded at the time of logging, and are encapsulated by the
 * #ZeitgeistSubject class. It's important to understand that it's just the
 * subject metadata at the time of logging, not necessarily the subject metadata
 * as it exists right now.
 *
 * In addition to the listed properties events may also carry a free form binary
 * <emphasis>payload</emphasis>. The usage of this is is application specific
 * and is generally useless unless you have some contextual information to
 * figure out what's in it.
 *
 * A large part of the Zeitgeist query and monitoring API revolves around a
 * concept of template matching. A query is simply a list of event templates
 * that you want to look for in the log.
 * An unset property on an event template indicates that anything is allowed
 * in that field. If the property is set it indicates that the property
 * must be an exact match.
 */
#if HAVE_CONFIG_H
#include <config.h>
#endif

#include <gio/gdesktopappinfo.h>

#include "zeitgeist-event.h"

G_DEFINE_TYPE (ZeitgeistEvent, zeitgeist_event, G_TYPE_INITIALLY_UNOWNED);
#define ZEITGEIST_EVENT_GET_PRIVATE(obj) \
  (G_TYPE_INSTANCE_GET_PRIVATE(obj, ZEITGEIST_TYPE_EVENT, ZeitgeistEventPrivate))

typedef enum
{
  ZEITGEIST_EVENT_ID,
  ZEITGEIST_EVENT_TIMESTAMP,
  ZEITGEIST_EVENT_INTERPRETATION,
  ZEITGEIST_EVENT_MANIFESTATION,
  ZEITGEIST_EVENT_ACTOR,
} ZeitgeistEventDataOffset;

typedef struct
{
  guint32     id;
  gint64      timestamp;
  gchar      *interpretation;
  gchar      *manifestation;
  gchar      *actor;
  GPtrArray  *subjects;
  GByteArray *payload;
} ZeitgeistEventPrivate;

/**
 * zeitgeist_event_get_id:
 * @event: The event to get the event id for
 *
 * Get the event id as assigned by the Zeitgeist engine.
 *
 * Returns: The event id or 0 if it's unset. An event retrieved from the
 *          Zeitgeist engine will always have an event id.
 */
guint32
zeitgeist_event_get_id (ZeitgeistEvent *event)
{
  g_return_val_if_fail (ZEITGEIST_IS_EVENT (event), 0);

  ZeitgeistEventPrivate* priv = ZEITGEIST_EVENT_GET_PRIVATE (event);
  return priv->id;
}

/**
 * zeitgeist_event_set_id:
 * @event: The event to get the event id for
 * @event_id: The event id to assign to @event
 *
 * Set the event id of an event. Note that it is an error to send an event
 * with a pre set event id to zeitgest_log_insert_events().
 */
void
zeitgeist_event_set_id (ZeitgeistEvent *event, guint32 id)
{
  g_return_if_fail (ZEITGEIST_IS_EVENT (event));

  ZeitgeistEventPrivate* priv = ZEITGEIST_EVENT_GET_PRIVATE (event);
  priv->id = id;
}

/**
 * zeitgeist_event_get_timestamp:
 * @event: The event to get the timestamp for
 *
 * Get the event timestamp. The timestamp is in milliseconds since the
 * Unix epoch. There are a few helpers available for converting to and
 * from other time representations such a #GTimeVal. See for example
 * zeitgeist_timestamp_to_timeval() and zeitgeist_timestamp_from_timeval().
 *
 * Returns: The event timestamp. Note that 0 is ambiguous as it denotes both
 *          an unset timestamp and the time of the Unix Epoch.
 */
gint64 
zeitgeist_event_get_timestamp (ZeitgeistEvent *event)
{
  g_return_val_if_fail (ZEITGEIST_IS_EVENT (event), 0);

  ZeitgeistEventPrivate* priv = ZEITGEIST_EVENT_GET_PRIVATE (event);
  return priv->timestamp;
}

/**
 * zeitgeist_event_set_timestamp:
 * @event: The event to set the timestamp for
 *
 * Set the event timestamp. The timestamp is in milliseconds since the
 * Unix epoch. There are a few helpers available for converting to and
 * from other time representations such a #GTimeVal. See for example
 * zeitgeist_timestamp_to_timeval() and zeitgeist_timestamp_from_timeval().
 *
 * Note that the if you insert events into the Zeitgeist log without a
 * timestamp set the Zeiteist daemon will automatically assign the timestamp
 * of the logging time to the event.
 */
void
zeitgeist_event_set_timestamp (ZeitgeistEvent *event, gint64 timestamp)
{
  g_return_if_fail (ZEITGEIST_IS_EVENT (event));

  ZeitgeistEventPrivate* priv = ZEITGEIST_EVENT_GET_PRIVATE (event);
  priv->timestamp = timestamp;
}

/**
 * zeitgeist_event_get_interpretation:
 * @event: The event to get the interpretation of
 *
 * The event interpretation represents &quot;what happened&quot;. It is encoded
 * as URI defined by the Zeitgeist Event Ontology.
 * Examples could be &quot;something was opened&quot; or
 * &quot;something was modified&quot;.
 *
 * FIXME: Needs link to ontology and some defines to help coding
 *
 * Returns: The event interpretation as a URI or %NULL if unset
 */
const gchar*
zeitgeist_event_get_interpretation (ZeitgeistEvent *event)
{
  g_return_val_if_fail (ZEITGEIST_IS_EVENT (event), NULL);

  ZeitgeistEventPrivate* priv = ZEITGEIST_EVENT_GET_PRIVATE (event);
  return priv->interpretation;
}

/**
 * zeitgeist_event_set_interpretation:
 * @event: The event to set the interpretation of
 * @interpretation: URI designating the interpretation type of the event
 *
 * The event interpretation represents &quot;what happened&quot;. It is encoded
 * as URI defined by the Zeitgeist Event Ontology.
 *
 * FIXME: Needs link to ontology and some defines to help coding
 */
void
zeitgeist_event_set_interpretation (ZeitgeistEvent *event,
                                    const gchar    *interpretation)
{
  g_return_if_fail (ZEITGEIST_IS_EVENT (event));

  ZeitgeistEventPrivate* priv = ZEITGEIST_EVENT_GET_PRIVATE (event);

  gchar* copy = g_strdup (interpretation);

  if (priv->interpretation)
    {
      g_free (priv->interpretation);
    }
  
  priv->interpretation = copy;
}

/**
 * zeitgeist_event_get_manifestation:
 * @event: The event to get the manifestation of
 *
 * The event manifestation represents &quot;how did it happen&quot;.
 * It is encoded as URI defined by the Zeitgeist Event Ontology. Examples
 * could be &quot;the user did it&quot; or
 * &quot;the system send a notification&quot;.
 *
 * FIXME: Needs link to ontology and some defines to help coding
 *
 * Returns: The event interpretation as a URI or %NULL if unset
 */
const gchar*
zeitgeist_event_get_manifestation (ZeitgeistEvent *event)
{
  g_return_val_if_fail (ZEITGEIST_IS_EVENT (event), NULL);

  ZeitgeistEventPrivate* priv = ZEITGEIST_EVENT_GET_PRIVATE (event);
  return priv->manifestation;
}

/**
 * zeitgeist_event_set_manifestation:
 * @event: The event to set the manifestation of
 * @interpretation: URI designating the manifestation type of the event
 *
 * The event manifestation represents &quot;how did it happen&quot;.
 * It is encoded as URI defined by the Zeitgeist Event Ontology.
 *
 * FIXME: Needs link to ontology and some defines to help coding
 */
void
zeitgeist_event_set_manifestation (ZeitgeistEvent *event,
                                   const gchar    *manifestation)
{
  g_return_if_fail (ZEITGEIST_IS_EVENT (event));

  ZeitgeistEventPrivate* priv = ZEITGEIST_EVENT_GET_PRIVATE (event);

  gchar* copy = g_strdup (manifestation);

  if (priv->manifestation)
    {
      g_free (priv->manifestation);
    }
  
  priv->manifestation = copy;
}

/**
 * zeitgeist_event_get_actor:
 * @event: The event to set the actor for
 *
 * Get the event actor. The actor represents the party responsible for
 * triggering the event. When the actor is an application
 * (which it almost always is) the actor is encoded in the
 * <emphasis>app://</emphasis> URI scheme with the base name of the .desktop
 * file for the application appended. Eg. <emphasis>app://firefox.desktop</emphasis>
 *
 * Returns: A URI designating the actor of the event
 */
const gchar* 
zeitgeist_event_get_actor (ZeitgeistEvent *event)
{
  g_return_val_if_fail (ZEITGEIST_IS_EVENT (event), NULL);

  ZeitgeistEventPrivate* priv = ZEITGEIST_EVENT_GET_PRIVATE (event);
  return priv->actor;
}

/**
 * zeitgeist_event_set_actor:
 * @event: The event to set the actor for
 * @actor: URI designating the actor triggering the event.
 *         Fx. <emphasis>app://firefox.desktop</emphasis>
 *
 * Get the event actor. The actor represents the party responsible for
 * triggering the event. When the actor is an application
 * (which it almost always is) the actor is encoded in the
 * <emphasis>app://</emphasis> URI scheme with the base name of the .desktop
 * file for the application appended. Eg. <emphasis>app://firefox.desktop</emphasis>
 */
void
zeitgeist_event_set_actor (ZeitgeistEvent *event,
                           const gchar    *actor)
{
  g_return_if_fail (ZEITGEIST_IS_EVENT (event));

  ZeitgeistEventPrivate* priv = ZEITGEIST_EVENT_GET_PRIVATE (event);

  gchar* copy = g_strdup (actor);

  if (priv->actor)
    {
      g_free (priv->actor);
    }
  
  priv->actor = copy;
}

void
zeitgeist_event_set_actor_from_app_info (ZeitgeistEvent *event,
                                         GAppInfo       *appinfo)
{
  const gchar *app_id;
  gchar       *copy;
  
  g_return_if_fail (ZEITGEIST_IS_EVENT (event));
  g_return_if_fail (G_IS_APP_INFO (appinfo));

  ZeitgeistEventPrivate* priv = ZEITGEIST_EVENT_GET_PRIVATE (event);

  copy = NULL;
  app_id = g_app_info_get_id (appinfo);
  
  if (app_id != NULL)
    {      
      copy = g_strconcat ("application://", app_id, NULL);
    }
  else if (G_IS_DESKTOP_APP_INFO (appinfo) &&
           g_desktop_app_info_get_filename (G_DESKTOP_APP_INFO (appinfo)))
    {
      const gchar *path = g_desktop_app_info_get_filename (
                                                  G_DESKTOP_APP_INFO (appinfo));
      gchar *_app_id = g_path_get_basename (path);
      copy = g_strconcat ("application://", _app_id, NULL);
      g_free (_app_id);
    }
  else
    {
      /* Sometimes the name is set, but not the id... So try that */
      app_id = g_app_info_get_name (appinfo);
      if (app_id != NULL)
        {
          copy = g_strconcat ("application://", app_id, ".desktop", NULL);
        }
    }

  if (priv->actor)
    {
      g_free (priv->actor);
    }
  priv->actor = copy;
}

/**
 * zeitgeist_event_get_subject:
 * @event: The event to get a subject for
 * @index: The 0-based offset of the subject
 *
 * Get the n'th subject of this event. You can find the number of subjects
 * by calling zeitgeist_event_num_subjects().
 *
 * Returns: The subject at position @index. Do not free. If you want to
 *          keep the subject around you need to g_object_ref() it.
 */
ZeitgeistSubject*
zeitgeist_event_get_subject (ZeitgeistEvent *event,
                             gint            index)
{
  g_return_val_if_fail (ZEITGEIST_IS_EVENT (event), NULL);

  ZeitgeistEventPrivate *priv = ZEITGEIST_EVENT_GET_PRIVATE (event);

  g_return_val_if_fail (index < priv->subjects->len, NULL);
  return ZEITGEIST_SUBJECT (g_ptr_array_index (priv->subjects, index));
}

/**
 * zeitgeist_event_num_subjects:
 * @event: The event to get the number of subjects for
 *
 * Get the number of subjects for an event. This is a constant time operation.
 *
 * Returns: The number of subjects for this event.
 */
gint
zeitgeist_event_num_subjects (ZeitgeistEvent *event)
{
  g_return_val_if_fail (ZEITGEIST_IS_EVENT (event), 0);

  ZeitgeistEventPrivate *priv = ZEITGEIST_EVENT_GET_PRIVATE (event);
  return priv->subjects->len;
}

/**
 * zeitgeist_event_add_subject:
 * @event: The event to add a subject to
 * @subject: The subject to add
 *
 * Append a #ZeitgeistSubject to the list of subjects for @event. The
 * event will consume the floating reference on @subject when you call this
 * method.
 */
void
zeitgeist_event_add_subject (ZeitgeistEvent   *event,
                             ZeitgeistSubject *subject)
{
  g_return_if_fail (ZEITGEIST_IS_EVENT (event));

  ZeitgeistEventPrivate* priv = ZEITGEIST_EVENT_GET_PRIVATE (event);

  g_ptr_array_add (priv->subjects, subject);
  g_object_ref_sink (subject);
}

/**
 * zeitgeist_event_get_payload:
 * @event: The event to get the payload for
 *
 * Look up the free form binary payload of @event.
 *
 * Returns: The event payload or %NULL if unset. Do not free. If you want to
 *          keep the subject around you need to g_byte_array_ref() it.
 */

GByteArray*
zeitgeist_event_get_payload (ZeitgeistEvent *event)
{
  g_return_val_if_fail (ZEITGEIST_IS_EVENT (event), NULL);

  ZeitgeistEventPrivate* priv = ZEITGEIST_EVENT_GET_PRIVATE (event);

  return priv->payload;
}

/**
 * zeitgeist_event_set_payload:
 * @event: Event to add the payload to
 * @payload: (transfer-full): The payload to add to @event
 *
 * Attach a a free form binary payload to @event. Payloads are application
 * specific and can not be assumed to have any particular format unless
 * you have other contextual information about the event.
 *
 * The event will assume ownership of @payload. You should never call
 * g_byte_array_free() on @payload and only call g_byte_array_unref() on it if
 * you have added an extra reference to it.
 */
void
zeitgeist_event_set_payload (ZeitgeistEvent *event,
                             GByteArray     *payload)
{
  g_return_if_fail (ZEITGEIST_IS_EVENT (event));

  ZeitgeistEventPrivate* priv = ZEITGEIST_EVENT_GET_PRIVATE (event);

  if (priv->payload)
    {
      g_byte_array_unref (priv->payload);
    }

  priv->payload = payload;
}

static void
zeitgeist_event_init (ZeitgeistEvent *object)
{
  ZeitgeistEventPrivate *priv;
    
  priv = ZEITGEIST_EVENT_GET_PRIVATE (object);

  priv->id = 0;
  priv->timestamp = 0;
  priv->interpretation = NULL;
  priv->manifestation = NULL;
  priv->actor = NULL;
  priv->subjects = g_ptr_array_new_with_free_func (
                                                (GDestroyNotify)g_object_unref);
  priv->payload = NULL;
}

static void
zeitgeist_event_finalize (GObject *object)
{
  ZeitgeistEvent *event = ZEITGEIST_EVENT (object);
  ZeitgeistEventPrivate *priv;
  
  priv = ZEITGEIST_EVENT_GET_PRIVATE (event);

  if (priv->subjects)
    {
      /* Subjects are unreffed by the free-func of the GPtrArray  */
      g_ptr_array_unref (priv->subjects);
      priv->subjects = NULL;
    }
  
  zeitgeist_event_set_interpretation (event, NULL);
  zeitgeist_event_set_manifestation (event, NULL);
  zeitgeist_event_set_actor (event, NULL);  
  zeitgeist_event_set_payload (event, NULL);

  G_OBJECT_CLASS (zeitgeist_event_parent_class)->finalize (object);
  
}

static void
zeitgeist_event_class_init (ZeitgeistEventClass *klass)
{
  GObjectClass* object_class = G_OBJECT_CLASS (klass);
  
  object_class->finalize = zeitgeist_event_finalize;

  g_type_class_add_private (object_class, sizeof (ZeitgeistEventPrivate));
}

/** 
 * zeitgeist_event_new:
 *
 * Create a new empty event structure
 *
 * Returns: A newly create #ZeitgeistEvent instance. The returned event will
 *          have a floating reference which will be consumed if you pass the
 *          event to any of the methods provided by this library. If you
 *          do not do that then you must free the event youself with
 *          g_object_unref()
 */
ZeitgeistEvent* 
zeitgeist_event_new (void)
{
  ZeitgeistEvent *self;

  self = g_object_new (ZEITGEIST_TYPE_EVENT, NULL);
  return self;
}

/** 
 * zeitgeist_event_new_full:
 * @interpretation: The interpretation type of the event.
 *                  See #ZEITGEIST_ZG_EVENT_INTERPRETATION for a list of
 *                  event interpretation types
 * @manifestation: The manifestation type of the event.
 *                 See #ZEITGEIST_ZG_EVENT_MANIFESTATION for a list of
 *                  event manifestation types
 * @actor: The actor triggering the event. See zeitgeist_event_set_actor()
 *         for details on how to encode this.
 * @VarArgs: A list of #ZeitgeistSubject instances terminated by a %NULL
 * 
 * Create a new event structure with predefined data
 *
 * Returns: A newly create #ZeitgeistEvent instance. The returned event will
 *          have a floating reference which will be consumed if you pass the
 *          event to any of the methods provided by this library. If you
 *          do not do that then you must free the event youself with
 *          g_object_unref()
 */
ZeitgeistEvent* 
zeitgeist_event_new_full (const gchar *interpretation,
                          const gchar *manifestation,
                          const gchar *actor,
                          ...)
{
  va_list         args;
  ZeitgeistEvent *self;

  va_start (args, actor);
  self = zeitgeist_event_new_full_valist (interpretation, manifestation,
                                          actor, args);
  va_end (args);
    
  return self;
}
  
/** 
 * zeitgeist_event_new_full_valist:
 * interpretation: The interpretation type of the event.
 *                  See #ZEITGEIST_ZG_EVENT_INTERPRETATION for a list of
 *                  event interpretation types
 * @manifestation: The manifestation type of the event.
 *                 See #ZEITGEIST_ZG_EVENT_MANIFESTATION for a list of
 *                  event manifestation types
 * @actor: The actor triggering the event. See zeitgeist_event_set_actor()
 *         for details on how to encode this.
 * @args: A %va_list of #ZeitgeistSubject<!-- -->s terminated by %NULL
 * 
 * As zeitgeist_event_new_full() but intended for language bindings
 */
ZeitgeistEvent* 
zeitgeist_event_new_full_valist (const gchar *interpretation,
                                 const gchar *manifestation,
                                 const gchar *actor,
                                 va_list      args)
{
  ZeitgeistEvent *self;
  ZeitgeistSubject *subject = NULL;
  
  self = g_object_new (ZEITGEIST_TYPE_EVENT, NULL);
  zeitgeist_event_set_interpretation (self, interpretation);
  zeitgeist_event_set_manifestation (self, manifestation);
  zeitgeist_event_set_actor (self, actor);

  subject = va_arg (args, ZeitgeistSubject*);
  while (subject != NULL)
    {
      g_return_val_if_fail (ZEITGEIST_IS_SUBJECT (subject), NULL);
      zeitgeist_event_add_subject (self, subject);
      subject = va_arg (args, ZeitgeistSubject*);
    } 
  
    
  return self;
}
