/* -*- Mode: C; indent-tabs-mode: nil; c-basic-offset: 2; tab-width: 2 -*- */
/*
 * Copyright (C) 2009 Canonical, Ltd.
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License
 * version 3.0 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3.0 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library. If not, see
 * <http://www.gnu.org/licenses/>.
 *
 * Authored by Neil Jagdish Patel <neil.patel@canonical.com>
 */

/**
 * SECTION:launcher-application
 * @short_description: Abstract representation of an application
 * @include: launcher-application.h
 *
 * #LauncherApplication objects exist in order to acquire data about specific
 * applications and to launch applications. it also provides a method for
 * returning a libwnck application
 */
#if HAVE_CONFIG_H
#include <config.h>
#endif

#include "launcher-application.h"
#include "launcher-appman.h"

#include <gio/gdesktopappinfo.h>
#include <libwncksync/libwncksync.h>
#include <time.h>


#define TYPE_GS_LIST gs_list_get_type()

static gpointer gs_list_copy (gpointer boxed)
{
  return g_memdup (boxed, sizeof (GSList));
}

GType gs_list_get_type (void)
{
  static GType type = 0;

  if (!type) {
    type = g_boxed_type_register_static ("GSList",
       gs_list_copy, g_free);
  }

  return type;
}

enum
{
  PROP_0,

  PROP_NAME,
  PROP_EXEC,
  PROP_ICON_NAME,
  PROP_COMMENT,
  PROP_DESKTOP_FILE_PATH,
  PROP_UNIQUE_STRING,
  PROP_RUNNING,
  PROP_FAVORITE,
  PROP_FOCUSED,
};

enum
{
  OPENED,
  CLOSED,
  FOCUS_CHANGED,
  RUNNING_CHANGED,
  URGENT_CHANGED,
  INFO_CHANGED,
  ICON_CHANGED,

  LAST_SIGNAL
};

static guint application_signals[LAST_SIGNAL] = { 0 };

static LauncherApplicationWindowActivateFunc activate_func = NULL;
static void    *                             activate_data = NULL;

G_DEFINE_TYPE (LauncherApplication, launcher_application, G_TYPE_OBJECT);
#define LAUNCHER_APPLICATION_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE(obj, \
LAUNCHER_TYPE_APPLICATION, LauncherApplicationPrivate))

struct _LauncherApplicationWindow
{
  GTimeVal timestamp;
  WnckWindow *window;
};

struct _LauncherApplicationPrivate
{
  gchar    *name;
  gchar    *exec;
  gchar    *icon_name;
  gchar    *comment;
  gchar    *desktop_file_path;
  gchar    *unique_string;
  GSList   *managed_windows;
  gboolean  running;
  gboolean  favorite;
  gboolean  focused;

  gboolean has_gathered_windows;

  WnckScreen  *screen;
  gint        in_desktop_wd; // inotify desktop file watch descriptor
  gint        in_icon_wd; // inotify icon file watch descriptor
};


typedef struct _UpdateWindowsNotify UpdateWindowsNotify;
struct _UpdateWindowsNotify
{
  LauncherApplication              *application;
  LauncherApplicationNotifyFinished callback;
  gpointer                          user_data;
};

static void
on_active_window_changed (WnckScreen *screen,
                          WnckWindow *previous,
                          LauncherApplication *app)
{
  launcher_application_ensure_state (app);
}

static void
on_window_closed (WnckScreen          *screen,
                  WnckWindow          *window,
                  LauncherApplication *app)
{
  LauncherApplicationPrivate *priv;

  g_return_if_fail (LAUNCHER_IS_APPLICATION (app));

  priv = app->priv;

  priv->managed_windows = g_slist_remove (priv->managed_windows, window);

  g_signal_emit (app, application_signals[CLOSED], 0, window);
  launcher_application_ensure_state (app);
}

static void
on_watch_file_changed (LauncherAppman *appman, gint wd, LauncherApplication *app)
{
  LauncherApplicationPrivate *priv;
  g_return_if_fail (LAUNCHER_IS_APPLICATION (app));
  g_return_if_fail (LAUNCHER_IS_APPMAN (appman));

  priv = app->priv;
  if (wd == priv->in_desktop_wd)
    {
      launcher_application_set_desktop_file (app, priv->desktop_file_path, TRUE);
    }
  else if (wd == priv->in_icon_wd)
    {
      g_signal_emit (app, application_signals[ICON_CHANGED], 0);
    }
}

static void
launcher_application_init (LauncherApplication *object)
{
  LauncherApplicationPrivate *priv;
  LauncherAppman *appman = launcher_appman_get_default ();
  priv = object->priv = LAUNCHER_APPLICATION_GET_PRIVATE (object);

  priv->has_gathered_windows = FALSE;
  priv->screen = wnck_screen_get_default ();

  g_signal_connect (appman, "watch-file-modified",
                    G_CALLBACK (on_watch_file_changed), object);

  g_signal_connect (priv->screen, "active-window-changed",
                    G_CALLBACK (on_active_window_changed), object);
  g_signal_connect (priv->screen, "window-closed",
                    G_CALLBACK (on_window_closed), object);
}

static void
launcher_application_finalize (GObject *object)
{
  LauncherApplication *app = LAUNCHER_APPLICATION (object);
  LauncherApplicationPrivate *priv;

  priv = LAUNCHER_APPLICATION_GET_PRIVATE (app);

  G_OBJECT_CLASS (launcher_application_parent_class)->finalize (object);

}

static void
launcher_application_set_property (GObject *object,
                                   guint prop_id,
                                   const GValue *value,
                                   GParamSpec *pspec)
{
  LauncherApplication *application = LAUNCHER_APPLICATION(object);
  g_return_if_fail (LAUNCHER_IS_APPLICATION (application));


  switch (prop_id)
    {
    case PROP_NAME:
      if (application->priv->name)
        g_free(application->priv->name);
      application->priv->name = g_value_dup_string(value);
      break;
    case PROP_EXEC:
      if (application->priv->exec)
        g_free(application->priv->exec);
      application->priv->exec = g_value_dup_string(value);
      break;
    case PROP_ICON_NAME:
      if (application->priv->icon_name)
        g_free(application->priv->icon_name);
      application->priv->icon_name = g_value_dup_string(value);
      break;
    case PROP_COMMENT:
      if (application->priv->comment)
        g_free(application->priv->comment);
      application->priv->comment = g_value_dup_string(value);
      break;
    case PROP_DESKTOP_FILE_PATH:
      launcher_application_set_desktop_file(application,
                                            g_value_dup_string(value),
                                            FALSE);
      break;
    case PROP_UNIQUE_STRING:
      break;
    case PROP_FAVORITE:
      application->priv->favorite = g_value_get_boolean(value);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
    }
}

static void
launcher_application_get_property (GObject *object,
                                   guint prop_id,
                                   GValue *value,
                                   GParamSpec *pspec)
{
  LauncherApplication *application = LAUNCHER_APPLICATION(object);
  LauncherApplicationPrivate *priv = application->priv;

  g_return_if_fail (LAUNCHER_IS_APPLICATION (application));

  switch (prop_id)
    {
    case PROP_NAME:
      g_value_set_string(value, priv->name);
      break;
    case PROP_EXEC:
      g_value_set_string(value, priv->exec);
      break;
    case PROP_ICON_NAME:
      g_value_set_string(value, priv->icon_name);
      break;
    case PROP_COMMENT:
      g_value_set_string(value, priv->comment);
      break;
    case PROP_DESKTOP_FILE_PATH:
      g_value_set_string(value, priv->desktop_file_path);
      break;
    case PROP_UNIQUE_STRING:
      {
        const gchar *str;
        str = launcher_application_get_unique_string(application);
        g_value_set_string(value, str);
      }
      break;
    case PROP_RUNNING:
      g_value_set_boolean(value, priv->running);
      break;
    case PROP_FAVORITE:
      g_value_set_boolean(value, priv->favorite);
      break;
    case PROP_FOCUSED:
      g_value_set_boolean (value, priv->focused);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
    }
}

static void
launcher_application_opened (LauncherApplication *self,
                             WnckWindow *window)
{
  /* TODO: Add default signal handler implementation here */
}

static void
launcher_application_closed (LauncherApplication *self,
                             WnckWindow *window)
{
  /* TODO: Add default signal handler implementation here */
}

static void
launcher_application_class_init (LauncherApplicationClass *klass)
{
  GObjectClass* object_class = G_OBJECT_CLASS (klass);

  g_type_class_add_private (object_class, sizeof (LauncherApplicationPrivate));

  object_class->finalize = launcher_application_finalize;
  object_class->set_property = launcher_application_set_property;
  object_class->get_property = launcher_application_get_property;

  klass->opened = launcher_application_opened;
  klass->closed = launcher_application_closed;

  g_object_class_install_property (object_class,
                                   PROP_NAME,
                                   g_param_spec_string ("name",
                                                        "name",
                                                        "name of the application",
                                                        "",
                                                        G_PARAM_READABLE));

  g_object_class_install_property (object_class,
                                   PROP_EXEC,
                                   g_param_spec_string ("exec",
                                                        "exec",
                                                        "the exec thats called in order to launch the application",
                                                        "",
                                                        G_PARAM_READABLE));

  g_object_class_install_property (object_class,
                                   PROP_ICON_NAME,
                                   g_param_spec_string ("icon-name",
                                                        "icon-name",
                                                        "the standard \"icon-name\" for this application",
                                                        "",
                                                        G_PARAM_READABLE));

  g_object_class_install_property (object_class,
                                   PROP_COMMENT,
                                   g_param_spec_string ("comment",
                                                        "comment",
                                                        "a general comment about the application, usually helpful information",
                                                        "",
                                                        G_PARAM_READABLE));

  g_object_class_install_property (object_class,
                                   PROP_DESKTOP_FILE_PATH,
                                   g_param_spec_string ("desktop_file_path",
                                                        "desktop_file_path",
                                                        "the path to the desktop file accociated with this application",
                                                        "",
                                                        G_PARAM_READABLE|G_PARAM_WRITABLE));

  g_object_class_install_property (object_class,
                                   PROP_UNIQUE_STRING,
                                   g_param_spec_string ("unique_string",
                                                        "unique_string",
                                                        "A Unique string to identify this application",
                                                        "",
                                                        G_PARAM_READABLE));
  g_object_class_install_property (object_class,
                                   PROP_RUNNING,
                                   g_param_spec_boolean ("running",
                                                        "running",
                                                        "Returns TRUE if the application is running",
                                                        FALSE,
                                                        G_PARAM_READABLE|G_PARAM_WRITABLE));

  g_object_class_install_property (object_class,
                                   PROP_FOCUSED,
                                   g_param_spec_boolean ("focused",
                                                         "focused",
                                                         "returns TRUE if the application contains a focused window",
                                                         FALSE,
                                                         G_PARAM_READABLE|G_PARAM_WRITABLE));

  g_object_class_install_property (object_class,
                                   PROP_FAVORITE,
                                   g_param_spec_boolean ("favorite",
                                                        "favorite",
                                                        "returns TRUE if the application is listed as a favorite application",
                                                        FALSE,
                                                        G_PARAM_READABLE));

  application_signals[OPENED] =
      g_signal_new ("opened",
                    G_OBJECT_CLASS_TYPE (klass),
                    0,
                    G_STRUCT_OFFSET (LauncherApplicationClass, opened),
                    NULL, NULL,
                    g_cclosure_marshal_VOID__OBJECT,
                    G_TYPE_NONE, 1, WNCK_TYPE_WINDOW
                    );

  application_signals[CLOSED] =
      g_signal_new ("closed",
                    G_OBJECT_CLASS_TYPE (klass),
                    0,
                    G_STRUCT_OFFSET (LauncherApplicationClass, closed),
                    NULL, NULL,
                    g_cclosure_marshal_VOID__OBJECT,
                    G_TYPE_NONE, 1, WNCK_TYPE_WINDOW
                    );

  application_signals[FOCUS_CHANGED] =
      g_signal_newv ("focus-changed",
                     G_OBJECT_CLASS_TYPE (klass),
                     0,
                     NULL, NULL, NULL,
                     g_cclosure_marshal_VOID__VOID,
                     G_TYPE_NONE,
                     0,
                     NULL);

  application_signals[RUNNING_CHANGED] =
      g_signal_newv ("running-changed",
                     G_OBJECT_CLASS_TYPE (klass),
                     0,
                     NULL, NULL, NULL,
                     g_cclosure_marshal_VOID__VOID,
                     G_TYPE_NONE,
                     0,
                     NULL);

  application_signals[URGENT_CHANGED] =
      g_signal_newv ("urgent-changed",
                     G_OBJECT_CLASS_TYPE (klass),
                     0,
                     NULL, NULL, NULL,
                     g_cclosure_marshal_VOID__VOID,
                     G_TYPE_NONE,
                     0,
                     NULL);

  application_signals[INFO_CHANGED] =
      g_signal_newv ("info-changed",
                     G_OBJECT_CLASS_TYPE (klass),
                     0,
                     NULL, NULL, NULL,
                     g_cclosure_marshal_VOID__VOID,
                     G_TYPE_NONE,
                     0,
                     NULL);

  application_signals[ICON_CHANGED] =
      g_signal_newv ("icon-changed",
                     G_OBJECT_CLASS_TYPE (klass),
                     0,
                     NULL, NULL, NULL,
                     g_cclosure_marshal_VOID__VOID,
                     G_TYPE_NONE,
                     0,
                     NULL);

}

void
on_window_name_changed (WnckWindow *window,
                        LauncherApplication *app)
{
  LauncherApplicationPrivate *priv;

  g_return_if_fail (WNCK_IS_WINDOW (window));
  g_return_if_fail (LAUNCHER_IS_APPLICATION (app));

  priv = app->priv;

  if (priv->name)
    g_free (priv->name);
  priv->name = g_strdup (wnck_window_get_name (window));
}

/*
 * Constructors
 */
LauncherApplication *
launcher_application_new (void)
{
  LauncherApplication *application;

  application = g_object_new (LAUNCHER_TYPE_APPLICATION,
                              "running", FALSE,
                              NULL);



  return application;
}

/**
 * launcher_application_new_from_wnck_window:
 * @window: A #WnckWindow object
 *
 * creates a new #LauncherApplication object based on information from @window
 */
LauncherApplication *
launcher_application_new_from_wnck_window (WnckWindow *window)
{
  LauncherApplication *application;
  gchar *desktop_file;

  g_return_val_if_fail (WNCK_IS_WINDOW (window), NULL);

  desktop_file = wncksync_proxy_get_desktop_file (wncksync_proxy_get_default (),
                                                  wnck_window_get_xid (window));

  application = g_object_new (LAUNCHER_TYPE_APPLICATION,
                              NULL);

  application->priv->managed_windows = g_slist_prepend (application->priv->managed_windows, window);
  /* give ourself a primary window to key off of if we have no desktop file to work with */
  if (!desktop_file || !g_file_test (desktop_file, G_FILE_TEST_EXISTS))
    {
      application->priv->name = g_strdup (wnck_window_get_name (window));

      g_signal_connect (window, "name-changed",
                        G_CALLBACK (on_window_name_changed), application);
    }
  else
    {
      launcher_application_set_desktop_file (application, desktop_file, TRUE);
    }

  launcher_application_update_windows (application);
  launcher_application_ensure_state (application);

  return application;
}

/**
 * launcher_application_new_from_destkop_file
 * @desktop_file: the file path to the desktop file used to build the application
 * @dont_check_windows: if set to true, will not communicate with wncksync to get a window list
 *
 * This will create a new #LauncherApplication object using the information
 * contained within the given desktop file at @desktop_file
 *
 * Returns: (transfer full) A new #LauncherApplication object
 */
LauncherApplication *
launcher_application_new_from_desktop_file (const gchar *desktop_file,
                                            gboolean dont_check_windows)
{
  LauncherApplication *application;
  /* we can now make our application */
  application = g_object_new (LAUNCHER_TYPE_APPLICATION,
                              NULL);
  launcher_application_set_desktop_file (application,
                                         desktop_file, dont_check_windows);
  launcher_application_ensure_state (application);
  return application;
}

/*
 * Public Methods
 */

/**
 * launcher_application_launch
 * @application: a #LauncherApplication object
 * @error: a #GError error
 *
 * This method will attempt to launch the application using the exec string
 * associated with it
 *
 * Returns: A truth value dictating whether the launching was successful
 */
gboolean
launcher_application_launch (LauncherApplication *application, GError **error)
{
  GDesktopAppInfo     *info;
  gchar               *icon_name;
  gchar               *desktop_file_path;
  GdkAppLaunchContext *context;

  g_return_val_if_fail (application, FALSE);
  g_return_val_if_fail (*error == NULL, FALSE);

  g_object_get(application, "desktop_file_path", &desktop_file_path, NULL);

  info = g_desktop_app_info_new_from_filename (desktop_file_path);

  if (!info)
    {
      static GQuark quark;
      if (!quark)
        quark = g_quark_from_static_string ("launcher_application_launch_error");

      *error = g_error_new (quark, 1, "Unable to load GDesktopAppInfo for %s",
                            desktop_file_path);
      return FALSE;
    }

  context = gdk_app_launch_context_new ();
  gdk_app_launch_context_set_screen (context, gdk_screen_get_default ());
  gdk_app_launch_context_set_timestamp (context, time (NULL));

  g_object_get(application, "icon-name", &icon_name, NULL);
  gdk_app_launch_context_set_icon_name (context, icon_name);

  g_app_info_launch ((GAppInfo *)info, NULL, (GAppLaunchContext*)context,
                     error);

  g_object_unref (context);
  g_object_unref (info);

  return TRUE;
}

/**
 * launcher_application_get_unique_string
 * @app: a #LauncherApplication
 *
 * Provides a unique string that can be used to identify this application
 *
 * Returns: a unique string
 */
const gchar *
launcher_application_get_unique_string (LauncherApplication *application)
{
  g_return_val_if_fail (application, NULL);
  if (!application->priv->unique_string)
    application->priv->unique_string = g_utf8_strdown (application->priv->name,
                                                       -1);

  return application->priv->unique_string;
}

gboolean
launcher_application_owns_window (LauncherApplication *application,
                                  WnckWindow          *window)
{
  LauncherApplicationPrivate *priv;

  g_return_val_if_fail (LAUNCHER_IS_APPLICATION (application), FALSE);
  g_return_val_if_fail (WNCK_IS_WINDOW (window), FALSE);

  priv = application->priv;

  return g_slist_find (priv->managed_windows, window) != NULL;
}

/**
 * launcher_application_get_windows:
 * @app: a #LauncherApplication
 *
 * Provides a list of #WnckApplications associated with this @app
 *
 * Returns: (transfer none): a #GSList containing #WnckApplications
 */
GSList *
launcher_application_get_windows (LauncherApplication *application)
{
  g_return_val_if_fail (LAUNCHER_IS_APPLICATION (application), NULL);

  return application->priv->managed_windows;
}

gboolean
launcher_application_get_urgent (LauncherApplication *application)
{
  LauncherApplicationPrivate *priv;
  GSList *l = NULL;
  WnckWindow *window;

  g_return_val_if_fail (LAUNCHER_IS_APPLICATION (application), FALSE);

  priv = application->priv;

  for (l = priv->managed_windows; l; l = l->next)
    {
      window = l->data;

      if (wnck_window_needs_attention (window))
        return TRUE;
    }
  return FALSE;
}

void
launcher_application_ensure_state (LauncherApplication *application)
{
  LauncherApplicationPrivate *priv;
  WnckWindow *active_window;
  GSList *l = NULL;
  gboolean prev_focus, prev_running;

  g_return_if_fail (LAUNCHER_IS_APPLICATION (application));

  priv = application->priv;
  prev_focus = priv->focused;
  prev_running = priv->running;

  active_window = wnck_screen_get_active_window (priv->screen);

  if (priv->managed_windows)
    {
      priv->running = TRUE;
      priv->focused = FALSE;

      if (active_window)
        {
        for (l = priv->managed_windows; l; l = l->next)
          {
            if (active_window == l->data)
              {
                priv->focused = TRUE;
                break;
              }
          }
        }
    }
  else
    {
      priv->running = FALSE;
      priv->focused = FALSE;
    }

  if (prev_focus != priv->focused)
    {
      g_signal_emit (application, application_signals[FOCUS_CHANGED], 0);
    }
  if (prev_running != priv->running)
    {
      g_signal_emit (application, application_signals[RUNNING_CHANGED], 0);
    }
}

void on_window_state_changed (WnckWindow *window,
                              WnckWindowState change_mask,
                              WnckWindowState new_state,
                              LauncherApplication *application)
{
  if (change_mask & WNCK_WINDOW_STATE_URGENT)
    {
      g_signal_emit (application, application_signals[URGENT_CHANGED], 0);
    }
}

void
connect_window_events (LauncherApplication *application)
{
  LauncherApplicationPrivate *priv;
  GSList *l = NULL;
  WnckWindow *window;

  g_return_if_fail (LAUNCHER_IS_APPLICATION (application));

  priv = application->priv;

  for (l = priv->managed_windows; l; l = l->next)
    {
      window = l->data;
      g_signal_connect (window, "state-changed",
                        G_CALLBACK (on_window_state_changed), application);
    }
}

void disconnect_window_events (LauncherApplication *application)
{
  LauncherApplicationPrivate *priv;
  GSList *l = NULL;
  WnckWindow *window;

  g_return_if_fail (LAUNCHER_IS_APPLICATION (application));

  priv = application->priv;

  for (l = priv->managed_windows; l; l = l->next)
    {
      window = l->data;
      g_signal_handlers_disconnect_by_func (window,
                                            G_CALLBACK (on_window_state_changed),
                                            application);
    }
}

gboolean
launcher_application_has_gathered_windows (LauncherApplication *application)
{
  g_return_val_if_fail (LAUNCHER_IS_APPLICATION (application), FALSE);
  return application->priv->has_gathered_windows;
}

static void
update_windows_callback (WncksyncProxy *proxy,
                         GArray *xids,
                         UpdateWindowsNotify *notify_data)
{
  LauncherApplication *application;
  LauncherApplicationPrivate *priv;
  GSList *old_windows = NULL, *l = NULL;
  int i;
  WnckWindow *window;
  WnckWindowType type;
  guint32 xid;

  application = notify_data->application;

  g_return_if_fail (LAUNCHER_IS_APPLICATION (application));

  priv = notify_data->application->priv;

  priv->has_gathered_windows = TRUE;
  disconnect_window_events (application);

  old_windows = priv->managed_windows;
  priv->managed_windows = NULL;
  for (i = 0; i < xids->len; i++)
    {
      xid = g_array_index (xids, guint32, i);
      window = wnck_window_get (xid);

      if (!WNCK_IS_WINDOW (window))
        continue;

      type = wnck_window_get_window_type (window);
      if (type == WNCK_WINDOW_DESKTOP || type == WNCK_WINDOW_DOCK)
        continue;

      priv->managed_windows = g_slist_prepend (priv->managed_windows, window);
    }


  if (g_slist_length (priv->managed_windows) > g_slist_length (old_windows))
    {
      for (l = priv->managed_windows; l; l = l->next)
        {
          if (g_slist_find (old_windows, l->data))
            continue;
          g_signal_emit (application, application_signals[OPENED], 0, l->data);
          break;
        }
    }
  else if (g_slist_length (priv->managed_windows) < g_slist_length (old_windows))
    {
      for (l = old_windows; l; l = l->next)
        {
          if (g_slist_find (priv->managed_windows, l->data))
            continue;
          g_signal_emit (application, application_signals[CLOSED], 0, l->data);
          break;
        }
    }

  if (old_windows)
    g_slist_free (old_windows);

  connect_window_events (application);

  launcher_application_ensure_state (application);

  g_array_free (xids, TRUE);

  if (notify_data->callback)
    {
      (notify_data->callback) (application, notify_data->user_data);
    }
  g_free (notify_data);
}

void
launcher_application_update_windows_with_callback (LauncherApplication *application,
                                                   LauncherApplicationNotifyFinished notify,
                                                   gpointer user_data)
{
  LauncherApplicationPrivate *priv;
  UpdateWindowsNotify *notify_data;

  g_return_if_fail (LAUNCHER_IS_APPLICATION (application));

  priv = application->priv;

  if (!priv->desktop_file_path)
    {
      if (notify)
        notify (application, user_data);
      return;
    }

  notify_data = (UpdateWindowsNotify *) g_malloc0 (sizeof (UpdateWindowsNotify));

  notify_data->application = application;
  notify_data->callback = notify;
  notify_data->user_data = user_data;

  wncksync_proxy_get_xids_async (wncksync_proxy_get_default (),
                                 priv->desktop_file_path,
                                 (WncksyncArrayCallback) update_windows_callback,
                                 notify_data);

}

void
launcher_application_update_windows (LauncherApplication *application)
{
  LauncherApplicationPrivate *priv;
  UpdateWindowsNotify *notify_data;

  g_return_if_fail (LAUNCHER_IS_APPLICATION (application));

  priv = application->priv;

  if (!priv->desktop_file_path)
    {
      return;
    }

  notify_data = (UpdateWindowsNotify *) g_malloc0 (sizeof (UpdateWindowsNotify));
  notify_data->application = application;

  wncksync_proxy_get_xids_async (wncksync_proxy_get_default (),
                                 priv->desktop_file_path,
                                 (WncksyncArrayCallback) update_windows_callback,
                                 notify_data);

}

/**
 * launcher_application_get_name:
 * @application: a #LauncherApplication object
 *
 * Get the internationalized name of the application. Suitable for display.
 *
 * Returns: a string containing the name
 */
const gchar *
launcher_application_get_name (LauncherApplication *application)
{
  g_return_val_if_fail (application, NULL);

  return application->priv->name;
}

/**
 * launcher_application_get_comment
 * @application: A #LauncherApplication object
 *
 * Get the internationalized comment string assigned to this application.
 * Suitable for display as a short description of the application.
 *
 * Returns: An internationalized string containing the comment
 */
const gchar *
launcher_application_get_comment (LauncherApplication *application)
{
  g_return_val_if_fail (application, NULL);

  return application->priv->comment;
}

/**
 * launcher_application_get_icon_name
 * @application: A #LauncherApplication object
 *
 * Get icon-name associated with the application
 *
 * Returns: an icon-name string
 */
const gchar *
launcher_application_get_icon_name (LauncherApplication *application)
{
  g_return_val_if_fail (application, NULL);

  return application->priv->icon_name;
}

/**
 * launcher_application_get_exec_string
 * @application: A #LauncherApplication
 *
 * Get the command line used to start this application
 *
 * Returns: the command line used to start this application
 */
const gchar *
launcher_application_get_exec_string (LauncherApplication *application)
{
  g_return_val_if_fail (application, NULL);

  return application->priv->exec;
}

/**
 * launcher_application_get_desktop_file
 * @application: A #LauncherApplication
 *
 * Get the absolute path to the .desktop file for this application
 *
 * Returns: the path to the .desktop file
 */
const gchar *
launcher_application_get_desktop_file (LauncherApplication *application)
{
  g_return_val_if_fail (application, NULL);

  return application->priv->desktop_file_path;
}

gboolean
proxy_launcher_application_update_windows (LauncherApplication *application)
{
  launcher_application_update_windows (application);
  return FALSE;
}

/**
 * launcher_application_set_desktop_file
 * @application: A #LauncherApplication
 * @desktop_file: A string describing the path to a desktop file
 *
 * sets the desktop file for this application and will also regenerate its
 * information
 *
 */
void
launcher_application_set_desktop_file (LauncherApplication *application,
                                       const gchar *desktop_file_path,
                                       gboolean dont_check_windows)
{
  GKeyFile  *desktop_keyfile;
  FILE      *desktop_disk_file;
  gchar     *desktop_disk_buffer;
  size_t    memsize;
  long int  numbytes;
  GError    *error = NULL;
  gchar     *name = NULL;
  gchar     *exec = NULL;
  gchar     *comment = NULL;
  gchar     *icon_name = NULL;
  gchar     *desktop_file;
  LauncherApplicationPrivate *priv;

  g_return_if_fail (application);
  desktop_file = g_strdup (desktop_file_path);
  g_strstrip (desktop_file);
  g_return_if_fail (desktop_file != NULL);

  priv = application->priv;

  desktop_keyfile = g_key_file_new ();

  /* load our desktop file from standard C instead of gio so we don't init gio
   * during startup
   */
  desktop_disk_file = fopen (desktop_file, "r");

  if (!desktop_disk_file)
    {
      g_warning ("launcher-application.c: Unable to read from destop file %s",
                 desktop_file);
      return;
    }

  fseek (desktop_disk_file, 0L, SEEK_END);
  numbytes = ftell (desktop_disk_file);
  fseek (desktop_disk_file, 0L, SEEK_SET);
  desktop_disk_buffer = g_malloc ((numbytes) * sizeof(gchar));
  memsize = fread (desktop_disk_buffer, sizeof (gchar), numbytes, desktop_disk_file);
  fclose(desktop_disk_file);
  g_key_file_load_from_data (desktop_keyfile, desktop_disk_buffer,
                             numbytes, 0, &error);

  /* free the desktop file buffer */
  g_free(desktop_disk_buffer);

  if (error)
    {
      g_warning ("launcher-application.c: Unable to read from desktop file %s: %s",
                desktop_file,
                error->message);
      g_error_free (error);
      return;
    }

  name = g_key_file_get_locale_string (desktop_keyfile,
                                      G_KEY_FILE_DESKTOP_GROUP,
                                      G_KEY_FILE_DESKTOP_KEY_NAME,
                                      NULL,
                                      &error);
  if (error)
    {
      g_warning ("Unable to read from desktop file %s: %s",
                desktop_file,
                error->message);
      g_error_free (error);
      name = "";
    }


  icon_name = g_key_file_get_string (desktop_keyfile,
                                     G_KEY_FILE_DESKTOP_GROUP,
                                     G_KEY_FILE_DESKTOP_KEY_ICON,
                                     NULL);

  exec = g_key_file_get_string (desktop_keyfile,
                                G_KEY_FILE_DESKTOP_GROUP,
                                G_KEY_FILE_DESKTOP_KEY_EXEC,
                                NULL);

  comment = g_key_file_get_locale_string (desktop_keyfile,
                                          G_KEY_FILE_DESKTOP_GROUP,
                                          G_KEY_FILE_DESKTOP_KEY_COMMENT,
                                          NULL,
                                          &error);

  if (error)
  {
    g_warning ("Unable to read comment from desktop file %s: %s",
              desktop_file,
              error->message);
    g_error_free (error);
    comment = "";
  }

  priv->name = name;
  if (g_strcmp0 (application->priv->icon_name, icon_name))
    {
      // figure out file path for icon
      LauncherAppman *appman= launcher_appman_get_default ();
      launcher_appman_rm_file_watch (appman, priv->in_icon_wd);
      // we only watch on absolute paths because we can't deal with choosing
      // the right size from the icon theme.
      if (g_file_test (icon_name, G_FILE_TEST_EXISTS))
        {
          // we have a file path
          priv->in_icon_wd = launcher_appman_add_file_watch (appman,
                                                                icon_name);
        }
    }
  priv->icon_name = icon_name;
  priv->exec = exec;
  priv->comment = comment;

  if (g_strcmp0 (application->priv->desktop_file_path, desktop_file))
    {
      LauncherAppman *appman= launcher_appman_get_default ();
      launcher_appman_rm_file_watch (appman, priv->in_desktop_wd);
      priv->in_desktop_wd = launcher_appman_add_file_watch (appman, desktop_file);
    }

  if (application->priv->desktop_file_path != NULL)
    g_free (application->priv->desktop_file_path);

  application->priv->desktop_file_path = desktop_file;
  g_signal_emit (application, application_signals[INFO_CHANGED], 0);
  if (!dont_check_windows)
    launcher_application_update_windows (application);

  launcher_application_ensure_state (application);
}


/**
 * launcher_application_get_running
 * @application: A #LauncherApplication
 *
 * Method to tell if the @application is currently running or not
 *
 * Returns: A Truth value
 */
gboolean
launcher_application_get_running (LauncherApplication *application)
{
  g_return_val_if_fail (application, FALSE);
  if (g_slist_length (application->priv->managed_windows))
    {
      return TRUE;
    }
  else
    {
      return FALSE;
    }
}

/**
 * launcher_application_get_favorite
 * @application: A #LauncherApplication
 *
 * Method to tell if the @application is a favorite or not
 *
 * Returns: A Truth value
 */
gboolean
launcher_application_get_favorite (LauncherApplication *application)
{
  g_return_val_if_fail (application, FALSE);

  return application->priv->favorite;
}

/**
 * launcher_application_get_focused
 * @application: A #LauncherApplication
 *
 * provides a method that will return true if any of this @applicaiton's windows
 * are currently "focused"
 *
 * Returns: a gboolean value
 */
gboolean
launcher_application_get_focused (LauncherApplication *application)
{
  g_return_val_if_fail (application, FALSE);

  return application->priv->focused;
}

/**
 * launcher_application_show
 * @application: a #LauncherApplication
 *
 * this method will focus the latest un-minimized window this @application has
 * all this @application's windows are minimized then this method will
 * unminimize them all
 */
void
launcher_application_show (LauncherApplication *application,
                           guint32 timestamp)
{
  LauncherApplicationPrivate *priv;
  WnckWindow *best = NULL;
  WnckWorkspace *workspace;
  GSList *j;
  GList *l;
  GList *stack;

  g_return_if_fail (LAUNCHER_IS_APPLICATION (application));

  priv = application->priv;
  workspace = wnck_screen_get_active_workspace (priv->screen);
  stack = wnck_screen_get_windows_stacked (priv->screen);

  for (l = stack; l; l = l->next)
    {
      j = g_slist_find (priv->managed_windows, l->data);
      if (j)
        {
          best = j->data;
        }
    }

  if (best)
    {
      if (!wnck_window_is_visible_on_workspace (best, workspace))
        wnck_workspace_activate (wnck_window_get_workspace (best), timestamp);
      if (activate_func != NULL)
        activate_func (best, timestamp, activate_data);
      else
        wnck_window_activate (best, timestamp);
    }
}

void
launcher_application_minimize (LauncherApplication *application)
{
  LauncherApplicationPrivate *priv;
  WnckWindow *window;
  GSList *l;

  g_return_if_fail (LAUNCHER_IS_APPLICATION (application));

  priv = application->priv;

  for (l = priv->managed_windows; l; l = l->next)
    {
      window = l->data;

      if (!WNCK_IS_WINDOW (window) || wnck_window_is_minimized (window))
        continue;

      wnck_window_minimize (window);
    }
}

gboolean
launcher_application_has_minimized (LauncherApplication *application)
{
  LauncherApplicationPrivate *priv;
  WnckWindow *window;
  GSList *l;

  g_return_val_if_fail (LAUNCHER_IS_APPLICATION (application), FALSE);

  priv = application->priv;

  for (l = priv->managed_windows; l; l = l->next)
    {
      window = l->data;

      if (!WNCK_IS_WINDOW (window))
        continue;

      if (wnck_window_is_minimized (window))
        return TRUE;
    }
  return FALSE;
}

void
launcher_application_restore (LauncherApplication *application,
                              guint32 timestamp)
{
  LauncherApplicationPrivate *priv;
  WnckWindow *window;
  WnckWorkspace *workspace;
  GSList *l;

  g_return_if_fail (LAUNCHER_IS_APPLICATION (application));

  priv = application->priv;
  workspace = wnck_screen_get_active_workspace (priv->screen);


  for (l = priv->managed_windows; l; l = l->next)
    {
      window = l->data;

      if (!WNCK_IS_WINDOW (window) ||
          !wnck_window_is_minimized (window))
        continue;

      wnck_window_unminimize (window, timestamp);
    }
}

void
launcher_application_close (LauncherApplication *application,
                            guint32 timestamp)
{
  LauncherApplicationPrivate *priv;
  WnckWindow *window;
  GSList *l;

  g_return_if_fail (LAUNCHER_IS_APPLICATION (application));
  priv = application->priv;

  for (l = priv->managed_windows; l; l=l->next)
    {
      window = l->data;
      if (!WNCK_IS_WINDOW (window)) continue;
      wnck_window_close (window, timestamp);
    }
}

/**
 * launcher_application_set_window_activate_func:
 * @func: a #LauncherApplicationWindowActivateFunc handler
 * @data: a pointer to pass to the @func handler
 *
 * Allows external control of showing windows
 **/
void
launcher_application_set_window_activate_func (LauncherApplicationWindowActivateFunc func,
                                               void *data)
{
  activate_func = func;
  activate_data = data;
}

