/*
 * Copyright (C) 2009 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by Neil Jagdish Patel <neil.patel@canonical.com>
 *
 */


#if HAVE_CONFIG_H
#include <config.h>
#endif

#include "ggadget-gadget.h"

#include "libgadget-host.h"

#include <ggadget/event.h>
#include <ggadget/gadget.h>
#include <ggadget/gadget_consts.h>
#include <ggadget/file_manager_interface.h>
#include <ggadget/file_manager_factory.h>
#include <ggadget/options_interface.h>
#include <ggadget/view_interface.h>
#include <ggadget/view_host_interface.h>

#include <ggadget/gtk/key_convert.h>
#include <ggadget/gtk/utilities.h>

using ggadget::Event;
using ggadget::EventResult;
using ggadget::MouseEvent;
using ggadget::SimpleEvent;
using ggadget::KeyboardEvent;
using ggadget::LibgadgetHost;
using ggadget::ViewInterface;
using ggadget::FileManagerInterface;
using ggadget::GetGlobalFileManager;
using ggadget::OptionsInterface;

using ggadget::gtk::ConvertGdkModifierToModifier;
using ggadget::gtk::ConvertGdkModifierToButton;
using ggadget::gtk::ConvertGdkKeyvalToKeyCode;

G_DEFINE_TYPE (GGadget, g_gadget, GADGET_TYPE_CAIRO);

#define G_GADGET_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj),\
  G_TYPE_GADGET, \
  GGadgetPrivate))

struct _GGadgetPrivate
{
  LibgadgetHost   *host;
  ggadget::Gadget *gadget;
  ViewInterface::HitTest mouse_hittest_;

  gint mouse_down_x;
  gint mouse_down_y;
};

/* Forwards */
static void button_press_event (GadgetProxy *self, GdkEventButton *event);
static void button_release_event (GadgetProxy *self, GdkEventButton *event);
static void enter_event          (GadgetProxy *self, GdkEventCrossing *event);
static void leave_event          (GadgetProxy *self, GdkEventCrossing *event);
static void key_press_event      (GadgetProxy *self, GdkEventKey *event);
static void key_release_event     (GadgetProxy *self, GdkEventKey *event);
static void scroll_event         (GadgetProxy *self, GdkEventScroll *event);
static void motion_event         (GadgetProxy *self, GdkEventMotion *event);
static void focus_in_event       (GadgetProxy *self);
static void focus_out_event      (GadgetProxy *self);
static GdkPixbuf * get_icon      (GadgetProxy *self);
static void draw                 (GadgetCairo *self, cairo_t *cr);

/* Globals */
static gint kDragThreshold = 3;

/* GObject stuff */
static void
g_gadget_finalize (GObject *object)
{
  GGadgetPrivate *priv = G_GADGET_GET_PRIVATE (object);
  OptionsInterface *iface;

  iface = priv->gadget->GetOptions ();
  iface->Flush ();

  delete priv->gadget;
  delete priv->host;

  G_OBJECT_CLASS (g_gadget_parent_class)->finalize (object);
}

static gboolean
g_gadget_real_constructed (GObject *object)
{
  GGadget        *gadget = G_GADGET (object);
  GGadgetPrivate *priv = G_GADGET_GET_PRIVATE (object);
  const gchar    *path;
  const gchar    *uid;
  std::string     name;

  path = gadget_proxy_get_path (GADGET_PROXY (object));
  uid  = gadget_proxy_get_uid  (GADGET_PROXY (object));

  priv->host = new LibgadgetHost (gadget);
  priv->gadget = priv->host->LoadGadget (path, uid, 0, false);

  if (priv->gadget)
    {
      name = priv->gadget->GetManifestInfo (ggadget::kManifestName);
      gadget_proxy_set_name (GADGET_PROXY (object), name.c_str ());
    }

  if (!priv->gadget)
    {
      g_warning ("Unable to load ggl gadget: %s", path);
    }

  return FALSE;
}

static void
g_gadget_constructed (GObject *object)
{
  ggadget::StringMap  manifest;
  const gchar        *path;

  LibgadgetHost::Init ();

  path = gadget_proxy_get_path (GADGET_PROXY (object));

  if (ggadget::Gadget::GetGadgetManifest (path, &manifest))
    {
      gadget_proxy_set_name (GADGET_PROXY (object),
                             manifest[ggadget::kManifestName].c_str ());
    }

  g_idle_add ((GSourceFunc)g_gadget_real_constructed, object);
}

static void
g_gadget_class_init (GGadgetClass *klass)
{
  GObjectClass      *obj_class = G_OBJECT_CLASS (klass);
  GadgetProxyClass *gad_class = GADGET_PROXY_CLASS (klass);
  GadgetCairoClass *cai_class = GADGET_CAIRO_CLASS (klass);

  /* Overrides */
  obj_class->finalize    = g_gadget_finalize;
  obj_class->constructed = g_gadget_constructed;

  gad_class->button_press_event   = button_press_event;
  gad_class->button_release_event = button_release_event;
  gad_class->enter_event          = enter_event;
  gad_class->leave_event          = leave_event;
  gad_class->key_press_event      = key_press_event;
  gad_class->key_release_event    = key_release_event;
  gad_class->scroll_event         = scroll_event;
  gad_class->motion_event         = motion_event;
  gad_class->focus_in_event       = focus_in_event;
  gad_class->focus_out_event      = focus_out_event;
  gad_class->get_icon             = get_icon;

  cai_class->draw                 = draw;

  /* Install Properties */

  /* Add signals */

  /* Add Private struct */
  g_type_class_add_private (obj_class, sizeof (GGadgetPrivate));
}

static void
g_gadget_init (GGadget *self)
{
  GGadgetPrivate *priv;

  priv = self->priv = G_GADGET_GET_PRIVATE (self);

  priv->mouse_hittest_ = ViewInterface::HT_CLIENT;
  priv->mouse_down_x = 0;
  priv->mouse_down_y = 0;
}

static void
button_press_event (GadgetProxy *self, GdkEventButton *event)
{
  GGadgetPrivate *priv;
  EventResult     result;
  Event::Type     type;
  gint            mod;
  gint            button;
  gint            x;
  gint            y;
  ViewInterface  *view_;

  g_return_if_fail (G_IS_GADGET (self));
  priv = G_GADGET (self)->priv;

  if (!priv->host || !priv->gadget)
    return;

  view_ = priv->host->lvh_->view_;

  result = ggadget::EVENT_RESULT_UNHANDLED;

  mod = ConvertGdkModifierToModifier (event->state);
  button = event->button == 1 ? MouseEvent::BUTTON_LEFT :
           event->button == 2 ? MouseEvent::BUTTON_MIDDLE :
           event->button == 3 ? MouseEvent::BUTTON_RIGHT :
           MouseEvent::BUTTON_NONE;

  type = Event::EVENT_MOUSE_DOWN;
  priv->mouse_down_x = event->x_root;
  priv->mouse_down_y = event->y_root;
  if (event->type == GDK_2BUTTON_PRESS)
    {
      if (button == MouseEvent::BUTTON_LEFT)
        type = Event::EVENT_MOUSE_DBLCLICK;
      else if (button == MouseEvent::BUTTON_RIGHT)
        type = Event::EVENT_MOUSE_RDBLCLICK;
    }

  x = event->x;
  y = event->y;

  if (button != MouseEvent::BUTTON_NONE  && type != Event::EVENT_INVALID)
    {
      MouseEvent e (type,
                    x, y,
                    0, 0,
                    button, mod, event);

      result = view_->OnMouseEvent (e);
      priv->mouse_hittest_ = view_->GetHitTest ();

      if (result == ggadget::EVENT_RESULT_UNHANDLED
          && button == MouseEvent::BUTTON_LEFT
          && type == Event::EVENT_MOUSE_DOWN)
        {
          if (priv->mouse_hittest_ == ViewInterface::HT_MENU)
            {
              priv->host->lvh_->ShowContextMenu (event->button);
            }
          else if (priv->mouse_hittest_ == ViewInterface::HT_CLOSE)
            {
              priv->host->lvh_->CloseView ();
            }
        }
    }
}

static void
button_release_event (GadgetProxy *self, GdkEventButton *event)
{
  GGadgetPrivate *priv;
  EventResult     result;
  EventResult     result2;
  Event::Type     type;
  gint            mod;
  gint            button;
  gint            x;
  gint            y;
  ViewInterface  *view_;

  g_return_if_fail (G_IS_GADGET (self));
  priv = G_GADGET (self)->priv;

  if (!priv->host || !priv->gadget)
    return;
  view_ = priv->host->lvh_->view_;

  result = ggadget::EVENT_RESULT_UNHANDLED;
  result2 = ggadget::EVENT_RESULT_UNHANDLED;

  mod = ConvertGdkModifierToModifier (event->state);
  button = event->button == 1 ? MouseEvent::BUTTON_LEFT :
           event->button == 2 ? MouseEvent::BUTTON_MIDDLE :
           event->button == 3 ? MouseEvent::BUTTON_RIGHT :
           MouseEvent::BUTTON_NONE;

  type = Event::EVENT_MOUSE_UP;

  x = event->x;
  y = event->y;

  if (button != MouseEvent::BUTTON_NONE)
    {
      MouseEvent e (type,
                    x,
                    y,
                    0,
                    0,
                    button,
                    mod,
                    event);
      result = view_->OnMouseEvent (e);

      MouseEvent e2 (button == MouseEvent::BUTTON_LEFT
                     ? Event::EVENT_MOUSE_CLICK : Event::EVENT_MOUSE_RCLICK,
                     x, y, 0, 0, button, mod);
      result2 = view_->OnMouseEvent (e2);
    }

  priv->mouse_hittest_ = ViewInterface::HT_CLIENT;
}

static void
enter_event (GadgetProxy *self, GdkEventCrossing *event)
{
  GGadgetPrivate *priv;
  EventResult    result;
  ViewInterface *view_;

  g_return_if_fail (G_IS_GADGET (self));
  priv = G_GADGET (self)->priv;

  if (!priv->host || !priv->gadget)
    return;

  view_ = priv->host->lvh_->view_;
  result = ggadget::EVENT_RESULT_UNHANDLED;

  MouseEvent e (Event::EVENT_MOUSE_OVER,
                event->x, event->y,
                0, 0,
                MouseEvent::BUTTON_NONE,
                ConvertGdkModifierToModifier (0));

  result = view_->OnMouseEvent (e);
}

static void
leave_event (GadgetProxy *self, GdkEventCrossing *event)
{
  GGadgetPrivate *priv;
  EventResult    result;
  ViewInterface *view_;

  g_return_if_fail (G_IS_GADGET (self));
  priv = G_GADGET (self)->priv;

  if (!priv->host || !priv->gadget)
    return;

  view_ = priv->host->lvh_->view_;
  result = ggadget::EVENT_RESULT_UNHANDLED;

  MouseEvent e (Event::EVENT_MOUSE_OUT,
                event->x, event->y,
                0, 0,
                MouseEvent::BUTTON_NONE,
                ConvertGdkModifierToModifier (0));

  result = view_->OnMouseEvent (e);
}

static void
key_press_event      (GadgetProxy *self, GdkEventKey *event)
{
  GGadgetPrivate *priv;
  EventResult    result;
  EventResult    result2;
  guint          key_code;
  guint32        key_char = 0;
  gint           mod;
  ViewInterface *view_;

  g_return_if_fail (G_IS_GADGET (self));
  priv = G_GADGET (self)->priv;

  if (!priv->host || !priv->gadget)
    return;

  view_ = priv->host->lvh_->view_;
  result = ggadget::EVENT_RESULT_UNHANDLED;
  result2 = ggadget::EVENT_RESULT_UNHANDLED;

  mod = ConvertGdkModifierToModifier (event->state);
  key_code = ConvertGdkKeyvalToKeyCode (event->keyval);

  if (key_code)
    {
      KeyboardEvent e (Event::EVENT_KEY_DOWN, key_code, mod, event);
      result = view_->OnKeyEvent (e);
    }
  else
    {
      g_debug ("Unknown key 0x%x", event->keyval);
    }

  if ((event->state & (GDK_CONTROL_MASK | GDK_MOD1_MASK)) == 0)
    {
      if (key_code == KeyboardEvent::KEY_ESCAPE
          || key_code == KeyboardEvent::KEY_RETURN
          || key_code == KeyboardEvent::KEY_BACK
          || key_code == KeyboardEvent::KEY_TAB)
        {
          key_char = key_code;
        }
      else
        {
          key_char = gdk_keyval_to_unicode (event->keyval);
        }
    }
  else if ((event->state & GDK_CONTROL_MASK)
           && key_code >= 'A'
           && key_code <= 'Z')
    {
      key_char = key_code - 'A' + 1;
    }

  if (key_char)
    {
      KeyboardEvent e2 (Event::EVENT_KEY_PRESS, key_char, mod, event);
      result2 = view_->OnKeyEvent (e2);
    }
}

static void
key_release_event     (GadgetProxy *self, GdkEventKey *event)
{
  GGadgetPrivate *priv;
  EventResult    result;
  guint          key_code;
  gint           mod;
  ViewInterface *view_;

  g_return_if_fail (G_IS_GADGET (self));
  priv = G_GADGET (self)->priv;

  if (!priv->host || !priv->gadget)
    return;

  view_ = priv->host->lvh_->view_;
  result = ggadget::EVENT_RESULT_UNHANDLED;

  mod = ConvertGdkModifierToModifier (event->state);
  key_code = ConvertGdkKeyvalToKeyCode (event->keyval);

  if (key_code)
    {
      KeyboardEvent e (Event::EVENT_KEY_UP, key_code, mod, event);
      result = view_->OnKeyEvent (e);
    }
  else
    {
      g_debug ("Unknown key 0x%x", event->keyval);
    }
}


static void
scroll_event (GadgetProxy *self, GdkEventScroll *event)
{
  GGadgetPrivate *priv;
  gint           delta_x = 0;
  gint           delta_y = 0;
  ViewInterface *view_;

  g_return_if_fail (G_IS_GADGET (self));
  priv = G_GADGET (self)->priv;

  if (!priv->host || !priv->gadget)
    return;

  view_ = priv->host->lvh_->view_;

  if (event->direction == GDK_SCROLL_UP)
    {
      delta_y = MouseEvent::kWheelDelta;
    }
  else if (event->direction == GDK_SCROLL_DOWN)
    {
      delta_y = -MouseEvent::kWheelDelta;
    }
  else if (event->direction == GDK_SCROLL_LEFT)
    {
      delta_x = MouseEvent::kWheelDelta;
    }
  else if (event->direction == GDK_SCROLL_RIGHT)
    {
      delta_x = -MouseEvent::kWheelDelta;
    }

  MouseEvent e (Event::EVENT_MOUSE_WHEEL,
                event->x, event->y,
                delta_x, delta_y,
                ConvertGdkModifierToButton (event->state),
                ConvertGdkModifierToModifier (event->state));
  view_->OnMouseEvent (e);
}

static void
motion_event (GadgetProxy *self, GdkEventMotion *event)
{
  GGadgetPrivate *priv;
  EventResult    result;
  gint           button;
  gint           mod;
  ViewInterface *view_;

  g_return_if_fail (G_IS_GADGET (self));
  priv = G_GADGET (self)->priv;

  if (!priv->host || !priv->gadget)
    return;

  view_ = priv->host->lvh_->view_;
  result = ggadget::EVENT_RESULT_UNHANDLED;

  button = ConvertGdkModifierToButton (event->state);
  mod = ConvertGdkModifierToModifier (event->state);

  MouseEvent e (Event::EVENT_MOUSE_MOVE,
                event->x, event->y,
                0, 0,
                MouseEvent::BUTTON_NONE,
                ConvertGdkModifierToModifier (0));

  result = view_->OnMouseEvent (e);

  if (result == ggadget::EVENT_RESULT_UNHANDLED
      && button != MouseEvent::BUTTON_NONE
      && (std::abs (event->x_root - priv->mouse_down_x) > kDragThreshold
          || std::abs (event->y_root - priv->mouse_down_y) > kDragThreshold
          ||priv->mouse_hittest_ != ViewInterface::HT_CLIENT))
    {
      MouseEvent e (Event::EVENT_MOUSE_UP,
                    event->x, event->y,
                    0, 0, button, mod);
      view_->OnMouseEvent (e);

      ViewInterface::HitTest hittest = priv->mouse_hittest_;
      gboolean resize_drag = FALSE;
      if (hittest == ViewInterface::HT_LEFT ||
          hittest == ViewInterface::HT_RIGHT ||
          hittest == ViewInterface::HT_TOP ||
          hittest == ViewInterface::HT_BOTTOM ||
          hittest == ViewInterface::HT_TOPLEFT ||
          hittest == ViewInterface::HT_TOPRIGHT ||
          hittest == ViewInterface::HT_BOTTOMLEFT ||
          hittest == ViewInterface::HT_BOTTOMRIGHT)
        {
          resize_drag = TRUE;
        }

      if (resize_drag)
        {
          priv->host->lvh_->BeginResizeDrag (button, hittest);
        }
      else
        {
          priv->host->lvh_->BeginMoveDrag (button);
        }

      priv->mouse_hittest_ = ViewInterface::HT_CLIENT;
    }
}

static void
focus_in_event (GadgetProxy *self)
{
  GGadgetPrivate *priv;
  ViewInterface *view_;

  g_return_if_fail (G_IS_GADGET (self));
  priv = G_GADGET (self)->priv;

  if (!priv->host || !priv->gadget)
    return;

  view_ = priv->host->lvh_->view_;

  SimpleEvent e (Event::EVENT_FOCUS_IN);
  view_->OnOtherEvent (e);
}

static void
focus_out_event (GadgetProxy *self)
{
  GGadgetPrivate *priv;
  ViewInterface *view_;

  g_return_if_fail (G_IS_GADGET (self));
  priv = G_GADGET (self)->priv;

  if (!priv->host || !priv->gadget)
    return;

  view_ = priv->host->lvh_->view_;

  SimpleEvent e (Event::EVENT_FOCUS_OUT);
  view_->OnOtherEvent (e);
}

static FileManagerInterface *
create_gadget_file_manager (const gchar *base_path)
{
  gchar *path;

  if (g_str_has_suffix (base_path, ".gmanifest"))
    path = g_path_get_dirname (base_path);
  else
    path = g_strdup (base_path);

  FileManagerInterface *fm = ggadget::CreateFileManager (path);
  g_free (path);

  return fm;
}

static GdkPixbuf *
get_pixbuf_for_icon_path (const gchar *icon_path, const gchar *gad_path)
{
  GdkPixbuf *pixbuf = NULL;
  std::string data;
  FileManagerInterface *gad_manager;

  gad_manager = create_gadget_file_manager (gad_path);
  if (gad_manager)
    {
      gad_manager->ReadFile(icon_path, &data);
      delete gad_manager;
    }

  if (data.empty())
    {
      FileManagerInterface *file_manager = GetGlobalFileManager ();
      if (file_manager)
        file_manager->ReadFile(ggadget::kGadgetsIcon, &data);
    }

  if (!data.empty())
    {
      pixbuf = ggadget::gtk::LoadPixbufFromData(data);
    }

  if (pixbuf)
    {
      gint width, height;

      width = gdk_pixbuf_get_width (pixbuf);
      height = gdk_pixbuf_get_height (pixbuf);

      if (width < 96 || height < 96)
        {
          GdkPixbuf *temp = pixbuf;

          pixbuf = gdk_pixbuf_scale_simple (temp,
                                            96,
                                            ((gfloat)height/width) * 96.0,
                                            GDK_INTERP_HYPER);
          g_object_unref (temp);
        }

    }
  return pixbuf;
}


static GdkPixbuf *
get_icon  (GadgetProxy *self)
{
  GGadgetPrivate *priv;
  const gchar *path;
  ggadget::StringMap manifest;

  g_return_val_if_fail (G_IS_GADGET (self), NULL);
  priv = G_GADGET (self)->priv;

  path = gadget_proxy_get_path (self);

  if (!ggadget::Gadget::GetGadgetManifest (path, &manifest))
    {
      g_debug ("Unable to get manifest information for path: %s", path);
      return NULL;
    }

  return get_pixbuf_for_icon_path (manifest[ggadget::kManifestIcon].c_str(),
                                   path);
}

static void
draw (GadgetCairo *self, cairo_t *cr)
{
  GGadgetPrivate *priv;

  g_return_if_fail (G_IS_GADGET (self));
  priv = G_GADGET (self)->priv;

  if (priv->host)
    priv->host->Draw (cr);
}

/*
 * Public methods
 */
