/****************************************************************************
 * Solarflare driver for Xen network acceleration
 *
 * Copyright 2006-2008: Solarflare Communications Inc,
 *                      9501 Jeronimo Road, Suite 250,
 *                      Irvine, CA 92618, USA
 *
 * Maintained by Solarflare Communications <linux-xen-drivers@solarflare.com>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 as published
 * by the Free Software Foundation, incorporated herein by reference.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 ****************************************************************************
 */

#include <linux/stddef.h>
#include <linux/errno.h>

#include <xen/xenbus.h>
#include <xen/evtchn.h>
#include <xen/gnttab.h>

#include "accel.h"
#include "accel_util.h"
#include "accel_msg_iface.h"
#include "accel_bufs.h"
#include "accel_ssr.h"
/* drivers/xen/netfront/netfront.h */
#include "netfront.h"

void netfront_accel_set_closing(netfront_accel_vnic *vnic) 
{

	vnic->frontend_state = XenbusStateClosing;
	net_accel_update_state(vnic->dev, XenbusStateClosing);
}
	

static void mac_address_change(struct xenbus_watch *watch,
			       const char **vec, unsigned int len)
{
	netfront_accel_vnic *vnic;
	struct xenbus_device *dev;
	int rc;

	DPRINTK("%s\n", __FUNCTION__);
	
	vnic = container_of(watch, netfront_accel_vnic, 
				mac_address_watch);
	dev = vnic->dev;

	rc = net_accel_xen_net_read_mac(dev, vnic->mac);

	if (rc != 0)
		EPRINTK("%s: failed to read mac (%d)\n", __FUNCTION__, rc);
}


static int setup_mac_address_watch(struct xenbus_device *dev,
				   netfront_accel_vnic *vnic)
{
	int err;

	DPRINTK("Setting watch on %s/%s\n", dev->nodename, "mac");

	err = xenbus_watch_path2(dev, dev->nodename, "mac", 
				 &vnic->mac_address_watch, 
				 mac_address_change);
	if (err) {
		EPRINTK("%s: Failed to register xenbus watch: %d\n",
			__FUNCTION__, err);
		goto fail;
	}

	return 0;
 fail:
	vnic->mac_address_watch.node = NULL;
	return err;
}


/* Grant access to some pages and publish through xenbus */
static int make_named_grant(struct xenbus_device *dev, void *page, 
			    const char *name, grant_ref_t *gnt_ref)
{
	struct xenbus_transaction tr;
	int err;
	grant_ref_t gnt;

	gnt = net_accel_grant_page(dev, virt_to_mfn(page), 0);
	if (gnt < 0)
		return gnt;

	do {
		err = xenbus_transaction_start(&tr);
		if (err != 0) {
			EPRINTK("%s: transaction start failed %d\n",
				__FUNCTION__, err);
			return err;
		}
		err = xenbus_printf(tr, dev->nodename, name, "%d", gnt);
		if (err != 0) {
			EPRINTK("%s: xenbus_printf failed %d\n", __FUNCTION__,
				err);
			xenbus_transaction_end(tr, 1);
			return err;
		}
		err = xenbus_transaction_end(tr, 0);
	} while (err == -EAGAIN);
	
	if (err != 0) {
		EPRINTK("%s: transaction end failed %d\n", __FUNCTION__, err);
		return err;
	}
	
	*gnt_ref = gnt;

	return 0;
}


static int remove_named_grant(struct xenbus_device *dev,
			      const char *name, grant_ref_t gnt_ref)
{
	struct xenbus_transaction tr;
	int err;

	net_accel_ungrant_page(gnt_ref);

	do {
		err = xenbus_transaction_start(&tr);
		if (err != 0) {
			EPRINTK("%s: transaction start failed %d\n",
				__FUNCTION__, err);
			return err;
		}
		err = xenbus_rm(tr, dev->nodename, name);
		if (err != 0) {
			EPRINTK("%s: xenbus_rm failed %d\n", __FUNCTION__,
				err);
			xenbus_transaction_end(tr, 1);
			return err;
		}
		err = xenbus_transaction_end(tr, 0);
	} while (err == -EAGAIN);
	
	if (err != 0) {
		EPRINTK("%s: transaction end failed %d\n", __FUNCTION__, err);
		return err;
	}

	return 0;
}


static 
netfront_accel_vnic *netfront_accel_vnic_ctor(struct net_device *net_dev,
					      struct xenbus_device *dev)
{
	struct netfront_info *np =
		(struct netfront_info *)netdev_priv(net_dev);
	netfront_accel_vnic *vnic;
	int err;

	/*
	 * A bug in earlier versions of Xen accel plugin system meant
	 * you could be probed twice for the same device on suspend
	 * cancel.  Be tolerant of that.
	 */ 
	if (np->accel_priv != NULL)
		return ERR_PTR(-EALREADY);

	/* Alloc mem for state */
	vnic = kzalloc(sizeof(netfront_accel_vnic), GFP_KERNEL);
	if (vnic == NULL) {
		EPRINTK("%s: no memory for vnic state\n", __FUNCTION__);
		return ERR_PTR(-ENOMEM);
	}

	spin_lock_init(&vnic->tx_lock);

	mutex_init(&vnic->vnic_mutex);
	mutex_lock(&vnic->vnic_mutex);

	/* Store so state can be retrieved from device */
	BUG_ON(np->accel_priv != NULL);
	np->accel_priv = vnic;
	vnic->dev = dev;
	vnic->net_dev = net_dev;
	spin_lock_init(&vnic->irq_enabled_lock);
	netfront_accel_ssr_init(&vnic->ssr_state);

	init_waitqueue_head(&vnic->state_wait_queue);
	vnic->backend_state = XenbusStateUnknown;
	vnic->frontend_state = XenbusStateClosed;
	vnic->removing = 0;
	vnic->domU_state_is_setup = 0;
	vnic->dom0_state_is_setup = 0;
	vnic->poll_enabled = 0;
	vnic->tx_enabled = 0;
	vnic->tx_skb = NULL;

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,20)
	INIT_WORK(&vnic->msg_from_bend, netfront_accel_msg_from_bend);
#else
	INIT_WORK(&vnic->msg_from_bend, netfront_accel_msg_from_bend, vnic);
#endif

	netfront_accel_debugfs_create(vnic);

	mutex_unlock(&vnic->vnic_mutex);

	err = net_accel_xen_net_read_mac(dev, vnic->mac);
	if (err) 
		goto fail_mac;

	/* Setup a watch on the frontend's MAC address */
	err = setup_mac_address_watch(dev, vnic);
	if (err)
		goto fail_mac;

	return vnic;

fail_mac:

	mutex_lock(&vnic->vnic_mutex);

	netfront_accel_debugfs_remove(vnic);

	netfront_accel_ssr_fini(vnic, &vnic->ssr_state);

	EPRINTK_ON(vnic->tx_skb != NULL);

	vnic->frontend_state = XenbusStateUnknown;
	net_accel_update_state(dev, XenbusStateUnknown);

	mutex_unlock(&vnic->vnic_mutex);

	np->accel_priv = NULL;
	kfree(vnic);

	return ERR_PTR(err);
}


static void netfront_accel_vnic_dtor(netfront_accel_vnic *vnic)
{
	struct net_device *net_dev = vnic->net_dev;
	struct netfront_info *np = 
		(struct netfront_info *)netdev_priv(net_dev);

	/*
	 * Now we don't hold the lock any more it is safe to remove
	 * this watch and synchonrise with the completion of
	 * watches
	 */
	DPRINTK("%s: unregistering xenbus mac watch\n", __FUNCTION__);
	unregister_xenbus_watch(&vnic->mac_address_watch);
	kfree(vnic->mac_address_watch.node);

	flush_workqueue(netfront_accel_workqueue);

	mutex_lock(&vnic->vnic_mutex);

	netfront_accel_debugfs_remove(vnic);

	netfront_accel_ssr_fini(vnic, &vnic->ssr_state);

	EPRINTK_ON(vnic->tx_skb != NULL);

	vnic->frontend_state = XenbusStateUnknown;
	net_accel_update_state(vnic->dev, XenbusStateUnknown);

	mutex_unlock(&vnic->vnic_mutex);

	np->accel_priv = NULL;
	kfree(vnic);
}


static int vnic_setup_domU_shared_state(struct xenbus_device *dev,
					netfront_accel_vnic *vnic)
{
	struct xenbus_transaction tr;
	int err;
	int msgs_per_queue;


	DPRINTK("Setting up domU shared state.\n");

	msgs_per_queue = (PAGE_SIZE/2) / sizeof(struct net_accel_msg);

	/* Allocate buffer state */
	vnic->tx_bufs = netfront_accel_init_bufs(&vnic->tx_lock);
	if (vnic->tx_bufs == NULL) {
		err = -ENOMEM;
		EPRINTK("%s: Failed to allocate tx buffers\n", __FUNCTION__);
		goto fail_tx_bufs;
	}

	vnic->rx_bufs = netfront_accel_init_bufs(NULL);
	if (vnic->rx_bufs == NULL) {
		err = -ENOMEM;
		EPRINTK("%s: Failed to allocate rx buffers\n", __FUNCTION__);
		goto fail_rx_bufs;
	}

	/* 
	 * This allocates two pages, one for the shared page and one
	 * for the message queue.
	 */
	vnic->shared_page = (struct net_accel_shared_page *)
		__get_free_pages(GFP_KERNEL, 1);
	if (vnic->shared_page == NULL) {
		EPRINTK("%s: no memory for shared pages\n", __FUNCTION__);
		err = -ENOMEM;
		goto fail_shared_page;
	}

	net_accel_msg_init_queue
		(&vnic->from_dom0, &vnic->shared_page->queue0, 
		 (struct net_accel_msg *)((u8*)vnic->shared_page + PAGE_SIZE),
		 msgs_per_queue);

	net_accel_msg_init_queue
		(&vnic->to_dom0, &vnic->shared_page->queue1,
		 (struct net_accel_msg *)((u8*)vnic->shared_page +
					  (3 * PAGE_SIZE / 2)),
		 msgs_per_queue);
	
	vnic->msg_state = NETFRONT_ACCEL_MSG_NONE;

	err = make_named_grant(dev, vnic->shared_page, "accel-ctrl-page",
			       &vnic->ctrl_page_gnt);
	if (err) {
		EPRINTK("couldn't make ctrl-page named grant\n");
		goto fail_ctrl_page_grant;
	}

	err = make_named_grant(dev, (u8*)vnic->shared_page + PAGE_SIZE,
			       "accel-msg-page", &vnic->msg_page_gnt);
	if (err) {
		EPRINTK("couldn't make msg-page named grant\n");
		goto fail_msg_page_grant;
	}

	/* Create xenbus msg event channel */
	err = bind_listening_port_to_irqhandler
		(dev->otherend_id, netfront_accel_msg_channel_irq_from_bend,
		 IRQF_SAMPLE_RANDOM, "vnicctrl", vnic);
	if (err < 0) {
		EPRINTK("Couldn't bind msg event channel\n");
		goto fail_msg_irq;
	}
	vnic->msg_channel_irq = err;
	vnic->msg_channel = irq_to_evtchn_port(vnic->msg_channel_irq);
	
	/* Create xenbus net event channel */
	err = bind_listening_port_to_irqhandler
		(dev->otherend_id, netfront_accel_net_channel_irq_from_bend,
		 IRQF_SAMPLE_RANDOM, "vnicfront", vnic);
	if (err < 0) {
		EPRINTK("Couldn't bind net event channel\n");
		goto fail_net_irq;
	}
	vnic->net_channel_irq = err;
	vnic->net_channel = irq_to_evtchn_port(vnic->net_channel_irq);
	/* Want to ensure we don't get interrupts before we're ready */
	netfront_accel_disable_net_interrupts(vnic);

	DPRINTK("otherend %d has msg ch %u (%u) and net ch %u (%u)\n",
		dev->otherend_id, vnic->msg_channel, vnic->msg_channel_irq, 
		vnic->net_channel, vnic->net_channel_irq);

	do {
		err = xenbus_transaction_start(&tr);
		if (err != 0) {
			EPRINTK("%s: Transaction start failed %d\n",
				__FUNCTION__, err);
			goto fail_transaction;
		}

		err = xenbus_printf(tr, dev->nodename, "accel-msg-channel",
				    "%u", vnic->msg_channel);
		if (err != 0) {
			EPRINTK("%s: event channel xenbus write failed %d\n",
				__FUNCTION__, err);
			xenbus_transaction_end(tr, 1);
			goto fail_transaction;
		}

		err = xenbus_printf(tr, dev->nodename, "accel-net-channel",
				    "%u", vnic->net_channel);
		if (err != 0) {
			EPRINTK("%s: net channel xenbus write failed %d\n",
				__FUNCTION__, err);
			xenbus_transaction_end(tr, 1);
			goto fail_transaction;
		}

		err = xenbus_transaction_end(tr, 0);
	} while (err == -EAGAIN);

	if (err != 0) {
		EPRINTK("%s: Transaction end failed %d\n", __FUNCTION__, err);
		goto fail_transaction;
	}

	DPRINTK("Completed setting up domU shared state\n");

	return 0;

fail_transaction:

	unbind_from_irqhandler(vnic->net_channel_irq, vnic);
fail_net_irq:

	unbind_from_irqhandler(vnic->msg_channel_irq, vnic);
fail_msg_irq:

	remove_named_grant(dev, "accel-ctrl-page", vnic->ctrl_page_gnt);
fail_msg_page_grant:

	remove_named_grant(dev, "accel-msg-page", vnic->msg_page_gnt);
fail_ctrl_page_grant:

	free_pages((unsigned long)vnic->shared_page, 1);
	vnic->shared_page = NULL;
fail_shared_page:

	netfront_accel_fini_bufs(vnic->rx_bufs);
fail_rx_bufs:

	netfront_accel_fini_bufs(vnic->tx_bufs);
fail_tx_bufs:

	/* Undo the memory allocation created when we got the HELLO */
	netfront_accel_free_buffer_mem(&vnic->bufpages,
				       vnic->rx_bufs,
				       vnic->tx_bufs);

	DPRINTK("Failed to setup domU shared state with code %d\n", err);

	return err;
}


static void vnic_remove_domU_shared_state(struct xenbus_device *dev, 
					  netfront_accel_vnic *vnic)
{
	struct xenbus_transaction tr;
	
	/*
	 * Don't remove any watches because we currently hold the
	 * mutex and the watches take the mutex.
	 */

	DPRINTK("%s: removing event channel irq handlers %d %d\n",
		__FUNCTION__, vnic->net_channel_irq, vnic->msg_channel_irq);
	do {
		if (xenbus_transaction_start(&tr) != 0)
			break;
		xenbus_rm(tr, dev->nodename, "accel-msg-channel");
		xenbus_rm(tr, dev->nodename, "accel-net-channel");
	} while (xenbus_transaction_end(tr, 0) == -EAGAIN);

	unbind_from_irqhandler(vnic->net_channel_irq, vnic);
	unbind_from_irqhandler(vnic->msg_channel_irq, vnic);

	/* ungrant pages for msg channel */
	remove_named_grant(dev, "accel-ctrl-page", vnic->ctrl_page_gnt);
	remove_named_grant(dev, "accel-msg-page", vnic->msg_page_gnt);
	free_pages((unsigned long)vnic->shared_page, 1);
	vnic->shared_page = NULL;

	/* ungrant pages for buffers, and free buffer memory */
	netfront_accel_free_buffer_mem(&vnic->bufpages,
				       vnic->rx_bufs,
				       vnic->tx_bufs);
	netfront_accel_fini_bufs(vnic->rx_bufs);
	netfront_accel_fini_bufs(vnic->tx_bufs);
}


static void vnic_setup_dom0_shared_state(struct xenbus_device *dev,
					netfront_accel_vnic *vnic)
{
	DPRINTK("Setting up dom0 shared state\n");

	netfront_accel_vi_ctor(vnic);

	/*
	 * Message processing will be enabled when this function
	 * returns, but we might have missed an interrupt.  Schedule a
	 * check just in case.
	 */
	queue_work(netfront_accel_workqueue, &vnic->msg_from_bend);
}


static void vnic_remove_dom0_shared_state(struct xenbus_device *dev,
					  netfront_accel_vnic *vnic)
{
	DPRINTK("Removing dom0 shared state\n");

	vnic_stop_fastpath(vnic);

	netfront_accel_vi_dtor(vnic);
}


/*************************************************************************/

/*
 * The following code handles accelstate changes between the frontend
 * and the backend.  In response to transitions, calls the following
 * functions in matching pairs:
 *
 *   vnic_setup_domU_shared_state
 *   vnic_remove_domU_shared_state
 *
 *   vnic_setup_dom0_shared_state
 *   vnic_remove_dom0_shared_state
 *
 * Valid state transitions for DomU are as follows:
 *
 * Closed->Init       on probe or in response to Init from dom0
 *
 * Init->Connected    in response to Init from dom0
 * Init->Closing      on error providing dom0 is in Init
 * Init->Closed       on remove or in response to Closing from dom0
 *
 * Connected->Closing on error/remove
 * Connected->Closed  in response to Closing from dom0
 *
 * Closing->Closed    in response to Closing from dom0
 *
 */


/* Function to deal with Xenbus accel state change in backend */
static void netfront_accel_backend_accel_changed(netfront_accel_vnic *vnic,
						 XenbusState backend_state)
{
	struct xenbus_device *dev = vnic->dev;
	XenbusState frontend_state;
	int state;

	DPRINTK("%s: changing from %s to %s. nodename %s, otherend %s\n",
		__FUNCTION__, xenbus_strstate(vnic->backend_state),
		xenbus_strstate(backend_state), dev->nodename, dev->otherend);

	/*
	 * Ignore duplicate state changes.  This can happen if the
	 * backend changes state twice in quick succession and the
	 * first watch fires in the frontend after the second
	 * transition has completed.
	 */
	if (vnic->backend_state == backend_state)
		return;

	vnic->backend_state = backend_state;
	frontend_state = vnic->frontend_state;

	switch (backend_state) {
	case XenbusStateInitialising:
		/*
		 * It's possible for us to miss the closed state from
		 * dom0, so do the work here.
		 */
		if (vnic->domU_state_is_setup) {
			vnic_remove_domU_shared_state(dev, vnic);
			vnic->domU_state_is_setup = 0;
		}

		if (frontend_state != XenbusStateInitialising) {
			/* Make sure the backend doesn't go away. */
			frontend_state = XenbusStateInitialising;
			net_accel_update_state(dev, frontend_state);
			xenbus_scanf(XBT_NIL, dev->otherend, "accelstate", "%d", &state);
			backend_state = (XenbusState)state;
			if (backend_state != XenbusStateInitialising)
				break;
		}

		/* Start the new connection. */
		if (!vnic->removing) {
			BUG_ON(vnic->domU_state_is_setup);
			if (vnic_setup_domU_shared_state(dev, vnic) == 0) {
				vnic->domU_state_is_setup = 1;
				frontend_state = XenbusStateConnected;
			} else
				frontend_state = XenbusStateClosing;
		}
		break;
	case XenbusStateConnected:
		if (vnic->domU_state_is_setup &&
		    !vnic->dom0_state_is_setup) {
			vnic_setup_dom0_shared_state(dev, vnic);
			vnic->dom0_state_is_setup = 1;
		}
		break;
	default:
	case XenbusStateClosing:
		if (vnic->dom0_state_is_setup) {
			vnic_remove_dom0_shared_state(dev, vnic);
			vnic->dom0_state_is_setup = 0;
		}
		frontend_state = XenbusStateClosed;
		break;
	case XenbusStateUnknown:
	case XenbusStateClosed:
		if (vnic->domU_state_is_setup) {
			vnic_remove_domU_shared_state(dev, vnic);
			vnic->domU_state_is_setup = 0;
		}
		break;
	}

	if (frontend_state != vnic->frontend_state) {
		DPRINTK("Switching from state %s (%d) to %s (%d)\n",
			xenbus_strstate(vnic->frontend_state),
			vnic->frontend_state,
			xenbus_strstate(frontend_state), frontend_state);
		vnic->frontend_state = frontend_state;
		net_accel_update_state(dev, frontend_state);
	}

	wake_up(&vnic->state_wait_queue);
}


static void backend_accel_state_change(struct xenbus_watch *watch,
				       const char **vec, unsigned int len)
{
	int state;
	netfront_accel_vnic *vnic;
	struct xenbus_device *dev;

	DPRINTK("%s\n", __FUNCTION__);

	vnic = container_of(watch, struct netfront_accel_vnic,
				backend_accel_watch);

	mutex_lock(&vnic->vnic_mutex);

	dev = vnic->dev;

	state = (int)XenbusStateUnknown;
	xenbus_scanf(XBT_NIL, dev->otherend, "accelstate", "%d", &state);
	netfront_accel_backend_accel_changed(vnic, state);

	mutex_unlock(&vnic->vnic_mutex);
}


static int setup_dom0_accel_watch(struct xenbus_device *dev,
				  netfront_accel_vnic *vnic)
{
	int err;

	DPRINTK("Setting watch on %s/%s\n", dev->otherend, "accelstate");

	err = xenbus_watch_path2(dev, dev->otherend, "accelstate", 
				 &vnic->backend_accel_watch, 
				 backend_accel_state_change);
	if (err) {
		EPRINTK("%s: Failed to register xenbus watch: %d\n",
			__FUNCTION__, err);
		goto fail;
	}
	return 0;
 fail:
	vnic->backend_accel_watch.node = NULL;
	return err;
}


int netfront_accel_probe(struct net_device *net_dev, struct xenbus_device *dev)
{
	netfront_accel_vnic *vnic;
	int err;

	DPRINTK("Probe passed device %s\n", dev->nodename);

	vnic = netfront_accel_vnic_ctor(net_dev, dev);
	if (IS_ERR(vnic))
		return PTR_ERR(vnic);

	/*
	 * Setup a watch on the backend accel state.  This sets things
	 * going.
	 */
	err = setup_dom0_accel_watch(dev, vnic);
	if (err) {
		netfront_accel_vnic_dtor(vnic);
		EPRINTK("%s: probe failed with code %d\n", __FUNCTION__, err);
		return err;
	}

	/*
	 * Indicate to the other end that we're ready to start unless
	 * the watch has already fired.
	 */
	mutex_lock(&vnic->vnic_mutex);
	VPRINTK("setup success, updating accelstate\n");
	if (vnic->frontend_state == XenbusStateClosed) {
		vnic->frontend_state = XenbusStateInitialising;
		net_accel_update_state(dev, XenbusStateInitialising);
	}
	mutex_unlock(&vnic->vnic_mutex);

	DPRINTK("Probe done device %s\n", dev->nodename);

	return 0;
}


int netfront_accel_remove(struct xenbus_device *dev)
{
	struct netfront_info *np = dev_get_drvdata(&dev->dev);
	netfront_accel_vnic *vnic = (netfront_accel_vnic *)np->accel_priv;

	DPRINTK("%s %s\n", __FUNCTION__, dev->nodename);

	BUG_ON(vnic == NULL);

	mutex_lock(&vnic->vnic_mutex);

	/* Reject any attempts to connect. */
	vnic->removing = 1;

	/* Close any existing connection. */
	if (vnic->frontend_state == XenbusStateConnected) {
		vnic->frontend_state = XenbusStateClosing;
		net_accel_update_state(dev, XenbusStateClosing);
	}

	mutex_unlock(&vnic->vnic_mutex);

	DPRINTK("%s waiting for release of %s\n", __FUNCTION__, dev->nodename);

	/*
	 * Wait for the xenbus watch to release the shared resources.
	 * This indicates that dom0 has made the transition
	 * Closing->Closed or that dom0 was in Closed or Init and no
	 * resources were mapped.
	 */
	wait_event(vnic->state_wait_queue,
		   !vnic->domU_state_is_setup);

	/*
	 * Now we don't need this watch anymore it is safe to remove
	 * it (and so synchronise with it completing if outstanding)
	 */
	DPRINTK("%s: unregistering xenbus accel watch\n",
		__FUNCTION__);
	unregister_xenbus_watch(&vnic->backend_accel_watch);
	kfree(vnic->backend_accel_watch.node);

	netfront_accel_vnic_dtor(vnic);

	DPRINTK("%s done %s\n", __FUNCTION__, dev->nodename);

	return 0;
}
