/*
 * Copyright (C) 2008 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by Neil Jagdish Patel <neil.patel@canonical.com>
 *
 */

#if HAVE_CONFIG_H
#include <config.h>
#endif

#include "nl-gadget-loader.h"

#include <glib.h>
#include <glib-object.h>

#include <glib/gi18n-lib.h>
#include <clutk/clutk.h>

#include <clutter/clutter.h>
#include <clutter-gtk/clutter-gtk.h>
#include <launcher/launcher.h>
#include <libgadget/libgadget.h>

#include "nl-favorite-gadget.h"
#include "nl-gadget-item.h"

G_DEFINE_TYPE (NlGadgetLoader, nl_gadget_loader,
               NL_TYPE_FAVORITES_LOADER);

#define NL_GADGET_LOADER_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj), \
  NL_TYPE_GADGET_LOADER, \
  NlGadgetLoaderPrivate))

struct _NlGadgetLoaderPrivate
{
  NlShell      *shell;
  NlGadgetItem *item;
};

enum
{
  PROP_0,
  PROP_SHELL
};

/* Forwards */
static gboolean     handles_type (NlFavoritesLoader *self,
                                  const gchar       *type);
static NlFavorite * load         (NlFavoritesLoader *self,
                                  const gchar       *uid);
static gchar      * uid_for_path (NlFavoritesLoader *self,
                                  const gchar       *path);

/* GObject stuff */
static void
set_property (GObject      *object,
              guint         prop_id,
              const GValue *value,
              GParamSpec   *pspec)
{
  NlGadgetLoaderPrivate *priv;

  g_return_if_fail (NL_IS_GADGET_LOADER (object));
  priv = NL_GADGET_LOADER_GET_PRIVATE (object);

  switch (prop_id)
    {
    case PROP_SHELL:
      priv->shell = g_value_get_pointer (value);
      break;

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
    }
}

static void
get_property (GObject      *object,
              guint         prop_id,
              GValue       *value,
              GParamSpec   *pspec)
{
  NlGadgetLoaderPrivate *priv;

  g_return_if_fail (NL_IS_GADGET_LOADER (object));
  priv = NL_GADGET_LOADER_GET_PRIVATE (object);

  switch (prop_id)
    {
    case PROP_SHELL:
      g_value_set_pointer (value, priv->shell);
      break;

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
    }
}

static void
nl_gadget_loader_finalize (GObject *object)
{
  NlGadgetLoaderPrivate *priv;

  priv = NL_GADGET_LOADER_GET_PRIVATE (object);

  if (priv->item)
    {
      g_object_unref (priv->item);
      priv->item = NULL;
    }
  G_OBJECT_CLASS (nl_gadget_loader_parent_class)->finalize (object);
}

static void
nl_gadget_loader_constructed (GObject *object)
{
  NlGadgetLoaderPrivate *priv;

  priv = NL_GADGET_LOADER_GET_PRIVATE (object);

  nl_shell_add_favorites_loader (priv->shell,
                                 NL_FAVORITES_LOADER (object));

  priv->item = nl_gadget_item_new (priv->shell);
}

static void
nl_gadget_loader_class_init (NlGadgetLoaderClass *klass)
{
  GObjectClass                 *obj_class = G_OBJECT_CLASS (klass);
  NlFavoritesLoaderClass *fav_class;
  GParamSpec                   *pspec;

  fav_class = NL_FAVORITES_LOADER_CLASS (klass);

  obj_class->finalize     = nl_gadget_loader_finalize;
  obj_class->constructed  = nl_gadget_loader_constructed;
  obj_class->set_property = set_property;
  obj_class->get_property = get_property;

  fav_class->handles_type = handles_type;
  fav_class->load         = load;
  fav_class->uid_for_path = uid_for_path;

  pspec = g_param_spec_pointer ("shell", "shell", "shell",
                                G_PARAM_READWRITE | G_PARAM_CONSTRUCT);
  g_object_class_install_property (obj_class, PROP_SHELL, pspec);

  g_type_class_add_private (obj_class, sizeof (NlGadgetLoaderPrivate));
}

static void
nl_gadget_loader_init (NlGadgetLoader *loader)
{
  NlGadgetLoaderPrivate *priv;

  priv = loader->priv = NL_GADGET_LOADER_GET_PRIVATE (loader);
}

/*
 * Public methods
 */
GObject *
nl_gadget_loader_new (NlShell *shell)

{
  GObject *gadget_loader;

  g_return_val_if_fail (NL_IS_SHELL (shell), NULL);

  gadget_loader = g_object_new (NL_TYPE_GADGET_LOADER,
                                "shell", shell,
                                NULL);

  return gadget_loader;
}


/*
 * Private methods
 */
static gboolean
handles_type (NlFavoritesLoader *self,
              const gchar             *type)
{
  NlGadgetLoaderPrivate *priv;

  priv = NL_GADGET_LOADER (self)->priv;

  if (g_strcmp0 (type, "libgadget-gadget") == 0)
    return TRUE;

  return FALSE;
}

static NlFavorite *
load (NlFavoritesLoader *self,
      const gchar             *uid)
{
  NlGadgetLoaderPrivate *priv;
  NlFavorite *fav = NULL;

  g_return_val_if_fail (NL_IS_GADGET_LOADER (self), NULL);
  priv = NL_GADGET_LOADER (self)->priv;

  fav = nl_favorite_gadget_new (uid, priv->shell);

  return fav;
}

static gchar *
uid_for_path (NlFavoritesLoader *self, const gchar *path)
{
  GadgetManager     *manager;
  gchar             *uid = NULL;

  manager = gadget_manager_get_default ();
  gadget_manager_load_sources (manager);

  if (gadget_manager_can_handle_path (manager, path))
    {
      LauncherFavorites *favs;

      favs = launcher_favorites_get_default ();

      uid = g_strdup_printf ("gadget-%d", (gint)time (NULL));
      launcher_favorites_set_string (favs, uid, "type", "libgadget-gadget");
      launcher_favorites_set_string (favs, uid, "path", path);
      launcher_favorites_add_favorite (favs, uid);

      g_object_unref (favs);
    }

  g_object_unref (manager);
  return uid;
}

/*
 * Public Methods
 */
