/* -*- Mode: C; indent-tabs-mode: nil; c-basic-offset: 4 -*- */

/*
 *  Copyright (C) 2004 Hiroyuki Ikezoe
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "kz-entry.h"

#include "gtk-utils.h"
#include <glib/gi18n.h>

enum {
    PROP_0,
    PROP_BACKGROUND_TEXT
};

/* object class */
static void finalize       (GObject         *object);
static void set_property   (GObject         *object,
                            guint            prop_id,
                            const GValue    *value,
                            GParamSpec      *pspec);
static void get_property   (GObject         *object,
                            guint            prop_id,
                            GValue          *value,
                            GParamSpec      *pspec);

static gboolean expose     (GtkWidget        *widget,
                            GdkEventExpose   *event);

G_DEFINE_TYPE(KzEntry, kz_entry, GTK_TYPE_ENTRY) 

static void
kz_entry_class_init (KzEntryClass *klass)
{
    GObjectClass *gobject_class;
    GtkWidgetClass *widget_class;

    gobject_class = G_OBJECT_CLASS(klass);
    widget_class  = GTK_WIDGET_CLASS(klass);

    gobject_class->finalize     = finalize;
    gobject_class->set_property = set_property;
    gobject_class->get_property = get_property;

    widget_class->expose_event  = expose;

    g_object_class_install_property (gobject_class,
            PROP_BACKGROUND_TEXT,
            g_param_spec_string ("background-text",
                _("Background Text"),
                _("The background text of the entry"),
                "",
                G_PARAM_READWRITE));
}


static void
kz_entry_init (KzEntry *entry)
{
    entry->background_text = NULL;
    entry->with_arrow = FALSE;
}


static void
finalize (GObject *object)
{
    KzEntry *entry = KZ_ENTRY(object);

    g_free(entry->background_text);

    G_OBJECT_CLASS(kz_entry_parent_class)->finalize(object);
}


static void
set_property (GObject *object,
              guint prop_id,
              const GValue *value,
              GParamSpec *pspec)
{
    KzEntry *entry = KZ_ENTRY(object);

    switch (prop_id) {
    case PROP_BACKGROUND_TEXT:
        kz_entry_set_background_text(entry, g_value_get_string(value));
        break;
    default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
        break;
    }
}


static void
get_property (GObject *object,
              guint prop_id,
              GValue *value,
              GParamSpec *pspec)
{
    KzEntry *entry = KZ_ENTRY(object);

    switch (prop_id) {
    case PROP_BACKGROUND_TEXT:
        g_value_set_string(value, kz_entry_get_background_text(entry));
        break;
    default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
        break;
    }
}


GtkWidget *
kz_entry_new (void)
{
    return GTK_WIDGET(g_object_new(KZ_TYPE_ENTRY, NULL));
}

static gboolean
expose (GtkWidget *widget, GdkEventExpose *event)
{
    GtkEntry *entry = GTK_ENTRY(widget);

    if (!GTK_WIDGET_HAS_FOCUS(widget) &&
        (!gtk_entry_get_text(entry) || gtk_entry_get_text(entry)[0] == '\0')) {
        PangoLayout *layout;
        PangoAttrList *attrs;
        PangoAttribute *foreground_attr;
        GtkStyle *style;
        GdkColor *color;

        style = gtk_widget_get_style(widget);
        layout = gtk_entry_get_layout(GTK_ENTRY(entry));
        attrs = pango_layout_get_attributes(layout);
        color = &style->fg[GTK_STATE_INSENSITIVE];
        foreground_attr = pango_attr_foreground_new(color->red,
                color->green,
                color->blue);
        pango_attr_list_insert_before(attrs, foreground_attr);
        pango_layout_set_attributes(layout, attrs);
        pango_layout_set_text(layout, KZ_ENTRY(entry)->background_text, -1);
    }

    return GTK_WIDGET_CLASS(kz_entry_parent_class)->expose_event(widget, event);
}


void
kz_entry_set_background_text (KzEntry *entry,
                              const gchar *text)
{
    g_return_if_fail(KZ_IS_ENTRY(entry));
    g_return_if_fail(text != NULL);

    g_free(entry->background_text);
    entry->background_text = g_strdup(text);
}


void
kz_entry_set_arrow (KzEntry  *entry,
                    gboolean arrow)
{
    g_return_if_fail(KZ_IS_ENTRY(entry));

    entry->with_arrow = arrow;
}


G_CONST_RETURN gchar *
kz_entry_get_background_text (KzEntry *entry)
{
    g_return_val_if_fail(KZ_IS_ENTRY(entry), NULL);

    return entry->background_text;
}

/* 
vi:ts=4:nowrap:ai:expandtab:sw=4
*/
