/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2004 Hiroyuki Ikezoe
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <glib/gi18n.h>
#include <glib.h>
#include <gmodule.h>

#include <anthy/anthy.h>

#include <mecab.h>

#include "kazehakase.h"
#include "kz-anthy-trainer.h"
#include "utils.h"


typedef struct _KzAnthyTrainerPrivate	KzAnthyTrainerPrivate;
struct _KzAnthyTrainerPrivate
{
	gchar *dbname;
	mecab_t *mecab;
};

typedef struct _KzAnthyTrainerClass	KzAnthyTrainerClass;
struct _KzAnthyTrainerClass
{
	KzSearchClass parent_class;
};

#define KZ_ANTHY_TRAINER_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj), KZ_TYPE_ANTHY_TRAINER, KzAnthyTrainerPrivate))

#define KZ_ANTHY_TRAINER_CLASS(klass)	(G_TYPE_CHECK_CLASS_CAST ((klass), KZ_TYPE_ANTHY_TRAINER, KzAnthyTrainerClass))
#define KZ_IS_ANTHY_TRAINER_CLASS(klass) 	(G_TYPE_CHECK_CLASS_TYPE ((klass), KZ_TYPE_ANTHY_TRAINER))
#define KZ_ANTHY_TRAINER_GET_CLASS(obj)  	(G_TYPE_INSTANCE_GET_CLASS ((obj), KZ_TYPE_ANTHY_TRAINER, KzAnthyTrainerClass))

/* for module */
void        kz_search_module_init		(GTypeModule *module);
void        kz_search_module_exit		(void);
KzSearch   *kz_search_module_create		(void);

/* KzAnthyTrainer Class */
static void   kz_anthy_trainer_class_init (KzAnthyTrainerClass *klass);
static void   kz_anthy_trainer_init       (KzAnthyTrainer *search);

/* GObject Class */
static void     dispose      (GObject *object);

/* KzSearch Class */
static gboolean    register_document          (KzSearch *search,
					       const gchar *uri,
					       const gchar *title,
					       const gchar *contents,
					       GTime mtime);

static GObjectClass *parent_class;
static GType kz_anthy_trainer_type = 0;

static void
kz_anthy_trainer_register_type (GTypeModule *module)
{
	static const GTypeInfo kz_anthy_trainer_info =
	{
		sizeof (KzAnthyTrainerClass),
		NULL,		/* base_init */
		NULL,		/* base_finalize */
		(GClassInitFunc) kz_anthy_trainer_class_init,
		NULL,		/* class_finalize */
		NULL,		/* class_data */
		sizeof (KzAnthyTrainer),
		0,		/* n_preallocs */
		(GInstanceInitFunc) kz_anthy_trainer_init,
	};

	kz_anthy_trainer_type = g_type_module_register_type (module,
			KZ_TYPE_SEARCH,
			"KzAnthyTrainer",
			&kz_anthy_trainer_info, 0);
}

G_MODULE_EXPORT void 
kz_search_module_init (GTypeModule *module)
{
	kz_anthy_trainer_register_type(module);
}

G_MODULE_EXPORT void 
kz_search_module_exit (void)
{
}

G_MODULE_EXPORT KzSearch * 
kz_search_module_create (void)
{
	return kz_anthy_trainer_new();
}

GType
kz_anthy_trainer_get_type (void)
{
	return kz_anthy_trainer_type;
}

static void
kz_anthy_trainer_class_init (KzAnthyTrainerClass *klass)
{
	GObjectClass *object_class;
	KzSearchClass *search_class;

	parent_class = g_type_class_peek_parent (klass);
	object_class = G_OBJECT_CLASS(klass);
	search_class = KZ_SEARCH_CLASS(klass);

	object_class->dispose     = dispose;
	
	search_class->get_search_result_html     = NULL;
	search_class->get_search_result_bookmark = NULL;
	search_class->register_document          = register_document;
	search_class->unregister_document        = NULL;
	search_class->optimize_index             = NULL;
	search_class->make_index                 = NULL;
	search_class->exist_index_dir            = NULL;

	g_type_class_add_private (object_class, sizeof(KzAnthyTrainerPrivate));
}


static void
kz_anthy_trainer_init (KzAnthyTrainer *search)
{
	KzAnthyTrainerPrivate *priv = KZ_ANTHY_TRAINER_GET_PRIVATE(search);

	priv->dbname = g_build_filename(g_get_home_dir(), "."PACKAGE, "anthy", NULL);
	priv->mecab = mecab_new2("");
}

static void
dispose (GObject *object)
{
	KzAnthyTrainerPrivate *priv = KZ_ANTHY_TRAINER_GET_PRIVATE(object);
	
	if (priv->dbname)
		g_free(priv->dbname);

	if (priv->mecab)
		mecab_destroy(priv->mecab);

	priv->mecab  = NULL;
	priv->dbname = NULL;

	if (G_OBJECT_CLASS(parent_class)->dispose)
		G_OBJECT_CLASS(parent_class)->dispose(object);
}


KzSearch *
kz_anthy_trainer_new (void)
{
	return KZ_SEARCH(g_object_new(KZ_TYPE_ANTHY_TRAINER, NULL));
}


gboolean
register_document (KzSearch *search, const gchar *uri, const gchar *title, const gchar *contents, GTime mtime)
{
	KzAnthyTrainerPrivate *priv = KZ_ANTHY_TRAINER_GET_PRIVATE(search);
	mecab_node_t *node;
	gchar *text;
	text = html_to_text(contents);
	node = mecab_sparse_tonode(priv->mecab, text);
	if (!node) return FALSE;

	for (;  node; node = node->next)
	{
		gchar *surface;

		if (node->stat == MECAB_BOS_NODE ||
		    node->stat == MECAB_EOS_NODE)
			continue;

		surface = g_strndup(node->surface, node->length);
	
		g_free(surface);
	}

	g_free(text);
	return FALSE;
}

