/*
 * indicator-network
 * Copyright 2010-2012 Canonical Ltd.
 *
 * Authors:
 * Antti Kaijanmäki <antti.kaijanmaki@canonical.com>
 * Kalle Valo       <kalle.valo@canonical.com>
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 3, as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranties of
 * MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "service-manager.h"

#include <glib.h>

#include "service.h"
#include "manager.h"
#include "marshal.h"

#include "passphrase-dialog.h"

G_DEFINE_TYPE (ServiceManager, service_manager, G_TYPE_OBJECT)

#define GET_PRIVATE(o) \
  (G_TYPE_INSTANCE_GET_PRIVATE ((o), TYPE_SERVICE_MANAGER, 	\
				ServiceManagerPrivate))

typedef struct _ServiceManagerPrivate ServiceManagerPrivate;

struct _ServiceManagerPrivate {
  GList *services;
  Manager *network_service;
};

enum {
  STATE_CHANGED,
  STRENGTH_UPDATED,
  SERVICES_UPDATED,
  LAST_SIGNAL,
};

static guint signals[LAST_SIGNAL] = { 0 };

static void service_manager_free_all(GList *list);

static void passphrase_needed_cb(gpointer *object, gpointer user_data);

static void service_manager_dispose(GObject *object)
{
  G_OBJECT_CLASS (service_manager_parent_class)->dispose (object);
}

static void service_manager_finalize(GObject *object)
{
  ServiceManager *self = SERVICE_MANAGER(object);
  ServiceManagerPrivate *priv = GET_PRIVATE(self);

  service_manager_free_all(priv->services);
  priv->services = NULL;

  G_OBJECT_CLASS (service_manager_parent_class)->finalize (object);
}

static void service_manager_class_init(ServiceManagerClass *klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS (klass);

  g_type_class_add_private (klass, sizeof (ServiceManagerPrivate));

  object_class->dispose = service_manager_dispose;
  object_class->finalize = service_manager_finalize;

  signals[STATE_CHANGED] = g_signal_new("state-changed",
					G_TYPE_FROM_CLASS(klass),
					G_SIGNAL_RUN_LAST,
					0, NULL, NULL,
					_marshal_VOID__VOID,
					G_TYPE_NONE, 0);

  signals[STRENGTH_UPDATED] = g_signal_new("strength-updated",
					G_TYPE_FROM_CLASS(klass),
					G_SIGNAL_RUN_LAST,
					0, NULL, NULL,
					_marshal_VOID__UINT,
					G_TYPE_NONE, 1,
					G_TYPE_UINT);

  signals[SERVICES_UPDATED] = g_signal_new("services-updated",
					   G_TYPE_FROM_CLASS(klass),
					   G_SIGNAL_RUN_LAST,
					   0, NULL, NULL,
					   _marshal_VOID__VOID,
					   G_TYPE_NONE, 0);
}

static void service_manager_init(ServiceManager *self)
{
}

ServiceManager *service_manager_new(Manager *ns)
{
  ServiceManager *self;
  ServiceManagerPrivate *priv;

  self = g_object_new (TYPE_SERVICE_MANAGER, NULL);
  priv = GET_PRIVATE(self);

  priv->network_service = ns;

  return self;
}

static void service_manager_free_all(GList *list)
{
  Service *service;
  GList *iter;

  for (iter = list; iter != NULL; iter = iter->next) {
    service = iter->data;
    g_object_unref(service);
  }

  g_list_free(list);
}

static GList *service_manager_find(GList *list, const gchar *identifier)
{
  Service *service;
  GList *iter;

  for (iter = list; iter != NULL; iter = iter->next) {
    service = iter->data;
    if (g_strcmp0(identifier, service_get_identifier(service)) == 0)
      return iter;
  }

  return NULL;
}

void service_manager_remove_all(ServiceManager *self)
{
  ServiceManagerPrivate *priv = GET_PRIVATE(self);

  service_manager_free_all(priv->services);
  priv->services = NULL;
}

/*
 * Beginnings of proper sorting of services. Needs a lot more work, for example
 * need to give more priority for favourite services
 */
static gint cmp_services(gconstpointer a, gconstpointer b)
{
  Service *a_service, *b_service;
  AndroidNetworkType a_type, b_type;
  gboolean a_connected, b_connected;
  guint a_strength, b_strength;

  a_service = SERVICE(a);
  b_service = SERVICE(b);

  a_type = service_get_service_type(a_service);
  b_type = service_get_service_type(b_service);

  // make sure AP that is connected or connecting gets to the head of the list

  if (service_get_state(a_service) == ANDROID_NETWORK_STATE_CONNECTED ||
      service_get_state(a_service) == ANDROID_NETWORK_STATE_CONNECTING)
    a_connected = TRUE;
  else
    a_connected = FALSE;

  if (service_get_state(b_service) == ANDROID_NETWORK_STATE_CONNECTED ||
      service_get_state(b_service) == ANDROID_NETWORK_STATE_CONNECTING)
    b_connected = TRUE;
  else
    b_connected = FALSE;

  a_strength = service_get_strength(a_service);
  b_strength = service_get_strength(b_service);

  /* wifi */

  if (a_type == ANDROID_NETWORK_TYPE_WIFI &&
      b_type == ANDROID_NETWORK_TYPE_WIFI) {
    if (a_connected && !b_connected)
      return -1;
    else if (!a_connected && b_connected)
      return 1;
    else if (a_strength == b_strength)
      return g_strcmp0(service_get_name(a_service),
		       service_get_name(b_service));
    else 
      return b_strength - a_strength;
  }

  /* cellular */

  if (a_type == ANDROID_NETWORK_TYPE_MOBILE &&
      b_type == ANDROID_NETWORK_TYPE_WIFI)
    return 1;

  if (a_type == ANDROID_NETWORK_TYPE_MOBILE &&
      b_type == ANDROID_NETWORK_TYPE_MOBILE)
    return b_strength - a_strength;

  return 0;
}

static void service_state_changed(Service *service, gpointer user_data)
{
  ServiceManager *self = SERVICE_MANAGER(user_data);
  ServiceManagerPrivate *priv = GET_PRIVATE(self);

  g_signal_emit(self, signals[STATE_CHANGED], 0);

  /* services need to be sorted again due to the state change */
  priv->services = g_list_sort(priv->services, cmp_services);

  g_signal_emit(self, signals[SERVICES_UPDATED], 0);
}

static void service_strength_updated(Service *service, gpointer user_data)
{
  ServiceManager *self = SERVICE_MANAGER(user_data);

  g_signal_emit(self, signals[STRENGTH_UPDATED], 0,
		service_get_strength(service));
}

void service_manager_update_services(ServiceManager *self,
				     const GSList *services)
{
  ServiceManagerPrivate *priv = GET_PRIVATE(self);
  Service *service;
  AndroidServiceNetwork *network;
  const GSList *iter;
  gboolean found;

  GList *old_iter;

  /// @todo review this algorithm. Could be cleaner and faster

  // remove the ones that have disappeared
  old_iter = priv->services;
  while (old_iter != NULL) {
    service = old_iter->data;
    found = FALSE;

    for (iter = services; iter != NULL; iter = iter->next) {
      network = iter->data;
      if (g_strcmp0(service_get_identifier(service),
		    network->identifier) == 0) {
	    found = TRUE;
	    break;
      }
    }
    
    if (!found) {
      // step back one item. the item to delete is now old_iter->next
      old_iter = old_iter->prev;
      
      if (old_iter == NULL) {
	// removing the first item on the list
	service = priv->services->data;
	priv->services = g_list_delete_link(priv->services, priv->services);
	g_object_unref(service);
	old_iter = priv->services;
	continue;
      } else {
	priv->services = g_list_delete_link(priv->services, old_iter->next);
	g_object_unref(service);
	old_iter = old_iter->next;
	continue;
      }
    }
    old_iter = old_iter->next;
  }
  
  for (iter = services; iter != NULL; iter = iter->next) {
    network = iter->data;
    
    if (!service_manager_find(priv->services, network->identifier)) {
      service = service_new(network,
			    priv->network_service);
      
      g_signal_connect(network,
		       "passphrase-needed",
		       G_CALLBACK(passphrase_needed_cb),
		       self);

      priv->services = g_list_insert_sorted(priv->services, service,
					    cmp_services);
    }
  }

  // make sure the list is sorted properly
  priv->services = g_list_sort(priv->services, cmp_services);

  g_debug("services updated");

  g_signal_emit(self, signals[SERVICES_UPDATED], 0);
}

GList *service_manager_get_wireless(ServiceManager *self)
{
  ServiceManagerPrivate *priv = GET_PRIVATE(self);
  GList *list = NULL, *iter;
  Service *service;
  const gchar *connected_name;
  AndroidService *android_service = manager_get_android_service(priv->network_service);

  connected_name = android_service_get_connected_name(android_service);

  for (iter = priv->services; iter != NULL; iter = iter->next) {
    service = iter->data;
    if (service_get_service_type(service) == ANDROID_NETWORK_TYPE_WIFI) {

      if (g_strcmp0(connected_name, service_get_name(service)) == 0) {
	list = g_list_prepend(list, service); // add connected service to the top of the list
	continue;
      }

      list = g_list_append(list, service);
    }
  }

  return list;
}

GList *service_manager_get_cellular(ServiceManager *self)
{
  ServiceManagerPrivate *priv = GET_PRIVATE(self);
  GList *list = NULL, *iter;
  AndroidNetworkType type;
  Service *service;

  for (iter = priv->services; iter != NULL; iter = iter->next) {
    service = iter->data;
    type = service_get_service_type(service);

    if (type == ANDROID_NETWORK_TYPE_MOBILE) {
      list = g_list_append(list, service);
    }
  }

  return list;
}

gboolean service_manager_is_connecting(ServiceManager *self)
{
  ServiceManagerPrivate *priv = GET_PRIVATE(self);
  Service *service;
  GList *iter;

  for (iter = priv->services; iter != NULL; iter = iter->next) {
    service = iter->data;
    if (service_get_state(service) == ANDROID_NETWORK_STATE_CONNECTING) {
      return TRUE;
    }
  }

  return FALSE;
}

gboolean service_manager_is_connected(ServiceManager *self)
{
  return service_manager_get_connected(self) > 0;
}

guint service_manager_get_connected(ServiceManager *self)
{
  ServiceManagerPrivate *priv = GET_PRIVATE(self);
  Service *service;
  AndroidNetworkState state;
  GList *iter;
  guint result = 0;

  for (iter = priv->services; iter != NULL; iter = iter->next) {
    service = iter->data;
    state = service_get_state(service);

    if (state == ANDROID_NETWORK_STATE_CONNECTED) {
      result++;
    }
  }

  return result;
}


static void responded(GtkDialog *dialog, gint response_id, gpointer user_data)
{
  ServiceManager *self        = SERVICE_MANAGER(user_data);
  ServiceManagerPrivate *priv = GET_PRIVATE(self);

  PassphraseDialog *passphrase_dialog = PASSPHRASE_DIALOG(dialog);

  const gchar *passphrase;
  const gchar *identifier;
  AndroidService *android_service = NULL;

  const GSList *iter;
  AndroidServiceNetwork *network;

  g_debug("%s(): response_id %d", __func__, response_id);

  if (response_id == GTK_RESPONSE_ACCEPT) {

    identifier = passphrase_dialog_get_identifier(passphrase_dialog);
    passphrase = passphrase_dialog_get_passphrase(passphrase_dialog);

    android_service = manager_get_android_service(priv->network_service);

    for (iter = android_service_get_networks(android_service);
	 iter != NULL;
	 iter = iter->next) {

      network = iter->data;
      if (g_strcmp0(network->identifier, identifier) == 0) {
	g_free(network->passphrase);
	network->passphrase = g_strdup(passphrase);

	android_service_enable_network(android_service, network);
	break;
      }
    }
  }

  /// @todo fix me, the dialog can't really perform a harakiri here, right?
  g_object_unref(passphrase_dialog);
}

static void
passphrase_needed_cb(gpointer *object, gpointer user_data)
{
  ServiceManager *self        = SERVICE_MANAGER(user_data);
  ServiceManagerPrivate *priv = GET_PRIVATE(self);

  PassphraseDialog *dialog;

  AndroidServiceNetwork *network = ANDROID_SERVICE_NETWORK(object);

  g_debug("[%p] WANNA PASSPHRASE?", network);

  dialog = passphrase_dialog_new(network->ssid,
				 network->security,
				 network->identifier);

  g_signal_connect(G_OBJECT(dialog),
		   "response",
		   G_CALLBACK(responded),
		   self);

  passphrase_dialog_ask_passphrase(dialog);
}
