/*
 * indicator-network
 * Copyright 2010-2012 Canonical Ltd.
 *
 * Authors:
 * Antti Kaijanmäki <antti.kaijanmaki@canonical.com>
 * Kalle Valo <kalle.valo@canonical.com>
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 3, as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranties of
 * MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "passphrase-dialog.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <glib.h>
#include <glib-object.h>
#include <glib/gi18n.h>
#include <gtk/gtk.h>
#include <locale.h>

#include "dbus-shared-names.h"

G_DEFINE_TYPE(PassphraseDialog, passphrase_dialog, GTK_TYPE_DIALOG);

typedef struct _PassphraseDialogPrivate PassphraseDialogPrivate;

struct _PassphraseDialogPrivate
{
  GtkWidget *entry;
  GtkWidget *body;

  gchar *ssid;
  AndroidNetworkSecurity security;
  gchar *identifier;
};

#define PASSPHRASE_DIALOG_GET_PRIVATE(o)				\
  (G_TYPE_INSTANCE_GET_PRIVATE((o), PASSPHRASE_DIALOG_TYPE,		\
			       PassphraseDialogPrivate))

#define WEP_MIN_LEN 5
#define WEP_MAX_LEN 26
#define WPA_MIN_LEN 8
#define WPA_MAX_LEN 64

void passphrase_dialog_ask_passphrase(PassphraseDialog *self)
{
  PassphraseDialogPrivate *priv = PASSPHRASE_DIALOG_GET_PRIVATE(self);

  g_debug("%s()", __func__);

  gtk_entry_set_text(GTK_ENTRY(priv->entry), "");

  gtk_widget_show_all(GTK_WIDGET(self));
}

static void passphrase_dialog_init(PassphraseDialog *self)
{
  PassphraseDialogPrivate *priv = PASSPHRASE_DIALOG_GET_PRIVATE(self);

  memset(priv, 0, sizeof(*priv));

  return;
}

static void passphrase_dialog_dispose(GObject *object)
{
  PassphraseDialog *self = PASSPHRASE_DIALOG(object);
  PassphraseDialogPrivate *priv = PASSPHRASE_DIALOG_GET_PRIVATE(self);

  if (priv->body != NULL) {
    gtk_widget_destroy(priv->body);
    priv->body = NULL;
  }

  if (priv->entry != NULL) {
    gtk_widget_destroy(priv->entry);
    priv->entry = NULL;
  }

  G_OBJECT_CLASS(passphrase_dialog_parent_class)->dispose(object);
}

static void passphrase_dialog_finalize(GObject *object)
{
  PassphraseDialog *self = PASSPHRASE_DIALOG(object);
  PassphraseDialogPrivate *priv = PASSPHRASE_DIALOG_GET_PRIVATE(self);

  g_free(priv->ssid);
  g_free(priv->identifier);

  G_OBJECT_CLASS(passphrase_dialog_parent_class)->finalize(object);
}

static void passphrase_dialog_class_init(PassphraseDialogClass *klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS (klass);

  g_type_class_add_private(object_class, sizeof(PassphraseDialogPrivate));

  object_class->dispose = passphrase_dialog_dispose;
  object_class->finalize = passphrase_dialog_finalize;

  g_assert(klass != NULL);
}

static void responded(GtkDialog *dialog, gint response_id, gpointer user_data)
{
  PassphraseDialog *self = PASSPHRASE_DIALOG(user_data);
  PassphraseDialogPrivate *priv = PASSPHRASE_DIALOG_GET_PRIVATE(self);

  gtk_widget_hide(GTK_WIDGET(self));
}

const gchar *
passphrase_dialog_get_identifier(PassphraseDialog *self)
{
  PassphraseDialogPrivate *priv = PASSPHRASE_DIALOG_GET_PRIVATE(self);
  return priv->identifier;
}

const gchar *
passphrase_dialog_get_passphrase(PassphraseDialog *self)
{
  PassphraseDialogPrivate *priv = PASSPHRASE_DIALOG_GET_PRIVATE(self);
  return gtk_entry_get_text(GTK_ENTRY(priv->entry));
}

static void validate(GtkWidget *widget, gpointer user_data)
{
  PassphraseDialog *self = PASSPHRASE_DIALOG(user_data);
  PassphraseDialogPrivate *priv = PASSPHRASE_DIALOG_GET_PRIVATE(self);
  const gchar *text;
  guint len;

  text = gtk_entry_get_text(GTK_ENTRY(priv->entry));
  len = strlen(text);

  switch(priv->security) {

  case ANDROID_NETWORK_SECURITY_WEP:
    if (len >= WEP_MIN_LEN && len <= WEP_MAX_LEN) {
      gtk_dialog_set_default_response(GTK_DIALOG(self), GTK_RESPONSE_ACCEPT);
      gtk_dialog_set_response_sensitive(GTK_DIALOG(self), GTK_RESPONSE_ACCEPT,
					TRUE);
    } else {
      gtk_dialog_set_default_response(GTK_DIALOG(self), GTK_RESPONSE_NONE);
      gtk_dialog_set_response_sensitive(GTK_DIALOG(self), GTK_RESPONSE_ACCEPT,
					FALSE);
    }
    break;
  case ANDROID_NETWORK_SECURITY_WPA:
    if (len >= WPA_MIN_LEN && len <= WPA_MAX_LEN) {
      gtk_dialog_set_default_response(GTK_DIALOG(self), GTK_RESPONSE_ACCEPT);
      gtk_dialog_set_response_sensitive(GTK_DIALOG(self), GTK_RESPONSE_ACCEPT,
					TRUE);
    } else {
      gtk_dialog_set_default_response(GTK_DIALOG(self), GTK_RESPONSE_NONE);
      gtk_dialog_set_response_sensitive(GTK_DIALOG(self), GTK_RESPONSE_ACCEPT,
					FALSE);
    }
    break;
  default:
    g_warning("passphrase-dialog: unknown security mode: %d",
	      priv->security);
    gtk_dialog_set_default_response(GTK_DIALOG(self), GTK_RESPONSE_ACCEPT);
    gtk_dialog_set_response_sensitive(GTK_DIALOG(self), GTK_RESPONSE_ACCEPT,
				      TRUE);
  }

}

static void show_password_toggled(GtkToggleButton *button, gpointer user_data)
{
  PassphraseDialog *self = PASSPHRASE_DIALOG(user_data);
  PassphraseDialogPrivate *priv = PASSPHRASE_DIALOG_GET_PRIVATE(self);

  if (gtk_toggle_button_get_active(button))
    gtk_entry_set_visibility(GTK_ENTRY(priv->entry), TRUE);
  else
    gtk_entry_set_visibility(GTK_ENTRY(priv->entry), FALSE);
}

PassphraseDialog *passphrase_dialog_new(const gchar *ssid,
					AndroidNetworkSecurity security,
					const gchar *identifier)
{
  GtkWidget *vbox, *widget;
  PassphraseDialogPrivate *priv;
  PassphraseDialog *self;
  gchar *title;

  self = g_object_new(PASSPHRASE_DIALOG_TYPE, NULL);
  priv = PASSPHRASE_DIALOG_GET_PRIVATE(self);

  priv->ssid = g_strdup(ssid);
  priv->security = security;
  priv->identifier = g_strdup(identifier);

  title = g_strdup_printf("%s %s", _("Connect to"), priv->ssid);
  gtk_window_set_title(GTK_WINDOW(self), title);
  g_free(title);

  gtk_window_set_icon_name(GTK_WINDOW(self), "nm-signal-100");

  gtk_dialog_add_buttons(GTK_DIALOG(self),
			 GTK_STOCK_CANCEL, GTK_RESPONSE_REJECT,
			 GTK_STOCK_CONNECT, GTK_RESPONSE_ACCEPT,
			 NULL);
  gtk_dialog_set_default_response(GTK_DIALOG(self), GTK_RESPONSE_NONE);
  gtk_dialog_set_response_sensitive(GTK_DIALOG(self), GTK_RESPONSE_ACCEPT,
				    FALSE);

  vbox = gtk_dialog_get_content_area(GTK_DIALOG(self));

  priv->body = gtk_label_new(_("Enter the password."));
  gtk_misc_set_alignment(GTK_MISC(priv->body), 0.0, 0.5);
  gtk_box_pack_start(GTK_BOX(vbox), priv->body, TRUE, TRUE, 5);

  priv->entry = gtk_entry_new();
  gtk_entry_set_visibility(GTK_ENTRY(priv->entry), FALSE);
  gtk_entry_set_width_chars(GTK_ENTRY(priv->entry), WEP_MAX_LEN); /// @todo change if necessary
  gtk_entry_set_activates_default(GTK_ENTRY(priv->entry), TRUE);
  g_signal_connect(G_OBJECT(priv->entry), "changed", (GCallback) validate,
		   self);

  gtk_box_pack_start(GTK_BOX(vbox), priv->entry, TRUE, TRUE, 5);

  widget = gtk_check_button_new_with_label(_("Show password"));
  g_signal_connect(G_OBJECT(widget), "toggled",
		   G_CALLBACK(show_password_toggled), self);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(widget), FALSE);
  gtk_box_pack_start(GTK_BOX(vbox), widget, TRUE, TRUE, 5);

  g_signal_connect(G_OBJECT(self), "response", G_CALLBACK(responded),
		   self);

  return self;
}
