/*
 * indicator-network
 * Copyright 2010-2012 Canonical Ltd.
 *
 * Authors:
 * Antti Kaijanmäki <antti.kaijanmaki@canonical.com>
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 3, as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranties of
 * MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "android-service.h"

#include <string.h>
#include <gio/gio.h>

#define ANDROID_SERVICE_DBUS_SERVICE     "com.canonical.Android"
#define ANDROID_SERVICE_DBUS_OBJECT_PATH "/com/canonical/android/network/NetworkService"
#define ANDROID_SERVICE_DBUS_INTERFACE   "com.canonical.android.network.NetworkService"

typedef struct _AndroidServicePrivate AndroidServicePrivate;

struct _AndroidServicePrivate
{
  GDBusProxy *proxy;
  guint watch_id;

  AndroidNetworkState mobile_state;
  AndroidNetworkState wifi_state;

  GSList *networks;

  gchar *connected_name;

  gboolean connected;
  gint rssi;

  gboolean airplane_mode;
  gboolean mobile_data_enabled;
  gboolean wifi_enabled;

  GHashTable *connection_information;
};

enum {
  PROP_0,

  
};

enum {
  SIG_0,

  SIG_CONNECTIVITY,
  SIG_RSSI_CHANGED,
  SIG_SCAN_RESULTS_READY,
  SIG_NETWORK_STATE_CHANGED,

  SIG_AIRPLANE_MODE_CHANGED,
  SIG_MOBILE_DATA_ENABLED_CHANGED,
  SIG_WIFI_ENABLED_CHANGED,

  SIG_LAST
};
static int signals[SIG_LAST];

#define GET_PRIVATE(o)                                 \
  (G_TYPE_INSTANCE_GET_PRIVATE ((o),                   \
                                ANDROID_SERVICE_TYPE,  \
                                AndroidServicePrivate))

static void android_service_dispose (GObject *object);
static void android_service_finalize(GObject *object);

G_DEFINE_TYPE(AndroidService, android_service, G_TYPE_OBJECT);

static void get_network_states(AndroidService *self);
static void net_get_networks(AndroidService *self);

static void set_airplane_mode(AndroidService *self, gboolean value);
static void set_mobile_data_enabled(AndroidService *self, gboolean value);
static void set_wifi_enabled(AndroidService *self, gboolean value);

static void get_connection_information(AndroidService *self);

static gint
dbm2strength(gint32 dbm)
{
  if (dbm < -96)
    dbm = -96;

  if (dbm > -20)
    dbm = -20;

  return 100 * (96 + dbm) / 76;
}

static void
net_rssi_changed_handler(AndroidService *self,
			 gint32 value)
{
  AndroidServicePrivate *priv = GET_PRIVATE(self);

  
  priv->rssi = dbm2strength(value);
  g_signal_emit(G_OBJECT(self), signals[SIG_RSSI_CHANGED], 0);
}

static void
net_get_scan_results_cb(GObject *source_object,
			GAsyncResult *res,
			gpointer user_data)
{
  AndroidService *self        = NULL;
  AndroidServicePrivate *priv = NULL;
  GError *error               = NULL;
  GVariant *result            = NULL;
  
  GVariantIter *list_iter;
  
  gchar *ssid        = NULL;
  gchar *capabilities = NULL;
  gint32 dbm          = G_MININT32;

  gchar                  *identifier = NULL;
  AndroidNetworkSecurity  security   = ANDROID_NETWORK_SECURITY_NONE;
  gint                    strength   = 0;

  AndroidServiceNetwork *network = NULL;
  GSList                *element = NULL;

  // this list keeps track which networks have dissapeared since last call
  GSList *old_list = NULL;

  g_debug("net_get_scan_results_cb called");

  self = ANDROID_SERVICE(user_data);
  g_assert(self != NULL);

  priv = GET_PRIVATE(self);

  result = g_dbus_proxy_call_finish(priv->proxy,
				    res,
				    &error);

  if (error != NULL) {
    g_warning("net_get_scan_results_cb failed: %s",
	       error->message);
    g_error_free(error);

    g_slist_free_full(priv->networks,
		      (GDestroyNotify)g_object_unref);
    priv->networks = NULL;
    g_signal_emit(G_OBJECT(self), signals[SIG_SCAN_RESULTS_READY], 0);

    get_network_states(self);

    return;
  }

  g_debug("about to process scan results");

  old_list = g_slist_copy(priv->networks);

  g_variant_get(result, "(a(ssi))", &list_iter);

  while (g_variant_iter_loop(list_iter, "(ssi)", &ssid, &capabilities, &dbm)) {
    
    security = ANDROID_NETWORK_SECURITY_NONE;
    if (g_strrstr(capabilities, "WEP") != NULL) {
      security = ANDROID_NETWORK_SECURITY_WEP;
    } else if (g_strrstr(capabilities, "WPA") != NULL) {
      security = ANDROID_NETWORK_SECURITY_WPA;
    }

    strength = dbm2strength(dbm);


    /* Create the identifier by combining ssid with capabilities.
       Access Points with same ssid and capabilities are considered
       to be part of the same network. */
    identifier = g_strdup_printf("%s_%s", ssid, capabilities);

    
    for (element = priv->networks;
	 element != NULL;
	 element = g_slist_next(element)) {
      
      network = element->data;
    
      // check if there already exists an element with the given identifier
      if (g_strcmp0(network->identifier, identifier) == 0)
	break;
    } 


    if (G_LIKELY (element != NULL)) {

      /* The network exists in the list already. Either it's an previous entry
	 or there are multiple APs on the same network. Leave the existing entry
	 to the list and update strength if the new is a stronger one. */

      network = element->data;
      
      if (android_service_network_get_strength(network) < strength)
	android_service_network_set_strength(network, strength);
     

      // remove the network from dissapeared list as we have seen it now :)
      old_list = g_slist_remove(old_list, network);

    } else {

      network = android_service_network_new(identifier,
					    ssid,
					    security,
					    capabilities,
					    strength,
					    ANDROID_NETWORK_TYPE_WIFI);
 
      priv->networks = g_slist_prepend(priv->networks, network);      

    } 

    g_free(identifier);
    identifier   = NULL;
  }

  g_variant_iter_free(list_iter);
  g_variant_unref(result);
  priv->networks = g_slist_reverse(priv->networks);


  // free dissapeared networks from the list
  for (element = old_list;
       element != NULL;
       element = g_slist_next(element)) {

    network = element->data;

    priv->networks = g_slist_remove(priv->networks, network);
  }
  
  // and free the networks.
  g_slist_free_full(old_list,
		    (GDestroyNotify)g_object_unref);
  old_list = NULL;


  g_debug("emitted SIG_SCAN_RESULTS_READY...");
  g_signal_emit(G_OBJECT(self), signals[SIG_SCAN_RESULTS_READY], 0);

  get_network_states(self);
}

static void
net_get_scan_results(AndroidService *self)
{
  AndroidServicePrivate *priv = GET_PRIVATE(self);

  if (!priv->wifi_enabled)
    return;

  g_dbus_proxy_call(priv->proxy,
		    "GetScanResults",
		    NULL,
		    G_DBUS_CALL_FLAGS_NONE,
		    -1,
		    NULL,
		    net_get_scan_results_cb,
		    self);		    
}


static void
net_scan_results_ready_handler(AndroidService *self)
{
  net_get_scan_results(self);
}

static gchar *
unquote_dup_gvariant_str(GVariant *value)
{
    int l;
    gboolean quoteNeedRemove = FALSE;
    gchar *v = g_variant_dup_string(value, NULL);
    gchar *pv, *rv;

    if (v == NULL) return NULL;
    l = strlen(v);
    if (l == 0) return v;
    pv = v;

    if (pv[0] == '"') { // remove prefix double quote
        quoteNeedRemove = TRUE;
        pv++;
        l--;
        if (l == 0) {
          g_free(v); return NULL;
        }
    }
    if (pv[l - 1] == '"') { // remove postfix double qoute
        quoteNeedRemove = TRUE;
        pv[l - 1] = '\0';
        if (l - 1 == 0) {
          g_free(v); return NULL;
        }
         
    }
    if (quoteNeedRemove) {
        rv = g_strdup(pv);
        g_free(v);
        return rv;
    }
    return v;
}

static void
net_get_connected_name_cb(GObject *source_object,
			  GAsyncResult *res,
			  gpointer user_data)
{
  AndroidService * self       = NULL;
  AndroidServicePrivate *priv = NULL;

  GError   *error = NULL;
  GVariant *result;
  GVariant *value;

  GSList *element;
  AndroidServiceNetwork *network;

  g_debug("_connected_name_cb: called");

  self = ANDROID_SERVICE(user_data);
  g_assert(self != NULL);

  priv = GET_PRIVATE(self);

  result = g_dbus_proxy_call_finish(priv->proxy,
				    res,
				    &error);
  if (error != NULL) {
    g_critical("net_get_connected_name_cb failed: %s",
	       error->message);
    g_error_free(error);
    return;
  }

  g_debug("_connected_name_cb: proxy_call_finish completed");

  g_free(priv->connected_name);

  /* Single string 'out' parameter: name */
  g_assert(g_variant_n_children (result) == 1);

  value = g_variant_get_child_value(result, 0);
  if (value) {
    g_debug("_connected_name_cb: connected name returned!");
    priv->connected_name = unquote_dup_gvariant_str(value);
    g_variant_unref(value);

    for (element = priv->networks;
		(element != NULL && priv->connected_name != NULL);
		element = g_slist_next(element)) {
      /// @todo macthing by ssid is not reliable.. we should match by identifier, but we can't :(
      network = element->data;
      if (g_strcmp0(network->ssid, priv->connected_name) == 0)
	android_service_network_set_state(network,
					  ANDROID_NETWORK_STATE_CONNECTED);
      else
	android_service_network_set_state(network,
					  ANDROID_NETWORK_STATE_DISCONNECTED);
    }

  } else {
    g_critical("_connected_name_cb: couldn't get connected name!!!");
  }

  g_variant_unref(result);
  // see net_network_state_changed_handler()
  g_signal_emit(G_OBJECT(self), signals[SIG_NETWORK_STATE_CHANGED], 0);
}

typedef struct _state_helper
{
  AndroidService *service;
  AndroidNetworkType type;
} state_helper;

static void
net_get_network_state_cb(GObject *source_object,
			 GAsyncResult *res,
			 gpointer user_data)
{  
  AndroidService * self       = NULL;
  AndroidServicePrivate *priv = NULL;

  state_helper *helper;

  GError   *error = NULL;
  GVariant *result;

  AndroidNetworkType type;
  GVariant *value;
  gsize size;
  int state;

  helper = user_data;
  type   = helper->type;
  self   = ANDROID_SERVICE(helper->service);
  g_assert(self);
  g_free(helper);

  priv = GET_PRIVATE(self);

  result = g_dbus_proxy_call_finish(priv->proxy,
				    res,
				    &error);

  if (error != NULL) {
    g_critical("net_get_network_state_cb failed: %s",
	       error->message);
    g_error_free(error);
    return;
  }

  /* Single int32 'out' parameter: state */
  g_assert(g_variant_n_children (result) == 1);

  value = g_variant_get_child_value(result, 0);
  if (value) {

    state = g_variant_get_int32(value);

    g_assert (state == ANDROID_NETWORK_STATE_CONNECTING    ||
	      state == ANDROID_NETWORK_STATE_CONNECTED     ||
	      state == ANDROID_NETWORK_STATE_SUSPENDED     ||
	      state == ANDROID_NETWORK_STATE_DISCONNECTING ||
	      state == ANDROID_NETWORK_STATE_DISCONNECTED  ||
	      state == ANDROID_NETWORK_STATE_UNKNOWN);
    g_variant_unref(value);
  } else {
    g_debug("_network_state_cb: couldn't get out param!");
    state = ANDROID_NETWORK_STATE_UNKNOWN;
  }
 
  if (type == ANDROID_NETWORK_TYPE_WIFI) {
    priv->wifi_state   = state;
  } else if (type == ANDROID_NETWORK_TYPE_MOBILE) {
    priv->mobile_state = state;
  } else {
    g_assert_not_reached();
  }

  if (state == ANDROID_NETWORK_STATE_CONNECTED) {
    // if network connects we have to get the name. So let's delay the signal until we have it
    g_dbus_proxy_call(priv->proxy,
		      "GetConnectedName",
		      NULL,
		      G_DBUS_CALL_FLAGS_NONE,
		      -1,
		      NULL,
		      net_get_connected_name_cb,
		      self);
  } else {

    if (priv->wifi_state   != ANDROID_NETWORK_STATE_CONNECTED &&
	priv->mobile_state != ANDROID_NETWORK_STATE_CONNECTED) {
      g_free(priv->connected_name);
      priv->connected_name = g_strdup("");
    }

    g_signal_emit(G_OBJECT(self), signals[SIG_NETWORK_STATE_CHANGED], 0);
  }
  g_variant_unref(result);
}

static void
get_network_states(AndroidService *self)
{
  AndroidServicePrivate *priv = GET_PRIVATE(self);

  state_helper *helper;

  g_debug("get_network_states: called");

  /* we have to query the states by type :/ */

  helper          = g_malloc(sizeof(state_helper));
  helper->service = self;
  helper->type    = ANDROID_NETWORK_TYPE_WIFI;
  g_dbus_proxy_call(priv->proxy,
		    "GetNetworkState",
		    g_variant_new("(i)",
				  ANDROID_NETWORK_TYPE_WIFI),
		    G_DBUS_CALL_FLAGS_NONE,
		    -1,
		    NULL,
		    net_get_network_state_cb,
		    helper);

  helper          = g_malloc(sizeof(state_helper));
  helper->service = self;
  helper->type    = ANDROID_NETWORK_TYPE_MOBILE;
  g_dbus_proxy_call(priv->proxy,
		    "GetNetworkState",
		    g_variant_new("(i)",
				  ANDROID_NETWORK_TYPE_MOBILE),
		    G_DBUS_CALL_FLAGS_NONE,
		    -1,
		    NULL,
		    net_get_network_state_cb,
		    helper);

  // also update the connection information
  get_connection_information(self);
}

static void
net_connectivity_handler(AndroidService *self,
			 gboolean isConnected)
{
  AndroidServicePrivate *priv = GET_PRIVATE(self);
  GSList *element = NULL;

  get_network_states(self);

  // Ubuntu Service sends inverted value for some reason
  priv->connected = !isConnected;

  if(!priv->connected) {
    for (element = priv->networks; element != NULL; element = g_slist_next(element)) {
      AndroidServiceNetwork *network = element->data;
      if (android_service_network_get_state(network)
	  != ANDROID_NETWORK_STATE_DISCONNECTED)
	android_service_network_set_state(network,
					  ANDROID_NETWORK_STATE_DISCONNECTED);
    }
  } 
  g_signal_emit(G_OBJECT(self), signals[SIG_CONNECTIVITY], 0);
}


static void
net_network_state_changed_handler(AndroidService *self,
				  gint value)
{
  get_network_states(self);
}

static void
net_network_wifi_auth_fail_handler(AndroidService *self,
				  const gchar *ssid)
{
  AndroidServicePrivate *priv = GET_PRIVATE(self);
  AndroidServiceNetwork *network;
  GSList *iter;

  for (iter = priv->networks; iter != NULL; iter = iter->next) {
    network = iter->data;
    if (g_strcmp0(ssid, network->ssid) == 0) {
        g_signal_emit_by_name(G_OBJECT(network),
                              "passphrase-needed",
                              0); 
      return;
    }
  }
}


static void
proxy_signal_handler(GDBusProxy *proxy,
		     gchar *sender_name,
		     gchar *signal_name,
		     GVariant *parameters,
		     gpointer user_data)
{
  AndroidService     *self = NULL;
  GVariant           *value = NULL;

  g_debug("proxy_signal_handler: called");

  self = ANDROID_SERVICE(user_data);
  g_assert(self != NULL);

  if (g_strcmp0(signal_name, "Connectivity") == 0) {
    g_debug("proxy_signal_handler: Connectivity");

    // <arg name="isConnected" type="b" direction="out"/>
    g_assert(g_variant_n_children (parameters) == 1);

    value = g_variant_get_child_value(parameters, 0);
    if (value) {
      g_debug("Connectivity - firing handler");
      net_connectivity_handler(self, g_variant_get_boolean(value));
      g_variant_unref(value);
    } else {
      g_critical("Connectivity - couldn't get param!");
    }

  } else if (g_strcmp0(signal_name, "RSSIChanged") == 0) {

    // <arg name="signalStrength" type="i" direction="out"/>
    g_assert(g_variant_n_children (parameters) == 1);

    value = g_variant_get_child_value(parameters, 0);
    if (value) {
      g_debug("RSSIChanged - firing handler");
      net_rssi_changed_handler(self, g_variant_get_int32(value));
      g_variant_unref(value);
    } else {
      g_critical("RSSIChanged - couldn't get param!");
    }
  } else if (g_strcmp0(signal_name, "ScanResultsAvailable") == 0) {
    g_debug("proxy_signal_handler: ScanResultsReady");

    net_scan_results_ready_handler(self);

  } else if (g_strcmp0(signal_name, "NetworkStateChanged") == 0) {
    g_debug("proxy_signal_handler: NetworkStateChanged");

    // <arg name="networkState" type="i" direction="out"/>
    g_assert(g_variant_n_children (parameters) == 1);

    /* TODO: not sure if this return value needs to be freed? */
    value = g_variant_get_child_value(parameters, 0);
    if (value) {
      net_network_state_changed_handler(self,
					g_variant_get_int32(value));
      g_variant_unref(value);
    } else {
      g_critical("proxy_signal_handler: couldn't get RSSI parameter!");
    }
  } else if (g_strcmp0(signal_name, "WifiAuthenticationFail") == 0) {

    // <arg name="SSID" type="s" direction="out"/>
    g_assert(g_variant_n_children (parameters) == 1);

    value = g_variant_get_child_value(parameters, 0);
    if (value) {
      g_debug("WifiAuthenticationFail - firing handler");
      net_network_wifi_auth_fail_handler(self, g_variant_get_string(value,NULL));
      g_variant_unref(value);
    } else {
      g_critical("No SSID - couldn't get param!");
    }
 } else {
    // There MUST NOT be any unknown signals 
    g_assert_not_reached();
    return;
  }

}

static void
proxy_properties_changed_handler(GDBusProxy *proxy,
				 GVariant   *changed_properties,
				 GStrv       invalidated_properties,
				 gpointer    user_data)
{
  AndroidService        *self = NULL;
  AndroidServicePrivate *priv = NULL;

  GVariantIter *iter = NULL;

  gchar *property;
  GVariant *value;

  self = ANDROID_SERVICE(user_data);
  priv = GET_PRIVATE(self);

  g_variant_get(changed_properties, "a{sv}", &iter);

  if (iter == NULL) {
    g_warning("%s: could not get iter", __func__);
    return;
  }

  while(g_variant_iter_loop(iter,
			    "{sv}",
			    &property,
			    &value)) {

    if (g_strcmp0(property, "WifiEnabled") == 0) {

      set_wifi_enabled(self, g_variant_get_boolean(value));

    } else if (g_strcmp0(property, "AirplaneMode") == 0) {
	
      set_airplane_mode(self, g_variant_get_boolean(value));

    } else if (g_strcmp0(property, "MobileDataEnabled") == 0) {

      set_mobile_data_enabled(self, g_variant_get_boolean(value));

    } else {
      g_warning("%s: Unknown property change: %s",
		__func__,
		property);
      continue;
    }
  }

  g_variant_iter_free(iter);
}

static void
proxy_ready_cb(GObject *source_object,
	       GAsyncResult *res,
	       gpointer user_data)
{
  AndroidService        *self = NULL;
  AndroidServicePrivate *priv = NULL;
  GError *error = NULL;
  GVariant *value = NULL;

  g_debug("proxy_ready_cb: called");

  self = ANDROID_SERVICE(user_data);
  g_assert(self != NULL);
  
  priv = GET_PRIVATE(self);

  priv->proxy = g_dbus_proxy_new_for_bus_finish(res, &error);

  if (error != NULL) {
    priv->proxy = NULL;
    g_critical("Could not create proxy: %s", error->message);
    g_error_free(error);
    return;
  }

  g_signal_connect(priv->proxy,
		   "g-signal",
		   G_CALLBACK(proxy_signal_handler),
		   self);

  g_signal_connect(priv->proxy,
		   "g-properties-changed",
		   G_CALLBACK(proxy_properties_changed_handler),
		   self);

  value = g_dbus_proxy_get_cached_property(priv->proxy, "AirplaneMode");
  if (value == NULL) {
    g_warning("%s: could not get cached AirplaneMode", __func__);
  } else {
    set_airplane_mode(self, g_variant_get_boolean(value));
    g_variant_unref(value);
  }

  value = g_dbus_proxy_get_cached_property(priv->proxy, "WifiEnabled");
  if (value == NULL) {
    g_warning("%s: could not get cached WifiEnabled", __func__);
  } else {
    set_wifi_enabled(self, g_variant_get_boolean(value));
    g_variant_unref(value);
  }

  value = g_dbus_proxy_get_cached_property(priv->proxy, "MobileDataEnabled");
  if (value == NULL) {
    g_warning("%s: could not get cached MobileDataEnabled", __func__);
  } else {
    set_mobile_data_enabled(self, g_variant_get_boolean(value));
    g_variant_unref(value);
  }


  if (priv->wifi_enabled) {
    // calls get_network_states() eventually.
    net_get_scan_results(self);
  } else {
    get_network_states(self);
  }
}

static void
service_appeared(GDBusConnection *connection,
		 const gchar *name,
		 const gchar *name_owner,
		 gpointer user_data)
{
  AndroidService        *self = NULL;
  AndroidServicePrivate *priv = NULL;

  g_debug("service_appeared: called");

  self = ANDROID_SERVICE(user_data);
  g_assert(self != NULL);

  priv = GET_PRIVATE(self);

  g_assert(priv->proxy == NULL);

  g_dbus_proxy_new_for_bus(G_BUS_TYPE_SESSION,
			   G_DBUS_PROXY_FLAGS_NONE,
			   NULL,
			   ANDROID_SERVICE_DBUS_SERVICE,
			   ANDROID_SERVICE_DBUS_OBJECT_PATH,
			   ANDROID_SERVICE_DBUS_INTERFACE,
			   NULL,
			   proxy_ready_cb,
			   self);

  g_debug("service_appeared: proxy_new_for_bus called");
			   
}

static void
service_vanished(GDBusConnection *connection,
		 const gchar *name,
		 gpointer user_data)
{
  AndroidService        *self;
  AndroidServicePrivate *priv;

  g_debug("service_vanished: called");

  self = ANDROID_SERVICE(user_data);
  g_assert(self != NULL);

  priv = GET_PRIVATE(self);

  if(priv->proxy == NULL)
    return;

  g_object_unref(priv->proxy);
  priv->proxy = NULL;

}


static void
android_service_class_init(AndroidServiceClass *klass)
{
  GObjectClass *gobject_class = G_OBJECT_CLASS(klass);
  GParamSpec *spec;

  g_type_class_add_private(klass, sizeof(AndroidServicePrivate));

  gobject_class->dispose      = android_service_dispose;
  gobject_class->finalize     = android_service_finalize;

  signals[SIG_CONNECTIVITY] =
    g_signal_newv("connectivity",
		  G_TYPE_FROM_CLASS(gobject_class),
		  G_SIGNAL_RUN_FIRST,
		  NULL,
		  NULL,
		  NULL,
		  g_cclosure_marshal_VOID__VOID,
		  G_TYPE_NONE,
		  0,
		  NULL);

  signals[SIG_RSSI_CHANGED] =
    g_signal_newv("rssi-changed",
		  G_TYPE_FROM_CLASS(gobject_class),
		  G_SIGNAL_RUN_FIRST,
		  NULL,
		  NULL,
		  NULL,
		  g_cclosure_marshal_VOID__VOID,
		  G_TYPE_NONE,
		  0,
		  NULL);

  signals[SIG_SCAN_RESULTS_READY] =
    g_signal_newv("scan-results-ready",
		  G_TYPE_FROM_CLASS(gobject_class),
		  G_SIGNAL_RUN_FIRST,
		  NULL,
		  NULL,
		  NULL,
		  g_cclosure_marshal_VOID__VOID,
		  G_TYPE_NONE,
		  0,
		  NULL);

  signals[SIG_NETWORK_STATE_CHANGED] =
    g_signal_newv("network-state-changed",
		  G_TYPE_FROM_CLASS(gobject_class),
		  G_SIGNAL_RUN_FIRST,
		  NULL,
		  NULL,
		  NULL,
		  g_cclosure_marshal_VOID__VOID,
		  G_TYPE_NONE,
		  0,
		  NULL);

  signals[SIG_AIRPLANE_MODE_CHANGED] = 
    g_signal_new("airplane-mode-changed",
		 G_TYPE_FROM_CLASS(gobject_class),
		 G_SIGNAL_RUN_FIRST,
		 0,
		 NULL,
		 NULL,
		 g_cclosure_marshal_VOID__BOOLEAN,
		 G_TYPE_NONE,
		 1,
		 G_TYPE_BOOLEAN);

  signals[SIG_MOBILE_DATA_ENABLED_CHANGED] =
    g_signal_new("mobile-data-enabled-changed",
		 G_TYPE_FROM_CLASS(gobject_class),
		 G_SIGNAL_RUN_FIRST,
		 0,
		 NULL,
		 NULL,
		 g_cclosure_marshal_VOID__BOOLEAN,
		 G_TYPE_NONE,
		 1,
		 G_TYPE_BOOLEAN);
  
  signals[SIG_WIFI_ENABLED_CHANGED] =
    g_signal_new("wifi-enabled-changed",
		 G_TYPE_FROM_CLASS(gobject_class),
		 G_SIGNAL_RUN_FIRST,
		 0,
		 NULL,
		 NULL,
		 g_cclosure_marshal_VOID__BOOLEAN,
		 G_TYPE_NONE,
		 1,
		 G_TYPE_BOOLEAN);
}

static void
android_service_init(AndroidService *self)
{
  AndroidServicePrivate *priv = GET_PRIVATE(self);

  g_debug("android_service_init: called");

  priv->mobile_state = ANDROID_NETWORK_STATE_UNKNOWN;
  priv->wifi_state   = ANDROID_NETWORK_STATE_UNKNOWN;

  priv->connected_name = g_strdup("");

  priv->proxy = NULL;

  priv->networks = NULL;
  priv->rssi = 50;

  priv->connection_information = g_hash_table_new_full(g_str_hash,
						       g_str_equal,
						       g_free,
						       g_free);

  priv->watch_id = g_bus_watch_name(G_BUS_TYPE_SESSION,
				    ANDROID_SERVICE_DBUS_SERVICE,
				    G_BUS_NAME_WATCHER_FLAGS_AUTO_START,
				    service_appeared,
				    service_vanished,
				    self,
				    NULL);
}

static void
android_service_dispose(GObject *object)
{
  AndroidService *self        = ANDROID_SERVICE(object);
  AndroidServicePrivate *priv = GET_PRIVATE(self);

  g_debug("%s", __func__);

  if (priv->watch_id != 0) {
    g_bus_unwatch_name(priv->watch_id);
    priv->watch_id = 0;
  }

  if (priv->proxy != NULL) {
    g_object_unref(priv->proxy);
    priv->proxy = NULL;
  }

  G_OBJECT_CLASS(android_service_parent_class)->dispose(object);
}

static void
android_service_finalize(GObject *object)
{
  AndroidService *self = ANDROID_SERVICE(object);
  AndroidServicePrivate *priv = GET_PRIVATE(self);

  g_free(priv->connected_name);
 
  g_slist_free_full(priv->networks,
		    (GDestroyNotify)g_object_unref);


  g_hash_table_destroy(priv->connection_information);

  G_OBJECT_CLASS(android_service_parent_class)->finalize(object);
}

AndroidService *
android_service_new()
{
  return g_object_new(ANDROID_SERVICE_TYPE, NULL);
}


AndroidNetworkState
android_service_get_state(AndroidService *self,
			  AndroidNetworkType type)
{ 
  AndroidServicePrivate *priv = GET_PRIVATE(self);

  switch(type) {
  case ANDROID_NETWORK_TYPE_MOBILE:
    return priv->mobile_state;
  case ANDROID_NETWORK_TYPE_WIFI:
    return priv->wifi_state;
  default:
    g_assert_not_reached();
    return ANDROID_NETWORK_STATE_UNKNOWN;
  }
}

const gchar *
android_service_get_connected_name(AndroidService *self)
{
  AndroidServicePrivate *priv = GET_PRIVATE(self);

  return priv->connected_name;
}


static void
net_start_scan_cb(GObject *source_object,
		  GAsyncResult *res,
		  gpointer user_data)
{
  AndroidService *self        = NULL;
  AndroidServicePrivate *priv = NULL;
  GError *error    = NULL;
  GVariant *result = NULL;

  self = ANDROID_SERVICE(user_data);
  g_assert(self != NULL);

  priv = GET_PRIVATE(self);

  result = g_dbus_proxy_call_finish(priv->proxy,
				    res,
				    &error);

  if (error != NULL) {
    g_critical("service call failed: %s", error->message);
    g_error_free(error);
  } else {
	g_variant_unref(result); // we don't care
  }
}

void
android_service_start_scan(AndroidService *self)
{
  AndroidServicePrivate *priv = GET_PRIVATE(self);
  
  if (priv->proxy == NULL) {
    // we don't have connection to the proxy, yet
    return;
  }

  if (!priv->wifi_enabled)
    return;

  g_dbus_proxy_call(priv->proxy,
		    "StartScan",
		    NULL,
		    G_DBUS_CALL_FLAGS_NONE,
		    -1,
		    NULL,
		    net_start_scan_cb,
		    self);
}

gint
android_service_get_rssi(AndroidService *self)
{
  AndroidServicePrivate *priv = GET_PRIVATE(self);
  return priv->rssi;
}

const GSList *
android_service_get_networks(AndroidService *self)
{
  AndroidServicePrivate *priv = GET_PRIVATE(self);
  
  return priv->networks;
}

static void
net_add_network_cb(GObject *source_object,
		   GAsyncResult *res,
		   gpointer user_data)
{
  GDBusProxy *proxy = NULL;
  GError *error    = NULL;
  GVariant *result = NULL;
  AndroidServiceNetwork *network = NULL;
  gboolean success;
  
  proxy   = G_DBUS_PROXY(source_object);
  network = ANDROID_SERVICE_NETWORK(user_data);

  result = g_dbus_proxy_call_finish(proxy,
				    res,
				    &error);

  if (error != NULL) {
    g_critical("service call failed: %s", error->message);
    g_error_free(error);
  } else {
    g_variant_get(result, "(b)", &success);
    if (success) {
      g_debug("added network: %s", network->identifier);
    } else {
      g_warning("failed to add network %s", network->identifier);
    }      
  }

  if (result != NULL)
    g_variant_unref(result);

  g_object_unref(network);
}

#define AUTH_ALGO_OPEN   0x01
#define AUTH_ALGO_SHARED 0x02
#define AUTH_ALGO_LEAP   0x04

#define GROUP_CIPHER_WEP40  0x01
#define GROUP_CIPHER_WEP104 0x02
#define GROUP_CIPHER_TKIP   0x04
#define GROUP_CIPHER_CCMP   0x08

#define KEY_MGMT_NONE      0x01
#define KEY_MGMT_WPA_PSK   0x02
#define KEY_MGMT_WPA_EAP   0x04
#define KEY_MGMT_IEEE801XP 0x08

#define PAIRWISE_CIPHER_NONE 0x01
#define PAIRWISE_CIPHER_TKIP 0x02
#define PAIRWISE_CIPHER_CCMP 0x04

#define PROTOCOL_WPA 0x01
#define PROTOCOL_RSN 0x02

static GVariant *
create_wifi_configuration(const gchar *ssid,
			  AndroidNetworkSecurity security,
			  const gchar *passphrase)
{
  GVariant *conf = NULL;

  guint32 authAlgorithms  = 0;
  guint32 groupCiphers    = 0;
  guint32 keyMgmt         = 0;
  guint32 pairwiseCiphers = 0;
  guint32 protocols       = 0;

  /*
   * Force authAlgorithm to OPEN for now as we
   * currently don't support WPA-Enterprise, nor WEP Shared
   * key authentication.  Both WPA-PSK and WEP Open require
   * OPEN authentication.
   */
  authAlgorithms |= AUTH_ALGO_OPEN;

  switch (security) {
  case ANDROID_NETWORK_SECURITY_NONE:
    g_debug("NONE builder");

    keyMgmt |= KEY_MGMT_NONE;

    break;
  case ANDROID_NETWORK_SECURITY_WEP:
    g_debug("WEP builder");

    keyMgmt |= KEY_MGMT_NONE;

    groupCiphers |= GROUP_CIPHER_WEP40;
    groupCiphers |= GROUP_CIPHER_WEP104;

    break;
  case ANDROID_NETWORK_SECURITY_WPA:
    g_debug("WPA builder");

    keyMgmt |= KEY_MGMT_WPA_PSK;
    
    groupCiphers |= GROUP_CIPHER_TKIP;
    groupCiphers |= GROUP_CIPHER_CCMP;

    pairwiseCiphers |= PAIRWISE_CIPHER_TKIP;
    pairwiseCiphers |= PAIRWISE_CIPHER_CCMP;

    protocols |= PROTOCOL_WPA;
    protocols |= PROTOCOL_RSN;

    break;
  default:
    g_assert_not_reached();
  }

  conf = g_variant_new("((ssuuuuubisii)b)",
		       "",              // BSSID
		       ssid,
		       authAlgorithms,
		       groupCiphers,
		       keyMgmt,
		       pairwiseCiphers,
		       protocols,
		       FALSE,           // hiddenSSID
		       -1,              // networkId
		       (passphrase == NULL) ? "" : passphrase,
		       -1,              // priority
		       -1,              // status
		       TRUE);           // Auto-Enable
		
  return conf; // floating
}

/**
 * @returns  0 if request was sent successfully
 * @returns -1 if passpharase was missing, retry after setting passphrase
 * @returns -2 on other errors
 */
/// @todo this is private for now.
gint
android_service_add_network(AndroidService *self,
			    AndroidServiceNetwork *network)
{
  AndroidServicePrivate *priv = NULL;

  GVariant *configuration = NULL;

  g_debug("android_service_add_network()");

  priv = GET_PRIVATE(self);

  if (priv->proxy == NULL) 
    return -2; // can't add network if we don't have a proxy

  if (!priv->wifi_enabled)
    return -2;
  
  if (network->security == ANDROID_NETWORK_SECURITY_WEP ||
      network->security == ANDROID_NETWORK_SECURITY_WPA) {
    if (network->passphrase == NULL) {
      return -1;
    }
  }

  // floating.
  configuration = create_wifi_configuration(network->ssid,
					    network->security,
					    network->passphrase);

  g_debug("*********************************************");

  g_object_ref(network); // make sure network does not go away during the call
  g_dbus_proxy_call(priv->proxy,
		    "AddNetwork",
		    configuration,
		    G_DBUS_CALL_FLAGS_NONE,
		    -1,
		    NULL,
		    (GAsyncReadyCallback)net_add_network_cb,
		    network);
  return 0;
}

struct _EnableNetworkHelper
{
  AndroidService *self;
  AndroidServiceNetwork *network;
};

static void
net_enable_network_cb(GObject *source_object,
		      GAsyncResult *res,
		      gpointer user_data)
{

  GDBusProxy *proxy = NULL;
  struct _EnableNetworkHelper *helper = NULL;
  GError *error    = NULL;
  GVariant *result = NULL;
  gboolean success;
  gint32 ret;

  helper = user_data;
  proxy  = G_DBUS_PROXY(source_object);

  result = g_dbus_proxy_call_finish(proxy,
				    res,
				    &error);

  if (error != NULL) {
    g_critical("%s: failed: %s", __func__, error->message);
    g_error_free(error);
    android_service_network_set_state(helper->network,
				      ANDROID_NETWORK_STATE_DISCONNECTED);	
	
  } else {
    g_variant_get(result, "(b)", &success);
    if (success) {
      g_debug("%s: success: %s", __func__, helper->network->identifier);
    } else {
      ret = android_service_add_network(helper->self, helper->network);
      
      switch (ret) {
      case 0:
	break; // everything went well
      case -1:
	// passhprase missing
	g_signal_emit_by_name(G_OBJECT(helper->network),
			      "passphrase-needed",
			      0);
	// FALLTHROUGH
      case -2:
      default:
	android_service_network_set_state(helper->network,
					  ANDROID_NETWORK_STATE_DISCONNECTED);	
      }
    }
  }  

  if (result != NULL)
    g_variant_unref(result);

  g_free(helper);
}

void
android_service_enable_network(AndroidService *self,
			       AndroidServiceNetwork *network)
{
  AndroidServicePrivate *priv = GET_PRIVATE(self);
  struct _EnableNetworkHelper *helper = NULL;

  if (priv->proxy == NULL)
    return;

  if (!priv->wifi_enabled)
    return;

  g_debug("%s", __func__);

  /// @todo cellular

  android_service_network_set_state(network,
				    ANDROID_NETWORK_STATE_CONNECTING);
  
  helper = g_malloc0(sizeof(struct _EnableNetworkHelper));
  helper->self = self;
  helper->network = network;

  // make sure network does not go away while doing the call
  g_object_ref(helper->network);
  g_dbus_proxy_call(priv->proxy,
		    "EnableNetwork",
		    g_variant_new("(ss)",
				  network->ssid,
				  network->capabilities),
		    G_DBUS_CALL_FLAGS_NONE,
		    -1,
		    NULL,
		    (GAsyncReadyCallback)net_enable_network_cb,
		    helper);
  return;
}

void         
android_service_enable_network_with_identifier(AndroidService *self,
					       const gchar *identifier)
{
  AndroidServicePrivate *priv = GET_PRIVATE(self);
  AndroidServiceNetwork *network;
  GSList *iter;

  for (iter = priv->networks; iter != NULL; iter = iter->next) {
    network = iter->data;
    if (g_strcmp0(identifier, network->identifier) == 0) {
      android_service_enable_network(self, network);
      return;
    }
  }
}

void
android_service_disconnect(AndroidService *self, AndroidServiceNetwork *network)
{
  AndroidServicePrivate *priv = GET_PRIVATE(self);

  if (!priv->wifi_enabled)
    return;

  g_debug("android_service_disconnect(): %s", network->identifier);

  android_service_network_set_state(network,
				    ANDROID_NETWORK_STATE_DISCONNECTED);

  g_dbus_proxy_call(priv->proxy,
		    "Disconnect",
		    NULL,
		    G_DBUS_CALL_FLAGS_NONE,
		    -1,
		    NULL,
		    NULL,
		    self);
}

void
android_service_disconnect_with_identifier(AndroidService *self,
					   const gchar *identifier)
{
  AndroidServicePrivate *priv = GET_PRIVATE(self);
  AndroidServiceNetwork *network;
  GSList *iter;

  for (iter = priv->networks; iter != NULL; iter = iter->next) {
    network = iter->data;
    if (g_strcmp0(identifier, network->identifier) == 0) {
      android_service_disconnect(self, network);
      return;
    }
  }
}

static void
dbus_properties_set(AndroidService *self,
		    const gchar *property,
		    GVariant *value)
{
  AndroidServicePrivate *priv = GET_PRIVATE(self);

  gchar *name_owner = g_dbus_proxy_get_name_owner(priv->proxy);

  g_dbus_connection_call(g_dbus_proxy_get_connection(priv->proxy),
			 name_owner,
			 g_dbus_proxy_get_object_path(priv->proxy),
			 "org.freedesktop.DBus.Properties",
			 "Set",
			 g_variant_new("(ssv)",
				       ANDROID_SERVICE_DBUS_INTERFACE,
				       property,
				       value),
			 NULL,
			 G_DBUS_CALL_FLAGS_NONE,
			 -1,
			 NULL,
			 NULL, // don't care about the result
			 self);
  g_free(name_owner);
}

gboolean
android_service_get_airplane_mode(AndroidService *self)
{
  AndroidServicePrivate *priv = GET_PRIVATE(self);

  return priv->airplane_mode;
}

static void
set_airplane_mode(AndroidService *self, gboolean value)
{
  AndroidServicePrivate *priv = GET_PRIVATE(self);
  
  if (priv->airplane_mode == value)
    return;
  
  priv->airplane_mode = value;
  g_signal_emit(self,
		signals[SIG_AIRPLANE_MODE_CHANGED],
		0,
		priv->airplane_mode);
}

void
android_service_set_airplane_mode(AndroidService *self, gboolean value)
{
  AndroidServicePrivate *priv = GET_PRIVATE(self);

  if (!priv->proxy)
    return;
  
  if (priv->airplane_mode == value)
    return;

  dbus_properties_set(self, "AirplaneMode", g_variant_new("b", value));
}

gboolean
android_service_get_mobile_data_enabled(AndroidService *self)
{
  AndroidServicePrivate *priv = GET_PRIVATE(self);

  return priv->mobile_data_enabled;
}

static void set_mobile_data_enabled(AndroidService *self, gboolean value)
{
  AndroidServicePrivate *priv = GET_PRIVATE(self);

  if (priv->mobile_data_enabled == value)
    return;

  priv->mobile_data_enabled = value;
  g_signal_emit(self,
		signals[SIG_MOBILE_DATA_ENABLED_CHANGED],
		0,
		priv->mobile_data_enabled);
}

void
android_service_set_mobile_data_enabled(AndroidService *self, gboolean value)
{
  AndroidServicePrivate *priv = GET_PRIVATE(self);

  if (!priv->proxy)
    return;
  
  if (priv->mobile_data_enabled == value)
    return;

  dbus_properties_set(self, "MobileDataEnabled", g_variant_new("b", value));
}

gboolean
android_service_get_wifi_enabled(AndroidService *self)
{
  AndroidServicePrivate *priv = GET_PRIVATE(self);

  return priv->wifi_enabled;
}

static void
set_wifi_enabled(AndroidService *self, gboolean value)
{
  AndroidServicePrivate *priv = GET_PRIVATE(self);

  if (priv->wifi_enabled == value)
    return;

  priv->wifi_enabled = value;
  g_signal_emit(self,
		signals[SIG_WIFI_ENABLED_CHANGED],
		0,
		priv->wifi_enabled);

  if (!priv->wifi_enabled) {
    g_slist_free_full(priv->networks,
		      (GDestroyNotify)g_object_unref);
    priv->networks = NULL;
    g_signal_emit(G_OBJECT(self), signals[SIG_SCAN_RESULTS_READY], 0);
  }
}

void
android_service_set_wifi_enabled(AndroidService *self, gboolean value)
{
  AndroidServicePrivate *priv = GET_PRIVATE(self);

  if (!priv->proxy)
    return;
  
  if (priv->wifi_enabled == value)
    return;

  dbus_properties_set(self, "WifiEnabled", g_variant_new("b", value));
}

GHashTable *
android_service_get_connection_information(AndroidService *self)
{
  AndroidServicePrivate *priv = GET_PRIVATE(self);

  return priv->connection_information;
}

static gchar *
int_to_ipaddress(gint32 value)
{
  gint8 buf[4];
  int i;

  for (i = 0; i < 4; i++) {
    buf[i] = value & 0xff;
    value >>= 8;
  }

  return g_strdup_printf("%hhu.%hhu.%hhu.%hhu", buf[0], buf[1], buf[2], buf[3]);
}

static void
get_connection_information_cb(GObject *source_object,
			      GAsyncResult *res,
			      gpointer user_data)
{
  AndroidService        *self = user_data;
  AndroidServicePrivate *priv = GET_PRIVATE(self);
  
  GError   *error  = NULL;
  GVariant *result = NULL;

  GVariantIter *iter  = NULL;
  gchar        *key   = NULL;
  GVariant     *value = NULL;

  result = g_dbus_proxy_call_finish(priv->proxy,
				    res,
				    &error);

  if (error != NULL) {
    g_critical("%s: service call failed: %s", __func__, error->message);
    g_error_free(error);
  } else {

    g_hash_table_remove_all(priv->connection_information);

    g_variant_get(result, "(a{sv})", &iter);
    while (g_variant_iter_loop(iter,
			       "{sv}",
			       &key,
			       &value)) {

      if (g_strcmp0(key, "network-name") == 0) {

	g_hash_table_insert(priv->connection_information,
			    g_strdup(key),
			    g_variant_dup_string(value, NULL));

      } else if (g_strcmp0(key, "network-type") == 0) {

	g_hash_table_insert(priv->connection_information,
			    g_strdup(key),
			    g_variant_dup_string(value, NULL));

      } else if (g_strcmp0(key, "network-security") == 0) {

	g_hash_table_insert(priv->connection_information,
			    g_strdup(key),
			    g_variant_dup_string(value, NULL));

      } else if (g_strcmp0(key, "ipv4-address") == 0) {

	g_hash_table_insert(priv->connection_information,
			    g_strdup(key),
			    int_to_ipaddress(g_variant_get_int32(value)));

      } else if (g_strcmp0(key, "ipv4-broadcast") == 0) {

	/// @todo missing from the reply

      } else if (g_strcmp0(key, "ipv4-gateway") == 0) {

	g_hash_table_insert(priv->connection_information,
			    g_strdup(key),
			    int_to_ipaddress(g_variant_get_int32(value)));

      } else if (g_strcmp0(key, "ipv4-dns1") == 0) {

	g_hash_table_insert(priv->connection_information,
			    g_strdup(key),
			    int_to_ipaddress(g_variant_get_int32(value)));

      } else if (g_strcmp0(key, "ipv4-dns2") == 0) {

	g_hash_table_insert(priv->connection_information,
			    g_strdup(key),
			    int_to_ipaddress(g_variant_get_int32(value)));

      } else if (g_strcmp0(key, "ipv4-lease") == 0) {

	g_hash_table_insert(priv->connection_information,
			    g_strdup(key),
			    g_strdup_printf("%d", g_variant_get_int32(value)));

      } else if (g_strcmp0(key, "ipv4-netmask") == 0) {

	g_hash_table_insert(priv->connection_information,
			    g_strdup(key),
			    int_to_ipaddress(g_variant_get_int32(value)));
      } else {
	g_warning("%s: Unknown field: %s", __func__, key);
      }

    }

    g_variant_unref(result);
  }

}

static void
get_connection_information(AndroidService *self)
{
  AndroidServicePrivate *priv = GET_PRIVATE(self);

  g_dbus_proxy_call(priv->proxy,
		    "GetConnectionInformation",
		    NULL,
		    G_DBUS_CALL_FLAGS_NONE,
		    -1,
		    NULL,
		    (GAsyncReadyCallback)get_connection_information_cb,
		    self);  
}
