#
# Copyright 2009 Canonical Ltd.
#
# Written by:
#     Gustavo Niemeyer <gustavo.niemeyer@canonical.com>
#     Sidnei da Silva <sidnei.da.silva@canonical.com>
#
# This file is part of the Image Store Proxy.
#
# This program is free software: you can redistribute it and/or modify it 
# under the terms of the GNU General Public License version 3, as published 
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but 
# WITHOUT ANY WARRANTY; without even the implied warranties of 
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR 
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along 
# with this program.  If not, see <http://www.gnu.org/licenses/>.
#
import logging
import base64
import os

from urllib import quote

from twisted.internet.defer import Deferred

from imagestore.lib.service import (
    ServiceTask, ServiceError, Service, taskHandler)
from imagestore.lib.twistedutil import mergeDeferreds

from imagestore.model import ImageState, ImageStoreResponse, ImageRegistration
from imagestore.signatureservice import CheckImageSignatureTask
from imagestore.downloadservice import DownloadFileTask
from imagestore.eucaservice import BundleAndUploadImageTask
from imagestore.storageservice import (
    GetStoredImagesTask, GetStoredImageStatesTask, SetImageRegistrationTask,
    SetErrorMessageTask)


class InstallServiceError(ServiceError):
    pass


class InstallServiceTask(ServiceTask):
    pass

class GetStatesTask(InstallServiceTask):

    def __init__(self, uris):
        self.uris = uris

class InstallImageTask(InstallServiceTask):

    def __init__(self, image):
        self.image = image

class CancelChangesTask(InstallServiceTask):

    def __init__(self, imageUri):
        self.imageUri = imageUri


class InstallService(Service):

    def __init__(self, serviceHub):
        self._serviceHub = serviceHub
        self._downloads = {}
        self._downloading = set()
        self._installing = set()
        Service.__init__(self)

    @taskHandler(InstallImageTask)
    def _installImage(self, task):
        image = task.image
        self._downloading.add(image["uri"])
        deferred = Deferred()
        deferred.addCallback(self._checkSignature)
        deferred.addCallback(self._downloadFiles)
        deferred.addCallback(self._bundleImage)
        deferred.addCallback(self._installFinishedCallback, image["uri"])
        deferred.addErrback(self._installFailedErrback, image["uri"])
        deferred.callback(image)
        return deferred

    def _checkSignature(self, image):
        return self._serviceHub.addTask(CheckImageSignatureTask(image))

    def _downloadFiles(self, image):
        deferreds = []

        imageUri = image["uri"]
        imageRegistration = ImageRegistration(image)

        imageFiles = {}
        for imageFile in image["files"]:
            imageFiles[imageFile["kind"]] = imageFile.copy()

        downloads = self._downloads[imageUri] = []
        for kind in ("kernel", "ramdisk", "image"):
            imageFile = imageFiles[kind]

            if kind == "kernel":
                imagePart = imageRegistration.eki
            elif kind == "ramdisk":
                imagePart = imageRegistration.eri
            elif kind == "image":
                imagePart = imageRegistration.emi

            imagePart.size = imageFile["size-in-bytes"]
            imagePart.sha256 = imageFile["sha256"]
            imagePart.kind = kind

            downloadTask = DownloadFileTask(imageFile["url"],
                                            imageFile["size-in-bytes"],
                                            imageFile["sha256"])

            def setLocalPath(localPath, imagePart=imagePart):
                imagePart.path = localPath

            downloads.append(downloadTask)
            deferred = self._serviceHub.addTask(downloadTask)
            deferred.addCallback(setLocalPath)
            deferreds.append(deferred)

        resultDeferred = mergeDeferreds(deferreds)
        resultDeferred.addCallback(lambda result: imageRegistration)
        return resultDeferred

    def _bundleImage(self, imageRegistration):
        # Mark the image as installing and clear the internal state
        # that's keeping track of downloads.
        self._installing.add(imageRegistration.image["uri"])
        self._downloading.remove(imageRegistration.image["uri"])
        del self._downloads[imageRegistration.image["uri"]]

        def storeRegistration(result):
             deferred = self._serviceHub.addTask(
                 SetImageRegistrationTask(imageRegistration))
             deferred.addCallback(lambda result: imageRegistration)
             return deferred
        deferred = self._serviceHub.addTask(
            BundleAndUploadImageTask(imageRegistration))
        deferred.addCallback(storeRegistration)
        return deferred

    def _installFinishedCallback(self, imageRegistration, imageUri):
        self._installing.remove(imageUri)
        for path in (imageRegistration.eki.path,
                     imageRegistration.eri.path,
                     imageRegistration.emi.path):
            # Theoretically they necessarily must exist by now, but
            # checking makes testing easier.
            if path and os.path.exists(path):
                logging.debug("Removing downloaded file: %s" % (path,))
                os.unlink(path)
        return imageRegistration

    def _installFailedErrback(self, failure, imageUri):
        logging.error("Install failed:\n%s" % (failure.getBriefTraceback(),))
        wasCancelled = False
        for downloadTask in self._downloads.get(imageUri, ()):
            wasCancelled |= downloadTask.progress.wasCancelled()
            downloadTask.progress.cancel()
        self._downloads.pop(imageUri, None)
        self._downloading.discard(imageUri)
        self._installing.discard(imageUri)
        if wasCancelled:
            return None
        else:
            task = SetErrorMessageTask(imageUri, failure.getErrorMessage())
            deferred = self._serviceHub.addTask(task)
            deferred.addCallback(lambda result: failure)
            return deferred

    @taskHandler(GetStatesTask)
    def _getStates(self, task):
        deferred = self._serviceHub.addTask(GetStoredImageStatesTask(task.uris))
        deferred.addCallback(self._addInProgressStates)
        return deferred

    def _addInProgressStates(self, states):
        for state in states:
            state["actions"] = {}
            if state["image-uri"] in self._downloading:
                state["status"] = "downloading"
                self._addAction(state, "cancel")
                self._addProgress(state)
            elif state["image-uri"] in self._installing:
                state["status"] = "installing"
            elif state["status"] == "uninstalled":
                self._addAction(state, "install")

            if "error-message" in state:
                self._addAction(state, "clear-error")

        return states

    def _addAction(self, state, action):
        state["actions"][action] = (
            "http://localhost:52780/api/images/%s/%s" %
            (base64.urlsafe_b64encode(state["image-uri"]), action))

    def _addProgress(self, state):
        if state["image-uri"] in self._downloads:
            currentSize = 0
            totalSize = 0
            for downloadTask in self._downloads[state["image-uri"]]:
                currentSize += downloadTask.progress.getCurrentSize()
                totalSize += downloadTask.progress.getTotalSize()
            if totalSize > 0:
                state["progress-percentage"] = (currentSize * 100.0)/totalSize

    @taskHandler(CancelChangesTask)
    def _cancelChanges(self, task):
        for downloadTask in self._downloads[task.imageUri]:
            downloadTask.progress.cancel()
