#include "hrn.h"
#include "hrn-pin-manager.h"

enum {
    PROP_0,
};

enum {
    PIN_ADDED,
    PIN_REMOVED,
    PIN_CHANGED,
    PIN_SELECTED,
    LAST_SIGNAL
};

struct _HrnPinManagerPrivate {
    GHashTable *pins;
    int pin_count;

    gboolean dirty;
    char *current_pin;
};

#define GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj), HRN_TYPE_PIN_MANAGER, HrnPinManagerPrivate))
G_DEFINE_TYPE (HrnPinManager, hrn_pin_manager, G_TYPE_OBJECT);
static guint32 signals[LAST_SIGNAL] = {0, };

static void
hrn_pin_manager_finalize (GObject *object)
{
    HrnPinManager *self = (HrnPinManager *) object;
    HrnPinManagerPrivate *priv = self->priv;

    g_hash_table_destroy (priv->pins);

    G_OBJECT_CLASS (hrn_pin_manager_parent_class)->finalize (object);
}

static void
hrn_pin_manager_dispose (GObject *object)
{
    G_OBJECT_CLASS (hrn_pin_manager_parent_class)->dispose (object);
}

static void
hrn_pin_manager_set_property (GObject      *object,
                          guint         prop_id,
                          const GValue *value,
                          GParamSpec   *pspec)
{
    switch (prop_id) {

    default:
        break;
    }
}

static void
hrn_pin_manager_get_property (GObject    *object,
                          guint       prop_id,
                          GValue     *value,
                          GParamSpec *pspec)
{
    switch (prop_id) {

    default:
        break;
    }
}

static void
hrn_pin_manager_class_init (HrnPinManagerClass *klass)
{
    GObjectClass *o_class = (GObjectClass *)klass;

    o_class->dispose = hrn_pin_manager_dispose;
    o_class->finalize = hrn_pin_manager_finalize;
    o_class->set_property = hrn_pin_manager_set_property;
    o_class->get_property = hrn_pin_manager_get_property;

    g_type_class_add_private (klass, sizeof (HrnPinManagerPrivate));

    signals[PIN_ADDED] = g_signal_new ("pin-added",
                                       G_TYPE_FROM_CLASS (klass),
                                       G_SIGNAL_NO_RECURSE |
                                       G_SIGNAL_RUN_FIRST, 0, NULL, NULL,
                                       g_cclosure_marshal_VOID__POINTER,
                                       G_TYPE_NONE, 1,
                                       G_TYPE_POINTER);
    signals[PIN_REMOVED] = g_signal_new ("pin-removed",
                                         G_TYPE_FROM_CLASS (klass),
                                         G_SIGNAL_NO_RECURSE |
                                         G_SIGNAL_RUN_FIRST, 0, NULL, NULL,
                                         g_cclosure_marshal_VOID__POINTER,
                                         G_TYPE_NONE, 1,
                                         G_TYPE_POINTER);
    signals[PIN_SELECTED] = g_signal_new ("pin-selected",
                                          G_TYPE_FROM_CLASS (klass),
                                          G_SIGNAL_NO_RECURSE |
                                          G_SIGNAL_RUN_FIRST, 0, NULL, NULL,
                                          g_cclosure_marshal_VOID__POINTER,
                                          G_TYPE_NONE, 1,
                                          G_TYPE_POINTER);
}

static void
free_pin (gpointer value)
{
    HrnPin *pin = (HrnPin *) value;

    g_free (pin->group);
    g_free (pin->name);
    g_free (pin->query);
    g_free (pin->source);

    g_slice_free (HrnPin, pin);
}

static void
hrn_pin_manager_init (HrnPinManager *self)
{
    HrnPinManagerPrivate *priv;

    self->priv = GET_PRIVATE (self);
    priv = self->priv;

    /* FIXME: Need to add a destroy function for the pins */
    priv->pins = g_hash_table_new_full (g_str_hash, g_str_equal,
                                        NULL, /* The key is freed by the pin */
                                        free_pin);
}

static HrnZoomLevel
float_to_zoom_level (float zl)
{
    HrnZoomLevel zoom_level;

    if (ABS (zl - 0.5) < 0.01) {
        zoom_level = HRN_ZOOM_LEVEL_OVERVIEW;
    } else if (ABS (zl - 1.0) < 0.01) {
        zoom_level = HRN_ZOOM_LEVEL_COLLECTIONS;
    } else if (ABS (zl - 2.0) < 0.01) {
        zoom_level = HRN_ZOOM_LEVEL_SMALL_ITEMS;
    } else if (ABS (zl - 4.0) < 0.01) {
        zoom_level = HRN_ZOOM_LEVEL_LARGE_ITEMS;
    } else {
        zoom_level = HRN_ZOOM_LEVEL_THEATRE;
    }

    return zoom_level;
}

static void
load_from_file (HrnPinManager *manager,
                const char    *filename)
{
    HrnPinManagerPrivate *priv = manager->priv;
    GKeyFile *keyfile;
    GError *error = NULL;
    char **groups;
    int i;

    keyfile = g_key_file_new ();
    g_key_file_load_from_file (keyfile, filename,
                               G_KEY_FILE_KEEP_COMMENTS, &error);
    if (error != NULL) {
        g_warning ("Error loading keyfile: %s", error->message);
        g_error_free (error);
        return;
    }

    groups = g_key_file_get_groups (keyfile, NULL);
    for (i = 0; groups[i]; i++) {
        HrnPin *pin;
        GError *error = NULL;

        /* Hornsey used to mix last-state and queries in the same file:
           this is bad, so don't do it. */
        if (g_str_equal (groups[i], "hrn-last-state")) {
            continue;
        }

        pin = g_slice_new0 (HrnPin);

        pin->group = g_strdup (groups[i]);
        pin->name = g_key_file_get_string (keyfile, groups[i],
                                           "name", &error);
        if (error != NULL) {
            g_warning ("Error getting name %s: %s", groups[i],
                       error->message);
            g_error_free (error);
            error = NULL;
        }

        pin->query = g_key_file_get_string (keyfile, groups[i],
                                            "query", &error);
        if (error != NULL) {
            g_warning ("Error getting query %s: %s", groups[i],
                       error->message);
            g_error_free (error);
            error = NULL;
        }

        pin->source = g_key_file_get_string (keyfile, groups[i],
                                             "source", &error);
        if (error != NULL) {
            g_warning ("Error getting source %s: %s", groups[i],
                       error->message);
            g_error_free (error);
            error = NULL;
        }

        pin->filter = g_key_file_get_integer (keyfile, groups[i],
                                              "filter", &error);
        if (error != NULL) {
            g_warning ("Error getting filter %s: %s", groups[i],
                       error->message);
            g_error_free (error);
            error = NULL;
        }

        /* Look for the new zoom level first */
        pin->zoom_level = g_key_file_get_integer (keyfile, groups[i],
                                                  "zoomlevel", &error);
        if (error != NULL) {
            double zoom;

            g_error_free (error);
            error = NULL;

            /* If we don't have zoomlevel, look for zoom */
            zoom = g_key_file_get_double (keyfile, groups[i],
                                          "zoom", &error);
            if (error != NULL) {
                g_warning ("Error getting zoom %s: %s", groups[i],
                           error->message);
                g_error_free (error);
                error = NULL;

                pin->zoom_level = 0;
            } else {
                pin->zoom_level = float_to_zoom_level (zoom);
            }
        }

        priv->pin_count++;
        g_hash_table_insert (priv->pins, pin->group, pin);
    }

    g_strfreev (groups);
    g_key_file_free (keyfile);
}

HrnPinManager *
hrn_pin_manager_new (const char *filename)
{
    HrnPinManager *manager;

    manager = g_object_new (HRN_TYPE_PIN_MANAGER, NULL);
    load_from_file (manager, filename);

    return manager;
}

GList *
hrn_pin_manager_get_pins (HrnPinManager *manager)
{
    HrnPinManagerPrivate *priv = manager->priv;

    return g_hash_table_get_values (priv->pins);
}

HrnPin *
hrn_pin_manager_get_pin (HrnPinManager *manager,
                         const char    *name)
{
    HrnPinManagerPrivate *priv = manager->priv;

    return g_hash_table_lookup (priv->pins, name);
}

void
hrn_pin_manager_write_to_file (HrnPinManager *manager,
                               const char    *filename)
{
    HrnPinManagerPrivate *priv = manager->priv;
    GKeyFile *keyfile;
    GError *error = NULL;
    GList *pins, *p;
    char *data;

    if (priv->dirty == FALSE) {
        /* No need to write it out */
        return;
    }

    keyfile = g_key_file_new ();
    pins = g_hash_table_get_values (priv->pins);
    for (p = pins; p; p = p->next) {
        HrnPin *pin = p->data;

        g_key_file_set_string (keyfile, pin->group, "name", pin->name);
        g_key_file_set_string (keyfile, pin->group, "query", pin->query);
        g_key_file_set_string (keyfile, pin->group, "source", pin->source);
        g_key_file_set_integer (keyfile, pin->group, "filter", pin->filter);
        g_key_file_set_integer (keyfile, pin->group,
                                "zoomlevel", pin->zoom_level);
    }

    g_list_free (pins);

    data = g_key_file_to_data (keyfile, NULL, NULL);
    g_file_set_contents (filename, data, -1, &error);

    if (error != NULL) {
        g_warning ("Error writing %s: %s", filename, error->message);
        g_error_free (error);
    }

    g_free (data);
    g_key_file_free (keyfile);
}

void
hrn_pin_manager_add (HrnPinManager *manager,
                     const char    *name,
                     const char    *query,
                     const char    *source,
                     int            filter,
                     HrnZoomLevel   zoom_level)
{
    HrnPinManagerPrivate *priv = manager->priv;
    HrnPin *pin;

    pin = g_slice_new0 (HrnPin);

    pin->group = g_strdup_printf ("query%i", priv->pin_count);
    pin->name = g_strdup (name);
    pin->query = g_strdup (query);
    pin->source = g_strdup (source);
    pin->filter = filter;
    pin->zoom_level = zoom_level;

    g_hash_table_insert (priv->pins, pin->group, pin);
    priv->pin_count++;

    priv->dirty = TRUE;

    /* Select this pin */
    hrn_pin_manager_select_pin (manager, pin);

    g_signal_emit (manager, signals[PIN_ADDED], 0, pin);
}

void
hrn_pin_manager_remove (HrnPinManager *manager,
                        const char    *pin_group)
{
    HrnPinManagerPrivate *priv = manager->priv;
    HrnPin *pin;

    pin = g_hash_table_lookup (priv->pins, pin_group);
    if (pin == NULL) {
        return;
    }

    g_signal_emit (manager, signals[PIN_REMOVED], 0, pin);

    /* Pin is now destroyed */
    if (g_hash_table_remove (priv->pins, pin_group)) {
        priv->pin_count--;
    }

    /* This needs to happen last, as pin_group may be freed in a callback
       from HrnPinManager::pin-selected */
    if (priv->current_pin &&
        g_str_equal (priv->current_pin, pin_group)) {
        /* This is our current pin, so unselect it */
        hrn_pin_manager_select_pin (manager, NULL);
    }

    priv->dirty = TRUE;
}

void
hrn_pin_manager_rename_pin (HrnPinManager *manager,
                            HrnPin        *pin,
                            const char    *name)
{
    HrnPinManagerPrivate *priv = manager->priv;

    g_free (pin->name);
    pin->name = g_strdup (name);

    priv->dirty = TRUE;
    /* emit changed signal */
}

void
hrn_pin_manager_select_pin (HrnPinManager *manager,
                            HrnPin        *pin)

{
    HrnPinManagerPrivate *priv = manager->priv;

    g_free (priv->current_pin);
    if (pin) {
        priv->current_pin = g_strdup (pin->group);
    } else {
        priv->current_pin = NULL;
    }

    g_signal_emit (manager, signals[PIN_SELECTED], 0, pin);
}

const char *
hrn_pin_manager_get_selected_pin (HrnPinManager *manager)
{
    HrnPinManagerPrivate *priv = manager->priv;

    return priv->current_pin;
}
