/*
 * Hornsey - Moblin Media Player.
 * Copyright © 2009 Intel Corporation.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 2.1, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <clutter/clutter.h>
#include <math.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include "hrn.h"
#include "hrn-sidebar.h"
#include "hrn-sidebar-subitem.h"

#define STUCKBAR    1

G_DEFINE_TYPE (HrnSidebar, hrn_sidebar, NBTK_TYPE_BIN);

#define HRN_SIDEBAR_GET_PRIVATE(obj) \
  (G_TYPE_INSTANCE_GET_PRIVATE ((obj), HRN_TYPE_SIDEBAR, HrnSidebarPrivate))

struct _HrnSidebarPrivate
{
  NbtkWidget *scroll;
  NbtkWidget *flow;
  NbtkWidget *expanded;
};

typedef struct SidebarItem
{
  HrnSidebar   *sidebar;
  BklItem      *item;
  ClutterActor *actor;
}
SidebarItem;

static GObject *
hrn_sidebar_constructor (GType type, guint n_params,
                         GObjectConstructParam *params);
static void
hrn_sidebar_dispose (GObject *object);
static void
hrn_sidebar_allocate (ClutterActor *self, const ClutterActorBox *box,
                      ClutterAllocationFlags flags);
static void
hrn_sidebar_paint (ClutterActor *actor);
static void
hrn_sidebar_pick (ClutterActor *actor, const ClutterColor   *color);

static void
hrn_sidebar_map (ClutterActor *self)
{
  HrnSidebarPrivate *priv = HRN_SIDEBAR (self)->priv;

  CLUTTER_ACTOR_CLASS (hrn_sidebar_parent_class)->map (self);

  if (priv->scroll)
    clutter_actor_map (CLUTTER_ACTOR (priv->scroll));
  if (priv->expanded)
    clutter_actor_map (CLUTTER_ACTOR (priv->expanded));
}

static void
hrn_sidebar_unmap (ClutterActor *self)
{
  HrnSidebarPrivate *priv = HRN_SIDEBAR (self)->priv;

  CLUTTER_ACTOR_CLASS (hrn_sidebar_parent_class)->unmap (self);

  if (priv->scroll)
    clutter_actor_unmap (CLUTTER_ACTOR (priv->scroll));
  if (priv->expanded)
    clutter_actor_unmap (CLUTTER_ACTOR (priv->expanded));
}


static void
hrn_sidebar_class_init (HrnSidebarClass *klass)
{
  GObjectClass      *gobject_class = G_OBJECT_CLASS (klass);
  ClutterActorClass *actor_class   = CLUTTER_ACTOR_CLASS (klass);


  gobject_class->dispose     = hrn_sidebar_dispose;
  gobject_class->constructor = hrn_sidebar_constructor;
  actor_class->allocate      = hrn_sidebar_allocate;
  actor_class->paint         = hrn_sidebar_paint;
  actor_class->pick          = hrn_sidebar_pick;
  actor_class->map           = hrn_sidebar_map;
  actor_class->unmap         = hrn_sidebar_unmap;

  g_type_class_add_private (gobject_class, sizeof (HrnSidebarPrivate));
}

static void
hrn_sidebar_init (HrnSidebar *self)
{
  self->priv = HRN_SIDEBAR_GET_PRIVATE (self);
  memset (self->priv, 0, sizeof (self->priv));
}

static gboolean visible = TRUE;

gboolean
hrn_sidebar_visible (void)
{
  return visible;
}

#if STUCKBAR
static gboolean stuck = TRUE;
#else
static gboolean stuck = FALSE;
#endif

static gboolean
expanded_clicked_cb (NbtkButton *button, gpointer sidebar)
{
  if (nbtk_button_get_checked (button))
    {
      hrn_sidebar_wake ();
    }
  else
    {
#if STUCKBAR
#else
      stuck = FALSE;
#endif

      hrn_sidebar_hide ();
    }
  return TRUE;
}

static void
set_source (HrnSource *source)
{
  hrn_view_set_source (HRN_VIEW (hrn_view), source);

  hrn_toolbar_set_query ((HrnToolbar *) hrn_toolbar, NULL);

  hrn_toolbar_set_filter_mode ((HrnToolbar *) hrn_toolbar,
                               BKL_ITEM_TYPE_AUDIO |
                               BKL_ITEM_TYPE_IMAGE |
                               BKL_ITEM_TYPE_VIDEO);

  hrn_view_set_filter (HRN_VIEW (hrn_view),
                       BKL_ITEM_TYPE_AUDIO |
                       BKL_ITEM_TYPE_IMAGE |
                       BKL_ITEM_TYPE_VIDEO);
  hrn_set_zoom_hard (0.5);
}

static void
change_source (NbtkButton *button, gpointer userdata)
{
  HrnSource *source = (HrnSource *) userdata;

  set_source (source);
}

static void
change_source_audio (NbtkButton *button, gpointer userdata)
{
  HrnSource *source = (HrnSource *) userdata;

  set_source (source);
  hrn_toolbar_set_filter_mode ((HrnToolbar *) hrn_toolbar, BKL_ITEM_TYPE_AUDIO);
  hrn_view_set_filter (HRN_VIEW (hrn_view), BKL_ITEM_TYPE_AUDIO);
  hrn_toolbar_set_query ((HrnToolbar *) hrn_toolbar, "");
}

static void
change_source_video (NbtkButton *button, gpointer userdata)
{
  HrnSource *source = (HrnSource *) userdata;

  set_source (source);
  hrn_toolbar_set_filter_mode ((HrnToolbar *) hrn_toolbar, BKL_ITEM_TYPE_VIDEO);
  hrn_view_set_filter (HRN_VIEW (hrn_view), BKL_ITEM_TYPE_VIDEO);
  hrn_toolbar_set_query ((HrnToolbar *) hrn_toolbar, "");
}

static void
change_source_images (NbtkButton *button, gpointer userdata)
{
  HrnSource *source = (HrnSource *) userdata;

  set_source (source);
  hrn_toolbar_set_filter_mode ((HrnToolbar *) hrn_toolbar, BKL_ITEM_TYPE_IMAGE);
  hrn_view_set_filter (HRN_VIEW (hrn_view), BKL_ITEM_TYPE_IMAGE);
  hrn_toolbar_set_query ((HrnToolbar *) hrn_toolbar, "");
}

extern gboolean hrn_search_direct;

void
hrn_retrieve_query (const gchar *qname)
{
  gchar     *query = g_key_file_get_string (pinned_searches, qname, "query",
                                            NULL);
  gchar     *path = g_key_file_get_string (pinned_searches, qname, "source",
                                           NULL);
  gint       filter = g_key_file_get_integer (pinned_searches, qname, "filter",
                                              NULL);
  gdouble    zoom = g_key_file_get_double (pinned_searches, qname, "zoom",
                                           NULL);
  HrnSource *source;

  if (!path)
    return;

  if (g_str_equal (path, BKL_LOCAL_SOURCE_PATH))
    {
      source = hrn_get_local_source ();
    }
  else
    {
      source = hrn_get_source_for_path (path);
    }

  hrn_view_set_source (HRN_VIEW (hrn_view), source);

  hrn_toolbar_set_filter_mode ((HrnToolbar *) hrn_toolbar, filter);
  hrn_view_set_filter (HRN_VIEW (hrn_view), filter);

  hrn_search_direct = TRUE; /* hack that changes the behavior of
                               the call for this caller (and similar ones)
                             */

  hrn_toolbar_set_query ((HrnToolbar *) hrn_toolbar, query);
  hrn_search_direct = FALSE;
  hrn_set_zoom_hard (zoom);
  g_free (query);
  g_free (path);
}

static void
change_source_query (NbtkButton *button, gpointer userdata)
{
  const gchar *qname = hrn_sidebar_subitem_get_qname (
    (HrnSidebarSubitem *) button);

  hrn_retrieve_query (qname);

  hrn_toolbar_set_pinned ((HrnToolbar *) hrn_toolbar, TRUE);
}

void save_queries (void);

static void
item_renamed (HrnSidebarSubitem *item, const char        *old_title,
              const char        *new_title,
              HrnSidebar        *sidebar)
{
  const gchar *qname;

  qname = hrn_sidebar_subitem_get_qname (item);
  g_key_file_set_string (pinned_searches, qname, "name", new_title);

  save_queries ();

  hrn_sidebar_update_sources (HRN_SIDEBAR (hrn_sidebar));
}

static void
unpinned (NbtkButton *button, gboolean pinned, gpointer userdata)
{
  HrnSidebarSubitem *item  = (HrnSidebarSubitem *) button;
  const char        *qname = hrn_sidebar_subitem_get_qname (item);
  HrnView           *view  = HRN_VIEW (hrn_view);
  gchar             *query =
    g_key_file_get_string (pinned_searches, qname, "query",
                           NULL);
  gchar *source =
    g_key_file_get_string (pinned_searches, qname, "source",
                           NULL);
  gint filter =
    g_key_file_get_integer (pinned_searches, qname, "filter",
                            NULL);

  if (pinned)
    {
      return;
    }

  if (hrn_view_get_filter (view) == filter)
    {
      HrnSource  *view_src = hrn_view_get_source (view);
      const char *vs_path  = NULL;

      if (view_src)
        {
          vs_path = hrn_source_get_object_path (view_src);
        }

      if (source && vs_path && g_str_equal (source, vs_path))
        {
          const char *v_query = hrn_view_get_search (view);

          if (query && v_query && g_str_equal (query, v_query))
            {
              hrn_toolbar_set_pinned (HRN_TOOLBAR (hrn_toolbar), FALSE);
            }
        }
    }

  g_key_file_remove_group (pinned_searches, qname, NULL);

  save_queries ();

  hrn_sidebar_update_sources (HRN_SIDEBAR (hrn_sidebar));
}

static gboolean
sidebar_enter (ClutterActor *actor, ClutterEvent *event, gpointer data);
static gboolean
sidebar_leave (ClutterActor *actor, ClutterEvent *event, gpointer data);

#define SIDEBAR_WIDTH           170
#define SIDEBAR_WIDTH_PADDED    190

static void
add_source (HrnSidebar *sidebar, HrnSource  *source, gboolean is_local)
{
  HrnSidebarPrivate *priv = sidebar->priv;

  NbtkWidget        *button;

  if (is_local)
    button = nbtk_button_new_with_label (_ ("Your local media"));
  else
    {
      const char *name = hrn_source_get_name (source);
      button = nbtk_button_new_with_label (name);
    }

  nbtk_widget_set_style_class_name (button, "HrnSidebarItem");
  clutter_actor_set_reactive (CLUTTER_ACTOR (button), TRUE);

  clutter_actor_set_width (CLUTTER_ACTOR (button), SIDEBAR_WIDTH);
  clutter_container_add_actor (CLUTTER_CONTAINER (priv->flow),
                               CLUTTER_ACTOR (button));

  g_signal_connect (button, "clicked", G_CALLBACK (change_source), source);

  nbtk_bin_set_alignment (NBTK_BIN (button), NBTK_ALIGN_LEFT, NBTK_ALIGN_TOP);

  /* These predefined audio/video/image searches should be created on demand
   * when there is audio images or video present in the library
   */
  button =
    (NbtkWidget *) hrn_sidebar_subitem_new (source, "predefined", _ ("Audio"),
                                            "icon-audio", FALSE, FALSE);
  g_object_set_data (G_OBJECT (button), "source", source);

  clutter_container_add_actor (CLUTTER_CONTAINER (priv->flow),
                               CLUTTER_ACTOR (button));
  nbtk_bin_set_alignment (NBTK_BIN (button), NBTK_ALIGN_LEFT, NBTK_ALIGN_TOP);
  g_signal_connect (button, "clicked",
                    G_CALLBACK (change_source_audio), source);

  button =
    (NbtkWidget *) hrn_sidebar_subitem_new (source, "predefined", _ ("Video"),
                                            "icon-video", FALSE, FALSE);
  g_object_set_data (G_OBJECT (button), "source", source);

  clutter_container_add_actor (CLUTTER_CONTAINER (priv->flow),
                               CLUTTER_ACTOR (button));
  nbtk_bin_set_alignment (NBTK_BIN (button), NBTK_ALIGN_LEFT, NBTK_ALIGN_TOP);
  g_signal_connect (button, "clicked",
                    G_CALLBACK (change_source_video), source);

  button =
    (NbtkWidget *) hrn_sidebar_subitem_new (source, "predefined", _ ("Images"),
                                            "icon-images", FALSE, FALSE);
  g_object_set_data (G_OBJECT (button), "source", source);

  clutter_container_add_actor (CLUTTER_CONTAINER (priv->flow),
                               CLUTTER_ACTOR (button));
  nbtk_bin_set_alignment (NBTK_BIN (button), NBTK_ALIGN_LEFT, NBTK_ALIGN_TOP);
  g_signal_connect (button, "clicked",
                    G_CALLBACK (change_source_images), source);

  {
    gsize   n_groups;
    gchar **groups;
    gint    j;

    groups = g_key_file_get_groups (pinned_searches, &n_groups);

    for (j = 0; j < n_groups; j++)
      {
        const gchar *qname = groups[j];
        gchar       *sname = g_key_file_get_string (pinned_searches, qname,
                                                    "source", NULL);

        if (sname &&
            g_str_equal (sname, hrn_source_get_object_path (source)) &&
            !g_str_equal (qname, "hrn-last-state"))
          {
            int         filter = g_key_file_get_integer (pinned_searches, qname,
                                                         "filter", NULL);
            const char *iname;

            switch (filter)
              {
                case BKL_ITEM_TYPE_AUDIO:
                  iname = "icon-audio";
                  break;

                case BKL_ITEM_TYPE_IMAGE:
                  iname = "icon-images";
                  break;

                case BKL_ITEM_TYPE_VIDEO:
                  iname = "icon-video";
                  break;

                default:
                  iname = "icon-all";
              }

	    g_free (sname);

            /* If the group doesnt contain name, we fall back to the name of
             * the group.
             */
            sname = g_key_file_get_string (pinned_searches, qname, "name", NULL);
            if (sname)
              {
                char *name =
                  g_key_file_get_string (pinned_searches, qname, "name",
                                         NULL);
                button =
                  (NbtkWidget *) hrn_sidebar_subitem_new (source, qname, name,
                                                          iname, TRUE,
                                                          TRUE);
                g_free (sname);
                g_free (name);
              }
            else
              {
                button =
                  (NbtkWidget *) hrn_sidebar_subitem_new (source, qname, qname,
                                                          iname, TRUE,
                                                          TRUE);
              }

            g_object_set_data (G_OBJECT (button), "source", source);
            g_signal_connect (button, "clicked",
                              G_CALLBACK (change_source_query), source);

            g_signal_connect (button, "pinned",
                              G_CALLBACK (unpinned), NULL);
            g_signal_connect (button, "renamed",
                              G_CALLBACK (item_renamed), sidebar);

            clutter_container_add_actor (CLUTTER_CONTAINER (priv->flow),
                                         CLUTTER_ACTOR (button));
            nbtk_bin_set_alignment (NBTK_BIN (button),
                                    NBTK_ALIGN_LEFT,
                                    NBTK_ALIGN_TOP);
          }
      }

    if (groups)
      g_strfreev (groups);
  }
}

void
hrn_sidebar_update_sources (HrnSidebar *sidebar)
{
  GList             *sources, *s;
  HrnSidebarPrivate *priv = sidebar->priv;
  HrnSource         *source;


  {
    GList *c, *children =
      clutter_container_get_children (CLUTTER_CONTAINER (priv->flow));

    for (c = children; c; c = c->next)
      {
        clutter_actor_destroy (c->data);
      }
    g_list_free (children);
  }

  source = hrn_get_local_source ();
  if (source)
    {
      add_source (sidebar, source, TRUE);
    }

  sources = hrn_get_transient_sources ();
  for (s = sources; s; s = s->next)
    {
      add_source (sidebar, s->data, FALSE);
    }
}

static GObject *
hrn_sidebar_constructor (GType type, guint n_params,
                         GObjectConstructParam *params)
{
  HrnSidebarPrivate *priv;
  GObject           *object;
  HrnSidebar        *sidebar;

  object = G_OBJECT_CLASS (hrn_sidebar_parent_class)->constructor (
    type, n_params, params);

  sidebar = HRN_SIDEBAR (object);
  priv    = HRN_SIDEBAR_GET_PRIVATE (sidebar);

  /*clutter_actor_set_size (unused_state, 16, 16);*/

  priv->scroll = (NbtkWidget*) nbtk_scroll_view_new ();
  priv->flow   = nbtk_grid_new ();
  nbtk_grid_set_max_stride (NBTK_GRID (priv->flow), 1);
  clutter_actor_set_width (CLUTTER_ACTOR (priv->flow), SIDEBAR_WIDTH);
  clutter_actor_set_width (CLUTTER_ACTOR (priv->scroll), SIDEBAR_WIDTH);
  nbtk_widget_set_style_class_name (NBTK_WIDGET (sidebar), "HrnSidebar");
  clutter_actor_set_name (CLUTTER_ACTOR (sidebar), "HrnSidebar");

#if 1
  clutter_actor_set_parent (CLUTTER_ACTOR (priv->scroll),
                            CLUTTER_ACTOR (sidebar));
#else
  clutter_container_add_actor (CLUTTER_CONTAINER (object), priv->scroll);
#endif

  clutter_container_add_actor (CLUTTER_CONTAINER (priv->scroll),
                               CLUTTER_ACTOR (priv->flow));
  g_object_set_data (object, "HRN_DROP_MASK",
                     GINT_TO_POINTER (HRN_DROP_MASK_SIDEBAR));
  g_object_set_data (G_OBJECT (priv->flow), "HRN_DROP_MASK",
                     GINT_TO_POINTER (HRN_DROP_MASK_SIDEBAR));

  priv->expanded = nbtk_button_new ();
  clutter_actor_set_parent (CLUTTER_ACTOR (priv->expanded),
                            CLUTTER_ACTOR (sidebar));

  clutter_actor_set_anchor_point_from_gravity (CLUTTER_ACTOR (
                                                 priv->expanded),
                                               CLUTTER_GRAVITY_EAST);

  clutter_actor_set_position (CLUTTER_ACTOR (
                                priv->expanded), SIDEBAR_WIDTH_PADDED,
                              (clutter_actor_get_height (
                                 clutter_stage_get_default ()) - 61) / 2);

  nbtk_button_set_toggle_mode (NBTK_BUTTON (priv->expanded), TRUE);
  nbtk_widget_set_style_class_name (priv->expanded, "HrnSidebarExpanded");

  clutter_actor_set_size (CLUTTER_ACTOR (priv->expanded), 10, 600);

  g_signal_connect (priv->expanded, "clicked", G_CALLBACK (
                      expanded_clicked_cb), sidebar);

  /*  hrn_sidebar_update_sources (sidebar);*/

  /* hack to sync up size */
  {
    gfloat width  = SIDEBAR_WIDTH_PADDED;
    guint  height = clutter_actor_get_height (clutter_stage_get_default ()) -
                    61;

    clutter_actor_set_size (CLUTTER_ACTOR (priv->flow), width - 40, height);
    clutter_actor_set_size (CLUTTER_ACTOR (priv->scroll), width, height);
    clutter_actor_set_size (CLUTTER_ACTOR (sidebar), width, height);
  }
  clutter_actor_set_reactive (CLUTTER_ACTOR (object), TRUE);
  g_signal_connect (object, "enter-event", G_CALLBACK (sidebar_enter), NULL);
  g_signal_connect (object, "leave-event", G_CALLBACK (sidebar_leave), NULL);

  return object;
}


static void
hrn_sidebar_allocate (ClutterActor *self, const ClutterActorBox *box,
                      ClutterAllocationFlags flags)
{
  HrnSidebarPrivate *priv = HRN_SIDEBAR_GET_PRIVATE (self);
  ClutterActorClass *parent_class;

  parent_class = CLUTTER_ACTOR_CLASS (hrn_sidebar_parent_class);

  parent_class->allocate (self, box, flags);
  clutter_actor_allocate_preferred_size (CLUTTER_ACTOR (priv->scroll), flags);
  clutter_actor_allocate_preferred_size (CLUTTER_ACTOR (priv->expanded), flags);
}

static void
hrn_sidebar_dispose (GObject *object)
{
  G_OBJECT_CLASS (hrn_sidebar_parent_class)->dispose (object);
}

static void
hrn_sidebar_paint (ClutterActor *actor)
{
  HrnSidebar        *sidebar = HRN_SIDEBAR (actor);
  HrnSidebarPrivate *priv    = HRN_SIDEBAR_GET_PRIVATE (sidebar);

  CLUTTER_ACTOR_CLASS (hrn_sidebar_parent_class)->paint (actor);

  clutter_actor_paint (CLUTTER_ACTOR (priv->scroll));
  clutter_actor_paint (CLUTTER_ACTOR (priv->expanded));
}

#include <cogl/cogl.h>

static void
hrn_sidebar_pick (ClutterActor *actor, const ClutterColor   *color)
{
  gfloat             width;
  gfloat             height;
  HrnSidebarPrivate *priv = HRN_SIDEBAR_GET_PRIVATE (actor);

  clutter_actor_get_size (CLUTTER_ACTOR (actor), &width, &height);

  cogl_set_source_color ((void*) color);
  cogl_rectangle (0, 0, width, height);

  clutter_actor_paint (CLUTTER_ACTOR (priv->scroll));
  clutter_actor_paint (CLUTTER_ACTOR (priv->expanded));
}

HrnSidebar *
hrn_sidebar_new (void)
{
  HrnSidebar *sidebar = g_object_new (HRN_TYPE_SIDEBAR, NULL);

  return sidebar;
}


/**********/

static guint sidebar_hide_handler_id = 0;

gboolean
sidebar_hide_handler (gpointer foo)
{
  HrnSidebarPrivate *priv = HRN_SIDEBAR_GET_PRIVATE (hrn_sidebar);

  nbtk_button_set_checked (NBTK_BUTTON (priv->expanded), FALSE);

  clutter_actor_animate (hrn_sidebar, CLUTTER_EASE_IN_CUBIC,
                         SIDEBAR_OUT_DURATION,
                         "x", -180.0,
                         "opacity", 255,
                         NULL);

  clutter_actor_animate (hrn_content_area, CLUTTER_EASE_IN_CUBIC,
                         SIDEBAR_OUT_DURATION,
                         "depth", 0.0,
                         "x", 0.0,
                         NULL);

  sidebar_hide_handler_id = 0;
  visible                 = FALSE;
  return FALSE;
}

void
hrn_sidebar_wake (void)
{
  HrnSidebarPrivate *priv = HRN_SIDEBAR_GET_PRIVATE (hrn_sidebar);

  nbtk_button_set_checked (NBTK_BUTTON (priv->expanded), TRUE);

  if (sidebar_hide_handler_id)
    {
      g_source_remove (sidebar_hide_handler_id);
      sidebar_hide_handler_id = 0;
    }

  clutter_actor_animate (hrn_sidebar, CLUTTER_EASE_IN_OUT_QUINT,
                         SIDEBAR_IN_DURATION,
                         "x", 0.0,
                         NULL);

  if (hrn_theatre_get_active (HRN_THEATRE (hrn_theatre)))
    {
      clutter_actor_animate (hrn_sidebar, CLUTTER_EASE_IN_OUT_QUINT,
                             SIDEBAR_IN_DURATION,
                             "opacity", 190,
                             NULL);
    }
  else
    {
      clutter_actor_animate (hrn_content_area, CLUTTER_EASE_IN_OUT_QUINT,
                             SIDEBAR_IN_DURATION,
                             "depth", -200.0,
                             "x", 110.0,
                             NULL);
    }

  if (hrn_get_zoom () > 0.5 && !stuck)
    {
      sidebar_hide_handler_id = g_timeout_add (HRN_SIDEBAR_TIMEOUT,
                                               sidebar_hide_handler, NULL);
    }
  visible = TRUE;
}


void
hrn_sidebar_unpin (void)
{
  if (sidebar_hide_handler_id == 0 && !stuck)
    sidebar_hide_handler_id = g_timeout_add (HRN_SIDEBAR_TIMEOUT,
                                             sidebar_hide_handler, NULL);
}

void
hrn_sidebar_hide (void)
{
  if (sidebar_hide_handler_id != 0)
    g_source_remove (sidebar_hide_handler_id);
  sidebar_hide_handler (NULL);
}


static gboolean
sidebar_enter (ClutterActor *actor, ClutterEvent *event, gpointer data)
{
  if (sidebar_hide_handler_id)
    g_source_remove (sidebar_hide_handler_id);
  sidebar_hide_handler_id = 0;

  {
    NbtkButton *expanded = NBTK_BUTTON (HRN_SIDEBAR (actor)->priv->expanded);
    if (!nbtk_button_get_checked (expanded))
      {
        nbtk_button_set_checked (expanded, TRUE);
        expanded_clicked_cb (expanded, NULL);
      }
  }

  stuck = TRUE;
  return FALSE;
}

static gboolean
sidebar_leave (ClutterActor *actor, ClutterEvent *event, gpointer data)
{
#if STUCKBAR
#else
  stuck = FALSE;
  hrn_sidebar_unpin ();
#endif
  return FALSE;
}

static gboolean
sidebar_capture_cb (ClutterActor *actor, ClutterEvent *event, gpointer data)
{
  switch (event->any.type)
    {
      case CLUTTER_MOTION:
      case CLUTTER_BUTTON_PRESS:
      case CLUTTER_BUTTON_RELEASE:
      case CLUTTER_SCROLL:
        break;

      default:
        break;
    }
  return FALSE;
}

static gboolean
sidebar_release_cb (ClutterActor *actor, ClutterEvent *event, gpointer data)
{
  hrn_set_zoom (0.5);
  return TRUE;
}

void
sidebar_init (void)
{
  clutter_actor_set_reactive (hrn_sidebar, TRUE);
  g_signal_connect (hrn_sidebar, "captured-event",
                    G_CALLBACK (sidebar_capture_cb), NULL);
  g_signal_connect (hrn_sidebar, "button-release-event",
                    G_CALLBACK (sidebar_release_cb), NULL);
  hrn_sidebar_wake ();
}
