/*
 * Hornsey - Moblin Media Player.
 * Copyright © 2009 Intel Corporation.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 2.1, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <clutter/clutter.h>
#include <math.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include "hrn.h"
#include "hrn-searchbar.h"

G_DEFINE_TYPE (HrnSearchBar, hrn_search_bar, NBTK_TYPE_ENTRY);

#define HRN_SEARCH_BAR_GET_PRIVATE(obj)                 \
  (G_TYPE_INSTANCE_GET_PRIVATE ((obj), \
                                HRN_TYPE_SEARCH_BAR, \
                                HrnSearchBarPrivate))

struct _HrnSearchBarPrivate
{
  ClutterActor *pin_button;
  ClutterActor *clear_button;
};

static GObject * hrn_search_bar_constructor (GType type, guint n_params,
                                    GObjectConstructParam *params);
static void hrn_search_bar_dispose (GObject               *object);
static void hrn_searchbar_allocate (ClutterActor          *self,
                                    const ClutterActorBox *box,
                                    ClutterAllocationFlags flags);
static void hrn_searchbar_paint    (ClutterActor          *actor);
static void hrn_searchbar_pick     (ClutterActor          *actor,
                                    const ClutterColor    *color);


static void
hrn_searchbar_map (ClutterActor *self)
{
  HrnSearchBarPrivate *priv = HRN_SEARCH_BAR (self)->priv;

  CLUTTER_ACTOR_CLASS (hrn_search_bar_parent_class)->map (self);

  if (priv->pin_button)
    clutter_actor_map (priv->pin_button);
  if (priv->clear_button)
    clutter_actor_map (priv->clear_button);
}


static void
hrn_searchbar_unmap (ClutterActor *self)
{
  HrnSearchBarPrivate *priv = HRN_SEARCH_BAR (self)->priv;

  CLUTTER_ACTOR_CLASS (hrn_search_bar_parent_class)->unmap (self);

  if (priv->pin_button)
    clutter_actor_unmap (priv->pin_button);
  if (priv->clear_button)
    clutter_actor_unmap (priv->clear_button);
}

static gboolean
hrn_search_bar_style_changed (NbtkWidget *widget)
{
  HrnSearchBarPrivate *priv = HRN_SEARCH_BAR (widget)->priv;

  g_signal_emit_by_name (priv->pin_button, "style-changed", 0);
  g_signal_emit_by_name (priv->clear_button, "style-changed", 0);

  return FALSE;
}

static void
hrn_search_bar_class_init (HrnSearchBarClass *klass)
{
  GObjectClass      *gobject_class = G_OBJECT_CLASS (klass);
  ClutterActorClass *actor_class   = CLUTTER_ACTOR_CLASS (klass);

  gobject_class->dispose      = hrn_search_bar_dispose;
  gobject_class->constructor  = hrn_search_bar_constructor;
  actor_class->paint          = hrn_searchbar_paint;
  actor_class->pick           = hrn_searchbar_pick;
  actor_class->map            = hrn_searchbar_map;
  actor_class->unmap          = hrn_searchbar_unmap;
  actor_class->allocate       = hrn_searchbar_allocate;

  g_type_class_add_private (gobject_class, sizeof (HrnSearchBarPrivate));
}

static void
hrn_search_bar_init (HrnSearchBar *self)
{
  HrnSearchBarPrivate *priv = HRN_SEARCH_BAR_GET_PRIVATE (self);

  self->priv = priv;
  memset (priv, 0, sizeof (priv));
  g_signal_connect (self, "style-changed", G_CALLBACK (hrn_search_bar_style_changed), NULL);
}

void
save_queries (void)
{
  GFile *file;
  gchar *data;

  {
    gchar *search_path = g_strdup_printf ("%s/.hrn-queries", g_get_home_dir ());
    file = g_file_new_for_path (search_path);
    g_free (search_path);
  }
  data = g_key_file_to_data (pinned_searches, NULL, NULL);
  g_file_replace_contents (file, data, strlen (
                             data), NULL, TRUE, G_FILE_CREATE_NONE, NULL, NULL,
                           NULL);
  g_object_unref (file);
  g_free (data);
}


extern gboolean hrn_search_direct; /* hack */

static void
cleared (NbtkButton *togglebutton, HrnSearchBar *search_bar)
{
  hrn_search_direct = TRUE;
  nbtk_entry_set_text (NBTK_ENTRY (search_bar), "");
  hrn_search_direct = FALSE;
}

static gint
make_query_no (void)
{
  gint  test = 0;
  gchar name[10];

  for (test = 0; test < 1000; test++)
    {
      g_snprintf (name, 9, "query%i", test);
      if (!g_key_file_has_group (pinned_searches, name))
        return test;
    }
  return -1;
}


static void
save_state (const gchar *qname)
{
  const gchar *search;
  const gchar *name   = qname;
  const gchar *visual = "";
  const gchar *audio  = "";

  guint        filter = hrn_view_get_filter (HRN_VIEW (hrn_view));
  HrnSource   *client;
  gchar        groupname[32]; /* 32 is more than wide enough to fit
                                 "hrn-last-state as well as all potentially
                                 created query names
                               */

  if (qname == NULL)
    {
      BklItem *temp;

      temp = hrn_queue_get_active_audio (master_queue);
      if (temp)
        audio = bkl_item_get_uri (temp);
      temp = hrn_queue_get_active_visual (master_queue);
      if (temp)
        visual = bkl_item_get_uri (temp);
    }

  search = hrn_view_get_search (HRN_VIEW (hrn_view));
  client = hrn_view_get_source (HRN_VIEW (hrn_view));

  if (!client)
    return;

  g_snprintf (groupname, 31, "query%i", make_query_no ());
  if (qname == NULL)
    {
      g_sprintf (groupname, "hrn-last-state");

      g_debug ("query %s %s [%s] %i saved", groupname,
               hrn_source_get_object_path (client), search, filter);
      g_key_file_set_string (pinned_searches, groupname, "visual", visual);
      g_key_file_set_string (pinned_searches, groupname, "audio", audio);
    }
  else
    {
      if (search == NULL || search[0] == '\0')
        {
          switch (filter)
            {
              case BKL_ITEM_TYPE_IMAGE:
                name = "Images2";
                break;

              case BKL_ITEM_TYPE_VIDEO:
                name = "Video2";
                break;

              case BKL_ITEM_TYPE_AUDIO:
                name = "Audio2";
                break;

              default:
                /* the library heading itself is the only all query
                 * that should be needed */
                return;
            }
        }
      else
        name = search;
    }

  g_key_file_set_string (pinned_searches, groupname, "name", name ? name : "");
  g_key_file_set_string (pinned_searches, groupname, "query",
                         search ? search : "");
  g_key_file_set_string (pinned_searches, groupname, "source",
                         hrn_source_get_object_path (
                           client));
  g_key_file_set_integer (pinned_searches, groupname, "filter", filter);
  g_key_file_set_double (pinned_searches, groupname, "zoom", hrn_get_zoom ());

  save_queries ();

  if (qname)
    hrn_sidebar_update_sources (HRN_SIDEBAR (hrn_sidebar));
}

void
hrn_quit (void)
{
  save_state (NULL);
  clutter_main_quit ();
}

static guint autosave = 0;

static gboolean
hrn_autosave (gpointer data)
{
  autosave = 0;
  save_state (NULL);
  return FALSE;
}

void
hrn_store_state (void)
{
  if (autosave)
    {
      g_source_remove (autosave);
    }
  autosave = g_timeout_add (5000, hrn_autosave, NULL);
}


static void
pin_toggled (NbtkButton *togglebutton, HrnSearchBar *search_bar)
{
  const gchar *search;
  const gchar *name   = "test search";
  guint        filter = hrn_view_get_filter (HRN_VIEW (hrn_view));
  HrnSource   *client;

  search = nbtk_entry_get_text (NBTK_ENTRY (search_bar));

  client = hrn_view_get_source (HRN_VIEW (hrn_view));

  g_debug ("%s [%s] %i %s", hrn_source_get_object_path (
             client), search, filter,
           nbtk_button_get_checked (togglebutton) ? "pinned" : "");

  if (nbtk_button_get_checked (togglebutton))
    {
      if (search == NULL || search[0] == '\0')
        {
          switch (filter)
            {
              case BKL_ITEM_TYPE_IMAGE:
              case BKL_ITEM_TYPE_VIDEO:
              case BKL_ITEM_TYPE_AUDIO:
                break;

              default:
                nbtk_button_set_checked (togglebutton, FALSE);
                return;
            }
        }
      save_state (search);
    }
  else
    {
      /* this should only happen on click?! */
      /* find the correct query */
      /* remove it */
      name = search; /* XXX: for now depending on search term == query */
      g_key_file_remove_group (pinned_searches, name, NULL);
    }

  save_queries ();
  hrn_sidebar_update_sources (HRN_SIDEBAR (hrn_sidebar));
}

void
hrn_search_bar_unpin (HrnSearchBar *searchbar)
{
  HrnSearchBarPrivate *priv = HRN_SEARCH_BAR_GET_PRIVATE (searchbar);

  if (nbtk_button_get_checked (NBTK_BUTTON (priv->pin_button)))
    {
      nbtk_button_set_checked (NBTK_BUTTON (priv->pin_button), FALSE);
      if (0) pin_toggled (NBTK_BUTTON (priv->pin_button), searchbar);
    }
}


void
hrn_search_bar_set_pinned (HrnSearchBar *searchbar)
{
  HrnSearchBarPrivate *priv = HRN_SEARCH_BAR_GET_PRIVATE (searchbar);

  nbtk_button_set_checked (NBTK_BUTTON (priv->pin_button), TRUE);
}


static GObject *
hrn_search_bar_constructor (GType type, guint n_params,
                            GObjectConstructParam *params)
{
  GObject             *object;
  HrnSearchBar        *search_bar;
  HrnSearchBarPrivate *priv;

  object = G_OBJECT_CLASS (hrn_search_bar_parent_class)->constructor (
    type, n_params, params);

  search_bar         = HRN_SEARCH_BAR (object);
  priv               = HRN_SEARCH_BAR_GET_PRIVATE (object);
  priv->pin_button   = (void*) nbtk_button_new ();
  priv->clear_button = (void*) nbtk_button_new ();

  nbtk_button_set_toggle_mode (NBTK_BUTTON (priv->pin_button), TRUE);
  clutter_actor_set_name (CLUTTER_ACTOR (priv->pin_button),
                          "fav-toggle");
  clutter_actor_set_name (CLUTTER_ACTOR (priv->clear_button),
                          "clear-query");

  clutter_actor_set_parent (CLUTTER_ACTOR (priv->pin_button),
                            CLUTTER_ACTOR (object));
  clutter_actor_set_parent (CLUTTER_ACTOR (priv->clear_button),
                            CLUTTER_ACTOR (object));

  g_signal_connect (priv->pin_button, "clicked", G_CALLBACK (
                      pin_toggled), object);
  g_signal_connect (priv->clear_button, "clicked", G_CALLBACK (cleared), object);

  clutter_actor_set_size (priv->pin_button, 32, 32);
  clutter_actor_set_size (priv->clear_button, 32, 32);
  clutter_actor_set_anchor_point_from_gravity (priv->pin_button,
                                               CLUTTER_GRAVITY_NORTH_EAST);
  clutter_actor_set_anchor_point_from_gravity (priv->clear_button,
                                               CLUTTER_GRAVITY_NORTH_EAST);

  return object;
}

static void
hrn_search_bar_dispose (GObject *object)
{
  /* XXX: dispose of children */
  G_OBJECT_CLASS (hrn_search_bar_parent_class)->dispose (object);
}


static void
hrn_searchbar_allocate (ClutterActor *self, const ClutterActorBox *box,
                        ClutterAllocationFlags flags)
{
  HrnSearchBarPrivate *priv = HRN_SEARCH_BAR_GET_PRIVATE (self);

  CLUTTER_ACTOR_CLASS (hrn_search_bar_parent_class)->allocate (self, box, flags);

  clutter_actor_set_x (CLUTTER_ACTOR (priv->pin_button),
                       clutter_actor_get_width (self));
  clutter_actor_set_x (CLUTTER_ACTOR (priv->clear_button),
                       clutter_actor_get_x (priv->pin_button) -
                       20);
  clutter_actor_allocate_preferred_size (CLUTTER_ACTOR (
                                           priv->pin_button), flags);
  clutter_actor_allocate_preferred_size (CLUTTER_ACTOR (
                                           priv->clear_button), flags);
}

static void
hrn_searchbar_paint (ClutterActor *actor)
{
  HrnSearchBarPrivate *priv = HRN_SEARCH_BAR_GET_PRIVATE (actor);

  CLUTTER_ACTOR_CLASS (hrn_search_bar_parent_class)->paint (actor);
  clutter_actor_paint (CLUTTER_ACTOR (priv->pin_button));
  clutter_actor_paint (CLUTTER_ACTOR (priv->clear_button));
}

static void
hrn_searchbar_pick (ClutterActor *actor, const ClutterColor *color)
{
  hrn_searchbar_paint (actor);
}
