#include "hrn.h"
#include "hrn-view.h"
#include "hrn-tileable.h"
#include "hrn-tiler.h"
#include "hrn-viewer.h"
#include "hrn-marshal.h"

#include "hrn-artist-cluster.h"
#include "hrn-year-cluster.h"
#include "hrn-video-tile.h"

enum {
    PROP_0,
    PROP_HADJUST,
    PROP_VADJUST
};

enum {
    ACTIVATED,
    LAST_SIGNAL,
};

/* #define DEBUG_SIZE 1 */

struct _HrnViewPrivate {
    NbtkAdjustment *hadjustment;
    NbtkAdjustment *vadjustment;

    HrnClusterNode *root;
    guint32 child_added_id;
    guint32 child_removed_id;
    GHashTable *child_items;

    guint visible_children;
    HrnTileable *shown_child;

    double scale;

    guint items_per_row;
    int count;

#ifdef DEBUG_SIZE
    ClutterActor *bg;
#endif
};

#define DEFAULT_ITEMS_PER_ROW 5

#define DEFAULT_SCALE 1.0

#define X_GRID_TO_COORDS(x) (x * (ITEM_WIDTH + COL_GAP))
#define Y_GRID_TO_COORDS(y) (y * (ITEM_HEIGHT + ROW_GAP))

#define LEVEL_COUNT 3

static gboolean debug_visible = FALSE;

#define GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj), HRN_TYPE_VIEW, HrnViewPrivate))

static void scrollable_set_adjustments (NbtkScrollable *scrollable,
                                        NbtkAdjustment *hadjustment,
                                        NbtkAdjustment *vadjustment);
static void scrollable_get_adjustments (NbtkScrollable  *scrollable,
                                        NbtkAdjustment **hadjustment,
                                        NbtkAdjustment **vadjustment);
static void scrollable_interface_init (NbtkScrollableInterface *iface);
static void container_interface_init (ClutterContainerIface *iface);
static void tiler_interface_init (HrnTilerInterface *iface);
static void viewer_interface_init (HrnViewerInterface *iface);
G_DEFINE_TYPE_WITH_CODE (HrnView, hrn_view, NBTK_TYPE_WIDGET,
                         G_IMPLEMENT_INTERFACE (HRN_TYPE_TILER,
                                                tiler_interface_init)
                         G_IMPLEMENT_INTERFACE (HRN_TYPE_VIEWER,
                                                viewer_interface_init)
                         G_IMPLEMENT_INTERFACE (CLUTTER_TYPE_CONTAINER,
                                                container_interface_init)
                         G_IMPLEMENT_INTERFACE (NBTK_TYPE_SCROLLABLE,
                                                scrollable_interface_init));

static guint32 signals[LAST_SIGNAL] = {0,};

static void
hrn_view_finalize (GObject *object)
{
    G_OBJECT_CLASS (hrn_view_parent_class)->finalize (object);
}

static void
hrn_view_dispose (GObject *object)
{
    G_OBJECT_CLASS (hrn_view_parent_class)->dispose (object);
}

static void
hrn_view_set_property (GObject      *object,
                       guint         prop_id,
                       const GValue *value,
                       GParamSpec   *pspec)
{
    HrnView *self = (HrnView *) object;
    HrnViewPrivate *priv = self->priv;

    switch (prop_id) {
    case PROP_HADJUST:
        scrollable_set_adjustments (NBTK_SCROLLABLE (object),
                                    g_value_get_object (value),
                                    priv->vadjustment);
        break;

    case PROP_VADJUST:
        scrollable_set_adjustments (NBTK_SCROLLABLE (object),
                                    priv->hadjustment,
                                    g_value_get_object (value));
        break;

    default:
        break;
    }
}

static void
hrn_view_get_property (GObject    *object,
                       guint       prop_id,
                       GValue     *value,
                       GParamSpec *pspec)
{
    NbtkAdjustment *adj;

    switch (prop_id) {
    case PROP_HADJUST:
        scrollable_get_adjustments (NBTK_SCROLLABLE (object), &adj, NULL);
        g_value_set_object (value, adj);
        break;

    case PROP_VADJUST:
        scrollable_get_adjustments (NBTK_SCROLLABLE (object), NULL, &adj);
        g_value_set_object (value, adj);
        break;

    default:
        break;
    }
}

static void
hrn_view_paint (ClutterActor *actor)
{
    HrnView *view = (HrnView *) actor;
    HrnViewPrivate *priv = view->priv;
    GSequenceIter *iter;
    float x, y;
    ClutterActorBox view_box;

    if (priv->hadjustment) {
        x = nbtk_adjustment_get_value (priv->hadjustment);
    } else {
        x = 0;
    }

    if (priv->vadjustment) {
        y = nbtk_adjustment_get_value (priv->vadjustment);
    } else {
        y = 0;
    }

    CLUTTER_ACTOR_CLASS (hrn_view_parent_class)->paint (actor);

    clutter_actor_get_allocation_box (actor, &view_box);
    view_box.x2 = (view_box.x2 - view_box.x1) + x;
    view_box.x1 = x;
    view_box.y2 = (view_box.y2 - view_box.y1) + y;
    view_box.y1 = y;

    if (priv->root == NULL) {
        return;
    }

    iter = g_sequence_get_begin_iter (priv->root->children);
    while (g_sequence_iter_is_end (iter) == FALSE) {
        HrnClusterNode *child_node = g_sequence_get (iter);
        ClutterActor *child;
        ClutterActorBox child_box;

        if (child_node->hidden) {
            iter = g_sequence_iter_next (iter);
            continue;
        }

        child = g_hash_table_lookup (priv->child_items, child_node);
        if (child == NULL) {
            g_warning ("No child found for %s", child_node->name);
            iter = g_sequence_iter_next (iter);
            continue;
        }

        clutter_actor_get_allocation_box (child, &child_box);

        if ((child_box.y1 < view_box.y2)
            && (child_box.y2 > view_box.y1)
            && CLUTTER_ACTOR_IS_VISIBLE (child)) {
            clutter_actor_paint (child);
        }

        iter = g_sequence_iter_next (iter);
    }
}

static void
hrn_view_pick (ClutterActor       *actor,
               const ClutterColor *color)
{
    HrnView *view = (HrnView *) actor;
    HrnViewPrivate *priv = view->priv;
    GSequenceIter *iter;
    float x, y;
    ClutterActorBox view_box;

    if (priv->hadjustment) {
        x = nbtk_adjustment_get_value (priv->hadjustment);
    } else {
        x = 0;
    }

    if (priv->vadjustment) {
        y = nbtk_adjustment_get_value (priv->vadjustment);
    } else {
        y = 0;
    }

    CLUTTER_ACTOR_CLASS (hrn_view_parent_class)->pick (actor, color);

    clutter_actor_get_allocation_box (actor, &view_box);
    view_box.x2 = (view_box.x2 - view_box.x1) + x;
    view_box.x1 = x;
    view_box.y2 = (view_box.y2 - view_box.y1) + y;
    view_box.y1 = y;

    if (priv->root == NULL) {
        return;
    }

    iter = g_sequence_get_begin_iter (priv->root->children);
    while (g_sequence_iter_is_end (iter) == FALSE) {
        HrnClusterNode *child_node = g_sequence_get (iter);
        ClutterActor *child;
        ClutterActorBox child_box;

        if (child_node->hidden) {
            iter = g_sequence_iter_next (iter);
            continue;
        }

        child = g_hash_table_lookup (priv->child_items, child_node);
        if (child == NULL) {
            g_warning ("No child found for %s", child_node->name);
            iter = g_sequence_iter_next (iter);
            continue;
        }

        clutter_actor_get_allocation_box (child, &child_box);

        if ((child_box.y1 < view_box.y2)
            && (child_box.y2 > view_box.y1)
            && CLUTTER_ACTOR_IS_VISIBLE (child)) {
            clutter_actor_paint (child);
        }

        iter = g_sequence_iter_next (iter);
    }
}

static void
hrn_view_apply_transform (ClutterActor *actor,
                          CoglMatrix   *matrix)
{
    HrnView *view = (HrnView *) actor;
    HrnViewPrivate *priv = view->priv;
    double x, y;

    CLUTTER_ACTOR_CLASS (hrn_view_parent_class)->apply_transform (actor, matrix);

    if (priv->hadjustment) {
        x = nbtk_adjustment_get_value (priv->hadjustment);
    } else {
        x = 0;
    }

    if (priv->vadjustment) {
        y = nbtk_adjustment_get_value (priv->vadjustment);
    } else {
        y = 0;
    }

    cogl_matrix_translate (matrix, (int) -x, (int) -y, 0);
}

static void
hrn_view_do_allocate (ClutterActor          *actor,
                      const ClutterActorBox *box,
                      ClutterAllocationFlags flags,
                      gboolean               calculate_extents_only,
                      float                 *actual_width,
                      float                 *actual_height)
{
    HrnView *view = (HrnView *) actor;
    HrnViewPrivate *priv = view->priv;
    GSequenceIter *iter;
    int row_count, i;
    NbtkPadding padding;

    nbtk_widget_get_padding (NBTK_WIDGET (actor), &padding);

    i = 0;
    if (priv->shown_child) {
        ClutterActorBox child_box;
        float nat_w, nat_h;

        clutter_actor_get_preferred_width ((ClutterActor *) priv->shown_child,
                                           0, NULL, &nat_w);
        clutter_actor_get_preferred_height ((ClutterActor *) priv->shown_child,
                                            0, NULL, &nat_h);

        child_box.x1 = X_GRID_TO_COORDS (0) + padding.left;
        child_box.x2 = child_box.x1 + nat_w;
        child_box.y1 = Y_GRID_TO_COORDS (0) + padding.top;
        child_box.y2 = child_box.y1 + nat_h;

        clutter_actor_allocate ((ClutterActor *) priv->shown_child,
                                &child_box, flags);

        /* Let tileable know where we are */
        hrn_tileable_set_position (priv->shown_child, 0, 0);

        i = hrn_tileable_get_count (priv->shown_child);
    } else {
        if (priv->root) {
            iter = g_sequence_get_begin_iter (priv->root->children);
            while (g_sequence_iter_is_end (iter) == FALSE) {
                HrnClusterNode *child_node = g_sequence_get (iter);
                ClutterActor *child;
                ClutterActorBox child_box;
                float nat_w, nat_h;
                int x, y;

                if (child_node->hidden) {
                    iter = g_sequence_iter_next (iter);
                    continue;
                }

                child = g_hash_table_lookup (priv->child_items, child_node);
                if (child == NULL) {
                    g_warning ("No child found for %s", child_node->name);
                    iter = g_sequence_iter_next (iter);
                    continue;
                }

                x = i % priv->items_per_row;
                y = i / priv->items_per_row;

                clutter_actor_get_preferred_width (child, 0, NULL, &nat_w);
                clutter_actor_get_preferred_height (child, 0, NULL, &nat_h);

                child_box.x1 = X_GRID_TO_COORDS (x) + padding.left;
                child_box.x2 = child_box.x1 + nat_w;
                child_box.y1 = Y_GRID_TO_COORDS (y) + padding.top;
                child_box.y2 = child_box.y1 + nat_h;

                clutter_actor_allocate (child, &child_box, flags);

                /* Let tileable know where we are */
                hrn_tileable_set_position ((HrnTileable *) child, x, y);

                iter = g_sequence_iter_next (iter);
                i++;
            }
        }
    }

    row_count = i / priv->items_per_row;
    if (i % priv->items_per_row > 0) {
        row_count++;
    }

    if (actual_width) {
        *actual_width = (priv->items_per_row * (ITEM_WIDTH + COL_GAP)) - COL_GAP
          + padding.left + padding.right;
    }

    if (actual_height) {
        *actual_height = ITEM_HEIGHT + ((row_count - 1) * (ITEM_HEIGHT + ROW_GAP))
          + padding.top + padding.bottom;
    }
}

static void
hrn_view_get_preferred_width (ClutterActor *actor,
                              float         for_height,
                              float        *min_width_p,
                              float        *natural_width_p)
{
    float actual_width, actual_height;
    ClutterActorBox box;

    box.x1 = 0;
    box.y1 = 0;
    box.x2 = G_MAXFLOAT;
    box.y2 = for_height;

    hrn_view_do_allocate (actor, &box, FALSE, TRUE,
                          &actual_width, &actual_height);

    if (min_width_p) {
        *min_width_p = actual_width;
    }
    if (natural_width_p) {
        *natural_width_p = actual_width;
    }
}

static void
hrn_view_get_preferred_height (ClutterActor *actor,
                               float         for_width,
                               float        *min_height_p,
                               float        *natural_height_p)
{
    float actual_width, actual_height;
    ClutterActorBox box;

    box.x1 = 0;
    box.y1 = 0;
    box.x2 = for_width;
    box.y2 = G_MAXFLOAT;

    hrn_view_do_allocate (actor, &box, FALSE, TRUE,
                          &actual_width, &actual_height);

    if (min_height_p) {
        *min_height_p = actual_height;
    }
    if (natural_height_p) {
        *natural_height_p = actual_height;
    }
}

static void
hrn_view_allocate (ClutterActor          *actor,
                   const ClutterActorBox *box,
                   ClutterAllocationFlags flags)
{
    HrnView *view = (HrnView *) actor;
    HrnViewPrivate *priv = view->priv;
    ClutterActorBox alloc_box = *box;
    float width;
    int items_per_row;

    CLUTTER_ACTOR_CLASS (hrn_view_parent_class)->allocate (actor, box, flags);

    if (priv->vadjustment) {
        double prev_value;
        float height;

        hrn_view_do_allocate (actor, box, flags, TRUE, NULL, &height);
        alloc_box.y2 = alloc_box.y1 + height;

        g_object_set (priv->vadjustment,
                      "lower", 0.0,
                      "upper", height,
                      "page-size", box->y2 - box->y1,
                      "step-increment", ITEM_HEIGHT + ROW_GAP,
                      "page-increment", box->y2 - box->y1,
                      NULL);
        if (priv->hadjustment) {
            g_object_set (priv->hadjustment,
                          "lower", 0.0,
                          "upper", 0.0,
                          NULL);
        }

        prev_value = nbtk_adjustment_get_value (priv->vadjustment);
        nbtk_adjustment_set_value (priv->vadjustment, prev_value);
    }

    /* Calculate items per row */
    width = alloc_box.x2 - alloc_box.x1;
    items_per_row = (int) (width / (ITEM_WIDTH + COL_GAP));
    if (items_per_row == 0) {
        items_per_row = 1;
    }

    hrn_tiler_set_items_per_row ((HrnTiler *) actor, items_per_row);
}

static void
hrn_view_class_init (HrnViewClass *klass)
{
    GObjectClass *o_class = (GObjectClass *) klass;
    ClutterActorClass *a_class = (ClutterActorClass *) klass;

    o_class->dispose = hrn_view_dispose;
    o_class->finalize = hrn_view_finalize;
    o_class->set_property = hrn_view_set_property;
    o_class->get_property = hrn_view_get_property;

    a_class->paint = hrn_view_paint;
    a_class->pick = hrn_view_pick;
    a_class->apply_transform = hrn_view_apply_transform;
    a_class->get_preferred_width = hrn_view_get_preferred_width;
    a_class->get_preferred_height = hrn_view_get_preferred_height;
    a_class->allocate = hrn_view_allocate;

    g_type_class_add_private (klass, sizeof (HrnViewPrivate));

    g_object_class_override_property (o_class, PROP_HADJUST, "hadjustment");
    g_object_class_override_property (o_class, PROP_VADJUST, "vadjustment");

    signals[ACTIVATED] = g_signal_new ("activated",
                                       G_TYPE_FROM_CLASS (klass),
                                       G_SIGNAL_NO_RECURSE |
                                       G_SIGNAL_RUN_LAST, 0, NULL, NULL,
                                       hrn_marshal_VOID__INT_POINTER,
                                       G_TYPE_NONE, 2, G_TYPE_INT,
                                       G_TYPE_POINTER);
}

#if 0
static void
ensure_children_are_visible (HrnView *view)
{
    HrnViewPrivate *priv = view->priv;
    float x, y;
    ClutterActorBox view_b;

    if (priv->hadjustment) {
        x = nbtk_adjustment_get_value (priv->hadjustment);
    } else {
        x = 0;
    }

    if (priv->vadjustment) {
        y = nbtk_adjustment_get_value (priv->vadjustment);
    } else {
        y = 0;
    }

    clutter_actor_get_allocation_box ((ClutterActor *) view, &view_b);
    view_b.x2 = (view_b.x2 - view_b.x1) + x;
    view_b.x1 = 0;
    view_b.y2 = (view_b.y2 - view_b.y1) + y;
    view_b.y1 = 0;

#if 0
    for (i = 0; i < priv->children->len; i++) {
        ClutterActor *child = priv->children->pdata[i];
        ClutterActorBox child_b;

        /* Check if the child is "on screen" */
        clutter_actor_get_allocation_box (child, &child_b);

        if ((child_b.y1 < view_b.y2)
            && (child_b.y2 > child_b.y1)) {
            clutter_actor_show (child);
        } else {
            clutter_actor_hide (child);
        }
    }
#endif
}
#endif

/* scrollable interface */
static void
adj_value_notify_cb (NbtkAdjustment *adjustment,
                            GParamSpec     *pspec,
                            HrnView        *view)
{
    /* ensure_children_are_visible (view); */
    /* clutter_actor_queue_redraw ((ClutterActor *) view); */
}

static void
scrollable_set_adjustments (NbtkScrollable *scrollable,
                            NbtkAdjustment *hadjustment,
                            NbtkAdjustment *vadjustment)
{
    HrnView *view = (HrnView *) scrollable;
    HrnViewPrivate *priv = view->priv;

    if (hadjustment != priv->hadjustment) {
        if (priv->hadjustment) {
            g_signal_handlers_disconnect_by_func (priv->hadjustment,
                                                  adj_value_notify_cb,
                                                  scrollable);
            g_object_unref (priv->hadjustment);
        }

        if (hadjustment) {
            g_object_ref (hadjustment);
            g_signal_connect (hadjustment, "notify::value",
                              G_CALLBACK (adj_value_notify_cb), scrollable);
        }

        priv->hadjustment = hadjustment;
    }

    if (vadjustment != priv->vadjustment) {
        if (priv->vadjustment) {
            g_signal_handlers_disconnect_by_func (priv->vadjustment,
                                                  adj_value_notify_cb,
                                                  scrollable);
            g_object_unref (priv->vadjustment);
        }

        if (vadjustment) {
            g_object_ref (vadjustment);
            g_signal_connect (vadjustment, "notify::value",
                              G_CALLBACK (adj_value_notify_cb), scrollable);
        }

        priv->vadjustment = vadjustment;
    }
}

static void
scrollable_get_adjustments (NbtkScrollable  *scrollable,
                            NbtkAdjustment **hadjustment,
                            NbtkAdjustment **vadjustment)
{
    HrnView *view = (HrnView *) scrollable;
    HrnViewPrivate *priv = view->priv;
    ClutterActor *actor;

    actor = CLUTTER_ACTOR (scrollable);

    if (hadjustment) {
        if (priv->hadjustment) {
            *hadjustment = priv->hadjustment;
        } else {
            NbtkAdjustment *adj;
            double width, increment;

            width = clutter_actor_get_width (actor);
            increment = MAX (1.0, width);
            adj = nbtk_adjustment_new (0, 0, width, 1.0,
                                       increment, increment);
            scrollable_set_adjustments (scrollable, adj, priv->vadjustment);
            *hadjustment = adj;
        }
    }

    if (vadjustment) {
        if (priv->vadjustment) {
            *vadjustment = priv->vadjustment;
        } else {
            NbtkAdjustment *adj;
            double height, increment;

            height = clutter_actor_get_height (actor);
            increment = MAX (1.0, height);
            adj = nbtk_adjustment_new (0, 0, height, 1.0,
                                       increment, increment);
            scrollable_set_adjustments (scrollable, priv->hadjustment, adj);
            *vadjustment = adj;
        }
    }
}

static void
scrollable_interface_init (NbtkScrollableInterface *iface)
{
    iface->set_adjustments = scrollable_set_adjustments;
    iface->get_adjustments = scrollable_get_adjustments;
}

static void
hide_actor (ClutterAnimation *animation,
            gpointer          data)
{
    clutter_actor_hide ((ClutterActor *) data);
}

static void child_visibility_changed (HrnTileable *tileable,
                                      gboolean     hidden,
                                      HrnView     *view);

static void
hide_child (HrnView      *view,
            ClutterActor *actor)
{
    HrnViewPrivate *priv = view->priv;
    HrnClusterNode *node;

    /* FIXME: Should probably check that the child is onscreen */
    clutter_actor_animate (actor, CLUTTER_EASE_IN_OUT_CUBIC, 500,
                           "opacity", 0x00,
                           "signal::completed", hide_actor, actor,
                           NULL);

    priv->visible_children--;

    node = hrn_tileable_get_node ((HrnTileable *) actor);

    /* Block this as we're already hiding it */
    g_signal_handlers_block_by_func (actor, child_visibility_changed, view);

    /* We want all the children to be hidden, which will mark the parents
       as being hidden */
    hrn_cluster_node_set_children_hidden (node, TRUE);

    g_signal_handlers_unblock_by_func (actor, child_visibility_changed, view);
}

static void
show_child (HrnView      *view,
            ClutterActor *actor)
{
    HrnViewPrivate *priv = view->priv;
    HrnClusterNode *node;

    clutter_actor_show (actor);
    clutter_actor_animate (actor, CLUTTER_EASE_IN_OUT_CUBIC, 500,
                           "opacity", 0xff,
                           NULL);

    priv->visible_children++;

    node = hrn_tileable_get_node ((HrnTileable *) actor);

    g_signal_handlers_block_by_func (actor, child_visibility_changed, view);
    hrn_cluster_node_set_children_hidden (node, FALSE);
    g_signal_handlers_unblock_by_func (actor, child_visibility_changed, view);
}

static void
move_child (ClutterActor *actor,
            int           x,
            int           y)
{
    hrn_tileable_set_position ((HrnTileable *) actor, x, y);

    /* FIXME: Check that we're on screen */
    clutter_actor_animate (actor, CLUTTER_EASE_IN_OUT_CUBIC, 500,
                           "x", X_GRID_TO_COORDS (x),
                           "y", Y_GRID_TO_COORDS (y),
                           NULL);
}

static void
child_activated (HrnTileable      *tileable,
                 HrnTileableAction action,
                 gpointer          payload,
                 HrnView          *view)
{
    HrnViewPrivate *priv = view->priv;
    HrnClusterNode *payload_node;
    GSequenceIter *iter;

    switch (action) {
    case HRN_TILEABLE_ACTION_EXPAND:
        payload_node = hrn_tileable_get_node ((HrnTileable *) payload);

        iter = g_sequence_get_begin_iter (priv->root->children);
        while (g_sequence_iter_is_end (iter) == FALSE) {
            HrnClusterNode *child_node = g_sequence_get (iter);

            if (child_node != payload_node) {
                ClutterActor *child;

                child = g_hash_table_lookup (priv->child_items, child_node);
                hide_child (view, child);
            }

            iter = g_sequence_iter_next (iter);
        }

        priv->shown_child = (HrnTileable *) payload;

        priv->count = hrn_tileable_get_count (tileable);
        move_child ((ClutterActor *) payload, 0, 0);
        hrn_tiler_set_expanded ((HrnTiler *) payload, TRUE);

        g_signal_emit (view, signals[ACTIVATED], 0,
                       HRN_TILEABLE_ACTION_EXPANDED,
                       hrn_tileable_get_node ((HrnTileable *) payload));
        return;

    case HRN_TILEABLE_ACTION_UP_LEVEL:
        return;

    case HRN_TILEABLE_ACTION_SHOW_ALL:
        g_print ("View showing all\n");
        payload_node = hrn_tileable_get_node ((HrnTileable *) payload);
        /* hrn_view_reset (view); */
        /* hrn_tiler_show_all ((HrnTiler *) view); */

        iter = g_sequence_get_begin_iter (priv->root->children);
        while (g_sequence_iter_is_end (iter) == FALSE) {
            HrnClusterNode *child_node = g_sequence_get (iter);
            ClutterActor *child;

            child = g_hash_table_lookup (priv->child_items, child_node);
            if (child == NULL) {
                g_warning ("%s No child found for %s",
                           G_STRLOC, child_node->name);
                iter = g_sequence_iter_next (iter);
                continue;
            }

            if (child_node != payload_node) {
                hrn_tiler_set_expanded ((HrnTiler *) payload, FALSE);
            }

            show_child (view, child);
            iter = g_sequence_iter_next (iter);
        }

        priv->shown_child = NULL;

        g_signal_emit (view, signals[ACTIVATED], 0,
                       HRN_TILEABLE_ACTION_LEVEL_CHANGE, priv->root);
        return;

    default:
        break;
    }

    g_signal_emit (view, signals[ACTIVATED], 0, action, payload);
}

#if 0
static void
child_count_changed (HrnTileable *tileable,
                     int          old_count,
                     int          new_count,
                     HrnView     *view)
{
    HrnViewPrivate *priv = view->priv;

    priv->count -= (old_count - new_count);
    clutter_actor_queue_relayout ((ClutterActor *) view);
}

static guint
layout_view (HrnView *view)
{
#if 0
    HrnViewPrivate *priv = view->priv;
    guint count;
    int i;

    count = 0;
    for (i = 0; i < priv->children->len; i++) {
        ClutterActor *actor = priv->children->pdata[i];
        HrnClusterNode *node;
        int x, y;

        node = hrn_tileable_get_node ((HrnTileable *) actor);
        /* Only need to include the child in the layout calculations
           if it is visible */
        if (CLUTTER_ACTOR_IS_VISIBLE (actor) &&
            (priv->shown_child == NULL ||
             priv->shown_child == (HrnTileable *) actor)) {

            x = count % priv->items_per_row;
            y = count / priv->items_per_row;

            hrn_tileable_set_position ((HrnTileable *) actor, x, y);
            clutter_actor_set_position (actor, X_GRID_TO_COORDS (x),
                                        Y_GRID_TO_COORDS (y));

            count += hrn_tileable_get_count ((HrnTileable *) actor);
        }
    }

    return count;
#endif
    return 1;
}
#endif

static void
child_visibility_changed (HrnTileable *tileable,
                          gboolean     hidden,
                          HrnView     *view)
{
    HrnViewPrivate *priv = view->priv;
    HrnClusterNode *node = hrn_tileable_get_node (tileable);

    if (debug_visible) {
        g_print ("[view] %s %s\n", node->name, hidden ? "hidden" : "shown");
    }

    if (hidden) {
        priv->visible_children--;
        clutter_actor_hide ((ClutterActor *) tileable);
    } else {
        priv->visible_children++;

        clutter_actor_show ((ClutterActor *) tileable);
        clutter_actor_animate ((ClutterActor *) tileable,
                               CLUTTER_EASE_IN_OUT_CUBIC, 500,
                               "opacity", 0xff,
                               NULL);
    }

    clutter_actor_queue_relayout ((ClutterActor *) view);
}

static void
hrn_view_real_add (ClutterContainer *container,
                   ClutterActor     *actor)
{
    HrnView *view = (HrnView *) container;
    HrnViewPrivate *priv = view->priv;

    if (!HRN_IS_TILEABLE (actor)) {
        g_warning ("Cannot add %s to HrnView as it is not implement HrnTileable ",
                   G_OBJECT_TYPE_NAME (actor));
        return;
    }

    clutter_actor_set_parent (actor, (ClutterActor *) container);

    if (HRN_IS_TILER (actor)) {
        hrn_tiler_set_items_per_row ((HrnTiler *) actor, priv->items_per_row);
    }

#if 0
    x = priv->count % priv->items_per_row;
    y = priv->count / priv->items_per_row;

    hrn_tileable_set_position ((HrnTileable *) actor, x, y);
    clutter_actor_set_position (actor, X_GRID_TO_COORDS (x),
                                Y_GRID_TO_COORDS (y));

    g_ptr_array_add (priv->children, actor);
    priv->count += hrn_tileable_get_count ((HrnTileable *) actor);
    g_signal_connect (actor, "activated",
                      G_CALLBACK (child_activated), view);
    g_signal_connect (actor, "count-changed",
                      G_CALLBACK (child_count_changed), view);
    g_signal_connect (actor, "visibility-changed",
                      G_CALLBACK (child_visibility_changed), view);

    node = hrn_tileable_get_node ((HrnTileable *) actor);
    if (node->hidden) {
        clutter_actor_hide (actor);
    } else {
        priv->visible_children++;
    }
#endif
    clutter_actor_queue_relayout ((ClutterActor *) container);
}

static void
hrn_view_real_remove (ClutterContainer *container,
                      ClutterActor     *actor)
{
    g_warning ("FIXME: hrn_view_real_remove");
}

static void
hrn_view_real_foreach (ClutterContainer *container,
                       ClutterCallback   callback,
                       gpointer          userdata)
{
#if 0
    HrnView *view = (HrnView *) container;
    HrnViewPrivate *priv = view->priv;

    g_ptr_array_foreach (priv->children, (GFunc) callback, userdata);
#endif
}

static void
hrn_view_real_raise (ClutterContainer *container,
                     ClutterActor     *actor,
                     ClutterActor     *sibling)
{
}

static void
hrn_view_real_lower (ClutterContainer *container,
                     ClutterActor     *actor,
                     ClutterActor     *sibling)
{
}

static void
hrn_view_real_sort_depth_order (ClutterContainer *container)
{
}

static void
container_interface_init (ClutterContainerIface *iface)
{
    iface->add = hrn_view_real_add;
    iface->remove = hrn_view_real_remove;
    iface->foreach = hrn_view_real_foreach;
    iface->raise = hrn_view_real_raise;
    iface->lower = hrn_view_real_lower;
    iface->sort_depth_order = hrn_view_real_sort_depth_order;
}

static void
tiler_show_all (HrnTiler *tiler)
{
    HrnView *view = (HrnView *) tiler;
    HrnViewPrivate *priv = view->priv;
    GSequenceIter *iter;

    if (priv->root == NULL) {
        return;
    }

    iter = g_sequence_get_begin_iter (priv->root->children);
    while (g_sequence_iter_is_end (iter) == FALSE) {
        HrnClusterNode *child_node = g_sequence_get (iter);
        ClutterActor *child;

        hrn_cluster_node_set_children_hidden (child_node, FALSE);

        child = g_hash_table_lookup (priv->child_items, child_node);
        if (child == NULL) {
            g_warning ("No item for %s", child_node->name);
            iter = g_sequence_iter_next (iter);
            continue;
        }

        if (HRN_IS_TILER (child)) {
            hrn_tiler_set_expanded ((HrnTiler *) child, FALSE);
            /* hrn_tiler_show_all ((HrnTiler *) child); */
        }

        iter = g_sequence_iter_next (iter);
    }

    clutter_actor_queue_relayout ((ClutterActor *) view);
}

static void
tiler_show (HrnTiler    *tiler,
            HrnTileable *tileable)
{
}

static void
tiler_set_expanded (HrnTiler *tiler,
                    gboolean  expanded)
{
}

static void
tiler_set_items_per_row (HrnTiler *tiler,
                         guint     items_per_row)
{
    HrnView *view = HRN_VIEW (tiler);
    HrnViewPrivate *priv = view->priv;

    if (priv->items_per_row == items_per_row) {
        return;
    }

    priv->items_per_row = items_per_row;

    if (priv->root) {
        GSequenceIter *iter = g_sequence_get_begin_iter (priv->root->children);

        while (g_sequence_iter_is_end (iter) == FALSE) {
            HrnClusterNode *child_node = g_sequence_get (iter);
            ClutterActor *child;

            child = g_hash_table_lookup (priv->child_items, child_node);
            if (child == NULL) {
                g_warning ("No child found for %s", child_node->name);
                iter = g_sequence_iter_next (iter);
                continue;
            }

            if (HRN_IS_TILER (child)) {
                hrn_tiler_set_items_per_row ((HrnTiler *) child,
                                             priv->items_per_row);
            }

            iter = g_sequence_iter_next (iter);
        }
    }
}

static void
add_child (HrnTiler       *tiler,
           HrnClusterNode *node)
{
    HrnView *view = (HrnView *) tiler;
    HrnViewPrivate *priv = view->priv;
    ClutterActor *actor;

    switch (node->type) {
    case HRN_CLUSTER_NODE_TYPE_ARTIST:
        actor = g_object_new (HRN_TYPE_ARTIST_CLUSTER, NULL);
        hrn_tiler_add_items ((HrnTiler *) actor, node);
        hrn_cluster_full ((HrnCluster *) actor);

        break;

    case HRN_CLUSTER_NODE_TYPE_YEAR:
        actor = g_object_new (HRN_TYPE_YEAR_CLUSTER, NULL);
        hrn_tiler_add_items ((HrnTiler *) actor, node);
        hrn_cluster_full ((HrnCluster *) actor);
        break;

    case HRN_CLUSTER_NODE_TYPE_VIDEO:
        actor = g_object_new (HRN_TYPE_VIDEO_TILE, "node", node, NULL);
        break;

    default:
        return;
    }

    g_signal_connect (actor, "activated",
                      G_CALLBACK (child_activated), view);
    g_signal_connect (actor, "visibility-changed",
                      G_CALLBACK (child_visibility_changed), view);

    if (priv->shown_child) {
        hrn_cluster_node_set_children_hidden (node, TRUE);
        clutter_actor_hide (actor);
    }

    clutter_container_add_actor ((ClutterContainer *) tiler, actor);
    g_hash_table_insert (priv->child_items, node, actor);
    clutter_actor_queue_redraw ((ClutterActor *) view);
}

static void
child_added_cb (HrnClusterNode *parent,
                HrnClusterNode *child,
                HrnTiler       *tiler)
{
    add_child (tiler, child);
}

static void
child_removed_cb (HrnClusterNode *parent,
                  HrnClusterNode *child,
                  HrnTiler       *tiler)
{
    HrnView *view = (HrnView *) tiler;
    HrnViewPrivate *priv = view->priv;
    ClutterActor *actor;

    actor = g_hash_table_lookup (priv->child_items, child);
    if (actor == NULL) {
        g_warning ("No item found for %s", child->name);
        return;
    }

    g_hash_table_remove (priv->child_items, child);
    clutter_actor_unparent (actor);

    if (priv->shown_child == (HrnTileable *) actor) {
        hrn_tiler_show_all ((HrnTiler *) view);
        priv->shown_child = NULL;
    }

    clutter_actor_queue_redraw ((ClutterActor *) view);
}

static void
tiler_add_items (HrnTiler       *tiler,
                 HrnClusterNode *root)
{
    HrnView *view = (HrnView *) tiler;
    HrnViewPrivate *priv = view->priv;
    GSequenceIter *iter = g_sequence_get_begin_iter (root->children);

    if (priv->child_added_id > 0) {
        g_signal_handler_disconnect (priv->root, priv->child_added_id);
        priv->child_added_id = 0;
    }

    if (priv->child_removed_id > 0) {
        g_signal_handler_disconnect (priv->root, priv->child_removed_id);
        priv->child_removed_id = 0;
    }

    priv->root = root;
    priv->child_added_id = g_signal_connect (root, "child-added",
                                             G_CALLBACK (child_added_cb),
                                             tiler);
    priv->child_removed_id = g_signal_connect (root, "child-removed",
                                               G_CALLBACK (child_removed_cb),
                                               tiler);

    while (g_sequence_iter_is_end (iter) == FALSE) {
        HrnClusterNode *child = g_sequence_get (iter);

        add_child (tiler, child);

        iter = g_sequence_iter_next (iter);
    }
}

static void
tiler_interface_init (HrnTilerInterface *iface)
{
    iface->show_all = tiler_show_all;
    iface->show = tiler_show;
    iface->set_expanded = tiler_set_expanded;
    iface->set_items_per_row = tiler_set_items_per_row;
    iface->add_items = tiler_add_items;
}

static guint
viewer_get_level_count (HrnViewer *viewer)
{
    return LEVEL_COUNT;
}

static void
viewer_interface_init (HrnViewerInterface *iface)
{
    iface->get_level_count = viewer_get_level_count;
}

static void
hrn_view_init (HrnView *self)
{
    HrnViewPrivate *priv = GET_PRIVATE (self);

    self->priv = priv;

    priv->count = 0;
    priv->items_per_row = DEFAULT_ITEMS_PER_ROW;
    priv->scale = DEFAULT_SCALE;

    priv->child_items = g_hash_table_new (NULL, NULL);
}

void
hrn_view_clear (HrnView *view)
{
    HrnViewPrivate *priv = view->priv;
    GSequenceIter *iter;

    if (priv->root == NULL) {
        return;
    }

    priv->shown_child = NULL;
    iter = g_sequence_get_begin_iter (priv->root->children);
    while (g_sequence_iter_is_end (iter) == FALSE) {
        HrnClusterNode *child_node = g_sequence_get (iter);
        ClutterActor *actor;

        actor = g_hash_table_lookup (priv->child_items, child_node);
        if (actor == NULL) {
            g_warning ("No item found for %s", child_node->name);
            iter = g_sequence_iter_next (iter);
            continue;
        }

        clutter_actor_unparent (actor);
        iter = g_sequence_iter_next (iter);
    }
    g_hash_table_remove_all (priv->child_items);
    priv->count = 0;

    clutter_actor_queue_redraw ((ClutterActor *) view);
}

/* FIXME: This is a crap name.
   It "resets" the view to the initial, fully unexpanded toplevel
*/
void
hrn_view_reset (HrnView *view)
{
    HrnViewPrivate *priv = view->priv;
    GSequenceIter *iter;

    if (priv->root == NULL) {
        return;
    }

    priv->shown_child = NULL;

    iter = g_sequence_get_begin_iter (priv->root->children);
    while (g_sequence_iter_is_end (iter) == FALSE) {
        HrnClusterNode *child_node = g_sequence_get (iter);
        ClutterActor *child;

        hrn_cluster_node_set_hidden (child_node, FALSE);

        child = g_hash_table_lookup (priv->child_items, child_node);
        if (child == NULL) {
            g_warning ("No item for %s", child_node->name);
            iter = g_sequence_iter_next (iter);
            continue;
        }

        if (HRN_IS_TILER (child)) {
            hrn_tiler_set_expanded ((HrnTiler *) child, FALSE);
            hrn_tiler_show_all ((HrnTiler *) child);
        }

        iter = g_sequence_iter_next (iter);
    }

    clutter_actor_queue_relayout ((ClutterActor *) view);
}

void
hrn_view_close_shown_item (HrnView *view)
{
    HrnViewPrivate *priv = view->priv;

    if (priv->shown_child && HRN_IS_TILER (priv->shown_child)) {
        hrn_tiler_set_expanded ((HrnTiler *) priv->shown_child, FALSE);
        priv->shown_child = NULL;
    }
}
