#include <glib/gi18n.h>
#include "hrn-cluster-node.h"
#include "hrn-artist-cluster.h"
#include "hrn-tiler.h"

enum {
    PROP_0,
    PROP_NODE,
};

struct _HrnArtistClusterPrivate {
    HrnClusterNode *node;
    guint32 visibility_id;
    guint32 added_id;
    guint32 removed_id;
};

static gboolean debug_visible = FALSE;

#define GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj), HRN_TYPE_ARTIST_CLUSTER, HrnArtistClusterPrivate))
G_DEFINE_TYPE (HrnArtistCluster, hrn_artist_cluster, HRN_TYPE_CLUSTER);

static void
hrn_artist_cluster_finalize (GObject *object)
{
    HrnArtistCluster *artist = (HrnArtistCluster *) object;
    HrnArtistClusterPrivate *priv = artist->priv;

    if (priv->visibility_id) {
        g_signal_handler_disconnect (priv->node, priv->visibility_id);
        priv->visibility_id = 0;
    }

    if (priv->added_id) {
        g_signal_handler_disconnect (priv->node, priv->added_id);
        priv->added_id = 0;
    }

    if (priv->removed_id) {
        g_signal_handler_disconnect (priv->node, priv->removed_id);
        priv->removed_id = 0;
    }

    G_OBJECT_CLASS (hrn_artist_cluster_parent_class)->finalize (object);
}

static void
hrn_artist_cluster_dispose (GObject *object)
{
    G_OBJECT_CLASS (hrn_artist_cluster_parent_class)->dispose (object);
}

static void
node_visibility_changed (HrnClusterNode   *node,
                         gboolean          hidden,
                         HrnArtistCluster *cluster)
{
    if (debug_visible) {
        g_print ("[artist] %s %s\n", node->name, hidden ? "hidden" : "shown");
    }
    hrn_tileable_visibility_changed ((HrnTileable *) cluster, hidden);
}

static void
update_tile (HrnArtistCluster *cluster)
{
    HrnArtistClusterPrivate *priv = cluster->priv;
    char *album_count;
    int count;

    count = g_sequence_get_length (priv->node->children);
    album_count = g_strdup_printf (ngettext ("%d album", "%d albums", count), count);
    g_object_set (cluster,
                  "primary", priv->node->name,
                  "secondary", album_count,
                  NULL);
    g_free (album_count);
}

static void
child_count_changed (HrnClusterNode   *parent,
                     HrnClusterNode   *child,
                     HrnArtistCluster *cluster)
{
    update_tile (cluster);
}

static void
hrn_artist_cluster_set_property (GObject      *object,
                                 guint         prop_id,
                                 const GValue *value,
                                 GParamSpec   *pspec)
{
    HrnArtistCluster *self = (HrnArtistCluster *) object;
    HrnArtistClusterPrivate *priv = self->priv;

    switch (prop_id) {
    case PROP_NODE:
        priv->node = g_value_get_pointer (value);
        priv->visibility_id = g_signal_connect
            (priv->node, "visibility-changed",
             G_CALLBACK (node_visibility_changed), self);
        priv->added_id = g_signal_connect (priv->node, "child-added",
                                           G_CALLBACK (child_count_changed),
                                           self);
        priv->removed_id = g_signal_connect (priv->node, "child-removed",
                                             G_CALLBACK (child_count_changed),
                                             self);
        clutter_actor_set_name ((ClutterActor *) self, priv->node->name);

        update_tile (self);
        break;

    default:
        break;
    }
}

static void
hrn_artist_cluster_get_property (GObject    *object,
                                 guint       prop_id,
                                 GValue     *value,
                                 GParamSpec *pspec)
{
}

static HrnClusterNode *
hrn_artist_cluster_get_node (HrnCluster *cluster)
{
    HrnArtistCluster *artist_cluster = (HrnArtistCluster *) cluster;
    HrnArtistClusterPrivate *priv = artist_cluster->priv;

    return priv->node;
}

static const char *
hrn_artist_cluster_get_label (HrnCluster *cluster)
{
    return _("Queue Albums");
}

static void
hrn_artist_cluster_class_init (HrnArtistClusterClass *klass)
{
    GObjectClass *o_class = (GObjectClass *) klass;
    HrnClusterClass *c_class = (HrnClusterClass *) klass;

    o_class->dispose = hrn_artist_cluster_dispose;
    o_class->finalize = hrn_artist_cluster_finalize;
    o_class->set_property = hrn_artist_cluster_set_property;
    o_class->get_property = hrn_artist_cluster_get_property;

    c_class->get_node = hrn_artist_cluster_get_node;
    c_class->get_label = hrn_artist_cluster_get_label;

    g_type_class_add_private (klass, sizeof (HrnArtistClusterPrivate));

    g_object_class_override_property (o_class, PROP_NODE, "node");
}

static void
hrn_artist_cluster_init (HrnArtistCluster *self)
{
    HrnArtistClusterPrivate *priv = GET_PRIVATE (self);
    self->priv = priv;

    hrn_cluster_set_secondary_reactive ((HrnCluster *) self, FALSE);
}

