
/**********************************************************************
MPEG-4 Audio VM
Audio i/o module


   This software module was originally developed by

    Heiko Purnhagen (University of Hannover / ACTS-MoMuSys)

     and edited by

      in the course of development of the MPEG-2 AAC/MPEG-4 Audio standard
      ISO/IEC 13818-7, 14496-1,2 and 3. This software module is an
      implementation of a part of one or more MPEG-2 AAC/MPEG-4 Audio tools
      as specified by the MPEG-2 AAC/MPEG-4 Audio standard. ISO/IEC gives
      users of the MPEG-2 AAC/MPEG-4 Audio standards free license to this
      software module or modifications thereof for use in hardware or
      software products claiming conformance to the MPEG-2 AAC/ MPEG-4 Audio
      standards. Those intending to use this software module in hardware or
      software products are advised that this use may infringe existing
      patents. The original developer of this software module and his/her
      company, the subsequent editors and their companies, and ISO/IEC have
      no liability for use of this software module or modifications thereof
      in an implementation. Copyright is not released for non MPEG-2
      AAC/MPEG-4 Audio conforming products. The original developer retains
      full right to use the code for his/her own purpose, assign or donate
      the code to a third party and to inhibit third party from using the
      code for non MPEG-2 AAC/MPEG-4 Audio conforming products. This
      copyright notice must be included in all copies or derivative works.

       Copyright (c) 1996, 1999.



	  Source file: audio.c



	     Required libraries:
	     libtsp.a                AFsp audio file library

	      Required modules:
	      common.o                common module
	      austream.o              audio i/o streams (.au format)

	       Authors:
	       HP    Heiko Purnhagen, Uni Hannover <purnhage@tnt.uni-hannover.de>
	       BT    Bodo Teichmann, FhG/IIS <tmn@iis.fhg.de>

		Changes:
		21-jan-97   HP    born (using AFsp-V2R2)
		27-jan-97   HP    set unavailable samples to 0 in AudioReadData()
		03-feb-97   HP    fix bug AudioInit formatString=NULL
		19-feb-97   HP    made internal data structures invisible
		21-feb-97   BT    raw: big-endian
		12-sep-97   HP    fixed numSample bug for mch files in AudioOpenRead()
		30-dec-98   HP    uses austream for stdin/stdout, evaluates USE_AFSP
		07-jan-99   HP    AFsp-v4r1 (AFsp-V3R2 still supported)
		11-jan-99   HP    clipping & seeking for austream module
		17-jan-99   HP    fixed quantisation to 16 bit
		26-jan-99   HP    improved output file format evaluation
		17-may-99   HP    improved output file format detection
		09-aug-00   RS    added int24 support
		31-aug-00   HP    restored AFsp-V3R2 compatibility
		test for "file not found" to avoid abort in AFopenRead()
		16-oct-00   RS/HP made #ifdef {} things nicer ...
		28-may-03   DB    made some preparation for future 48 channel support
**********************************************************************/
/************************************************************************
  * Copyright 2005-2010 by Freescale Semiconductor, Inc.
  * All modifications are confidential and proprietary information
  * of Freescale Semiconductor, Inc.
************************************************************************/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#ifdef USE_AFSP
#include "AFoperation.h"		/* AFsp audio file library - definitions */
#include "libtsp.h"             /* AFsp audio file library */
#endif
#include "audio.h"              /* audio i/o module */
#include "common_m4a.h"         /* common module */


/* ---------- declarations ---------- */

#define SAMPLE_BUF_SIZE 16384   /* local sample buffer size */

#ifndef min
#define min(a,b) ((a) < (b) ? (a) : (b))

#endif
#ifndef max

#define max(a,b) ((a) > (b) ? (a) : (b))
#endif

#ifdef USE_AFSP
#ifdef FW_SUN
/* only AFsp-V3R2 available: map AFsp-V3R2 to AFsp-v4r1 */
#define AFsetNHpar AFsetNH
#define FTW_AU (FW_SUN/256)
#define FTW_WAVE (FW_WAVE/256)
#define FTW_AIFF_C (FW_AIFF_C/256)
#define FTW_NH_EB (FW_NH_EB/256)
#endif
#endif

/*static	char*          output_format     = "raw";
static	unsigned short output_channels   = 1;
static	unsigned short output_wordsize   = 16;
static	unsigned long  output_sampleRate = 48000;*/

/* ---------- declarations (structures) ---------- */

struct AudioFileStruct          /* audio file handle */
{
    AFILE *file;                  /* AFILE handle */
    AFILE *outFile[48];
    /*   NULL if AFsp used */
    int numChannel;               /* number of channels */
    long currentSample;           /* number of samples read/written */
    /* (samples per channel!) */
    int write;                    /* 0=read  1=write */
    long numClip;                 /* number of samples clipped */
    int numFC;                    /* number of front channels */
    int fCenter;                  /* 1 if front center speaker is present*/
    int numSC;                    /* number of side channels */
    int numBC;                    /* number of back channels */
    int bCenter;                  /* 1 if back center speaker is present */
    int numLFE;                   /* number of LFE channels */
    int multiC;                   /* 1 if more than 2 Channels present */
    int numC;                     /* number of Channels for multi channel use */
};


/* ---------- variables ---------- */


/* ---------- local functions ---------- */

static int isfmtstr (char *filename, char *fmtstr)
/* isfmtstr returns true if filename has extension fmtstr */
{
    int i;

    i = strlen(filename)-strlen(fmtstr);
    if (i<0)
	return 0;
    filename += i;
    while (*filename) {
	if (tolower(*filename) != *fmtstr)
	    return 0;
	filename++;
	fmtstr++;
    }
    return 1;
}


/* ---------- functions ---------- */




/* AudioOpenWrite() */
/* Open audio file for writing. */
/* Sample format: 16 bit twos complement, uniform quantisation */
/* Supported file formats: (matching substring of format) */
/*  au, snd:  Sun (AFsp) audio file */
/*  wav:      RIFF WAVE file */
/*  aif:      AIFF-C audio file */
/*  raw:      headerless (raw) audio file (native byte order) */

AudioFile *AudioOpenWrite (
                           char *fileName,              /* in: file name */
			   /*     "-": stdout (only 16 bit .au) */
                           char *format,                        /* in: file format (ignored if stdout) */
			   /*     (au, snd, wav, aif, raw) */
                           int numChannel,              /* in: number of channels */
                           float fSample,               /* in: sampling frequency [Hz] */
                           int int24flag)
			   /* returns: */
			   /*  audio file (handle) */
			   /*  or NULL if error */
{
    AudioFile *file;
    AFILE *af;
    int fmt = 0;
    int AUdebugLevel = 0;    /* debug level */
    struct {
	char *str;
	int fmt;
    } fmtstr[] = {
	{"au",FTW_AU*256},
	{"snd",FTW_AU*256},
	{"wav",FTW_WAVE*256},
	{"wave",FTW_WAVE*256},
	{"aif",FTW_AIFF_C*256},
	{"aiff",FTW_AIFF_C*256},
	{"aifc",FTW_AIFF_C*256},
	{"raw",FTW_NH_EB*256},      /* no header big-endian */
	{NULL,-1}
    };


    if (AUdebugLevel >= 1) {
	printf("AudioOpenWrite: fileName=\"%s\"  format=\"%s\"\n",fileName,format);
	printf("AudioOpenWrite: numChannel=%d  fSample=%.1f\n",
	    numChannel,fSample);
    }

    if (strcmp(fileName,"-")) {
	int fmti = 0;
	while (fmtstr[fmti].str && !isfmtstr(format,fmtstr[fmti].str)) {
	    fmti++;
	}
#ifdef USE_AFSP
	if (fmtstr[fmti].str) {
	    if (int24flag) {
#ifdef FW_SUN
		/* only AFsp-V3R2 available */
		CommonWarning("AudioOpenWrite: AFsp-V3R2 has no INT24 support");
		return NULL;
#else
		fmt = FD_INT24 + fmtstr[fmti].fmt;
#endif
	    }
	    else {
		fmt = FD_INT16 + fmtstr[fmti].fmt;
	    }
	}
#endif /*USE_AFSP*/
	if (!fmtstr[fmti].str) {
	    CommonWarning("AudioOpenWrite: unkown audio file format \"%s\"", format);
	    return (AudioFile*)NULL;
	}
    }

    if ((file=(AudioFile*)malloc(sizeof(AudioFile))) == NULL)
	CommonExit(1,"AudioOpenWrite: memory allocation error");

    af = AFopenWrite(fileName,fmt,numChannel,fSample,
	AUdebugLevel?stdout:(FILE*)NULL);

    if (af==NULL) {
	CommonWarning("AudioOpenWrite: error opening audio file %s",fileName);
	free(file);
	return (AudioFile*)NULL;
    }

    file->file = af;
    file->numChannel = numChannel;
    file->currentSample = 0;
    file->write = 1;
    file->numClip = 0;
    file->multiC = 0;

    return file;
}


AudioFile *AudioOpenWriteMC_BSAC (char *fileName, /* in: file name */
                                  /* "-": stdout (only 16 bit .au) */
                                  char *format,   /* in: file format (ignored if stdout) */
                                  /* (au, snd, wav, aif, raw) */
                                  float fSample,  /* in: sampling frequency [Hz] */
                                  int int24flag,
                                  int numFC,
                                  int fCenter,
                                  int numSC,
                                  int bCenter,
                                  int numBC,
                                  int numLFE)
				  /* returns: */
				  /*  audio file (handle) */
				  /*  or NULL if error */
{
    int i, j=0, lenBaseName;
    char extendedFileName[1024]={0};    /*new name of outfile*/
    AudioFile *file;
    int numberOfChannels = (numFC + numSC + numBC + numLFE);
    AFILE *af;
    int fmt = 0;
    int AUdebugLevel = 0;    /* debug level */
    struct {
	char *str;
	int fmt;
    } fmtstr[] = {
	{"au",FTW_AU*256},
	{"snd",FTW_AU*256},
	{"wav",FTW_WAVE*256},
	{"wave",FTW_WAVE*256},
	{"aif",FTW_AIFF_C*256},
	{"aiff",FTW_AIFF_C*256},
	{"aifc",FTW_AIFF_C*256},
	{"raw",FTW_NH_EB*256},      /* no header big-endian */
	{NULL,-1}
    };


    if (AUdebugLevel >= 1) {
	printf("AudioOpenWrite: fileName=\"%s\"  format=\"%s\"\n",fileName,format);
    }

    if (strcmp(fileName,"-")) {
	int fmti = 0;
	while (fmtstr[fmti].str && !isfmtstr(format,fmtstr[fmti].str)) {
	    fmti++;
	}
#ifdef USE_AFSP
	if (fmtstr[fmti].str) {
	    if (int24flag) {
#ifdef FW_SUN
		/* only AFsp-V3R2 available */
		CommonWarning("AudioOpenWrite: AFsp-V3R2 has no INT24 support");
		return NULL;
#else
		fmt = FD_INT24 + fmtstr[fmti].fmt;
#endif
	    } else {
		fmt = FD_INT16 + fmtstr[fmti].fmt;
	    }
	}
#endif /*USE_AFSP*/
	if (!fmtstr[fmti].str) {
	    CommonWarning("AudioOpenWrite: unkown audio file format \"%s\"", format);
	    return (AudioFile*)NULL;
	}
    }

    if ((file=(AudioFile*)malloc(sizeof(AudioFile))) == NULL)
	CommonExit(1,"AudioOpenWrite: memory allocation error");

    lenBaseName = strlen(fileName)-strlen(format);
    strncpy(extendedFileName, fileName, lenBaseName);


    /* preparing files for front channels */
    /* ================================== */
    for (i=((fCenter==1) ? 0 : 1); i<numFC + (fCenter?0:1); i++) {
	sprintf(extendedFileName+lenBaseName, "_%c%02d%s", 'f', i, format);

	af = AFopenWrite( extendedFileName, fmt, 1, fSample,
	    AUdebugLevel?stdout:(FILE*)NULL);
	if (af==NULL) {
	    CommonWarning("AudioOpenWrite: error opening audio file %s", extendedFileName);
	    free(file);
	    return (AudioFile*)NULL;
	}
#ifdef USE_AFSP
	file->outFile[j] = af;
#else
	file->outStream[j] = as;
#endif
	j++;
    }

    /* preparing files for LFE channels */
    /* ================================ */
    for (i = 0; i<numLFE; i++) {
	sprintf(extendedFileName+lenBaseName, "_%c%02d%s", 'l', i, format);

	af = AFopenWrite( extendedFileName, fmt, 1, fSample,
	    AUdebugLevel?stdout:(FILE*)NULL);
	if (af==NULL) {
	    CommonWarning("AudioOpenWrite: error opening audio file %s", extendedFileName);
	    free(file);
	    return (AudioFile*)NULL;
	}
#ifdef USE_AFSP
	file->outFile[j] = af;
#else
	file->outStream[j] = as;
#endif
	j++;
    }

    /* preparing files for back channels */
    /* ================================= */
    for (i=0; i<numBC; i++) {
	sprintf(extendedFileName+lenBaseName, "_%c%02d%s", 'b', i, format);

	af = AFopenWrite( extendedFileName, fmt, 1, fSample,
	    AUdebugLevel?stdout:(FILE*)NULL);

	if (af==NULL) {
	    CommonWarning("AudioOpenWrite: error opening audio file %s", extendedFileName);
	    free(file);
	    return (AudioFile*)NULL;
	}
#ifdef USE_AFSP
	file->outFile[j] = af;
#else
	file->outStream[j] = as;
#endif
	j++;
    }

    /* preparing files for side channels */
    /* ================================= */
    for (i = 0; i<numSC; i++) {
	sprintf(extendedFileName+lenBaseName, "_%c%02d%s", 's', i, format);

	af = AFopenWrite( extendedFileName, fmt, 1, fSample,
	    AUdebugLevel?stdout:(FILE*)NULL);
	if ( af==NULL) {
	    CommonWarning("AudioOpenWrite: error opening audio file %s", extendedFileName);
	    free(file);
	    return (AudioFile*)NULL;
	}
#ifdef USE_AFSP
	file->outFile[j] = af;
#else
	file->outStream[j] = as;
#endif
	j++;
    }

    file->numChannel = 1;
    file->currentSample = 0;
    file->write = 1;
    file->numClip = 0;
    file->numFC = numFC;
    file->fCenter = fCenter;
    file->numSC = numSC;
    file->bCenter = bCenter;
    file->numBC = numBC/*11*/;
    file->numLFE = numLFE;
    file->numC = numberOfChannels;
    file->multiC = 1;

    return file;

}



/* AudioWriteData() */
/* Write data to audio file. */

void AudioWriteData (
                     AudioFile *file,           /* in: audio file (handle) */
                     float *data,                      /* in: data[channel][sample] */
		     /*     (range [-32768 .. 32767]) */
                     long numSample)            /* in: number of samples to be written */
		     /*     (samples per channel!) */
{
    long tot,cur,num=0;
    long i;
    int j=1;
    int AUdebugLevel = 0;    /* debug level */

#ifdef USE_AFSP
    float buf[SAMPLE_BUF_SIZE];
#else
    short bufs[SAMPLE_BUF_SIZE];
    long numClip,tmp;
#endif
    if (AUdebugLevel >= 2)
	printf("AudioWriteData: numSample=%ld (currentSample=%ld)\n",
	numSample,file->currentSample);

    if (file->write != 1)
	CommonExit(1,"AudioWriteData: audio file not in write mode");

    tot = file->numChannel*numSample;
    cur = max(0,-file->numChannel*file->currentSample);
    while (cur < tot) {
	num = min(tot-cur,SAMPLE_BUF_SIZE);
	if(file->multiC){
	    for(j=0; j<file->numC; j++) {
		if (file->outFile[j]) {
		    for (i=0; i<num; i++)
			buf[i] = data[j*num + (cur+i)];
		    AFwriteData(file->outFile[j],buf,num);
		}
	    }
	} else {
	    if (file->file) {
		for (i=0; i<num; i++)
		    buf[i] = data[((cur+i)%file->numChannel)*( num/file->numChannel) + (cur+i)/file->numChannel];
		AFwriteData(file->file,buf,num);
	    }
	}

	cur += num;
    }

    file->currentSample += tot/file->numChannel;
}



/* AudioClose() */
/* Close audio file.*/

void AudioClose (
                 AudioFile *file)               /* in: audio file (handle) */
{
    int j;
    int AUdebugLevel = 0;    /* debug level */
    if (AUdebugLevel >= 1)
	printf("AudioClose: (currentSample=%ld)\n",file->currentSample);

    if (file->numClip)
	CommonWarning("AudioClose: %ld samples clipped",file->numClip);


    if(file->multiC) {
	for(j=0; j<file->numC; j++) {
	    if (file->outFile[j]) {
		AFclose(file->outFile[j]);
		/*free(file->outFile[j]);*/ /* SAMSUNG_2005-09-30 */
	    }
	}
    } else {
	if (file->file)
	    AFclose(file->file);
    }


  free(file);
}



/* end of audio.c */
