/* $Id: main.c,v 1.59 2009-01-27 16:27:03 potyra Exp $ 
 *
 * Copyright (C) 2005-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <inttypes.h>
#include <assert.h>
#include <stdbool.h>

#include "patternm.h"
#include "sig_match.h"
#include "sig_opt_rgb.h"
#include "sig_boolean.h"
#include "sig_string.h"
#include "glue-png.h"
#include "glue-shm.h"
#include "glue-log.h"

#define COMP	"test-patternm"

const char *progname;
const char *basedir = ".";

struct cpssp {
	struct sig_opt_rgb* video_out;
	struct sig_match* match_inout[16];
	/* first 4 are connected to patternm_text,
	 * remaining 4 to patternm_asc
	 */
	struct {
		struct sig_boolean* result_in;
		struct sig_string* text_out;
		char string[256];
	} text_pm[8];
	struct {
		struct sig_string *pattern_rectangle;
		struct sig_integer *match_rectangle_x;
		struct sig_integer *match_rectangle_y;
		struct sig_integer *match_rectangle_w;
		struct sig_integer *match_rectangle_h;
	} gfx[4];
};

void
time_stop(void)
{
}

void
time_cont(void)
{
}

static void
test_patternm_text_result(void *s, unsigned int val)
{
	const char *str = (const char*) s;

	if (val) {
		faum_log(FAUM_LOG_DEBUG, "test_patternm", "",
			"\"%s\" is visible\n", str);
	} else {
		faum_log(FAUM_LOG_DEBUG, "test_patternm", "",
			"\"%s\" is not visible\n", str);
	}
}

static void
test_patternm_gfx0_x_set(void *_cpssp, int val)
{
	printf("0 x: %d\n", val);
}

static void
test_patternm_gfx0_y_set(void *_cpssp, int val)
{
	printf("0 y: %d\n", val);
}

static void
test_patternm_gfx0_w_set(void *_cpssp, int val)
{
	printf("0 w: %d\n", val);
}

static void
test_patternm_gfx0_h_set(void *_cpssp, int val)
{
	printf("0 h: %d\n", val);
}

static void
test_patternm_opt_out(struct sig_opt_rgb* video_out, const char* picture)
{
	uint32_t *shot;
	uint32_t *walk;
	int x;
	int y;
	int w;
	int h;
	bool ret;

	shot = NULL;
	w = 0;
	h = 0;
	ret = png_read(&shot, &w, &h, picture);
	if (! ret) {
		faum_log(FAUM_LOG_ERROR, COMP, "",
			"error loading screenshot %s\n", picture);
		return;
	}

	walk = shot;
	sig_opt_rgb_size_set(video_out, NULL, w, h);
	for (y = 0; y < h; y++) {
		for (x = 0; x < w; x++) {
			sig_opt_rgb_pixel_set(video_out, NULL,
						x, y,
						GET_RED(*walk),
						GET_GREEN(*walk),
						GET_BLUE(*walk));
			walk++;
		}
	}
	free(shot);

	/* need to have 5 syncs in total, until pattern matcher triggers */
	sig_opt_rgb_sync(video_out, NULL);
	sig_opt_rgb_sync(video_out, NULL);
	sig_opt_rgb_sync(video_out, NULL);
	sig_opt_rgb_sync(video_out, NULL);
	sig_opt_rgb_sync(video_out, NULL);
}

static void
test_patternm_find_pic_in(const char* pic, const char* screenshot)
{
	struct cpssp *cpssp;

	cpssp = shm_map(COMP, 0, sizeof(struct cpssp), 0);

	sig_string_set(cpssp->gfx[0].pattern_rectangle, cpssp, pic);
	test_patternm_opt_out(cpssp->video_out, screenshot);
	sig_string_set(cpssp->gfx[0].pattern_rectangle, cpssp, "");

	shm_unmap(cpssp, 0);
}

static void
test_patternm_find_text(const char* text, const char* screenshot, int ind)
{
	struct cpssp *cpssp;

	cpssp = shm_map(COMP, 0, sizeof(struct cpssp), 0);

	strncpy(cpssp->text_pm[ind].string, text, 
		sizeof(cpssp->text_pm[ind].string) - 1);

	sig_string_set(cpssp->text_pm[ind].text_out, 
			cpssp->text_pm[ind].string, text);
	test_patternm_opt_out(cpssp->video_out, screenshot);
	sig_string_set(cpssp->text_pm[ind].text_out, 
			cpssp->text_pm[ind].string, "");

	shm_unmap(cpssp, 0);
}

static void
test_patternm_init(unsigned int nr)
{
	static const struct sig_boolean_funcs sf = {
		.set = test_patternm_text_result,
	};
	static const struct sig_integer_funcs gfx0_x = {
		.set = test_patternm_gfx0_x_set,
	};
	static const struct sig_integer_funcs gfx0_y = {
		.set = test_patternm_gfx0_y_set,
	};
	static const struct sig_integer_funcs gfx0_w = {
		.set = test_patternm_gfx0_w_set,
	};
	static const struct sig_integer_funcs gfx0_h = {
		.set = test_patternm_gfx0_h_set,
	};
	struct cpssp *cpssp;
	int i;

	cpssp = shm_map(COMP, nr, sizeof(struct cpssp), 0);
	assert(cpssp != NULL);

	/* setup signals */
	cpssp->video_out = sig_opt_rgb_init("sig_opt_rgb", nr);

	for (i = 0; i < 16; i++) {
		cpssp->match_inout[i] = sig_match_init("sig_match", i);
	}

	for (i = 0; i < 8; i++) {
		cpssp->text_pm[i].result_in = sig_boolean_init("sig_boolean", i);
		cpssp->text_pm[i].text_out = sig_string_init("sig_string", i);
	}

	cpssp->gfx[0].pattern_rectangle = sig_string_init(COMP "-pr0", nr);
	cpssp->gfx[0].match_rectangle_x = sig_integer_init(COMP "-mrx0", nr);
	cpssp->gfx[0].match_rectangle_y = sig_integer_init(COMP "-mry0", nr);
	cpssp->gfx[0].match_rectangle_w = sig_integer_init(COMP "-mrw0", nr);
	cpssp->gfx[0].match_rectangle_h = sig_integer_init(COMP "-mrh0", nr);
	cpssp->gfx[1].pattern_rectangle = sig_string_init(COMP "-pr1", nr);
	cpssp->gfx[1].match_rectangle_x = sig_integer_init(COMP "-mrx1", nr);
	cpssp->gfx[1].match_rectangle_y = sig_integer_init(COMP "-mry1", nr);
	cpssp->gfx[1].match_rectangle_w = sig_integer_init(COMP "-mrw1", nr);
	cpssp->gfx[1].match_rectangle_h = sig_integer_init(COMP "-mrh1", nr);
	cpssp->gfx[2].pattern_rectangle = sig_string_init(COMP "-pr2", nr);
	cpssp->gfx[2].match_rectangle_x = sig_integer_init(COMP "-mrx2", nr);
	cpssp->gfx[2].match_rectangle_y = sig_integer_init(COMP "-mry2", nr);
	cpssp->gfx[2].match_rectangle_w = sig_integer_init(COMP "-mrw2", nr);
	cpssp->gfx[2].match_rectangle_h = sig_integer_init(COMP "-mrh2", nr);
	cpssp->gfx[3].pattern_rectangle = sig_string_init(COMP "-pr3", nr);
	cpssp->gfx[3].match_rectangle_x = sig_integer_init(COMP "-mrx3", nr);
	cpssp->gfx[3].match_rectangle_y = sig_integer_init(COMP "-mry3", nr);
	cpssp->gfx[3].match_rectangle_w = sig_integer_init(COMP "-mrw3", nr);
	cpssp->gfx[3].match_rectangle_h = sig_integer_init(COMP "-mrh3", nr);

	patternm_init(0,
			cpssp->video_out, 
			cpssp->match_inout[0],
			cpssp->match_inout[1],
			cpssp->match_inout[2],
			cpssp->match_inout[3],
			cpssp->match_inout[4],
			cpssp->match_inout[5],
			cpssp->match_inout[6],
			cpssp->match_inout[7],
			cpssp->match_inout[8],
			cpssp->match_inout[9],
			cpssp->match_inout[10],
			cpssp->match_inout[11],
			cpssp->match_inout[12],
			cpssp->match_inout[13],
			cpssp->match_inout[14],
			cpssp->match_inout[15],
			cpssp->text_pm[0].text_out,
			cpssp->text_pm[0].result_in,
			cpssp->text_pm[1].text_out,
			cpssp->text_pm[1].result_in,
			cpssp->text_pm[2].text_out,
			cpssp->text_pm[2].result_in,
			cpssp->text_pm[3].text_out,
			cpssp->text_pm[3].result_in,
			cpssp->text_pm[4].text_out,
			cpssp->text_pm[4].result_in,
			cpssp->text_pm[5].text_out,
			cpssp->text_pm[5].result_in,
			cpssp->text_pm[6].text_out,
			cpssp->text_pm[6].result_in,
			cpssp->text_pm[7].text_out,
			cpssp->text_pm[7].result_in,
			cpssp->gfx[0].pattern_rectangle,
			cpssp->gfx[0].match_rectangle_x,
			cpssp->gfx[0].match_rectangle_y,
			cpssp->gfx[0].match_rectangle_w,
			cpssp->gfx[0].match_rectangle_h,
			cpssp->gfx[1].pattern_rectangle,
			cpssp->gfx[1].match_rectangle_x,
			cpssp->gfx[1].match_rectangle_y,
			cpssp->gfx[1].match_rectangle_w,
			cpssp->gfx[1].match_rectangle_h,
			cpssp->gfx[2].pattern_rectangle,
			cpssp->gfx[2].match_rectangle_x,
			cpssp->gfx[2].match_rectangle_y,
			cpssp->gfx[2].match_rectangle_w,
			cpssp->gfx[2].match_rectangle_h,
			cpssp->gfx[3].pattern_rectangle,
			cpssp->gfx[3].match_rectangle_x,
			cpssp->gfx[3].match_rectangle_y,
			cpssp->gfx[3].match_rectangle_w,
			cpssp->gfx[3].match_rectangle_h
	);

	/* text/asc signals */
	for (i = 0; i < 8; i++) {
		sig_boolean_connect_in(cpssp->text_pm[i].result_in,
				cpssp->text_pm[i].string, &sf);
	}

	/* gfx signals */
	sig_integer_connect_in(cpssp->gfx[0].match_rectangle_x, cpssp, &gfx0_x);
	sig_integer_connect_in(cpssp->gfx[0].match_rectangle_y, cpssp, &gfx0_y);
	sig_integer_connect_in(cpssp->gfx[0].match_rectangle_w, cpssp, &gfx0_w);
	sig_integer_connect_in(cpssp->gfx[0].match_rectangle_h, cpssp, &gfx0_h);
}

static void
test_patternm_create(const char *dir, unsigned int nr)
{
	int i;
	int ret;

	ret = shm_create(COMP, nr, sizeof(struct cpssp));
	assert(ret == 0);

	sig_opt_rgb_create("sig_opt_rgb", nr);
	for (i = 0; i < 16; i++) {
		sig_match_create("sig_match", i);
	}

	for (i = 0; i < 8; i++) {
		sig_boolean_create("sig_boolean", i);
		sig_string_create("sig_string", i);
	}

	sig_string_create(COMP "-pr0", nr);
	sig_integer_create(COMP "-mrx0", nr);
	sig_integer_create(COMP "-mry0", nr);
	sig_integer_create(COMP "-mrw0", nr);
	sig_integer_create(COMP "-mrh0", nr);
	sig_string_create(COMP "-pr1", nr);
	sig_integer_create(COMP "-mrx1", nr);
	sig_integer_create(COMP "-mry1", nr);
	sig_integer_create(COMP "-mrw1", nr);
	sig_integer_create(COMP "-mrh1", nr);
	sig_string_create(COMP "-pr2", nr);
	sig_integer_create(COMP "-mrx2", nr);
	sig_integer_create(COMP "-mry2", nr);
	sig_integer_create(COMP "-mrw2", nr);
	sig_integer_create(COMP "-mrh2", nr);
	sig_string_create(COMP "-pr3", nr);
	sig_integer_create(COMP "-mrx3", nr);
	sig_integer_create(COMP "-mry3", nr);
	sig_integer_create(COMP "-mrw3", nr);
	sig_integer_create(COMP "-mrh3", nr);

	/* patternm component */
	patternm_create(0, "test");
}

static void
test_patternm_destroy(unsigned int nr)
{
	int i;

	assert(nr == 0);
	patternm_destroy(nr);

	sig_opt_rgb_destroy("sig_opt_rgb", nr);
	for (i = 0; i < 16; i++) {
		sig_match_destroy("sig_match", i);
	}

	for (i = 0; i < 8; i++) {
		sig_boolean_destroy("sig_boolean", i);
		sig_string_destroy("sig_string", i);
	}

	sig_string_destroy(COMP "-pr0", nr);
	sig_integer_destroy(COMP "-mrx0", nr);
	sig_integer_destroy(COMP "-mry0", nr);
	sig_integer_destroy(COMP "-mrw0", nr);
	sig_integer_destroy(COMP "-mrh0", nr);
	sig_string_destroy(COMP "-pr1", nr);
	sig_integer_destroy(COMP "-mrx1", nr);
	sig_integer_destroy(COMP "-mry1", nr);
	sig_integer_destroy(COMP "-mrw1", nr);
	sig_integer_destroy(COMP "-mrh1", nr);
	sig_string_destroy(COMP "-pr2", nr);
	sig_integer_destroy(COMP "-mrx2", nr);
	sig_integer_destroy(COMP "-mry2", nr);
	sig_integer_destroy(COMP "-mrw2", nr);
	sig_integer_destroy(COMP "-mrh2", nr);
	sig_string_destroy(COMP "-pr3", nr);
	sig_integer_destroy(COMP "-mrx3", nr);
	sig_integer_destroy(COMP "-mry3", nr);
	sig_integer_destroy(COMP "-mrw3", nr);
	sig_integer_destroy(COMP "-mrh3", nr);

	shm_destroy(COMP, 0);
}

static void
test_patternm_main_screenshot(const char* pic, const char* screenshot)
{
	test_patternm_create(".", 0);
	test_patternm_init(0);
	test_patternm_find_pic_in(pic, screenshot);
	test_patternm_destroy(0);
}

static void
test_patternm_main_text(const char* text, const char* screenshot, int ind)
{
	test_patternm_create(".", 0);
	test_patternm_init(0);
	test_patternm_find_text(text, screenshot, ind);
	test_patternm_destroy(0);
}

static __attribute__((__noreturn__)) void
usage(int retval)
{
	fprintf(stderr, "usage: %s [args] <screenshot>\n", progname);
	fprintf(stderr, "       --check <text> check for <text> in "
						"<screenshot>\n");
	
	fprintf(stderr, "       --asc   <text> check for <text> in "
				"<screenshot> with. ascii patternm\n");
	fprintf(stderr, "       --find <picture>: check for picture\n");
	fprintf(stderr, "               location in <screenshot>\n");

	exit(retval);
}

int
main(int argc, char** argv)
{
	const char *pic = NULL;
	const char *screenshot = NULL;
	const char *text = NULL;
	enum { 
		MODE_ASC,
		MODE_TEXT,
		MODE_PIC
	} mode = MODE_ASC;

	/* Get program name. */
	progname = argv[0];
	argc--;
	argv++;

	/* Get options. */
	while (argv[0] != NULL) {
		if (! strcmp(argv[0], "--find")) {
			argc--;
			argv++;
			if (argv[0] == NULL) {
				usage(EXIT_FAILURE);
			}
			pic = argv[0];
			mode = MODE_PIC;

		} else if (! strcmp(argv[0], "--check")) {
			argc--;
			argv++;
			if (screenshot) {
				usage(EXIT_FAILURE);
			}
			text = argv[0];
			mode = MODE_TEXT;

		} else if (! strcmp(argv[0], "--asc")) {
			argc--;
			argv++;
			if (screenshot) {
				usage(EXIT_FAILURE);
			}
			text = argv[0];
			mode = MODE_ASC;

		} else {
			if (screenshot) {
				usage(EXIT_FAILURE);
			}
			screenshot = argv[0];
		}
		argv++;
		argc--;
	}

	if (screenshot == NULL) {
		usage(EXIT_FAILURE);
	}
	if (text == NULL
	 && mode != MODE_PIC) {
		usage(EXIT_FAILURE);
	}

	switch (mode) {
	case MODE_TEXT:
		test_patternm_main_text(text, screenshot, 0);
		break;
	case MODE_ASC:
		test_patternm_main_text(text, screenshot, 4);
		break;
	case MODE_PIC:
		test_patternm_main_screenshot(pic, screenshot);
		break;
	default:
		assert(0);
	}

	return 0;
}
