# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.
#
# Author: Philippe Normand <philippe@fluendo.com>

from elisa.core.utils.i18n import install_translation

from elisa.plugins.poblesec.actions import LinkAction, Action
from elisa.plugins.poblesec.actions import GoToNode, GoToAction, ActionNodesNotSetError

from elisa.plugins.flickr.utils import get_flickr_provider

from twisted.internet import defer

_ = install_translation('flickr')
_poblesec = install_translation('poblesec')

class PicturesGoToAction(GoToAction):
    """ This action will clear the poblesec history, goto the pictures
    menu and proceed with other goto nodes same way L{GoToAction}
    does.
    """

    def run(self):
        if not self.nodes:
            raise ActionNodesNotSetError()

        browser = self.controller.frontend.retrieve_controllers('/poblesec/browser')[0]
        browser.history.clear()

        # Select the pictures item in Elisa's main menu
        for i, link in enumerate(browser.history.current.model):
            if link.controller_path == '/poblesec/pictures_menu':
                browser.history.current.menu.selected_item_index = i
                break

        def _append_controller(result, *args, **kwargs):
            return browser.history.append_controller(*args, **kwargs)

        dfr = defer.succeed(True)
        for node in self.nodes:
            dfr.addCallback(_append_controller, node.path, node.title, **node.args)

        return dfr


class LoginAction(LinkAction):
    """ L{LinkAction} leading to the Flickr login form. Once the form has
    been validated, ask the user to go to his Flickr account.
    """

    title = _('Login to Flickr')
    path = '/poblesec/settings/flickr/username'
    icon = 'elisa.plugins.poblesec.key_icon'

    def __init__(self, controller):
        super(LoginAction, self).__init__(controller)

        login_actions = []

        flickr_section_action = PicturesGoToAction(controller)
        flickr_section_action.label = _("Go To My Flickr Account")

        new_history = [('/poblesec/pictures_menu', _poblesec('Pictures')),
                       ('/poblesec/pictures/internet', _poblesec('Internet')),
                       ('/poblesec/flickr/home', _('Flickr')),
                       ('/poblesec/flickr/account', _('My Flickr Account')),
                       ]
        for path, title in new_history:
            goto_node = GoToNode()
            goto_node.path = path
            goto_node.title = title
            flickr_section_action.nodes.append(goto_node)

        login_actions.append(flickr_section_action)

        self.args = {'service': 'Flickr',
                     'password_controller': '/poblesec/settings/flickr/password',
                     'login_result_controller': '/poblesec/settings/flickr/login_result',
                     'try_login': get_flickr_provider().login,
                     'login_actions': login_actions}

class LogoutAction(Action):
    """ Action used to clear the Flickr resource_provider
    authentication token and go back to the previous menu. This action
    is called from the 'My Account' menu.
    """

    title = _('Logout From Flickr')
    icon = 'elisa.plugins.poblesec.remove'

    def run(self):
        get_flickr_provider().logout()

        browser = self.controller.frontend.retrieve_controllers('/poblesec/browser')[0]
        browser.history.go_back()

        return defer.succeed(True)

class MyAccountAction(LinkAction):
    """ Action leading to the 'My Account' menu.
    """
    icon = 'elisa.plugins.poblesec.user_account'
    title = _('My Flickr Account')
    path = '/poblesec/flickr/account'
