# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.
#
# Author: Olivier Tilloy <olivier@fluendo.com>

"""
A standard loading (spinning) animation widget.
"""

from elisa.plugins.pigment.graph.image import Image
from elisa.plugins.pigment.widgets.theme import Theme

import pgm
from pgm.timing import implicit

import math

from twisted.internet import reactor


class LoadingAnimation(Image):

    """
    A standard loading (spinning) animation widget.
    """

    def __init__(self, delay=0.017,
                 resource='elisa.plugins.poblesec.scanning_icon'):
        """
        Constructor.

        @param delay:    the delay in seconds between two animation updates
        @type delay:     C{float}
        @param resource: the theme image resource to use as an animation
        @type resource:  C{str}
        """
        super(LoadingAnimation, self).__init__()

        self.delay = delay
        self.bg_a = 0

        self.animated = implicit.AnimatedObject(self)

        self.layout = pgm.IMAGE_SCALED
        theme = Theme.get_default()
        image_file = theme.get_resource(resource)
        self.set_from_file(image_file)

        self._rotation_matrix = pgm.mat4x4_new_identity()
        self._rotation_matrix.translate(0.5, 0.5, 0.0)
        self._rotation_matrix.rotate_z(-math.pi / 30.0)
        self._rotation_matrix.translate(-0.5, -0.5, 0.0)

        self._loop_call = None

    def clean(self):
        super(LoadingAnimation, self).clean()
        self._stopping = True

    def activate(self, active):
        """
        Start/stop the animation.
        This does not change the visibility of the widget.

        @param active: whether to start or stop the animation
        @type active:  C{bool}
        """
        if active:
            if (self._loop_call is None) or self._stopping:
                self._stopping = False
                self._loop_call = self._transform_mapping_matrix_cb()
        else:
            self._stopping = True

    def _transform_mapping_matrix_cb(self):
        if (self.opacity > 0) and self.visible and not self._stopping:
            # Only do the animation if visible
            self.mapping_matrix *= self._rotation_matrix
            self._loop_call = \
               reactor.callLater(self.delay, self._transform_mapping_matrix_cb)
