# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.


from cssutils import CSSParser
from elisa.plugins.pigment.widgets import const
from elisa.plugins.pigment.widgets.style import Style

from pkg_resources import resource_filename
import pkg_resources

import logging
import copy
import pprint
import os, os.path

_logger = logging.getLogger("CSSUTILS")
_logger.setLevel(logging.ERROR)

_DEFAULT_THEME = None


class Theme(object):
    """
    A theme object, that adds styles to widgets properties and to stock resources.

    A basic theme will be built from the default configuration files, providing
    necessary style information for the widgets: without that, widgets won't
    work.

    @ivar widget_styles: the styles for the widgets, for each state
    @type widget_styles: dict of states to L{elisa.plugins.pigment.widgets.Style}s
    @ivar stock_resources: the map of resource names to file paths
    @type stock_resources: a dict strings to strings
    @ivar fallback_themes: a dictionary of plugin names to Themes, cashing
                           information necessary to do the right fallback
                           for missing resources
    @type fallback_themes: dictionary of strings to
                            L{elisa.core.components.theme.Theme}
    """

    default_theme = None

    def __init__(self, module_name=None, styles_conf=None, resources_conf=None):
        """
        DOCME
        """
        self.widget_styles = {}
        self.stock_resources = {}
        self.fallback_themes = {}

        if styles_conf:
            self._init_styles(styles_conf)
        if module_name and resources_conf:
            self._init_resources(module_name, resources_conf)

    def _init_styles(self, styles_conf):
        """
        Read the configuration file and fill up the widgets styles.
        """
        tmp = Theme()
        parser = CSSParser()
        try:
            # new api in cssutils 0.9.5b3
            cfg = parser.parseFile(styles_conf)
        except AttributeError:
            cfg = parser.parse(styles_conf)

        for rule in cfg.cssRules:
            if rule.selectorText == 'defaults':
                pass
            else:
                if ':' in rule.selectorText:
                    widget, state = map(str, rule.selectorText.split(':'))
                    state = getattr(const, state.upper(), const.STATE_NORMAL)
                else:
                    widget, state = str(rule.selectorText), const.STATE_NORMAL

                if widget not in tmp.widget_styles:
                    tmp.widget_styles[widget] = {}

                props = {}
                for prop in rule.style:
                    if '-color' in prop.name:
                        value = tuple(map(int, prop.value.strip().split(',')))
                    elif prop.value.isdigit():
                        value = int(prop.value)
                    else:
                        try:
                            value = float(prop.value)
                        except ValueError:
                            value = str(prop.value).strip('"')
                    props[str(prop.name)] = value

                style = Style(**props)
                if state not in tmp.widget_styles[widget]:
                    tmp.widget_styles[widget][state] = style
                else:
                    tmp.widget_styles[widget][state].update(style)

        self.update(tmp)

    def _init_resources(self, module_name, resources_conf):
        """
        Read the configuration file and fill up the stock resources.
        """
        tmp = Theme()
        parser = CSSParser()
        try:
            # new api in cssutils 0.9.5b3
            cfg = parser.parseFile(resources_conf)
        except AttributeError:
            cfg = parser.parse(resources_conf)

        # allow redefinitions of "defaults" sections
        basedir = '.'
        for rule in cfg.cssRules:
            if rule.selectorText == 'defaults':
                dir = rule.style.getProperty('basedir')
                basedir = dir and dir.value.strip('"') or basedir
            else:
                resource_prefix = str(rule.selectorText)

                for prop in rule.style:
                    filepath = os.path.join(basedir, prop.value.strip('"'))
                    resource_name = ".".join([resource_prefix, str(prop.name)])
                    try:
                        absolute_path = resource_filename(module_name, filepath)
                        self.stock_resources[resource_name] = absolute_path
                    except:
                        pass

        self.update(tmp)

    @classmethod
    def load_from_module(cls, module_name):
        """
        Build a L{elisa.plugins.pigment.widgets.Theme} object using the 'styles.conf'
        and 'resources.conf' files found in the specified module.

        @param module_name: the module to search, in the absolute dotted
                            notation
        @type module_name: string
        @returns: the new theme, or None
        @rtype: L{elisa.plugins.pigment.widgets.Theme}
        """

        try:
            styles_conf = resource_filename(module_name, 'styles.conf')
        except (ImportError, KeyError), exc:
            # ImportError raised when resource not found in uninstalled plugin
            # KeyError raised when resource not found in an egg plugin
            styles_conf = ''
        else:
            logging.debug("Loading plugin theme from: %s" % styles_conf)
        if not os.path.isfile(styles_conf):
            logging.warning("Cannot find theme file: %s" % styles_conf)
            styles_conf = None

        try:
            resources_conf = resource_filename(module_name, 'resources.conf')
        except:
            resources_conf = ''
        else:
            logging.debug("Loading resource theme from: %s" % resources_conf)
        if not os.path.isfile(resources_conf):
            logging.warning("Cannot find resource file: %s" % resources_conf)
            resources_conf = None

        if not resources_conf and not styles_conf:
            logging.warning("Cannot find theme configuration files")
            return None

        module_theme = cls(module_name=module_name,
                           styles_conf=styles_conf,
                           resources_conf=resources_conf)

        return module_theme

    def update(self, other):
        """
        Merge in-place another theme.

        @param other: the theme from which to update
        @type other: L{elisa.plugins.pigment.widgets.Theme}
        """
        self.merge(other, inplace=True)

    def merge(self, other, inplace=False):
        """
        Merge with another theme, returning a new one.

        The new theme will have all the "properties" of the current style, with
        replaced values from the second, plus further "properties" coming from the
        other theme.

        @param other: the theme to merge
        @type other: L{elisa.plugins.pigment.widgets.Theme}
        @param inplace: whether to build another theme, or update the current
                        one
        @type inplate: boolean
        @returns: the new theme
        @rtype: L{elisa.plugins.pigment.widgets.Theme}
        """

        if inplace:
            new = self
        else:
            new = Theme()

        if not inplace:
            for key, value in self.stock_resources.items():
                new.stock_resources[key] = value
            for widget, styles in self.widget_styles.items():
                new.widget_styles[widget] = copy.deepcopy(styles)

        for key, value in other.stock_resources.items():
            new.stock_resources[key] = value
        for widget, styles in other.widget_styles.items():
            if widget not in new.widget_styles:
                new.widget_styles[widget] = other.widget_styles[widget]
            else:
                for state, style in styles.items():
                    new.widget_styles[widget][state] = style.merge(other.widget_styles[widget][state])

        return new

    def get_style_for_widget(self, widget, state=const.STATE_NORMAL, search=True):
        """
        Get the style for a widget class in the specified state.

        @param widget: the classname of the widget
        @type widget: string
        @param state: the state for which we want to retrieve the style
        @type widget: int (one of elisa.plugins.pigment.widgets.const.STATE_*)
        @returns: the associated style, or None
        @rtype: L{elisa.plugins.pigment.widgets.Style}
        """
        if not widget or not state:
            return

        widget_styles = self.widget_styles.get(widget)

        if not widget_styles:
            if not search:
                return

            return self.lookup(widget, 'style', state)

        return widget_styles.get(state)

    def get_resource(self, name, search=True):
        """
        Get the named resources, doing a lookup into the plugins' defaults if not
        found, or None.

        @param name: the name of the resource
        @type name: string
        @type search: whether to do the lookup into the plugins defaults
        @type search: boolean
        """
        media = self.stock_resources.get(name)

        if media:
            return media

        if not search:
            return

        return self.lookup(name, 'resource')

    def lookup(self, name, type, state=const.STATE_NORMAL):
        """
        Dynamically search for the named resource ('style' or 'resource').

        If a suitable module is found during the search, a Theme object will be
        built and cached for later use: it will be stored in a dictionary
        indexed by absolute module names (in the Python dotted notation).

        @param name: the full qualified name to look for
                     (eg.: 'elisa.plugins.pigment.widgets.Button'
                      or 'elisa.plugins.shelf.icon')
        @type name: string
        @param type: 'style' or 'resource'
        @type type: Enum(['style', 'resource'])
        @param state: the state of the widget. Only used if type == 'style'
        @type state: Enum([state for state in const.STATE_*])
        @returns: the found resource (filepath or style), if any
        @rtype: string, or L{elisa.plugins.pigment.widgets.Style}, or None
        """

        if type == 'style' and ':' in name:
            name = name.split(':')[0]
            state = getattr(const, name.split(':')[1], const.STATE_NORMAL)

        module_name, resource_name = '.'.join(name.split('.')[0:-1]), name.split('.')[-1]
        if module_name:
            try:
                module_theme = self.fallback_themes[module_name]
            except KeyError:
                module_theme = None
            else:
                # we already searched for the theme, without finding it
                if not self.fallback_themes[module_name]:
                    return
            if module_theme is None:
                module_theme = Theme.load_from_module(module_name)
                # remember that we couldn't find the theme
                self.fallback_themes[module_name] = module_theme

                if not module_theme:
                    return

            if type == 'resource':
                #item = module_theme.get_resource(name.split('.')[-1], False)
                item = module_theme.get_resource(name, False)
            elif type == 'style':
                item = module_theme.get_style_for_widget(name, state, False)

            return item

    def __repr__(self):
        r = {}
        r['Theme'] = [{'widget_styles' : self.widget_styles},
                      {'stock_resources' : self.stock_resources}]

        return pprint.pformat(r)

    @classmethod
    def get_default(cls):
        """Get the default theme."""
        global _DEFAULT_THEME

        if not _DEFAULT_THEME:
            _DEFAULT_THEME = cls()

        return _DEFAULT_THEME

    @staticmethod
    def set_default(theme):
        """Set the default theme."""
        global _DEFAULT_THEME

        _DEFAULT_THEME = theme


if __name__ == '__main__':
    logger = logging.getLogger()
    logger.setLevel(logging.DEBUG)

    theme = Theme()

