/*******************************************************************************
 * Copyright (c) 2009 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.pde.api.tools.internal.provisional.search;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.SubMonitor;
import org.eclipse.pde.api.tools.internal.builder.Reference;
import org.eclipse.pde.api.tools.internal.builder.ReferenceResolver;
import org.eclipse.pde.api.tools.internal.provisional.ApiPlugin;
import org.eclipse.pde.api.tools.internal.provisional.builder.IReference;
import org.eclipse.pde.api.tools.internal.provisional.model.ApiTypeContainerVisitor;
import org.eclipse.pde.api.tools.internal.provisional.model.IApiBaseline;
import org.eclipse.pde.api.tools.internal.provisional.model.IApiComponent;
import org.eclipse.pde.api.tools.internal.provisional.model.IApiElement;
import org.eclipse.pde.api.tools.internal.provisional.model.IApiMember;
import org.eclipse.pde.api.tools.internal.provisional.model.IApiType;
import org.eclipse.pde.api.tools.internal.provisional.model.IApiTypeRoot;
import org.eclipse.pde.api.tools.internal.search.SearchMessages;
import org.eclipse.pde.api.tools.internal.util.Util;

import com.ibm.icu.text.MessageFormat;

/**
 * Engine used to search for API use 
 * 
 * @since 1.0.0
 */
public final class ApiSearchEngine {
	
	/**
	 * Default empty array for no search matches
	 */
	public static final IReference[] NO_REFERENCES = new IReference[0];
	
	/**
	 * Constant used for controlling tracing in the search engine
	 */
	private static boolean DEBUG = Util.DEBUG;
	
	/**
	 * Visitor used to extract references from the component is is passed to
	 */
	class ReferenceExtractor extends ApiTypeContainerVisitor {
		private List collector = null;
		private IApiSearchRequestor requestor = null;
		private IApiSearchReporter reporter = null;
		IApiElement element = null;
		private SubMonitor monitor = null;
		
		/**
		 * Constructor
		 */
		public ReferenceExtractor(IApiSearchRequestor requestor, IApiSearchReporter reporter, IApiElement element, IProgressMonitor monitor) {
			collector = new ArrayList();
			this.requestor = requestor;
			this.reporter = reporter;
			this.element = element;
			this.monitor = (SubMonitor) monitor;
		}
		
		/* (non-Javadoc)
		 * @see org.eclipse.pde.api.tools.internal.provisional.model.ApiTypeContainerVisitor#visit(java.lang.String, org.eclipse.pde.api.tools.internal.provisional.model.IApiTypeRoot)
		 */
		public void visit(String packageName, IApiTypeRoot typeroot) {
			if(monitor.isCanceled()) {
				return;
			}
			try {
				IApiType type = typeroot.getStructure();
				if(!requestor.acceptMember(type)) {
					return;
				}				
				collector.addAll(acceptReferences(requestor, 
						type, 
						getResolvedReferences(requestor, type, monitor.newChild(1)), 
						monitor.newChild(1)));
				
			}
			catch(CoreException ce) {
				ApiPlugin.log(ce);
			}
		}
		
		/**
		 * @see org.eclipse.pde.api.tools.internal.provisional.model.ApiTypeContainerVisitor#visit(org.eclipse.pde.api.tools.internal.provisional.model.IApiComponent)
		 */
		public boolean visit(IApiComponent component) {
			return requestor.acceptComponent(component);
		}
		
		/* (non-Javadoc)
		 * @see org.eclipse.pde.api.tools.internal.provisional.model.ApiTypeContainerVisitor#endVisitPackage(java.lang.String)
		 */
		public void endVisitPackage(String packageName) {
			reporter.reportResults(this.element, (IReference[]) collector.toArray(new IReference[collector.size()]));
			collector.clear();
		}
	}
	
	/**
	 * Method used for initializing tracing
	 */
	public static void setDebug(boolean debugValue) {
		DEBUG = debugValue || Util.DEBUG;
	}
	
	/**
	 * Simple string used for reporting what is being searched
	 */
	private String fRequestorContext = null;
	
	/**
	 * Returns the set of resolved references for the given {@link IApiType}
	 * @param requestor
	 * @param type
	 * @param monitor
	 * @return The listing of resolved references from the given {@link IApiType}
	 * @throws CoreException
	 */
	private List getResolvedReferences(IApiSearchRequestor requestor, IApiType type, IProgressMonitor monitor) throws CoreException {
		String name = type.getSimpleName();
		SubMonitor localmonitor = SubMonitor.convert(monitor, 
				MessageFormat.format(SearchMessages.ApiSearchEngine_extracting_refs_from, new String[] {(name == null ? SearchMessages.ApiSearchEngine_anonymous_type : name)}), 2);
		try {
			List refs = type.extractReferences(requestor.getReferenceKinds(), localmonitor.newChild(1));
			ReferenceResolver.resolveReferences(refs, localmonitor.newChild(1));
			return refs;
		}
		finally {
			localmonitor.done();
		}
	}
	
	/**
	 * Runs the given list of references through the search requestor to determine if they should be kept or not
	 * @param requestor
	 * @param type
	 * @param references
	 * @param monitor
	 * @return
	 * @throws CoreException
	 */
	private List acceptReferences(IApiSearchRequestor requestor, IApiType type, List references, IProgressMonitor monitor) throws CoreException {
		ArrayList refs = new ArrayList();
		Reference ref = null;
		SubMonitor localmonitor = SubMonitor.convert(monitor, references.size());
		IApiMember member = null;
		try {
			for (Iterator iter = references.iterator(); iter.hasNext();) {
				if(localmonitor.isCanceled()) {
					return Collections.EMPTY_LIST;
				}
				ref = (Reference) iter.next();
				member = ref.getResolvedReference();
				if(member == null) {
					continue;
				}
				localmonitor.setTaskName(MessageFormat.format(SearchMessages.ApiSearchEngine_searching_for_use_from, new String[] {fRequestorContext, type.getName()}));
				if(requestor.acceptReference(ref)) {
					refs.add(ref);
				}
				localmonitor.worked(1);
			}
		}
		finally {
			localmonitor.done();
		}
		return refs;
	}
	
	/**
	 * Searches for all accepted {@link IReference}s from the given {@link IApiElement}
	 * @param requestor
	 * @param element
	 * @param monitor
	 * @return the collection of accepted {@link IReference}s or an empty list, never <code>null</code>
	 * @throws CoreException
	 */
	private void searchReferences(IApiSearchRequestor requestor, IApiElement element, IApiSearchReporter reporter, IProgressMonitor monitor) throws CoreException {
		List refs = null;
		SubMonitor localmonitor = SubMonitor.convert(monitor, 3);
		try {
			switch(element.getType()) {
				case IApiElement.TYPE: {
					if(localmonitor.isCanceled()) {
						reporter.reportResults(element, NO_REFERENCES);
					}
					IApiType type = (IApiType) element;
					refs = acceptReferences(requestor, 
							type, 
							getResolvedReferences(requestor, type, localmonitor.newChild(1)),
							localmonitor.newChild(1));
					reporter.reportResults(element, (IReference[]) refs.toArray(new IReference[refs.size()]));
					break;
				}
				case IApiComponent.COMPONENT: {
					if(localmonitor.isCanceled()) {
						reporter.reportResults(element, NO_REFERENCES);
					}
					ReferenceExtractor visitor = new ReferenceExtractor(requestor, reporter, element, localmonitor.newChild(1));
					IApiComponent comp = (IApiComponent) element;
					comp.accept(visitor);
					localmonitor.worked(1);
					break;
				}
				case IApiElement.FIELD:
				case IApiElement.METHOD: {
					if(localmonitor.isCanceled()) {
						reporter.reportResults(element, NO_REFERENCES);
					}
					IApiMember member = (IApiMember) element;
					IApiType type = member.getEnclosingType();
					if(type != null) {
						refs = acceptReferences(requestor, 
								type, 
								getResolvedReferences(requestor, type, localmonitor.newChild(1)), 
								localmonitor.newChild(1));
					}
					if (refs != null) {
						reporter.reportResults(element, (IReference[]) refs.toArray(new IReference[refs.size()]));
					}
					break;
				}
			}
			localmonitor.worked(1);
		}
		finally {
			localmonitor.done();
		}
	}
	
	/**
	 * Searches for all of the use of API or internal code from the given 
	 * {@link IApiComponent} within the given {@link IApiBaseline}
	 * 
	 * @param baseline the baseline to search within
	 * @param requestor the requestor to use for the search
	 * @param reporter the reporter to use when reporting any search results to the user
	 * @param monitor the monitor to report progress to
	 * @throws CoreException if the search fails
	 */
	public void search(IApiBaseline baseline, IApiSearchRequestor requestor, IApiSearchReporter reporter, IProgressMonitor monitor) throws CoreException {
		if(baseline == null || reporter == null || requestor == null) {
			return;
		}
		IApiSearchScope scope = requestor.getScope();
		if(scope == null) {
			return;
		}
		fRequestorContext = SearchMessages.ApiSearchEngine_api_internal;
		if(requestor.includesAPI() && !requestor.includesInternal()) {
			fRequestorContext = SearchMessages.ApiSearchEngine_api;
		}
		if(requestor.includesInternal() && !requestor.includesAPI()) {
			fRequestorContext = SearchMessages.ApiSearchEngine_internal;
		}
		IApiElement[] scopeelements = scope.getScope();
		SubMonitor localmonitor = SubMonitor.convert(monitor, 
				MessageFormat.format(SearchMessages.ApiSearchEngine_searching_projects, new String[] {fRequestorContext}), scopeelements.length*2+1);
		try {
			long start = System.currentTimeMillis();
			long loopstart = 0;
			String taskname = null;
			for (int i = 0; i < scopeelements.length; i++) {
				taskname = MessageFormat.format(SearchMessages.ApiSearchEngine_searching_project, new String[] {scopeelements[i].getApiComponent().getId(), fRequestorContext});
				localmonitor.setTaskName(taskname);
				if(DEBUG) {
					loopstart = System.currentTimeMillis();
					System.out.println("Searching "+scopeelements[i].getApiComponent().getId()+"..."); //$NON-NLS-1$ //$NON-NLS-2$
				}
				searchReferences(requestor, scopeelements[i], reporter, localmonitor.newChild(1));
				localmonitor.setTaskName(taskname);
				if(localmonitor.isCanceled()) {
					reporter.reportResults(scopeelements[i], NO_REFERENCES);
					return;
				}
				localmonitor.worked(1);
				if(DEBUG) {
					System.out.println(Math.round((((float)(i+1))/scopeelements.length)*100)+"% done in "+(System.currentTimeMillis()-loopstart)+" ms"); //$NON-NLS-1$ //$NON-NLS-2$
				}
			}
			if(DEBUG) {
				System.out.println("Total Search Time: "+((System.currentTimeMillis()-start)/1000)+" seconds");  //$NON-NLS-1$//$NON-NLS-2$
			}
		}
		finally {
			localmonitor.done();
		}
	}
}
