/*
 * Easy Network
 * Copyright (c) <2009>, Intel Corporation.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */
#include "AccessPointSorter.h"
#include "DataCenter.h"

/**************************************************************************************************
* Function Name : AccessPointSorter                                                                    *
* Description  : Constructor                                  *
* Date   :                                                                                        *
* Parameter     :                                                                                 *
* Return Code  :                                                                                  *
* Author   :                                                                                      *
**************************************************************************************************/
AccessPointSorter::AccessPointSorter(int columnCount)
{
    assert(columnCount > 0);
    m_baSortEnable = new bool[columnCount];
    m_baSortOrder = new bool[columnCount];
    m_iColumnCount = columnCount;
    Reset();
}

/**************************************************************************************************
* Function Name : ~AccessPointSorter                                                                    *
* Description  : Destructor                                  *
* Date   :                                                                                        *
* Parameter     :                                                                                 *
* Return Code  :                                                                                  *
* Author   :                                                                                      *
**************************************************************************************************/
AccessPointSorter::~AccessPointSorter()
{
    delete[] m_baSortEnable;
    delete[] m_baSortOrder;
}

/**************************************************************************************************
* Function Name : Reset                                                                    *
* Description  :                                   *
* Date   :                                                                                        *
* Parameter     :                                                                                 *
* Return Code  :                                                                                  *
* Author   :                                                                                      *
**************************************************************************************************/
void AccessPointSorter::Reset()
{
    for ( int i = 0; i < m_iColumnCount; i++ )
    {
        m_baSortEnable[i] = true;
        m_baSortOrder[i] = true;
    }
}

/**************************************************************************************************
* Function Name : IsSortEnable                                                                    *
* Description  : Checking whether can sort by a specific column                                   *
* Date   :                                                                                        *
* Parameter     :                                                                                 *
* Return Code  :                                                                                  *
* Author   :                                                                                      *
**************************************************************************************************/
bool AccessPointSorter::IsSortEnable(int column) const
{
    if ( (column < 0) || (column >= m_iColumnCount) )
    {
        return false;
    }
    return m_baSortEnable[column];
}

/**************************************************************************************************
* Function Name : SetSortEnable                                                                     *
* Description  : Setting whether can sort by a specific column                                                                 *
* Date   :                                                                                        *
* Parameter     :                                                                                 *
* Return Code  :                                                                                  *
* Author   :                                                                                      *
**************************************************************************************************/
bool AccessPointSorter::SetSortEnable(int column, bool enable)
{
    if ( (column < 0) || (column >= m_iColumnCount) )
    {
        return false;
    }
    m_baSortEnable[column] = enable;
    return true;
}

/**************************************************************************************************
* Function Name : SetSortOrder                                                                     *
* Description  : Set order when sorting                                                                 *
* Date   :                                                                                        *
* Parameter     :                                                                                 *
* Return Code  :                                                                                  *
* Author   :                                                                                      *
**************************************************************************************************/
bool AccessPointSorter::SetSortOrder(int column, bool ascending)
{
    if ( (column < 0) || (column >= m_iColumnCount) )
    {
        return false;
    }
    m_baSortOrder[column] = ascending;
    return true;
}

/**************************************************************************************************
* Function Name : Sort                                                                     *
* Description  : sort by specific column                                                                     *
* Date   :                                                                                        *
* Parameter     :                                                                                 *
* Return Code  :                                                                                  *
* Author   :                                                                                      *
**************************************************************************************************/
bool AccessPointSorter::Sort(APStatus* apArray, int arraySize, int column, bool insc)
{
    if ( (apArray == NULL) || (arraySize == 0) )
    {
        return false;
    }
    if ( (column < 0) || (column >= m_iColumnCount) )
    {
        return false;
    }
    if ( !m_baSortEnable[column] )
    {
        return false;
    }

    APStatus temp;
    APStatus* pCurrentAP = DataCenter::GetInstance()->GetCurrentAPStatus();
    WlanSsid currentSsid;
    if ( pCurrentAP != NULL )
    {
        currentSsid = pCurrentAP->ssid;
    }
    else
    {
        memset(&currentSsid, 0, sizeof(WlanSsid));
    }
    bool flag = false;
    for ( int i = 0; i < arraySize - 1; i++ )
    {
        flag = false;
        for ( int j = arraySize - 1; j > i; j-- )
        {
            int comp = 0;
            switch ( column )
            {
            case 0: // name
                comp = CompareStringNoCase((const char*)apArray[j - 1].ssid.ucSsid,
                                            (const char*)apArray[j].ssid.ucSsid);
            //    comp = strcmp((char*)apArray[j - 1].ssid.ucSsid, (char*)apArray[j].ssid.ucSsid);
                break;
            case 1: // signal
                comp = apArray[j].signalQuality - apArray[j- 1].signalQuality;
                break;
            case 2: // encrypted
                comp = apArray[j].auth - apArray[j - 1].auth;
                break;
            case 3: // connected
                comp = (apArray[j].ssid == currentSsid) - (apArray[j - 1].ssid == currentSsid);
                break;
            case 4: // type
                comp = apArray[j].type - apArray[j - 1].type;
                break;
            case 5: // channel
                comp = apArray[j].channel - apArray[j - 1].channel;
                break;
            case 6: // mac address
                comp = CompareMacAddress(apArray[j - 1].macAddress, apArray[j].macAddress);
                break;
            default:
                break;
            }
            if ( !m_baSortOrder[column] )
            {
                comp *= -1;
            }
            if ( comp > 0 )
            {
                temp = apArray[j - 1];
                apArray[j - 1] = apArray[j];
                apArray[j] = temp;
                flag = true;
            }
        }
        if ( !flag )
        {
            break;
        }
    }
    m_baSortOrder[column] = !m_baSortOrder[column];
    return true;
}

/**************************************************************************************************
* Function Name : CompareStringNoCase                                                                     *
* Description  : compare str1 with str2                                                                      *
* Date   :                                                                                        *
* Parameter     :                                                                                 *
* Return Code  :                                                                                  *
* Author   :                                                                                      *
**************************************************************************************************/
int AccessPointSorter::CompareStringNoCase(const char* str1, const char* str2)
{
#if defined(__WXMSW__)
    return _stricmp(str1, str2);
#else // LINUX
    return strcasecmp(str1, str2);
#endif
}
