# -*- coding: utf-8 -*-
# (c) 2013 Canonical Ltd.
# Authors: Yang Bai <hamo@canonical.com>
# License: GPL v2 or later

import logging, os, os.path

import XKit.xutils
import XKit.xorgparser
from jockey.oslib import OSLib
from jockey.xorg_driver import XorgDriverHandler
from NvidiaDetector.alternatives import Alternatives
from NvidiaDetector.alternatives import MultiArchUtils
import subprocess

# dummy stub for xgettext
def _(x): return x

class FglrxDriver(XorgDriverHandler):
    def __init__(self, backend, package=None):
        self._free = False

        name=_('ATI/AMD proprietary FGLRX graphics driver')
        # description will fetch from the package's description.
        # description=_('3D-accelerated proprietary graphics driver for '
        #               'ATI cards.')
        # rationale=_('This driver is required to fully utilise the 3D '
        #             'potential of some ATI graphics cards, as well as provide '
        #             '2D acceleration of newer cards.')

        XorgDriverHandler.__init__(self, backend, (package and
            package.replace('-', '_') or 'fglrx'), (package and
            package or 'fglrx'), None, None, add_modules=[],
            disable_modules=[], name=name)
            # disable_modules=[], name=name,
            # description=description,
            # rationale=rationale)

        (self._alternatives, self._other_alternatives) = self._get_alternatives()
        self.needs_kernel_headers = True

    def _get_alternatives(self):
        '''Get multi-arch alternatives names'''
        arch_utils = MultiArchUtils()
        main_name = arch_utils.get_main_alternative_name()
        other_name = arch_utils.get_other_alternative_name()
        return Alternatives(main_name), Alternatives(other_name)

    def available(self):
        # we don't offer fglrx in a life CD environment, as we will run out of
        # RAM trying to download and install all the packages in the RAM disk.
        if os.path.isdir('/rofs'):
            logging.debug('Disabling fglrx driver on live system')
            return False

        logging.debug('fglrx.available: falling back to default')
        return XorgDriverHandler.available(self)

    def enable_config_hook(self):
        # Call aticonfig --initial
        subprocess.call(['aticonfig', '--initial'])

        # re-read xorg.conf since aticonfig will overwrite it
        self.xorg_conf = XKit.xutils.XUtils(OSLib.inst.xorg_conf_path)
        
    def enable(self):
        XorgDriverHandler.enable(self)
        
        # Set the alternative to FGLRX
        fglrx_alternative = self._alternatives.get_alternative_by_name('fglrx')
        if not fglrx_alternative:
            logging.error('%s: get_alternative_by_name(%s) returned nothing' % (
                self.id(), self.package))
            return
        self._alternatives.set_alternative(fglrx_alternative)
        other_fglrx_alternative = self._other_alternatives.get_alternative_by_name('fglrx')
        if other_fglrx_alternative:
            self._other_alternatives.set_alternative(other_fglrx_alternative)
        subprocess.call(['update-initramfs', '-u'])
        subprocess.call(['update-initramfs', '-u', '-k', os.uname()[2]])

    def enabled(self):
        # See if fglrx is the current alternative
        target_alternative = self._alternatives.get_alternative_by_name('fglrx')
        current_alternative = self._alternatives.get_current_alternative()
        other_target_alternative = self._other_alternatives.get_alternative_by_name('fglrx')
        other_current_alternative = self._other_alternatives.get_current_alternative()

        logging.debug('fglrx.enabled(%s): target_alt %s current_alt %s other target alt %s other current alt %s',
                self.module, target_alternative, current_alternative,
                other_target_alternative, other_current_alternative)

        if current_alternative is None:
            logging.debug('current alternative of %s is None, not enabled', self.module)
            return False
        if current_alternative != target_alternative or \
           other_current_alternative != other_target_alternative:
            logging.debug('%s is not the alternative in use', self.module)
            return False

        return XorgDriverHandler.enabled(self)

    def disable(self):
        XorgDriverHandler.disable(self)

        logging.debug('fglrx: Skip open drivers')
        # Set the alternative back to open drivers
        # open_drivers = self._alternatives.get_open_drivers_alternative()
        # logging.debug('fglrx.disable(%s): open_drivers: %s', self.module, open_drivers)
        # if open_drivers:
        #     self._alternatives.set_alternative(open_drivers)
        # other_open_drivers = self._other_alternatives.get_open_drivers_alternative()
        # logging.debug('fglrx.disable(%s): other_open_drivers: %s', self.module, other_open_drivers)
        # if other_open_drivers:
        #     self._other_alternatives.set_alternative(other_open_drivers)
        subprocess.call(['update-initramfs', '-u'])
        subprocess.call(['update-initramfs', '-u', '-k', os.uname()[2]])

        return False

    def enables_composite(self):
        '''Return whether this driver supports the composite extension.'''

        if not self.xorg_conf:
            return False

        # the radeon X.org driver supports composite nowadays, so don't force
        # installation of fglrx upon those users. Treat absent driver
        # configuration as radeon, since that's what X.org should autodetect.
        # Only suggest fglrx if people use something else, like vesa.
        # try:
        #     if self.xorg_conf.getDriver('Device', 0) in ['fglrx', 'ati', 'radeon', None]:
        #         return False
        # except (XKit.xorgparser.OptionException, XKit.xorgparser.SectionException) as error:
        #     return False # unconfigured driver -> defaults to ati

        return True

class FglrxDriverUpdate(FglrxDriver):
    def __init__(self, backend):
        FglrxDriver.__init__(self, backend, 'fglrx-updates')
    def available(self):
        # We do not want end users to see the fglrx packages in
        # official repos
        logging.debug('fglrx.available: Hide fglrx-updates from official repo')
        return False

class FglrxDriverExperimental9(FglrxDriver):
    def __init__(self, backend):
        FglrxDriver.__init__(self, backend, 'fglrx-experimental-9')
    def available(self):
        # We do not want end users to see the fglrx packages in
        # official repos
        logging.debug('fglrx.available: Hide fglrx-experimental-9 from official repo')
        return False

class FglrxDriverExperimental12(FglrxDriver):
    def __init__(self, backend):
        FglrxDriver.__init__(self, backend, 'fglrx-experimental-12')
    def available(self):
        # We do not want end users to see the fglrx packages in
        # official repos
        logging.debug('fglrx.available: Hide fglrx-experimental-12 from official repo')
        return False
