#ifndef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdlib.h>
#include <string.h>

#include <glib-object.h>
#include <gobject/gvaluecollector.h>

#include <clutter/clutter-color.h>

#include "tidy-style.h"
#include "tidy-marshal.h"
#include "tidy-debug.h"

enum
{
  CHANGED,

  LAST_SIGNAL
};

#define TIDY_STYLE_GET_PRIVATE(obj) \
        (G_TYPE_INSTANCE_GET_PRIVATE ((obj), TIDY_TYPE_STYLE, TidyStylePrivate))

typedef struct {
  GType value_type;
  gchar *value_name;
  GValue value;
} StyleProperty;

struct _TidyStylePrivate
{
  GHashTable *properties;
};

static guint style_signals[LAST_SIGNAL] = { 0, };

static const gchar *tidy_default_font_name          = "Sans 12px";
static const ClutterColor tidy_default_text_color   = { 0x00, 0x00, 0x00, 0xff };
static const ClutterColor tidy_default_bg_color     = { 0xcc, 0xcc, 0xcc, 0xff };
static const ClutterColor tidy_default_active_color = { 0xf5, 0x79, 0x00, 0xff };

static TidyStyle *default_style = NULL;

G_DEFINE_TYPE (TidyStyle, tidy_style, G_TYPE_OBJECT);

static StyleProperty *
style_property_new (const gchar *value_name,
                    GType        value_type)
{
  StyleProperty *retval;

  retval = g_slice_new0 (StyleProperty);
  retval->value_type = value_type;
  retval->value_name = g_strdup (value_name);
  g_value_init (&retval->value, value_type);

  return retval;
}

static void
style_property_free (gpointer data)
{
  if (G_LIKELY (data))
    {
      StyleProperty *sp = data;

      g_free (sp->value_name);
      g_value_unset (&sp->value);
    }
}

static void
init_defaults (TidyStyle *style)
{
  TidyStylePrivate *priv = style->priv;
  
  {
    StyleProperty *sp;

    sp = style_property_new (TIDY_FONT_NAME, G_TYPE_STRING);
    g_value_set_string (&sp->value, tidy_default_font_name);

    g_hash_table_insert (priv->properties, sp->value_name, sp);
  }

  {
    StyleProperty *sp;

    sp = style_property_new (TIDY_BACKGROUND_COLOR, CLUTTER_TYPE_COLOR);
    g_value_set_boxed (&sp->value, &tidy_default_bg_color);

    g_hash_table_insert (priv->properties, sp->value_name, sp);
  }

  {
    StyleProperty *sp;

    sp = style_property_new (TIDY_ACTIVE_COLOR, CLUTTER_TYPE_COLOR);
    g_value_set_boxed (&sp->value, &tidy_default_active_color);

    g_hash_table_insert (priv->properties, sp->value_name, sp);
  }

  {
    StyleProperty *sp;

    sp = style_property_new (TIDY_TEXT_COLOR, CLUTTER_TYPE_COLOR);
    g_value_set_boxed (&sp->value, &tidy_default_text_color);

    g_hash_table_insert (priv->properties, sp->value_name, sp);
  }
}

static gboolean
tidy_style_load_from_file (TidyStyle    *style,
                           const gchar  *filename,
                           GError      **error)
{
  GKeyFile *rc_file;
  GError *internal_error;

  rc_file = g_key_file_new ();
  
  internal_error = NULL;
  g_key_file_load_from_file (rc_file, filename, 0, &internal_error);
  if (internal_error)
    {
      /* if the specified files does not exist then just ignore it
       * and fall back to the default values; if, instead, the file
       * is not accessible or is malformed, propagate the error
       */
      if (internal_error->domain == G_FILE_ERROR &&
          internal_error->code == G_FILE_ERROR_NOENT)
        {
          g_error_free (internal_error);
          return TRUE;
        }

      g_propagate_error (error, internal_error);
      return FALSE;
    }

  g_key_file_free (rc_file);

  return TRUE;
}

static void
tidy_style_load (TidyStyle *style)
{
  const gchar *env_var;
  gchar *rc_file = NULL;
  GError *error;
  
  init_defaults (style);

  env_var = g_getenv ("TIDY_RC_FILE");
  if (env_var && *env_var)
    rc_file = g_strdup (env_var);
  
  if (!rc_file)
    rc_file = g_build_filename (g_get_user_config_dir (),
                                "tidy",
                                "tidyrc",
                                NULL);

  error = NULL;
  if (!tidy_style_load_from_file (style, rc_file, &error))
    {
      g_critical ("Unable to load resource file `%s': %s",
                  rc_file,
                  error->message);
      g_error_free (error);
    }
}

static void
tidy_style_finalize (GObject *gobject)
{
  TidyStylePrivate *priv = TIDY_STYLE (gobject)->priv;

  g_hash_table_destroy (priv->properties);

  G_OBJECT_CLASS (tidy_style_parent_class)->finalize (gobject);
}

static void
tidy_style_class_init (TidyStyleClass *klass)
{
  GObjectClass *gobject_class = G_OBJECT_CLASS (klass);

  g_type_class_add_private (klass, sizeof (TidyStylePrivate));

  gobject_class->finalize = tidy_style_finalize;

  style_signals[CHANGED] =
    g_signal_new ("changed",
                  G_TYPE_FROM_CLASS (klass),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (TidyStyleClass, changed),
                  NULL, NULL,
                  _tidy_marshal_VOID__VOID,
                  G_TYPE_NONE, 0);
}

static void
tidy_style_init (TidyStyle *style)
{
  TidyStylePrivate *priv;

  style->priv = priv = TIDY_STYLE_GET_PRIVATE (style);

  priv->properties = g_hash_table_new_full (g_str_hash, g_str_equal,
                                            NULL,
                                            style_property_free);

  tidy_style_load (style);
}

/* need to unref */
TidyStyle *
tidy_style_new (void)
{
  return g_object_new (TIDY_TYPE_STYLE, NULL);
}

/* never ref/unref */
TidyStyle *
tidy_style_get_default (void)
{
  if (G_LIKELY (default_style))
    return default_style;

  default_style = g_object_new (TIDY_TYPE_STYLE, NULL);
  
  return default_style;
}

gboolean
tidy_style_has_property (TidyStyle   *style,
                         const gchar *property_name)
{
  g_return_val_if_fail (TIDY_IS_STYLE (style), FALSE);
  g_return_val_if_fail (property_name != NULL, FALSE);

  return (g_hash_table_lookup (style->priv->properties, property_name) != NULL);
}

void
tidy_style_add_property (TidyStyle   *style,
                         const gchar *property_name,
                         GType        property_type)
{
  StyleProperty *property;

  g_return_if_fail (TIDY_IS_STYLE (style));
  g_return_if_fail (property_name != NULL);
  g_return_if_fail (property_type != G_TYPE_INVALID);

  property = g_hash_table_lookup (style->priv->properties, property_name);
  if (property)
    {
      g_warning ("A property named `%s', with type %s already exists.",
                 property->value_name,
                 g_type_name (property->value_type));
      return;
    }

  property = style_property_new (property_name, property_type);
  g_hash_table_insert (style->priv->properties, property->value_name, property);

  g_signal_emit (style, style_signals[CHANGED], 0);
}

void
tidy_style_get_property (TidyStyle   *style,
                         const gchar *property_name,
                         GValue      *value)
{
  StyleProperty *property;

  g_return_if_fail (TIDY_IS_STYLE (style));
  g_return_if_fail (property_name != NULL);
  g_return_if_fail (value != NULL);

  property = g_hash_table_lookup (style->priv->properties, property_name);
  if (!property)
    {
      g_warning ("No style property named `%s' found.", property_name);
      return;
    }

  g_value_init (value, property->value_type);
  g_value_copy (&property->value, value);
}

void
tidy_style_set_property (TidyStyle    *style,
                         const gchar  *property_name,
                         const GValue *value)
{
  StyleProperty *property;

  g_return_if_fail (TIDY_IS_STYLE (style));
  g_return_if_fail (property_name != NULL);
  g_return_if_fail (value != NULL);

  property = g_hash_table_lookup (style->priv->properties, property_name);
  if (!property)
    {
      g_warning ("No style property named `%s' found.", property_name);
      return;
    }

  g_value_copy (value, &property->value);

  g_signal_emit (style, style_signals[CHANGED], 0);
}
