/*
 * Copyright (C) 2008 Dell Inc.
 * Copyright (C) 2008 Canonical Ltd
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Authored by Neil Jagdish Patel <neil.patel@canonical.com>
 *
 */

#include <glib/gi18n.h>
#include <math.h>

#include <tidy/tidy.h>

#include "launcher-catbar.h"

#include "clutter-drag-server.h"
#include "clutter-focus-group.h"
#include "clutter-focus-manager.h"
#include "launcher-add.h"
#include "launcher-app.h"
#include "launcher-behave.h"
#include "launcher-category.h"
#include "launcher-category-editor.h"
#include "launcher-config.h"
#include "launcher-defines.h"
#include "launcher-util.h"
#include "launcher-wm.h"

static void clutter_focus_group_iface_init (ClutterFocusGroupIface *iface);

G_DEFINE_TYPE_WITH_CODE (LauncherCatbar, 
                         launcher_catbar, 
                         CLUTTER_TYPE_GROUP,
                         G_IMPLEMENT_INTERFACE (CLUTTER_TYPE_FOCUS_GROUP,
                                              clutter_focus_group_iface_init));

#define LAUNCHER_CATBAR_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj),\
  LAUNCHER_TYPE_CATBAR, \
  LauncherCatbarPrivate))

struct _LauncherCatbarPrivate
{
  gint n_cats;
  GList *cats;

  ClutterActor *bg_texture;
  ClutterActor *bg;

  ClutterActor *add;

  ClutterActor *hilight_texture;
  ClutterActor *hilight;

  ClutterActor *context;
  gint offset;

  LauncherMenuCategory *current;
  LauncherCategory     *current_cat;
  LauncherCategory     *focused;

  gint x;
  gint y;

  ClutterTimeline  *reorder_time;
  ClutterBehaviour *reorder_behave;
};

enum
{
  CAT_SELECTED,

  LAST_SIGNAL
};
static guint _catbar_signals[LAST_SIGNAL] = { 0 };

/* Forwards */
static gboolean on_drag (ClutterActor *, ClutterMotionEvent *, LauncherCatbar*);

static void
on_add_reponse (GtkWidget *dialog, gint res, LauncherCatbar *bar)
{
  LauncherCatbarPrivate *priv;

  if (!LAUNCHER_IS_CATBAR (bar))
  {
    gtk_widget_destroy (dialog);
    return;
  }
  priv = bar->priv;

  if (res == GTK_RESPONSE_YES)
  {

     gchar *name = NULL, *icon = NULL;
     g_object_get (dialog, 
                   "category-name", &name,
                   "category-icon", &icon,
                   NULL);
     launcher_menu_add_category (launcher_menu_get_default (),
                                 name, icon);
     g_free (name);
     g_free (icon);
  }

  gtk_widget_destroy (dialog);
}

static gboolean
on_add_clicked (ClutterActor       *actor, 
                LauncherCatbar     *bar)
{
  LauncherCatbarPrivate *priv;
  LauncherConfig *cfg = launcher_config_get_default ();
  GtkWidget *dialog;

  g_return_val_if_fail (LAUNCHER_IS_CATBAR (bar), FALSE);
  priv = bar->priv;

  if (priv->n_cats >= cfg->n_cats)
  {
    gchar *msg_text;

    msg_text = g_strdup_printf ("<b>%s</b>\n%s",
               _("The maximum number of categories has been reached"),
               _("Please rename or delete a category by right-clicking on it,"
                 " then making a selection from the popup menu."));
    dialog = gtk_message_dialog_new (NULL,
                                     0,
                                     GTK_MESSAGE_WARNING,
                                     GTK_BUTTONS_CLOSE,
                                     msg_text, NULL);
    g_free (msg_text);

    g_object_set (dialog, 
                  "title", _("Unable to add Launcher"),
                  "use-markup", TRUE,
                  NULL);
    g_signal_connect (dialog, "response", G_CALLBACK (on_add_reponse), bar);
  }
  else
  {
    dialog = launcher_category_editor_new ();
    gtk_dialog_add_buttons (GTK_DIALOG (dialog), 
                            "gtk-cancel", GTK_RESPONSE_CANCEL,
                            GTK_STOCK_APPLY, GTK_RESPONSE_YES,
                            NULL);
    g_object_set (dialog, 
                  "title", _("Add Category to Launcher"),
                  NULL);
    g_signal_connect (dialog, "response", G_CALLBACK (on_add_reponse), bar);
  }
  launcher_util_present_window (GTK_WINDOW (dialog));

  return TRUE;
}


static gboolean
on_drop (ClutterActor       *stage, 
         ClutterButtonEvent *event, 
         LauncherCatbar     *bar)
{
  LauncherCatbarPrivate *priv;
  LauncherConfig *cfg = launcher_config_get_default ();
  gint cat_width = cfg->cat_width;
  GList *c;
  GList *new_order = NULL;
  gint contextx;

  g_return_val_if_fail (LAUNCHER_IS_CATBAR (bar), FALSE);
  priv = bar->priv;

  contextx = clutter_actor_get_x (priv->context);
  contextx -= (contextx % cat_width);
  clutter_actor_set_x (priv->context, contextx);

  for (c = priv->cats; c; c = c->next)
  {
    ClutterActor *cat;
    
    cat = c->data;
  
    new_order = g_list_append (new_order, 
                               g_object_get_data (G_OBJECT (cat), "category"));
  }

  launcher_menu_set_categories (launcher_menu_get_default (), new_order);

  launcher_category_reset_after_drag (LAUNCHER_CATEGORY (priv->context));
  clutter_ungrab_pointer ();
  g_signal_handlers_disconnect_by_func (stage, on_drag, bar);
  g_signal_handlers_disconnect_by_func (stage, on_drop, bar);
  
  return TRUE;
}

static gboolean
on_drag (ClutterActor       *stage, 
         ClutterMotionEvent *event, 
         LauncherCatbar     *bar)
{
  LauncherCatbarPrivate *priv;
  gint x = event->x;
  gint cat_width = (launcher_config_get_default ())->cat_width;
  GList *c;
  gint i = 0;

  g_return_val_if_fail (LAUNCHER_IS_CATBAR (bar), FALSE);
  priv = bar->priv;

  x = CLAMP (x, 0, cat_width * (priv->n_cats-1));
  
  clutter_actor_set_x (priv->context, x);

  x -= x % cat_width;

  priv->cats = g_list_remove (priv->cats, priv->context);
  priv->cats = g_list_insert (priv->cats, priv->context, x/cat_width);

  for (c = priv->cats; c; c = c->next)
  {
    ClutterActor *cat = c->data;
    
    if (cat != priv->context)
    {
      g_object_set_data (G_OBJECT (cat), 
                         "end-x", GINT_TO_POINTER (i * cat_width));
    }

    i++;
  }
  if (!clutter_timeline_is_playing (priv->reorder_time))
    clutter_timeline_start (priv->reorder_time);

  return TRUE;
}

static void
reorder_func (ClutterBehaviour *behave,
              guint32           alpha_value,
              LauncherCatbar   *bar)
{
  LauncherCatbarPrivate *priv;
  GList *c;
  
  g_return_if_fail (LAUNCHER_IS_CATBAR (bar));
  priv = bar->priv;

  for (c = priv->cats; c; c = c->next)
  {
    ClutterActor *actor = c->data;
    gint x, endx, newx;

    if (actor == priv->context)
      continue;

    x = clutter_actor_get_x (actor);
    
    endx = GPOINTER_TO_INT (g_object_get_data (G_OBJECT (actor), "end-x"));

    newx = alpha_value * (endx - x)
            / CLUTTER_ALPHA_MAX_ALPHA
            + x;

    clutter_actor_set_x (actor, newx);
  }
}


static gboolean
on_motion (ClutterActor       *actor,
           ClutterMotionEvent *event,
           LauncherCatbar     *bar)
{
  LauncherCatbarPrivate *priv;
  ClutterActor *stage = clutter_stage_get_default ();
  
  g_return_val_if_fail (LAUNCHER_IS_CATBAR (bar), FALSE);
  priv = bar->priv;

  if (event->modifier_state & CLUTTER_BUTTON1_MASK)
  {
    priv->context = actor;
    priv->offset = event->x;
    
    clutter_actor_raise_top (actor);
    clutter_grab_pointer (stage);
    g_signal_connect (stage, "motion-event",
                      G_CALLBACK (on_drag), bar);
    g_signal_connect (stage, "button-release-event", 
                      G_CALLBACK (on_drop), bar);   
    return TRUE;
  }

  return FALSE;
}

static void
set_active_category (LauncherCatbar *bar, LauncherCategory *hit)
{
  LauncherMenuCategory *cat = g_object_get_data (G_OBJECT (hit), "category");
  GList *c, *cats;

  if (launcher_category_get_active (LAUNCHER_CATEGORY (hit)))
  {
    launcher_category_set_active (LAUNCHER_CATEGORY (hit), FALSE);
    launcher_app_reset (launcher_app_get_default ());

    return;
  }

  cats= clutter_container_get_children (CLUTTER_CONTAINER (bar));
  for (c = cats; c; c= c->next)
  {
    LauncherCategory *lcat = c->data;

    if (LAUNCHER_IS_CATEGORY (lcat))
    {
      if (lcat == LAUNCHER_CATEGORY (hit))
        launcher_category_set_active (lcat, TRUE);
      else
      {
        launcher_category_set_active (lcat, FALSE);
        launcher_category_set_flipped (lcat, FALSE);
      }
    }

  }

  bar->priv->current = cat;
  bar->priv->current_cat = hit;
  g_signal_emit (bar, _catbar_signals[CAT_SELECTED], 0, cat);

  g_list_free (cats);

 }

static gboolean
on_category_clicked (ClutterActor *hit,
                     ClutterButtonEvent *event,
                     LauncherCatbar *bar)
{
  g_return_val_if_fail (LAUNCHER_IS_CATBAR (bar), FALSE);

  if (event->button != 1)
    return FALSE;

  set_active_category (bar, LAUNCHER_CATEGORY (hit));

  return TRUE;
}

static void
reset_categories (LauncherWm *wm, ClutterContainer *bar)
{
  GList *c, *cats;

  g_return_if_fail (LAUNCHER_IS_CATBAR (bar));

  cats= clutter_container_get_children (CLUTTER_CONTAINER (bar));
  for (c = cats; c; c= c->next)
  {
    LauncherCategory *lcat = c->data;

    if (LAUNCHER_IS_CATEGORY (lcat))
    {
        launcher_category_set_active (lcat, FALSE);
    }
  }
  g_list_free (cats);
}

void   
launcher_catbar_set (LauncherCatbar   *bar,
                     LauncherCategory *cat)
{
  g_return_if_fail (LAUNCHER_IS_CATBAR (bar));
  set_active_category (bar, cat);
}

void     
launcher_catbar_unset (LauncherCatbar *bar)
{
  g_return_if_fail (LAUNCHER_IS_CATBAR (bar));

  reset_categories (NULL, CLUTTER_CONTAINER (bar));
}

static void
on_cat_flipped (LauncherCategory *flipped, LauncherCatbar *bar)
{
  LauncherCatbarPrivate *priv;
  GList *cats, *c;

  g_return_if_fail (LAUNCHER_IS_CATBAR (bar));
  priv = bar->priv;

  cats = clutter_container_get_children (CLUTTER_CONTAINER (bar));
  for (c = cats; c; c = c->next)
  {
    LauncherCategory *cat= c->data;

    if (cat != flipped && LAUNCHER_IS_CATEGORY (cat))
      launcher_category_set_flipped (cat, FALSE);
  }
  g_list_free (cats);
}


static LauncherCategory *
_append_category (LauncherCatbar       *bar, 
                  LauncherMenuCategory *category,
                  gint                  size,
                  gint                  offset)
{
  ClutterActor *hit;
  
  /* Hit area */
  hit = launcher_category_new (category);
  clutter_container_add_actor (CLUTTER_CONTAINER (bar), hit);

  clutter_actor_set_position (hit, offset, PANEL_HEIGHT);
  clutter_actor_set_reactive (hit, TRUE);

  clutter_actor_show (hit);
  g_object_set_data (G_OBJECT (hit), "category", category);

  g_signal_connect (hit, "button-release-event", 
                    G_CALLBACK (on_category_clicked), bar);
  g_signal_connect (hit, "motion-event", 
                    G_CALLBACK (on_motion), bar);
  g_signal_connect (hit, "flipped", G_CALLBACK (on_cat_flipped), bar);

  return LAUNCHER_CATEGORY (hit);
}
 

static void
load_categories (LauncherCatbar *catbar)
{
  LauncherCatbarPrivate *priv = catbar->priv;
  LauncherConfig *cfg = launcher_config_get_default ();
  LauncherMenu *menu = launcher_menu_get_default ();
  LauncherCategory *cur = NULL;
  GList *l;
  gint bar_width = (launcher_config_get_default ())->cat_width;
  gint i = 0;
  gboolean current = FALSE;

  for (l = launcher_menu_get_categories (menu); l; l = l->next)
  {
    LauncherMenuCategory *cat = l->data;
    LauncherCategory *category;

    category = _append_category (catbar, cat, bar_width, (bar_width * i));

    if (priv->current == cat && !cur)
    {
      current = TRUE;
      cur = category;
      launcher_category_set_active (category, TRUE);
    }
    if (cat->is_new)
    {
      launcher_category_set_active (cur, FALSE);
      current = TRUE;
      cur = category;
      priv->current = cat;
      cat->is_new = FALSE;
    }

    priv->cats = g_list_append (priv->cats, category);
    
    i++;
    if (i >= cfg->n_cats)
      break;;
  }
  priv->n_cats = i;

  launcher_set_enabled (LAUNCHER_ADD(priv->add), 
                        priv->n_cats < cfg->n_cats);

  if (!current && launcher_menu_get_categories (menu))
  {
    priv->current = launcher_menu_get_categories (menu)->data;
    //launcher_category_set_active (priv->cats->data, TRUE);
  }
  else
  {
    launcher_category_set_active (cur, TRUE);
  }
}

static void
on_menu_changed (LauncherMenu *menu, LauncherCatbar *bar)
{
  GList *actors, *a;
  
  g_return_if_fail (LAUNCHER_IS_CATBAR (bar));

  actors = clutter_container_get_children (CLUTTER_CONTAINER (bar));
  for (a = actors; a; a = a->next)
  {
    if (LAUNCHER_IS_CATEGORY (a->data))
    {
      clutter_drag_server_remove_drag_dest (clutter_drag_server_get_default(),
                                            CLUTTER_DRAG_DEST (a->data));
      clutter_actor_destroy (a->data);
    }
  }
  g_list_free (actors);
  g_list_free (bar->priv->cats);
  bar->priv->cats = NULL;

  load_categories (bar);

  g_signal_emit (bar, _catbar_signals[CAT_SELECTED], 0, bar->priv->current);
}

/*
 * FOCUS STUFF
 */
static void
launcher_catbar_set_focus (ClutterFocusGroup *group, gboolean has_focus)
{
  LauncherCatbarPrivate *priv = LAUNCHER_CATBAR (group)->priv;

  if (has_focus)
  {
    priv->focused = priv->cats ? priv->cats->data : NULL;
  }
  else
  {
    if (priv->focused != priv->current_cat)
      launcher_category_set_active (LAUNCHER_CATEGORY (priv->focused), FALSE);
  }
}

static gboolean
launcher_catbar_direction_event (ClutterFocusGroup     *group,
                                 ClutterFocusDirection  dir)
{
  LauncherCatbarPrivate *priv = LAUNCHER_CATBAR (group)->priv;
  LauncherCategory *new;
  GList *children, *c;
  gint current;
  gint next;

  switch (dir)
  {
    case CLUTTER_DIRECTION_UP:
    case CLUTTER_DIRECTION_DOWN:
      return FALSE;
    default:
      ;
  }
  /* Move to the next or previous category, 
   *  if category is first or last, return FALSE
   *  page up and page down takes you to top and bottom
   */
  children = priv->cats;
  current = next = g_list_index (children, priv->focused);

  switch (dir)
  {
    case CLUTTER_DIRECTION_LEFT:
      next -= 1;
      break;
    case CLUTTER_DIRECTION_RIGHT:
      next += 1;
      break;
    case CLUTTER_DIRECTION_PAGE_UP:
      next = 0;
      break;
    case CLUTTER_DIRECTION_PAGE_DOWN:
      next = g_list_length (children) -1;
      break;
    default:
      break;
  }
  if (next < 0) next = 0;
  next = CLAMP (next, 0, g_list_length (children)-1);
  new = g_list_nth_data (children, next);
  priv->focused = new;
  launcher_category_set_active (new, TRUE);

  for (c = children; c; c= c->next)
  {
    LauncherCategory *category = c->data;

    if (category == LAUNCHER_CATEGORY (priv->current_cat)
        || category == new)
      ;
    else
      launcher_category_set_active (category, FALSE);
  }

  return TRUE;
}

static gboolean
launcher_catbar_action_event (ClutterFocusGroup *group)
{
  LauncherCatbarPrivate *priv = LAUNCHER_CATBAR (group)->priv;

  if (priv->current_cat != priv->focused)
    launcher_category_set_active (priv->focused, FALSE); 
  set_active_category (LAUNCHER_CATBAR (group), priv->focused);
  
  return TRUE;
}

static gboolean
launcher_catbar_key_event (ClutterFocusGroup *group, const gchar *string)
{
  g_debug ("String event: %s", string);

  return TRUE;
}                        

static void
clutter_focus_group_iface_init (ClutterFocusGroupIface *iface)
{
  iface->set_focus       = launcher_catbar_set_focus;
  iface->direction_event = launcher_catbar_direction_event;
  iface->key_event       = launcher_catbar_key_event;
  iface->action_event    = launcher_catbar_action_event;
}

/*
 * /FOCUS STUFF
 */
 
/* GObject stuff */
static void
launcher_catbar_class_init (LauncherCatbarClass *klass)
{
  GObjectClass        *obj_class = G_OBJECT_CLASS (klass);

	_catbar_signals[CAT_SELECTED] =
		g_signal_new ("category-selected",
			      G_OBJECT_CLASS_TYPE (obj_class),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET (LauncherCatbarClass, category_selected),
			      NULL, NULL,
			      g_cclosure_marshal_VOID__POINTER, 
			      G_TYPE_NONE,1, G_TYPE_POINTER);

  g_type_class_add_private (obj_class, sizeof (LauncherCatbarPrivate));
}

     
static void
launcher_catbar_init (LauncherCatbar *catbar)
{
  LauncherCatbarPrivate *priv;
  LauncherMenu *menu = launcher_menu_get_default ();
  LauncherConfig *cfg = launcher_config_get_default ();
  ClutterAlpha *alpha;
  ClutterFocusManager *manager = CLUTTER_FOCUS_MANAGER (clutter_focus_manager_get_default ());
  gint cat_height = cfg->cat_height;
  ClutterActor *shadow;

  priv = catbar->priv = LAUNCHER_CATBAR_GET_PRIVATE (catbar);

  priv->n_cats = 0;
  priv->cats = NULL;
  priv->offset = 0;

  shadow = launcher_util_texture_new_from_file (PKGDATADIR"/shadow.svg");
  clutter_container_add_actor (CLUTTER_CONTAINER (catbar), shadow);
  shadow = tidy_texture_frame_new (CLUTTER_TEXTURE (shadow), 20, 20, 20, 20);
  clutter_container_add_actor (CLUTTER_CONTAINER (catbar), shadow);
  clutter_actor_set_size (shadow, CSW()+20, cat_height+10);
  clutter_actor_set_position (shadow, -10, PANEL_HEIGHT-5);
  if (!cfg->new_theme)
    clutter_actor_show (shadow);

  priv->bg =launcher_util_texture_new_from_file (cfg->catbarbg);
  clutter_container_add_actor (CLUTTER_CONTAINER (catbar), priv->bg);
  clutter_actor_set_size (priv->bg, CSW (), cat_height);
  clutter_actor_set_position (priv->bg, 0, PANEL_HEIGHT);
  clutter_actor_set_opacity (priv->bg, 200);
  clutter_actor_show (priv->bg);

  priv->add = launcher_add_new ();
  clutter_container_add_actor (CLUTTER_CONTAINER (catbar), priv->add);
  clutter_actor_set_size (priv->add, cat_height/4, cat_height/4);
  clutter_actor_set_anchor_point_from_gravity (priv->add, 
                                               CLUTTER_GRAVITY_CENTER);
  clutter_actor_set_position (priv->add, 
                              CSW()-((CSW()/8)/2),
                              (cat_height/2)+PANEL_HEIGHT);
  clutter_actor_show (priv->add);
  clutter_actor_set_reactive (priv->add, TRUE);
  g_signal_connect (priv->add, "clicked",
                    G_CALLBACK (on_add_clicked), catbar);
  clutter_focus_manager_insert_group(CLUTTER_FOCUS_MANAGER (manager),
                                     CLUTTER_FOCUS_GROUP (priv->add),
                                     1);


  load_categories (catbar);

  clutter_actor_set_anchor_point (CLUTTER_ACTOR (catbar), 0, 0);

  priv->reorder_time = launcher_util_timeline_new_slow ();
  alpha = clutter_alpha_new_full (priv->reorder_time, clutter_sine_inc_func,
                                  NULL, NULL);
  priv->reorder_behave = launcher_behave_new (alpha, 
                                      (LauncherBehaveAlphaFunc)reorder_func,
                                      (gpointer)catbar);


  g_signal_connect (menu, "menu-changed",
                    G_CALLBACK (on_menu_changed), catbar);
}

ClutterActor *
launcher_catbar_get_default (void)

{
  static ClutterActor *catbar = NULL;

  if (!LAUNCHER_IS_CATBAR (catbar))
    catbar = g_object_new (LAUNCHER_TYPE_CATBAR, 
                       NULL);

  return catbar;
}
