#
# This file is part of Canola
# Copyright (C) 2007-2009 Instituto Nokia de Tecnologia
# Contact: Renato Chencarek <renato.chencarek@openbossa.org>
#          Eduardo Lima (Etrunko) <eduardo.lima@openbossa.org>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#

import ecore

from terra.core.controller import Controller
from terra.core.manager import Manager
from terra.core.terra_object import TerraObject


mger = Manager()
KineticListWidget = mger.get_class("Widget/KineticList")
ScrollbarWidget = mger.get_class("Widget/Scrollbar")
KnobWidget = mger.get_class("Widget/Knob")
BasicPanel = mger.get_class("Widget/BasicPanel")


class ListPanel(BasicPanel, TerraObject):
    terra_type = "Widget/ListPanel"
    SCROLLBAR_TIMEOUT = 0.7
    SCROLLBAR_FIRST_TIMEOUT = 1.5

    def __init__(self, main_window, title, elements, row_renderer,
                 group=None, theme=None, header_text=None):
        BasicPanel.__init__(self, main_window, title, group=group, theme=theme,
                            header_text=header_text)
        TerraObject.__init__(self)

        self.elements = elements
        self.callback_clicked = None
        self._row_renderer = row_renderer
        self._hide_scroll_timer = None

        self._setup_gui_scrollbar()
        self._setup_gui_list()

        for r in self._list.renderers:
            r.state_default()
            r.unselect()

        self._show_scrollbar(self.SCROLLBAR_FIRST_TIMEOUT)

    def _setup_gui_scrollbar(self):
        knob = KnobWidget(self, group="panel_list_scrollbar")
        self._scroll = ScrollbarWidget(self, knob)
        self._contents.part_swallow("scrollbar", self._scroll)

    def _setup_gui_list(self):
        def renderer_new(canvas):
            return self._row_renderer(self)
        self._list = KineticListWidget(self, renderer_new, self.elements)
        self._list.on_resize_add(self._cb_list_resized)
        self._list.position_changed_cb_set(self._cb_position_changed)
        self._list.clicked_cb_set(self._cb_clicked)
        self._contents.part_swallow("contents", self._list)

    def _cb_list_resized(self, lst):
        self._scroll.scale_set(lst.visible_rows_scale_get())

    def _show_scrollbar(self, timeout=SCROLLBAR_TIMEOUT):
        if self._list.visible_rows_count() < len(self.elements):
            self._hide_scroll_timer = \
                ecore.timer_add(timeout, self._hide_scroll_cb)
            self._contents.signal_emit("scrollbar,transition,in", "")

    def _cb_position_changed(self, lst, percent):
        self._scroll.value_set(percent)
        if self._hide_scroll_timer is not None:
            self._hide_scroll_timer.delete()
        self._hide_scroll_timer = ecore.timer_add(self.SCROLLBAR_TIMEOUT,
                                                  self._hide_scroll_cb)
        self._contents.signal_emit("scrollbar,transition,in", "")

    def _hide_scroll_cb(self):
        self._contents.signal_emit("scrollbar,transition,out", "")
        self._hide_scroll_timer = None
        return False

    def _cb_clicked(self, lst, index):
        self.callback_clicked(self, index)

    def handle_key_down(self, event):
        if event.key == "Escape" and self.callback_escape:
            self.callback_escape()

    def model_updated(self):
        self._list.model_updated()
        self._show_scrollbar()

    def delete(self):
        self._list.delete()
        self._scroll.delete()
        self.elements = None
        BasicPanel.delete(self)

    def theme_changed(self):
        BasicPanel.theme_changed(self)
        self._list.theme_changed()

    def select_item(self, index, end_callback=None):
        self._list.select_item(index, end_callback)


RowRendererFactory = mger.get_class("Widget/SimpleRowRendererFactory")
def row_value_set(self, value):
    self.part_text_set("text", value.name)

class ListPanelController(Controller):
    terra_type = "Controller/Settings/Folder"
    row_renderer = RowRendererFactory(row_value_set, "panel_list_item_text",
                                      provide_resizable_row=False)

    def __init__(self, model, canvas, parent):
        Controller.__init__(self, model, canvas, parent)
        self.model.load()
        self._setup_view()

        # should be after setup UI
        self.model.changed_callback_add(self._update_ui)
        self.model.callback_state_changed = self._model_state_changed

    def _setup_view(self):
        title = self.model.name
        self.view = ListPanel(self.parent.window, title,
                              self.model.children, self.row_renderer)
        self.view.callback_clicked = self.cb_on_clicked
        self.view.callback_escape = self.back

    def _update_ui(self, model):
        self.view.model_updated()

    def _model_state_changed(self, model):
        if not model.is_valid:
            self.parent.propagate_error(self, model.state_reason)

    def delete(self):
        self.model.callback_state_changed = None

        # should be before deleting UI
        self.model.changed_callback_del(self._update_ui)

        self.view.callback_clicked = None
        self.view.delete()
        self.view = None
        self.model.unload()
        self.model = None

    def back(self, end_callback=None):
        self.parent.back(end_callback)

    def use(self, model, end_callback=None):
        self.parent.use(model, end_callback)

    def cb_on_clicked(self, view, index):
        model = self.model.children[index]
        self.model.current = index
        self.view.select_item(index)

        # NOTE: after the animation happens, return the list to its
        # default position. In Screen this is in transition_from, but
        # here, since the list is visible even when there is another.
        def cb(*ignored):
            r = self.view._list.renderer_for_index(self.model.current)
            if r is not None:
                r.force_redraw()
                r.unselect()
        self.use(model, end_callback=cb)


# TODO: remove completely this Settings vs. Options difference
class OptionsFolderController(ListPanelController):
    terra_type = "Controller/Options/Folder"
