/** *********************************************************************
 * Copyright (C) 2003 Catalyst IT                                       *
 *                                                                      *
 * This program is free software; you can redistribute it and/or modify *
 * it under the terms of the GNU General Public License as published by *
 * the Free Software Foundation; either version 2 of the License, or    *
 * (at your option) any later version.                                  *
 *                                                                      *
 * This program is distributed in the hope that it will be useful,      *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 * GNU General Public License for more details.                         *
 *                                                                      *
 * You should have received a copy of the GNU General Public License    *
 * along with this program; if not, write to:                           *
 *   The Free Software Foundation, Inc., 59 Temple Place, Suite 330,    *
 *   Boston, MA  02111-1307  USA                                        *
 ************************************************************************/
package nz.net.catalyst.lucene.server;

import java.io.File;
import java.io.IOException;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import nz.net.catalyst.Log;
import nz.net.catalyst.Util; 
import org.apache.lucene.analysis.Analyzer;
import org.apache.lucene.document.Document;
import org.apache.lucene.index.Term;
import org.apache.lucene.queryParser.ParseException;
import org.apache.lucene.queryParser.QueryParser;
import org.apache.lucene.search.BooleanQuery;
import org.apache.lucene.search.BooleanClause;
import org.apache.lucene.search.Hits;
import org.apache.lucene.search.RangeQuery;
import org.apache.lucene.search.Searcher;
import org.apache.lucene.search.TermQuery;
import org.apache.lucene.search.SortField;
import org.apache.lucene.search.Sort;

/**
 * Execute a Lucene Query.
 * 
 * With the query response the following fields are ALWAYS returned:<p>
 * I: a counter representing the document's place in the result set.<p>
 * RANK: the document's rank relative to the other documents<p>
 * Domian: the domain the document belongs too.  Don't know why this is
 * returned!<p>
 * [your optional fields here]<p>
 */

public class Query implements IPackage, Constants
{
  private final Transmission input;
  private Application application;

  Query(Transmission transmission)
  {
    input = transmission;
  }

  Transmission execute()
  {
    long queryStart = System.currentTimeMillis();
    Transmission response = new Transmission(ECommand.QUERY_RESPONSE);
    response.setSerial(input.getSerial());

    String appName = input.get(APPLICATION, NO_APP);

    try
    {
      application = Application.getAppOrDefault(appName);
    }
    catch (ApplicationMissingException e)
    {
      return error(e.getMessage());
    }
    input.setApplication(application);

    Analyzer analyzer = Application.getAnalyzer(input); // Decide upon an analyzer
    													// THIS IS WHERE WE COULD BOLT IN DIFFERENT ANALYZERS.

    String defaultField = input.get(DEFAULT_FIELD);
    if (defaultField == null)
      defaultField = "UNDEFINED_FIELD"; // Make one up -- it won't matter!

    // Get list of field definitions.
    Map fieldMap = mapFields(input.getFields());

    long beforeParse = System.currentTimeMillis();

    org.apache.lucene.search.Query query;
    try {
        query = buildQuery(analyzer, defaultField, fieldMap);
    } catch (QueryException e) {
        return error(e.getMessage());
    } 

  	long beforeOpenSearcher = System.currentTimeMillis();

    File luceneStoreDir = Application.getIndexDirectory(application);
    int maxSearchers    = Application.maxSearcherInstances(application);
    
    Searcher searcher = null;

    List returnFields = getReturnFields(fieldMap); // List of FieldDef

    
    //hu.origo.lucenetools.iac.IndexAccessControl iac;
  	long beforeSearch;
  	long beforeOutput;
    try {
      // closing idle IndexWriter
      WriterControl.closeIdleWriter(luceneStoreDir);
      
      // opening Index
      
      //searcher = new IndexSearcher(luceneStoreDir.getPath());
      //TESTING USING INDEX SEARCHER CONTROL.  THIS SHOULD LIMIT NUMBER OF FILES OPENED.
      
      //This one (iac) had problems when accessed from multiple threads.
      //iac = hu.origo.lucenetools.iac.IndexAccessControl.getInstance(luceneStoreDir);
      //searcher = iac.getSearcher();
      
      try {
	      nz.net.catalyst.lucene.cache.IndexSearcherCache isc = nz.net.catalyst.lucene.cache.IndexSearcherCache.getInstance(luceneStoreDir);
	      searcher = isc.getSearcher(maxSearchers);
      } catch (IOException e) {
      		Log.error("Query performed on non-existing index!");
      		Log.error("An index needs to have at least one document successfully indexed before Querying.");
      		Log.error(e.getMessage());
      		String message = "Error during query: Index not found. Index documents first or check settings.";
    	  	Log.error(message);
	      	return error(message);
      }

      // Assemble any sort fields specified for this search..
      SortField[] sortfields = getSortFields();     
     
      // Execute either a sorted or unsorted query..
      // searching Index
      beforeSearch = System.currentTimeMillis();
      Hits hits = null;
      int hitCount = 0;
      if (sortfields != null) {
        hits = searcher.search(query, new Sort(sortfields));
        hitCount = hits.length();
      }
      else {
        hits = searcher.search(query);
        hitCount = hits.length();
      }

      Log.info("Found " + hitCount + " results in " + (System.currentTimeMillis() - beforeSearch) + "ms");

      // Find out how many results to return.  Default is all of them
      // need to do this now to ensure the Sort-Limit values make sense.
      int limit = -1;
      try	{
    	  limit = Integer.parseInt(input.get(LIMIT));
    	  if (limit < 0) limit = -1;
      }  catch (Exception e) { /* Ignore */ }

    	// Find out where to starting returning results from.
    	// Default is from result 1 (the first).
    	int first = 1;
    	try {
    	  first = Integer.parseInt(input.get(FIRST));
    	  if (first < 1)
    		first = 1;
    	} catch (Exception e) { /* Ignore */ }
    	--first;


    	beforeOutput = System.currentTimeMillis();

      // retrieving results
      int last = first + limit;
      response.add(COUNT, String.valueOf(hitCount));

      for (int i = first; i != last && i < hitCount; ++i) {
        int index = i;

        Document d = hits.doc(index);
        float score = hits.score(index);
        response.add(I, String.valueOf(i+1)); 
        response.add(RANK, String.valueOf(score));
        for (Iterator fld = returnFields.iterator(); fld.hasNext(); ) {
          FieldDef returnField = (FieldDef)fld.next();
          String value = d.get(returnField.name);

          if (value == null) {
            value = "";
          } else if (returnField.date) {
	            try {
		            long msec = DateField.stringToTime(value);
		            value = String.valueOf(msec / 1000);
	            } catch (java.lang.NumberFormatException e) {
	            	String errorMsg = "Invalid Timestamp for datefield:" + returnField.name + ". " + e.getMessage();
	            	Log.error(errorMsg);
	            	return error(errorMsg);
	            }
          }
          response.add(returnField.name, value);
        }
      }

    } catch (IOException e) {
      String message = "Error during query: " + e.toString();
      Log.error(message);
      return error(message);
    } finally {
      if (searcher != null) {
        try {
          // We must always close the IndexReader!
          searcher.close();
        } catch (Throwable e) {
          String message = "Error while closing IndexReader: " + e.toString();
          Log.error(message);
        }
      }
    }

	long end = System.currentTimeMillis();
	 
    if (Log.willDebug()) {
        Log.debug("ReadQuery: " + String.valueOf(beforeParse - queryStart));
      	Log.debug("Parse/BuildQuery: " +  String.valueOf(beforeOpenSearcher - beforeParse));
      	Log.debug("OpenSearcher: " + String.valueOf(beforeSearch - beforeOpenSearcher));
      	Log.debug("Search: " + String.valueOf(beforeOutput - beforeSearch));
      	Log.debug("Output: " + String.valueOf(end - beforeOutput));
    }
    else {
        Log.info("Query time: "+ String.valueOf(end - queryStart));
    }

    return response;
  }

  /**
   * Generate a BooleanQuery from all the Query terms.
   * Field parameter is the default field if not otherwise specified.
   */
  private org.apache.lucene.search.Query buildQuery(
    Analyzer a, String field, Map fieldMap)
  {

	try {
	  int clauseLimitSystem = Integer.parseInt(System.getProperty(PACKAGE + "ClauseLimitSystem"));
	  BooleanQuery.setMaxClauseCount(clauseLimitSystem);
	  
	  Log.debug("setting max clause count to:" + clauseLimitSystem);
	} catch (NumberFormatException e) { 
	  Log.error("invalid setting for ClauseLimitSystem:" 
	            + System.getProperty(PACKAGE + "ClauseLimitSystem"));
	}

    QueryParser qp = new QueryParser(field, a);
    BooleanQuery result = new BooleanQuery();
    boolean anyTerms = false;

    String domain = input.get(DOMAIN);
    if (domain != null)
    {
      Term domainSearch = new Term(DOMAIN, domain);
      result.add(new TermQuery(domainSearch), BooleanClause.Occur.MUST);
      anyTerms = true;
    }
    
    String query = input.get(QUERY);
    Log.debug("Query (Unparsed): " + query);
    if (query != null)
    {
      try {
        result.add(qp.parse(query), BooleanClause.Occur.MUST);
        anyTerms = true;
	  } catch (BooleanQuery.TooManyClauses e) {
		//Log the error.
		Log.error("Too many clauses in query: " + query);
		
		org.apache.lucene.search.BooleanClause[] b = result.getClauses();
		Log.error(" - Number of clauses before error: " + b.length);
		Log.error(" - Try increasing ClauseLimitSystem in Server.config.");
		throw new QueryException("Too many clauses in query \"" + query +
								 "\"", e);
      } catch (ParseException e) {
      	//Log the error.
      	Log.warn("Error parsing query: " + query);
      	Log.warn(e.getMessage());
      	
		org.apache.lucene.search.BooleanClause[] b = result.getClauses();
		Log.error(" Number of clauses before error: " + b.length);
      	
        throw new QueryException("Error parsing query \"" + query +
                                 "\": " + e.getMessage(), e);
      } catch (org.apache.lucene.queryParser.TokenMgrError e) {
    	  //Log the error.
    	  Log.warn("Error parsing query: " + query);
    	  Log.warn(e.getMessage());
    	  throw new QueryException("Error parsing query \"" + query +
    							   "\": " + e.getMessage(), e);
      }
      
    }

    for (Iterator it = getRangeQueries(fieldMap).iterator(); it.hasNext(); )
    {
		org.apache.lucene.search.Query theRange = null;
	  try {
		theRange = (org.apache.lucene.search.Query)it.next();
		result.add(theRange, BooleanClause.Occur.MUST);
		anyTerms = true;
	  } catch (BooleanQuery.TooManyClauses e) {
		//Log the error.
		Log.error("Too many clauses in query when adding range: " + theRange);
		org.apache.lucene.search.BooleanClause[] b = result.getClauses();
		Log.error(" - Number of clauses before error: " + b.length);
		Log.error(" - Try increasing ClauseLimitSystem in Server.config.");
		throw new QueryException("Too many clauses in query when adding range \"" + theRange +
								 "\"", e);
	  }
    }

    if (!anyTerms)
      throw new QueryException("No search expression!");

	Log.info("Query (Parsed):  " + result.toString(""));
    if (Log.willDebug()) {
        org.apache.lucene.search.BooleanClause[] b = result.getClauses();
		Log.debug(" Number of clauses in query: " + b.length);
  	}
    
    return result;
  }

  private Map mapFields(List fieldList)
  {
    Map result = new HashMap();
    for (Iterator it = fieldList.iterator(); it.hasNext(); )
    {
      FieldDef field = (FieldDef)it.next();
      result.put(field.name, field);
    }
    return result;
  }
  /**
   * Obtain a List of FieldDef objects defining the fields to return.
   * If the list of fields names to return includes any that are not
   * in the list of defined fields, then a default FieldDef is made.
   *
   * @param fieldMap The Map of FieldDef objects describing the
   *                  definedfields.
   * @return A List of FieldDef object of the items to be returned to
   *         the client.  These should all be fields that were stored
   *         in the index.
   */

  private List getReturnFields(Map fieldMap)
  {
    // Get names of fields to return.
    List requestedFields = Arrays.asList(input.get(RETURN, USE_APP, SPLIT));
    List returnFields = new LinkedList(requestedFields);

    // Check for mandatory return field "Id". If not present then add.
    if (!returnFields.contains(ID))
      returnFields.add(0, ID);

    // Build a list of FieldDef objects corresponding to field names.
    List result = new LinkedList();

    // Now scan through the list of field names to return looking for
    // the field definitions for those field names.  If they don't
    // exist, then a new, default field definition is created.
    for (Iterator it = returnFields.iterator(); it.hasNext(); ) {
      String name = (String)it.next();
      FieldDef field = (FieldDef)fieldMap.get(name);

      if (field == null)
        field = new FieldDef(name);

      result.add(field);
    }

    // Return a List of field definitions corresponding to the "Return"
    // header values.
    return result;
  }

  /**
   * Obtain a List of Lucene-Query objects representing the Range searches
   * requested.
   */

  private List getRangeQueries(Map fieldMap) {
    List result = new LinkedList();

    for (Iterator it = input.getRanges().iterator(); it.hasNext(); ) {
      RangeDef range = (RangeDef)it.next();
      if (range.from == null && range.to == null)
        throw new QueryException("Missing From or To value for Range field." + range.name);

      FieldDef field = (FieldDef)fieldMap.get(range.name);
      boolean date = field != null && field.date;

      try {
        String from = range.from;
        if (date && from != null)
          from = DateField.dateToString(Application.makeDate(from));

        String to = range.to;
        if (date && to != null)
          to = DateField.dateToString(Application.makeDate(to));

        result.add(new RangeQuery(
                     from == null ? null : new Term(range.name, from),
                     to == null ? null : new Term(range.name, to),
                     true));
      } catch (IllegalArgumentException e) {
        throw new QueryException("Error in range field \"" + range.name +
                                 "\": " + e.getMessage());
      }
    }
    return result;
  }

  /**
   * Obtain a sort-key specification.  Returns null if we're doing a
   * default descending RANK sort.
   */

  private SortField[] getSortFields()
  {
    String[] fieldSpecs = input.get(SORT, USE_APP, SPLIT);
    if (fieldSpecs.length == 0)
      return null;
    
    SortField[] result = new SortField[fieldSpecs.length];

    for (int i = 0; i < result.length; ++i)
    {
      String[] subSpec = Util.split(fieldSpecs[i], ":");

      if (subSpec.length < 1 || subSpec.length > 2)
        throw new QueryException("Invalid sort spec: \"" + input.get(SORT) + "\"");

      String field = subSpec[0];
      Log.debug("Sorting by " + field + ".");

      if (field.equals(RANK))
      {
          result[i] = new SortField(null, SortField.SCORE);
      } else {
        boolean descending = false;
    
        if (subSpec.length == 2)
        {
          String direction = subSpec[1].toLowerCase();
          if (direction.startsWith("d"))
            Log.debug("Setting sort order for field(" + field + ") to descending.");
            descending = true;
        }
        result[i] = new SortField(field, descending);
      }
    }

    // Check if we're doing a descending simple RANK sort (the default)
    if (result.length == 1 && result[0].getType() == SortField.SCORE && result[0].getReverse() == false) {
        Log.debug("Sort term specified was the default ordering.  Skipping sort request.");
        return null;
    } else {
        return result;
    }

  }
  
  /**
   * Build an error response for sending back to the client.
   *
   * @param message The text of the error message
   * @return An INDEX-RESPONSE Transmission
   */

  private Transmission error(String message)
  {
    Transmission response = new Transmission(ECommand.QUERY_RESPONSE);
    response.setSerial(input.getSerial());
    response.add(ERROR, message);
    return response;
  }

}

