/*
* Copyright 2012-15 Advanced Micro Devices, Inc.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE COPYRIGHT HOLDER(S) OR AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR
 * OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
 * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 *
 * Authors: AMD
 *
 */
#include "dm_services.h"

#include "resource.h"
#include "include/irq_service_interface.h"
#include "link_encoder.h"
#include "stream_encoder.h"
#include "opp.h"
#include "transform.h"
#include "video_csc_types.h"

#if defined(CONFIG_DRM_AMD_DAL_DCE8_0)
#include "dce80/dce80_resource.h"
#endif
#if defined(CONFIG_DRM_AMD_DAL_DCE10_0)
#include "dce100/dce100_resource.h"
#endif
#if defined(CONFIG_DRM_AMD_DAL_DCE11_0)
#include "dce110/dce110_resource.h"
#endif
#if defined(CONFIG_DRM_AMD_DAL_DCE11_2)
#include "dce112/dce112_resource.h"
#endif

enum dce_version resource_parse_asic_id(struct hw_asic_id asic_id)
{
	enum dce_version dc_version = DCE_VERSION_UNKNOWN;
	switch (asic_id.chip_family) {

#if defined(CONFIG_DRM_AMD_DAL_DCE8_0)
	case FAMILY_CI:
	case FAMILY_KV:
		dc_version = DCE_VERSION_8_0;
		break;
#endif
#if defined(CONFIG_DRM_AMD_DAL_DCE11_0)
	case FAMILY_CZ:
		dc_version = DCE_VERSION_11_0;
		break;
#endif

	case FAMILY_VI:
#if defined(CONFIG_DRM_AMD_DAL_DCE10_0)
		if (ASIC_REV_IS_TONGA_P(asic_id.hw_internal_rev) ||
				ASIC_REV_IS_FIJI_P(asic_id.hw_internal_rev)) {
			dc_version = DCE_VERSION_10_0;
			break;
		}
#endif
#if defined(CONFIG_DRM_AMD_DAL_DCE11_2)
		if (ASIC_REV_IS_POLARIS10_P(asic_id.hw_internal_rev) ||
				ASIC_REV_IS_POLARIS11_M(asic_id.hw_internal_rev)) {
			dc_version = DCE_VERSION_11_2;
		}
#endif
		break;
	default:
		dc_version = DCE_VERSION_UNKNOWN;
		break;
	}
	return dc_version;
}

bool dc_construct_resource_pool(struct adapter_service *adapter_serv,
				struct core_dc *dc,
				int num_virtual_links,
				enum dce_version dc_version)
{

	switch (dc_version) {
#if defined(CONFIG_DRM_AMD_DAL_DCE8_0)
	case DCE_VERSION_8_0:
		return dce80_construct_resource_pool(
			adapter_serv, num_virtual_links, dc, &dc->res_pool);
#endif
#if defined(CONFIG_DRM_AMD_DAL_DCE10_0)
	case DCE_VERSION_10_0:
		return dce100_construct_resource_pool(
			adapter_serv, num_virtual_links, dc, &dc->res_pool);
#endif
#if defined(CONFIG_DRM_AMD_DAL_DCE11_0)
	case DCE_VERSION_11_0:
		return dce110_construct_resource_pool(
			adapter_serv, num_virtual_links, dc, &dc->res_pool);
#endif
#if defined(CONFIG_DRM_AMD_DAL_DCE11_2)
	case DCE_VERSION_11_2:
		return dce112_construct_resource_pool(
			adapter_serv, num_virtual_links, dc, &dc->res_pool);
#endif
	default:
		break;
	}

	return false;
}

void resource_unreference_clock_source(
		struct resource_context *res_ctx,
		struct clock_source *clock_source)
{
	int i;
	for (i = 0; i < res_ctx->pool.clk_src_count; i++) {
		if (res_ctx->pool.clock_sources[i] != clock_source)
			continue;

		res_ctx->clock_source_ref_count[i]--;

		if (res_ctx->clock_source_ref_count[i] == 0)
			clock_source->funcs->cs_power_down(clock_source);

		break;
	}

	if (res_ctx->pool.dp_clock_source == clock_source) {
		res_ctx->dp_clock_source_ref_count--;

		if (res_ctx->dp_clock_source_ref_count == 0)
			clock_source->funcs->cs_power_down(clock_source);
	}
}

void resource_reference_clock_source(
		struct resource_context *res_ctx,
		struct clock_source *clock_source)
{
	int i;
	for (i = 0; i < res_ctx->pool.clk_src_count; i++) {
		if (res_ctx->pool.clock_sources[i] != clock_source)
			continue;

		res_ctx->clock_source_ref_count[i]++;
		break;
	}

	if (res_ctx->pool.dp_clock_source == clock_source)
		res_ctx->dp_clock_source_ref_count++;
}

bool resource_are_streams_timing_synchronizable(
	const struct core_stream *stream1,
	const struct core_stream *stream2)
{
	if (stream1->public.timing.h_total != stream2->public.timing.h_total)
		return false;

	if (stream1->public.timing.v_total != stream2->public.timing.v_total)
		return false;

	if (stream1->public.timing.h_addressable
				!= stream2->public.timing.h_addressable)
		return false;

	if (stream1->public.timing.v_addressable
				!= stream2->public.timing.v_addressable)
		return false;

	if (stream1->public.timing.pix_clk_khz
				!= stream2->public.timing.pix_clk_khz)
		return false;

	if (stream1->phy_pix_clk != stream2->phy_pix_clk
			&& !dc_is_dp_signal(stream1->signal)
			&& !dc_is_dp_signal(stream2->signal))
		return false;

	return true;
}

static bool is_sharable_clk_src(
	const struct pipe_ctx *pipe_with_clk_src,
	const struct pipe_ctx *pipe)
{
	if (pipe_with_clk_src->clock_source == NULL)
		return false;

	if (dc_is_dp_signal(pipe_with_clk_src->stream->signal))
		return false;

	if (dc_is_hdmi_signal(pipe_with_clk_src->stream->signal)
			&& dc_is_dvi_signal(pipe->stream->signal))
		return false;

	if (dc_is_hdmi_signal(pipe->stream->signal)
			&& dc_is_dvi_signal(pipe_with_clk_src->stream->signal))
		return false;

	if (!resource_are_streams_timing_synchronizable(
			pipe_with_clk_src->stream, pipe->stream))
		return false;

	return true;
}

struct clock_source *resource_find_used_clk_src_for_sharing(
					struct resource_context *res_ctx,
					struct pipe_ctx *pipe_ctx)
{
	int i;

	for (i = 0; i < MAX_PIPES; i++) {
		if (is_sharable_clk_src(&res_ctx->pipe_ctx[i], pipe_ctx))
			return res_ctx->pipe_ctx[i].clock_source;
	}

	return NULL;
}

static enum pixel_format convert_pixel_format_to_dalsurface(
		enum surface_pixel_format surface_pixel_format)
{
	enum pixel_format dal_pixel_format = PIXEL_FORMAT_UNKNOWN;

	switch (surface_pixel_format) {
	case SURFACE_PIXEL_FORMAT_GRPH_PALETA_256_COLORS:
		dal_pixel_format = PIXEL_FORMAT_INDEX8;
		break;
	case SURFACE_PIXEL_FORMAT_GRPH_ARGB1555:
		dal_pixel_format = PIXEL_FORMAT_RGB565;
		break;
	case SURFACE_PIXEL_FORMAT_GRPH_RGB565:
		dal_pixel_format = PIXEL_FORMAT_RGB565;
		break;
	case SURFACE_PIXEL_FORMAT_GRPH_ARGB8888:
		dal_pixel_format = PIXEL_FORMAT_ARGB8888;
		break;
	case SURFACE_PIXEL_FORMAT_GRPH_BGRA8888:
		dal_pixel_format = PIXEL_FORMAT_ARGB8888;
		break;
	case SURFACE_PIXEL_FORMAT_GRPH_ARGB2101010:
		dal_pixel_format = PIXEL_FORMAT_ARGB2101010;
		break;
	case SURFACE_PIXEL_FORMAT_GRPH_ABGR2101010:
		dal_pixel_format = PIXEL_FORMAT_ARGB2101010;
		break;
	case SURFACE_PIXEL_FORMAT_GRPH_ABGR2101010_XR_BIAS:
		dal_pixel_format = PIXEL_FORMAT_ARGB2101010_XRBIAS;
		break;
	case SURFACE_PIXEL_FORMAT_GRPH_ARGB16161616:
		dal_pixel_format = PIXEL_FORMAT_FP16;
		break;
	case SURFACE_PIXEL_FORMAT_GRPH_ABGR16161616F:
		dal_pixel_format = PIXEL_FORMAT_FP16;
		break;

	case SURFACE_PIXEL_FORMAT_VIDEO_420_YCbCr:
		dal_pixel_format = PIXEL_FORMAT_420BPP12;
		break;
	case SURFACE_PIXEL_FORMAT_VIDEO_420_YCrCb:
		dal_pixel_format = PIXEL_FORMAT_420BPP12;
		break;

	default:
		dal_pixel_format = PIXEL_FORMAT_UNKNOWN;
		break;
	}
	return dal_pixel_format;
}

static void calculate_viewport(
		const struct dc_surface *surface,
		struct pipe_ctx *pipe_ctx)
{
	const struct rect src = surface->src_rect;
	const struct rect clip = surface->clip_rect;
	const struct rect dst = surface->dst_rect;

	/* offset = src.ofs + (clip.ofs - dst.ofs) * scl_ratio
	 * num_pixels = clip.num_pix * scl_ratio
	 */
	pipe_ctx->scl_data.viewport.x = src.x + (clip.x - dst.x) * src.width / dst.width;
	pipe_ctx->scl_data.viewport.width = clip.width * src.width / dst.width;

	pipe_ctx->scl_data.viewport.y = src.y + (clip.y - dst.y) * src.height / dst.height;
	pipe_ctx->scl_data.viewport.height = clip.height * src.height / dst.height;

	/* Minimum viewport such that 420/422 chroma vp is non 0 */
	if (pipe_ctx->scl_data.viewport.width < 2)
		pipe_ctx->scl_data.viewport.width = 2;
	if (pipe_ctx->scl_data.viewport.height < 2)
		pipe_ctx->scl_data.viewport.height = 2;
}

static void calculate_recout(
		const struct dc_surface *surface,
		struct pipe_ctx *pipe_ctx)
{
	struct core_stream *stream = pipe_ctx->stream;

	pipe_ctx->scl_data.recout.x = stream->public.dst.x;
	if (stream->public.src.x < surface->clip_rect.x)
		pipe_ctx->scl_data.recout.x += (surface->clip_rect.x
			- stream->public.src.x) * stream->public.dst.width
						/ stream->public.src.width;

	pipe_ctx->scl_data.recout.width = surface->clip_rect.width *
			stream->public.dst.width / stream->public.src.width;
	if (pipe_ctx->scl_data.recout.width + pipe_ctx->scl_data.recout.x >
			stream->public.dst.x + stream->public.dst.width)
		pipe_ctx->scl_data.recout.width =
			stream->public.dst.x + stream->public.dst.width
						- pipe_ctx->scl_data.recout.x;

	pipe_ctx->scl_data.recout.y = stream->public.dst.y;
	if (stream->public.src.y < surface->clip_rect.y)
		pipe_ctx->scl_data.recout.y += (surface->clip_rect.y
			- stream->public.src.y) * stream->public.dst.height
						/ stream->public.src.height;

	pipe_ctx->scl_data.recout.height = surface->clip_rect.height *
			stream->public.dst.height / stream->public.src.height;
	if (pipe_ctx->scl_data.recout.height + pipe_ctx->scl_data.recout.y >
			stream->public.dst.y + stream->public.dst.height)
		pipe_ctx->scl_data.recout.height =
			stream->public.dst.y + stream->public.dst.height
						- pipe_ctx->scl_data.recout.y;
}

static void calculate_scaling_ratios(
		const struct dc_surface *surface,
		struct pipe_ctx *pipe_ctx)
{
	struct core_stream *stream = pipe_ctx->stream;
	const uint32_t in_w = stream->public.src.width;
	const uint32_t in_h = stream->public.src.height;
	const uint32_t out_w = stream->public.dst.width;
	const uint32_t out_h = stream->public.dst.height;

	pipe_ctx->scl_data.ratios.horz = dal_fixed31_32_from_fraction(
					surface->src_rect.width,
					surface->dst_rect.width);
	pipe_ctx->scl_data.ratios.vert = dal_fixed31_32_from_fraction(
					surface->src_rect.height,
					surface->dst_rect.height);

	if (surface->stereo_format == PLANE_STEREO_FORMAT_SIDE_BY_SIDE)
		pipe_ctx->scl_data.ratios.horz.value *= 2;
	else if (surface->stereo_format == PLANE_STEREO_FORMAT_TOP_AND_BOTTOM)
		pipe_ctx->scl_data.ratios.vert.value *= 2;

	pipe_ctx->scl_data.ratios.vert.value = div64_s64(
		pipe_ctx->scl_data.ratios.vert.value * in_h, out_h);
	pipe_ctx->scl_data.ratios.horz.value = div64_s64(
		pipe_ctx->scl_data.ratios.horz.value * in_w, out_w);

	pipe_ctx->scl_data.ratios.horz.value++;
	pipe_ctx->scl_data.ratios.vert.value++;

	pipe_ctx->scl_data.ratios.horz_c = pipe_ctx->scl_data.ratios.horz;
	pipe_ctx->scl_data.ratios.vert_c = pipe_ctx->scl_data.ratios.vert;

	if (pipe_ctx->scl_data.format == PIXEL_FORMAT_420BPP12) {
		pipe_ctx->scl_data.ratios.horz_c.value /= 2;
		pipe_ctx->scl_data.ratios.vert_c.value /= 2;
	}
}

void resource_build_scaling_params(
	const struct dc_surface *surface,
	struct pipe_ctx *pipe_ctx)
{
	/* Important: scaling ratio calculation requires pixel format,
	 * lb depth calculation requires recout and taps require scaling ratios.
	 */
	pipe_ctx->scl_data.format = convert_pixel_format_to_dalsurface(surface->format);

	calculate_viewport(surface, pipe_ctx);

	calculate_scaling_ratios(surface, pipe_ctx);

	calculate_recout(surface, pipe_ctx);

	pipe_ctx->scl_data.h_active = pipe_ctx->stream->public.timing.h_addressable;
	pipe_ctx->scl_data.v_active = pipe_ctx->stream->public.timing.v_addressable;

	/* Check if scaling is required update taps if not */
	if (dal_fixed31_32_u2d19(pipe_ctx->scl_data.ratios.horz) == 1 << 19)
		pipe_ctx->scl_data.taps.h_taps = 1;
	else
		pipe_ctx->scl_data.taps.h_taps = surface->scaling_quality.h_taps;

	if (dal_fixed31_32_u2d19(pipe_ctx->scl_data.ratios.horz_c) == 1 << 19)
		pipe_ctx->scl_data.taps.h_taps_c = 1;
	else
		pipe_ctx->scl_data.taps.h_taps_c = surface->scaling_quality.h_taps_c;

	if (dal_fixed31_32_u2d19(pipe_ctx->scl_data.ratios.vert) == 1 << 19)
		pipe_ctx->scl_data.taps.v_taps = 1;
	else
		pipe_ctx->scl_data.taps.v_taps = surface->scaling_quality.v_taps;

	if (dal_fixed31_32_u2d19(pipe_ctx->scl_data.ratios.vert_c) == 1 << 19)
		pipe_ctx->scl_data.taps.v_taps_c = 1;
	else
		pipe_ctx->scl_data.taps.v_taps_c = surface->scaling_quality.v_taps_c;

	dal_logger_write(pipe_ctx->stream->ctx->logger,
				LOG_MAJOR_DCP,
				LOG_MINOR_DCP_SCALER,
				"%s: Viewport:\nheight:%d width:%d x:%d "
				"y:%d\n dst_rect:\nheight:%d width:%d x:%d "
				"y:%d\n",
				__func__,
				pipe_ctx->scl_data.viewport.height,
				pipe_ctx->scl_data.viewport.width,
				pipe_ctx->scl_data.viewport.x,
				pipe_ctx->scl_data.viewport.y,
				surface->dst_rect.height,
				surface->dst_rect.width,
				surface->dst_rect.x,
				surface->dst_rect.y);
}


void resource_build_scaling_params_for_context(
	const struct core_dc *dc,
	struct validate_context *context)
{
	int i;

	for (i = 0; i < MAX_PIPES; i++) {
		if (context->res_ctx.pipe_ctx[i].surface != NULL &&
				context->res_ctx.pipe_ctx[i].stream != NULL)
			resource_build_scaling_params(
				&context->res_ctx.pipe_ctx[i].surface->public,
				&context->res_ctx.pipe_ctx[i]);
	}
}

bool resource_attach_surfaces_to_context(
		struct dc_surface *surfaces[],
		int surface_count,
		const struct dc_target *dc_target,
		struct validate_context *context)
{
	int i, j, k;
	struct dc_target_status *target_status = NULL;

	if (surface_count > MAX_SURFACE_NUM) {
		dm_error("Surface: can not attach %d surfaces! Maximum is: %d\n",
			surface_count, MAX_SURFACE_NUM);
		return false;
	}

	for (i = 0; i < context->target_count; i++)
		if (&context->targets[i]->public == dc_target) {
			target_status = &context->target_status[i];
			break;
		}
	if (target_status == NULL) {
		dm_error("Existing target not found; failed to attach surfaces\n");
		return false;
	}

	/* retain new surfaces */
	for (i = 0; i < surface_count; i++)
		dc_surface_retain(surfaces[i]);

	/* release existing surfaces*/
	for (i = 0; i < target_status->surface_count; i++) {
		dc_surface_release(target_status->surfaces[i]);
		target_status->surfaces[i] = NULL;
	}

	/* assign new surfaces*/
	for (i = 0; i < surface_count; i++)
		target_status->surfaces[i] = surfaces[i];

	target_status->surface_count = surface_count;

	for (i = 0; i < dc_target->stream_count; i++) {
		k = 0;
		for (j = 0; j < MAX_PIPES; j++) {
			struct core_surface *surface = NULL;

			/* Skip surface assignment for child pipes */
			if (context->res_ctx.pipe_ctx[j].primary_pipe != NULL)
				continue;

			if (surface_count)
				surface = DC_SURFACE_TO_CORE(surfaces[k]);

			if (context->res_ctx.pipe_ctx[j].stream !=
				DC_STREAM_TO_CORE(dc_target->streams[i]))
				continue;

			context->res_ctx.pipe_ctx[j].surface = surface;
			k++;
		}
	}

	return true;
}

static bool are_streams_same(
	const struct core_stream *stream_a, const struct core_stream *stream_b)
{
	if (stream_a == stream_b)
		return true;

	if (stream_a == NULL || stream_b == NULL)
		return false;

	if (memcmp(stream_a, stream_b, sizeof(struct core_stream)) == 0)
		return true;

	return false;
}

static bool is_target_unchanged(
	const struct core_target *old_target, const struct core_target *target)
{
	int i;

	if (old_target == target)
		return true;
	if (old_target->public.stream_count != target->public.stream_count)
		return false;

	for (i = 0; i < old_target->public.stream_count; i++) {
		const struct core_stream *old_stream = DC_STREAM_TO_CORE(
				old_target->public.streams[i]);
		const struct core_stream *stream = DC_STREAM_TO_CORE(
				target->public.streams[i]);

		if (!are_streams_same(old_stream, stream))
			return false;
	}

	return true;
}

bool resource_validate_attach_surfaces(
		const struct dc_validation_set set[],
		int set_count,
		const struct validate_context *old_context,
		struct validate_context *context)
{
	int i, j;

	for (i = 0; i < set_count; i++) {
		context->targets[i] = DC_TARGET_TO_CORE(set[i].target);
		dc_target_retain(&context->targets[i]->public);
		context->target_count++;

		for (j = 0; j < old_context->target_count; j++)
			if (is_target_unchanged(
					old_context->targets[j],
					context->targets[i])) {
				if (!resource_attach_surfaces_to_context(
						(struct dc_surface **) old_context->
							target_status[j].surfaces,
						old_context->target_status[j].surface_count,
						&context->targets[i]->public,
						context))
					return false;
				context->target_status[i] = old_context->target_status[j];
			}
		if (set[i].surface_count != 0)
			if (!resource_attach_surfaces_to_context(
					(struct dc_surface **) set[i].surfaces,
					set[i].surface_count,
					&context->targets[i]->public,
					context))
				return false;

	}

	return true;
}

/* Maximum TMDS single link pixel clock 165MHz */
#define TMDS_MAX_PIXEL_CLOCK_IN_KHZ 165000

static void set_stream_engine_in_use(
		struct resource_context *res_ctx,
		struct stream_encoder *stream_enc)
{
	int i;

	for (i = 0; i < res_ctx->pool.stream_enc_count; i++) {
		if (res_ctx->pool.stream_enc[i] == stream_enc)
			res_ctx->is_stream_enc_acquired[i] = true;
	}
}

/* TODO: release audio object */
static void set_audio_in_use(
		struct resource_context *res_ctx,
		struct audio *audio)
{
	int i;
	for (i = 0; i < res_ctx->pool.audio_count; i++) {
		if (res_ctx->pool.audios[i] == audio) {
			res_ctx->is_audio_acquired[i] = true;
		}
	}
}

static int acquire_first_free_pipe(
		struct resource_context *res_ctx,
		struct core_stream *stream)
{
	int i;

	for (i = 0; i < res_ctx->pool.pipe_count; i++) {
		if (!res_ctx->pipe_ctx[i].stream) {
			struct pipe_ctx *pipe_ctx = &res_ctx->pipe_ctx[i];

			pipe_ctx->tg = res_ctx->pool.timing_generators[i];
			pipe_ctx->mi = res_ctx->pool.mis[i];
			pipe_ctx->ipp = res_ctx->pool.ipps[i];
			pipe_ctx->xfm = res_ctx->pool.transforms[i];
			pipe_ctx->opp = res_ctx->pool.opps[i];
			pipe_ctx->dis_clk = res_ctx->pool.display_clock;
			pipe_ctx->pipe_idx = i;

			pipe_ctx->stream = stream;
			return i;
		}
	}
	return -1;
}

static struct stream_encoder *find_first_free_match_stream_enc_for_link(
		struct resource_context *res_ctx,
		struct core_stream *stream)
{
	int i;
	int j = -1;
	struct core_link *link = stream->sink->link;

	for (i = 0; i < res_ctx->pool.stream_enc_count; i++) {
		if (!res_ctx->is_stream_enc_acquired[i] &&
					res_ctx->pool.stream_enc[i]) {
			/* Store first available for MST second display
			 * in daisy chain use case */
			j = i;
			if (res_ctx->pool.stream_enc[i]->id ==
					link->link_enc->preferred_engine)
				return res_ctx->pool.stream_enc[i];
		}
	}

	/*
	 * below can happen in cases when stream encoder is acquired:
	 * 1) for second MST display in chain, so preferred engine already
	 * acquired;
	 * 2) for another link, which preferred engine already acquired by any
	 * MST configuration.
	 *
	 * If signal is of DP type and preferred engine not found, return last available
	 *
	 * TODO - This is just a patch up and a generic solution is
	 * required for non DP connectors.
	 */

	if (j >= 0 && dc_is_dp_signal(stream->signal))
		return res_ctx->pool.stream_enc[j];

	return NULL;
}

static struct audio *find_first_free_audio(struct resource_context *res_ctx)
{
	int i;
	for (i = 0; i < res_ctx->pool.audio_count; i++) {
		if (res_ctx->is_audio_acquired[i] == false) {
			return res_ctx->pool.audios[i];
		}
	}

	return 0;
}

static void update_stream_signal(struct core_stream *stream)
{
	const struct dc_sink *dc_sink = stream->public.sink;

	stream->signal = dc_sink->sink_signal;
	/* For asic supports dual link DVI, we should adjust signal type
	 * based on timing pixel clock. If pixel clock more than 165Mhz,
	 * signal is dual link, otherwise, single link.
	 */
	if (dc_sink->sink_signal == SIGNAL_TYPE_DVI_SINGLE_LINK ||
			dc_sink->sink_signal == SIGNAL_TYPE_DVI_DUAL_LINK) {
		if (stream->public.timing.pix_clk_khz >
						TMDS_MAX_PIXEL_CLOCK_IN_KHZ)
			stream->signal = SIGNAL_TYPE_DVI_DUAL_LINK;
		else
			stream->signal = SIGNAL_TYPE_DVI_SINGLE_LINK;
	}
}

bool resource_is_stream_unchanged(
	const struct validate_context *old_context, struct core_stream *stream)
{
	int i, j;

	for (i = 0; i < old_context->target_count; i++) {
		struct core_target *old_target = old_context->targets[i];

		for (j = 0; j < old_target->public.stream_count; j++) {
			struct core_stream *old_stream =
				DC_STREAM_TO_CORE(old_target->public.streams[j]);

			if (are_streams_same(old_stream, stream))
				return true;
		}
	}

	return false;
}

static void copy_pipe_ctx(
	const struct pipe_ctx *from_pipe_ctx, struct pipe_ctx *to_pipe_ctx)
{
	struct core_surface *surface = to_pipe_ctx->surface;
	struct core_stream *stream = to_pipe_ctx->stream;

	*to_pipe_ctx = *from_pipe_ctx;
	to_pipe_ctx->stream = stream;
	if (surface != NULL)
		to_pipe_ctx->surface = surface;
}

static struct core_stream *find_pll_sharable_stream(
		const struct core_stream *stream_needs_pll,
		struct validate_context *context)
{
	int i, j;

	for (i = 0; i < context->target_count; i++) {
		struct core_target *target = context->targets[i];

		for (j = 0; j < target->public.stream_count; j++) {
			struct core_stream *stream_has_pll =
				DC_STREAM_TO_CORE(target->public.streams[j]);

			/* We are looking for non dp, non virtual stream */
			if (resource_are_streams_timing_synchronizable(
						stream_needs_pll, stream_has_pll)
				&& !dc_is_dp_signal(stream_has_pll->signal)
				&& stream_has_pll->sink->link->public.connector_signal
							!= SIGNAL_TYPE_VIRTUAL)
					return stream_has_pll;
		}
	}

	return NULL;
}

static int get_norm_pix_clk(const struct dc_crtc_timing *timing)
{
	uint32_t pix_clk = timing->pix_clk_khz;
	uint32_t normalized_pix_clk = pix_clk;

	if (timing->pixel_encoding == PIXEL_ENCODING_YCBCR420)
		pix_clk /= 2;

	switch (timing->display_color_depth) {
	case COLOR_DEPTH_888:
		normalized_pix_clk = pix_clk;
		break;
	case COLOR_DEPTH_101010:
		normalized_pix_clk = (pix_clk * 30) / 24;
		break;
	case COLOR_DEPTH_121212:
		normalized_pix_clk = (pix_clk * 36) / 24;
		break;
	case COLOR_DEPTH_161616:
		normalized_pix_clk = (pix_clk * 48) / 24;
		break;
	default:
		ASSERT(0);
		break;
	}

	return normalized_pix_clk;
}

static void calculate_phy_pix_clks(
		const struct core_dc *dc,
		struct validate_context *context)
{
	int i, j;

	for (i = 0; i < context->target_count; i++) {
		struct core_target *target = context->targets[i];

		for (j = 0; j < target->public.stream_count; j++) {
			struct core_stream *stream =
				DC_STREAM_TO_CORE(target->public.streams[j]);

			update_stream_signal(stream);

			/* update actual pixel clock on all streams */
			if (dc_is_hdmi_signal(stream->signal))
				stream->phy_pix_clk = get_norm_pix_clk(
					&stream->public.timing);
			else
				stream->phy_pix_clk =
						stream->public.timing.pix_clk_khz;
		}
	}
}

enum dc_status resource_map_pool_resources(
		const struct core_dc *dc,
		struct validate_context *context)
{
	int i, j, k;

	calculate_phy_pix_clks(dc, context);

	for (i = 0; i < context->target_count; i++) {
		struct core_target *target = context->targets[i];

		for (j = 0; j < target->public.stream_count; j++) {
			struct core_stream *stream =
				DC_STREAM_TO_CORE(target->public.streams[j]);

			if (!resource_is_stream_unchanged(dc->current_context, stream))
				continue;

			/* mark resources used for stream that is already active */
			for (k = 0; k < MAX_PIPES; k++) {
				struct pipe_ctx *pipe_ctx =
					&context->res_ctx.pipe_ctx[k];
				const struct pipe_ctx *old_pipe_ctx =
					&dc->current_context->res_ctx.pipe_ctx[k];

				if (!are_streams_same(old_pipe_ctx->stream, stream))
					continue;

				pipe_ctx->stream = stream;
				copy_pipe_ctx(old_pipe_ctx, pipe_ctx);

				set_stream_engine_in_use(
					&context->res_ctx,
					pipe_ctx->stream_enc);

				/* Switch to dp clock source only if there is
				 * no non dp stream that shares the same timing
				 * with the dp stream.
				 */
				if (dc_is_dp_signal(pipe_ctx->stream->signal) &&
					!find_pll_sharable_stream(stream, context))
					pipe_ctx->clock_source =
						context->res_ctx.pool.dp_clock_source;

				resource_reference_clock_source(
					&context->res_ctx,
					pipe_ctx->clock_source);

				set_audio_in_use(&context->res_ctx,
					pipe_ctx->audio);
			}
		}
	}

	for (i = 0; i < context->target_count; i++) {
		struct core_target *target = context->targets[i];

		for (j = 0; j < target->public.stream_count; j++) {
			struct core_stream *stream =
				DC_STREAM_TO_CORE(target->public.streams[j]);
			struct pipe_ctx *pipe_ctx = NULL;
			int pipe_idx = -1;

			if (resource_is_stream_unchanged(dc->current_context, stream))
				continue;
			/* acquire new resources */
			pipe_idx = acquire_first_free_pipe(
						&context->res_ctx, stream);
			if (pipe_idx < 0)
				return DC_NO_CONTROLLER_RESOURCE;


			pipe_ctx = &context->res_ctx.pipe_ctx[pipe_idx];

			pipe_ctx->stream_enc =
				find_first_free_match_stream_enc_for_link(
					&context->res_ctx, stream);

			if (!pipe_ctx->stream_enc)
				return DC_NO_STREAM_ENG_RESOURCE;

			set_stream_engine_in_use(
					&context->res_ctx,
					pipe_ctx->stream_enc);

			/* TODO: Add check if ASIC support and EDID audio */
			if (!stream->sink->converter_disable_audio &&
						dc_is_audio_capable_signal(pipe_ctx->stream->signal) &&
						stream->public.audio_info.mode_count) {
				pipe_ctx->audio = find_first_free_audio(
						&context->res_ctx);

				/*
				 * Audio assigned in order first come first get.
				 * There are asics which has number of audio
				 * resources less then number of pipes
				 */
				if (pipe_ctx->audio)
					set_audio_in_use(
						&context->res_ctx,
						pipe_ctx->audio);
			}
		}
	}

	return DC_OK;
}

/* first target in the context is used to populate the rest */
void validate_guaranteed_copy_target(
		struct validate_context *context,
		int max_targets)
{
	int i;

	for (i = 1; i < max_targets; i++) {
		context->targets[i] = context->targets[0];

		copy_pipe_ctx(&context->res_ctx.pipe_ctx[0],
			      &context->res_ctx.pipe_ctx[i]);
		context->res_ctx.pipe_ctx[i].stream =
				context->res_ctx.pipe_ctx[0].stream;

		dc_target_retain(&context->targets[i]->public);
		context->target_count++;
	}
}

static void translate_info_frame(const struct hw_info_frame *hw_info_frame,
	struct encoder_info_frame *encoder_info_frame)
{
	memset(
		encoder_info_frame, 0, sizeof(struct encoder_info_frame));

	/* For gamut we recalc checksum */
	if (hw_info_frame->gamut_packet.valid) {
		uint8_t chk_sum = 0;
		uint8_t *ptr;
		uint8_t i;

		memmove(
						&encoder_info_frame->gamut,
						&hw_info_frame->gamut_packet,
						sizeof(struct hw_info_packet));

		/*start of the Gamut data. */
		ptr = &encoder_info_frame->gamut.sb[3];

		for (i = 0; i <= encoder_info_frame->gamut.sb[1]; i++)
			chk_sum += ptr[i];

		encoder_info_frame->gamut.sb[2] = (uint8_t) (0x100 - chk_sum);
	}

	if (hw_info_frame->avi_info_packet.valid) {
		memmove(
						&encoder_info_frame->avi,
						&hw_info_frame->avi_info_packet,
						sizeof(struct hw_info_packet));
	}

	if (hw_info_frame->vendor_info_packet.valid) {
		memmove(
						&encoder_info_frame->vendor,
						&hw_info_frame->vendor_info_packet,
						sizeof(struct hw_info_packet));
	}

	if (hw_info_frame->spd_packet.valid) {
		memmove(
						&encoder_info_frame->spd,
						&hw_info_frame->spd_packet,
						sizeof(struct hw_info_packet));
	}

	if (hw_info_frame->vsc_packet.valid) {
		memmove(
						&encoder_info_frame->vsc,
						&hw_info_frame->vsc_packet,
						sizeof(struct hw_info_packet));
	}
}

static void set_avi_info_frame(
	struct hw_info_packet *info_packet,
		struct pipe_ctx *pipe_ctx)
{
	struct core_stream *stream = pipe_ctx->stream;
	enum dc_color_space color_space = COLOR_SPACE_UNKNOWN;
	struct info_frame info_frame = { {0} };
	uint32_t pixel_encoding = 0;
	enum scanning_type scan_type = SCANNING_TYPE_NODATA;
	enum dc_aspect_ratio aspect = ASPECT_RATIO_NO_DATA;
	bool itc = false;
	uint8_t cn0_cn1 = 0;
	uint8_t *check_sum = NULL;
	uint8_t byte_index = 0;

	if (info_packet == NULL)
		return;

	color_space = pipe_ctx->stream->public.output_color_space;

	/* Initialize header */
	info_frame.avi_info_packet.info_packet_hdmi.bits.header.
			info_frame_type = INFO_FRAME_AVI;
	/* InfoFrameVersion_3 is defined by CEA861F (Section 6.4), but shall
	* not be used in HDMI 2.0 (Section 10.1) */
	info_frame.avi_info_packet.info_packet_hdmi.bits.header.version =
			INFO_FRAME_VERSION_2;
	info_frame.avi_info_packet.info_packet_hdmi.bits.header.length =
			INFO_FRAME_SIZE_AVI;

	/*
	 * IDO-defined (Y2,Y1,Y0 = 1,1,1) shall not be used by devices built
	 * according to HDMI 2.0 spec (Section 10.1)
	 */

	switch (stream->public.timing.pixel_encoding) {
	case PIXEL_ENCODING_YCBCR422:
		pixel_encoding = 1;
		break;

	case PIXEL_ENCODING_YCBCR444:
		pixel_encoding = 2;
		break;
	case PIXEL_ENCODING_YCBCR420:
		pixel_encoding = 3;
		break;

	case PIXEL_ENCODING_RGB:
	default:
		pixel_encoding = 0;
	}

	/* Y0_Y1_Y2 : The pixel encoding */
	/* H14b AVI InfoFrame has extension on Y-field from 2 bits to 3 bits */
	info_frame.avi_info_packet.info_packet_hdmi.bits.Y0_Y1_Y2 =
		pixel_encoding;

	/* A0 = 1 Active Format Information valid */
	info_frame.avi_info_packet.info_packet_hdmi.bits.A0 =
		ACTIVE_FORMAT_VALID;

	/* B0, B1 = 3; Bar info data is valid */
	info_frame.avi_info_packet.info_packet_hdmi.bits.B0_B1 =
		BAR_INFO_BOTH_VALID;

	info_frame.avi_info_packet.info_packet_hdmi.bits.SC0_SC1 =
			PICTURE_SCALING_UNIFORM;

	/* S0, S1 : Underscan / Overscan */
	/* TODO: un-hardcode scan type */
	scan_type = SCANNING_TYPE_UNDERSCAN;
	info_frame.avi_info_packet.info_packet_hdmi.bits.S0_S1 = scan_type;

	/* C0, C1 : Colorimetry */
	if (color_space == COLOR_SPACE_YCBCR709)
		info_frame.avi_info_packet.info_packet_hdmi.bits.C0_C1 =
				COLORIMETRY_ITU709;
	else if (color_space == COLOR_SPACE_YCBCR601)
		info_frame.avi_info_packet.info_packet_hdmi.bits.C0_C1 =
				COLORIMETRY_ITU601;
	else
		info_frame.avi_info_packet.info_packet_hdmi.bits.C0_C1 =
				COLORIMETRY_NO_DATA;

	/* TODO: un-hardcode aspect ratio */
	aspect = stream->public.timing.aspect_ratio;

	switch (aspect) {
	case ASPECT_RATIO_4_3:
	case ASPECT_RATIO_16_9:
		info_frame.avi_info_packet.info_packet_hdmi.bits.M0_M1 = aspect;
		break;

	case ASPECT_RATIO_NO_DATA:
	case ASPECT_RATIO_64_27:
	case ASPECT_RATIO_256_135:
	default:
		info_frame.avi_info_packet.info_packet_hdmi.bits.M0_M1 = 0;
	}

	/* Active Format Aspect ratio - same as Picture Aspect Ratio. */
	info_frame.avi_info_packet.info_packet_hdmi.bits.R0_R3 =
			ACTIVE_FORMAT_ASPECT_RATIO_SAME_AS_PICTURE;

	/* TODO: un-hardcode cn0_cn1 and itc */
	cn0_cn1 = 0;
	itc = false;

	if (itc) {
		info_frame.avi_info_packet.info_packet_hdmi.bits.ITC = 1;
		info_frame.avi_info_packet.info_packet_hdmi.bits.CN0_CN1 =
			cn0_cn1;
	}

	/* TODO: un-hardcode q0_q1 */
	if (color_space == COLOR_SPACE_SRGB)
		info_frame.avi_info_packet.info_packet_hdmi.bits.Q0_Q1 =
						RGB_QUANTIZATION_FULL_RANGE;
	else if (color_space == COLOR_SPACE_SRGB_LIMITED)
		info_frame.avi_info_packet.info_packet_hdmi.bits.Q0_Q1 =
						RGB_QUANTIZATION_LIMITED_RANGE;
	else
		info_frame.avi_info_packet.info_packet_hdmi.bits.Q0_Q1 =
						RGB_QUANTIZATION_DEFAULT_RANGE;

	/* TODO : We should handle YCC quantization,
	 * but we do not have matrix calculation */
	info_frame.avi_info_packet.info_packet_hdmi.bits.YQ0_YQ1 =
					YYC_QUANTIZATION_LIMITED_RANGE;

	info_frame.avi_info_packet.info_packet_hdmi.bits.VIC0_VIC7 =
					stream->public.timing.vic;

	/* pixel repetition
	 * PR0 - PR3 start from 0 whereas pHwPathMode->mode.timing.flags.pixel
	 * repetition start from 1 */
	info_frame.avi_info_packet.info_packet_hdmi.bits.PR0_PR3 = 0;

	/* Bar Info
	 * barTop:    Line Number of End of Top Bar.
	 * barBottom: Line Number of Start of Bottom Bar.
	 * barLeft:   Pixel Number of End of Left Bar.
	 * barRight:  Pixel Number of Start of Right Bar. */
	info_frame.avi_info_packet.info_packet_hdmi.bits.bar_top =
			stream->public.timing.v_border_top;
	info_frame.avi_info_packet.info_packet_hdmi.bits.bar_bottom =
		(stream->public.timing.v_border_top
			- stream->public.timing.v_border_bottom + 1);
	info_frame.avi_info_packet.info_packet_hdmi.bits.bar_left =
			stream->public.timing.h_border_left;
	info_frame.avi_info_packet.info_packet_hdmi.bits.bar_right =
		(stream->public.timing.h_total
			- stream->public.timing.h_border_right + 1);

	/* check_sum - Calculate AFMT_AVI_INFO0 ~ AFMT_AVI_INFO3 */
	check_sum =
		&info_frame.
		avi_info_packet.info_packet_hdmi.packet_raw_data.sb[0];
	*check_sum = INFO_FRAME_AVI + INFO_FRAME_SIZE_AVI
			+ INFO_FRAME_VERSION_2;

	for (byte_index = 1; byte_index <= INFO_FRAME_SIZE_AVI; byte_index++)
		*check_sum += info_frame.avi_info_packet.info_packet_hdmi.
				packet_raw_data.sb[byte_index];

	/* one byte complement */
	*check_sum = (uint8_t) (0x100 - *check_sum);

	/* Store in hw_path_mode */
	info_packet->hb0 =
		info_frame.avi_info_packet.info_packet_hdmi.packet_raw_data.hb0;
	info_packet->hb1 =
		info_frame.avi_info_packet.info_packet_hdmi.packet_raw_data.hb1;
	info_packet->hb2 =
		info_frame.avi_info_packet.info_packet_hdmi.packet_raw_data.hb2;

	for (byte_index = 0; byte_index < sizeof(info_packet->sb); byte_index++)
		info_packet->sb[byte_index] = info_frame.avi_info_packet.
		info_packet_hdmi.packet_raw_data.sb[byte_index];

	info_packet->valid = true;
}

static void set_vendor_info_packet(struct core_stream *stream,
		struct hw_info_packet *info_packet)
{
	uint32_t length = 0;
	bool hdmi_vic_mode = false;
	uint8_t checksum = 0;
	uint32_t i = 0;
	enum dc_timing_3d_format format;

	ASSERT_CRITICAL(stream != NULL);
	ASSERT_CRITICAL(info_packet != NULL);

	format = stream->public.timing.timing_3d_format;

	/* Can be different depending on packet content */
	length = 5;

	if (stream->public.timing.hdmi_vic != 0
			&& stream->public.timing.h_total >= 3840
			&& stream->public.timing.v_total >= 2160)
		hdmi_vic_mode = true;

	/* According to HDMI 1.4a CTS, VSIF should be sent
	 * for both 3D stereo and HDMI VIC modes.
	 * For all other modes, there is no VSIF sent.  */

	if (format == TIMING_3D_FORMAT_NONE && !hdmi_vic_mode)
		return;

	/* 24bit IEEE Registration identifier (0x000c03). LSB first. */
	info_packet->sb[1] = 0x03;
	info_packet->sb[2] = 0x0C;
	info_packet->sb[3] = 0x00;

	/*PB4: 5 lower bytes = 0 (reserved). 3 higher bits = HDMI_Video_Format.
	 * The value for HDMI_Video_Format are:
	 * 0x0 (0b000) - No additional HDMI video format is presented in this
	 * packet
	 * 0x1 (0b001) - Extended resolution format present. 1 byte of HDMI_VIC
	 * parameter follows
	 * 0x2 (0b010) - 3D format indication present. 3D_Structure and
	 * potentially 3D_Ext_Data follows
	 * 0x3..0x7 (0b011..0b111) - reserved for future use */
	if (format != TIMING_3D_FORMAT_NONE)
		info_packet->sb[4] = (2 << 5);
	else if (hdmi_vic_mode)
		info_packet->sb[4] = (1 << 5);

	/* PB5: If PB4 claims 3D timing (HDMI_Video_Format = 0x2):
	 * 4 lower bites = 0 (reserved). 4 higher bits = 3D_Structure.
	 * The value for 3D_Structure are:
	 * 0x0 - Frame Packing
	 * 0x1 - Field Alternative
	 * 0x2 - Line Alternative
	 * 0x3 - Side-by-Side (full)
	 * 0x4 - L + depth
	 * 0x5 - L + depth + graphics + graphics-depth
	 * 0x6 - Top-and-Bottom
	 * 0x7 - Reserved for future use
	 * 0x8 - Side-by-Side (Half)
	 * 0x9..0xE - Reserved for future use
	 * 0xF - Not used */
	switch (format) {
	case TIMING_3D_FORMAT_HW_FRAME_PACKING:
	case TIMING_3D_FORMAT_SW_FRAME_PACKING:
		info_packet->sb[5] = (0x0 << 4);
		break;

	case TIMING_3D_FORMAT_SIDE_BY_SIDE:
	case TIMING_3D_FORMAT_SBS_SW_PACKED:
		info_packet->sb[5] = (0x8 << 4);
		length = 6;
		break;

	case TIMING_3D_FORMAT_TOP_AND_BOTTOM:
	case TIMING_3D_FORMAT_TB_SW_PACKED:
		info_packet->sb[5] = (0x6 << 4);
		break;

	default:
		break;
	}

	/*PB5: If PB4 is set to 0x1 (extended resolution format)
	 * fill PB5 with the correct HDMI VIC code */
	if (hdmi_vic_mode)
		info_packet->sb[5] = stream->public.timing.hdmi_vic;

	/* Header */
	info_packet->hb0 = 0x81; /* VSIF packet type. */
	info_packet->hb1 = 0x01; /* Version */

	/* 4 lower bits = Length, 4 higher bits = 0 (reserved) */
	info_packet->hb2 = (uint8_t) (length);

	/* Calculate checksum */
	checksum = 0;
	checksum += info_packet->hb0;
	checksum += info_packet->hb1;
	checksum += info_packet->hb2;

	for (i = 1; i <= length; i++)
		checksum += info_packet->sb[i];

	info_packet->sb[0] = (uint8_t) (0x100 - checksum);

	info_packet->valid = true;
}

void resource_validate_ctx_destruct(struct validate_context *context)
{
	int i, j;

	for (i = 0; i < context->target_count; i++) {
		for (j = 0; j < context->target_status[i].surface_count; j++)
			dc_surface_release(
				context->target_status[i].surfaces[j]);

		context->target_status[i].surface_count = 0;
		dc_target_release(&context->targets[i]->public);
	}
}

/*
 * Copy src_ctx into dst_ctx and retain all surfaces and targets referenced
 * by the src_ctx
 */
void resource_validate_ctx_copy_construct(
		const struct validate_context *src_ctx,
		struct validate_context *dst_ctx)
{
	int i, j;

	*dst_ctx = *src_ctx;

	for (i = 0; i < dst_ctx->res_ctx.pool.pipe_count; i++) {
		struct pipe_ctx *cur_pipe = &dst_ctx->res_ctx.pipe_ctx[i];

		if (cur_pipe->primary_pipe)
			cur_pipe->primary_pipe =  &dst_ctx->res_ctx.pipe_ctx[cur_pipe->primary_pipe->pipe_idx];

		if (cur_pipe->secondary_pipe)
			cur_pipe->secondary_pipe = &dst_ctx->res_ctx.pipe_ctx[cur_pipe->secondary_pipe->pipe_idx];

	}

	for (i = 0; i < dst_ctx->target_count; i++) {
		dc_target_retain(&dst_ctx->targets[i]->public);
		for (j = 0; j < dst_ctx->target_status[i].surface_count; j++)
			dc_surface_retain(
				dst_ctx->target_status[i].surfaces[j]);
	}
}

struct clock_source *dc_resource_find_first_free_pll(
		struct resource_context *res_ctx)
{
	int i;

	for (i = 0; i < res_ctx->pool.clk_src_count; ++i) {
		if (res_ctx->clock_source_ref_count[i] == 0)
			return res_ctx->pool.clock_sources[i];
	}

	return NULL;
}

void resource_build_info_frame(struct pipe_ctx *pipe_ctx)
{
	enum signal_type signal = SIGNAL_TYPE_NONE;
	struct hw_info_frame info_frame = { { 0 } };

	/* default all packets to invalid */
	info_frame.avi_info_packet.valid = false;
	info_frame.gamut_packet.valid = false;
	info_frame.vendor_info_packet.valid = false;
	info_frame.spd_packet.valid = false;
	info_frame.vsc_packet.valid = false;

	signal = pipe_ctx->stream->signal;

	/* HDMi and DP have different info packets*/
	if (signal == SIGNAL_TYPE_HDMI_TYPE_A) {
		set_avi_info_frame(
			&info_frame.avi_info_packet, pipe_ctx);
		set_vendor_info_packet(
			pipe_ctx->stream, &info_frame.vendor_info_packet);
	}

	translate_info_frame(&info_frame,
			&pipe_ctx->encoder_info_frame);
}

enum dc_status resource_map_clock_resources(
		const struct core_dc *dc,
		struct validate_context *context)
{
	int i, j, k;

	/* acquire new resources */
	for (i = 0; i < context->target_count; i++) {
		struct core_target *target = context->targets[i];

		for (j = 0; j < target->public.stream_count; j++) {
			struct core_stream *stream =
				DC_STREAM_TO_CORE(target->public.streams[j]);

			if (resource_is_stream_unchanged(dc->current_context, stream))
				continue;

			for (k = 0; k < MAX_PIPES; k++) {
				struct pipe_ctx *pipe_ctx =
					&context->res_ctx.pipe_ctx[k];

				if (context->res_ctx.pipe_ctx[k].stream != stream)
					continue;

				if (dc_is_dp_signal(pipe_ctx->stream->signal)
					|| pipe_ctx->stream->signal == SIGNAL_TYPE_VIRTUAL)
					pipe_ctx->clock_source =
						context->res_ctx.pool.dp_clock_source;
				else {
					pipe_ctx->clock_source =
						resource_find_used_clk_src_for_sharing(
							&context->res_ctx,
							pipe_ctx);

					if (pipe_ctx->clock_source == NULL)
						pipe_ctx->clock_source =
							dc_resource_find_first_free_pll(&context->res_ctx);
				}

				if (pipe_ctx->clock_source == NULL)
					return DC_NO_CLOCK_SOURCE_RESOURCE;

				resource_reference_clock_source(
						&context->res_ctx,
						pipe_ctx->clock_source);

				/* only one cs per stream regardless of mpo */
				break;
			}
		}
	}

	return DC_OK;
}

static bool check_timing_change(struct core_stream *cur_stream,
		struct core_stream *new_stream)
{
	if (cur_stream == NULL)
		return true;

	/* If sink pointer changed, it means this is a hotplug, we should do
	 * full hw setting.
	 */
	if (cur_stream->sink != new_stream->sink)
		return true;

	return memcmp(
		&cur_stream->public.timing,
		&new_stream->public.timing,
		sizeof(struct dc_crtc_timing)) != 0;
}

/*
 * Note: We need to disable output if clock sources change,
 * since bios does optimization and doesn't apply if changing
 * PHY when not already disabled.
 */
bool pipe_need_reprogram(
		struct pipe_ctx *pipe_ctx_old,
		struct pipe_ctx *pipe_ctx)
{
	if (pipe_ctx_old->stream->sink != pipe_ctx->stream->sink)
		return true;

	if (pipe_ctx_old->stream->signal != pipe_ctx->stream->signal)
		return true;

	if (pipe_ctx_old->audio != pipe_ctx->audio)
		return true;

	if (pipe_ctx_old->clock_source != pipe_ctx->clock_source
			&& pipe_ctx_old->stream != pipe_ctx->stream)
		return true;

	if (pipe_ctx_old->stream_enc != pipe_ctx->stream_enc)
		return true;

	if (check_timing_change(pipe_ctx_old->stream, pipe_ctx->stream))
		return true;


	return false;
}
